% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/kriging.R
\name{Limit.Kriging}
\alias{Limit.Kriging}
\title{Limit Kriging}
\usage{
Limit.Kriging(
  X,
  y,
  fit = TRUE,
  kernel = NULL,
  kernel.parameters = list(),
  nlopt.parameters = list()
)
}
\arguments{
\item{X}{a matrix for input (feature)}

\item{y}{a vector for output (target), only one-dimensional output is supported}

\item{fit}{whether to fit the length scale parameters from data}

\item{kernel}{a kernel class object}

\item{kernel.parameters}{a list of parameters required for the kernel, if no kernel class object is provided}

\item{nlopt.parameters}{a list of parameters required for NLopt, including choice of optimization algorithm and maximum number of evaluation}
}
\value{
A Limit Kriging Class Object.
}
\description{
This functions fits the limit kriging model to the data.
}
\details{
Limit kriging avoids the mean reversion issue of ordinary kriging. Please see Joseph (2006) for details.
Only interpolation is available. Noisy output is not supported for limit kriging.  

The kernel choices are required and can be specified by 
(i) providing the kernel class object to \code{kernel}
or (ii) specifying the kernel type and other parameters in \code{kernel.parameters}. 
Please see examples section of \link{Fit.Kriging} for detail usages. 

When the lengthscale / correlation parameters are unknown, 
all parameters including the constant mean can be estimated via Maximum Likelihood method by setting \code{fit=TRUE}. 
The initial / lower bound / upper bound of the lengthscale parameters can be provided in \code{kernel.parameters}, 
otherwise a good initial and range would be estimated from the data. 
The optimization is performed via \href{https://nlopt.readthedocs.io/en/latest/}{NLopt}, 
a open-source library for nonlinear optimization. 
All gradient-free optimization methods in \href{https://nlopt.readthedocs.io/en/latest/}{NLopt} 
are supported and can be specified in \code{nlopt.parameters}.
See \code{nloptr::nloptr.print.options()} for the list of available derivative-free algorithms (prefix with NLOPT_GN or NLOPT_LN).  
The maximum number of optimization steps can also be defined in \code{nlopt.parameters}.
Please see examples section of \link{Fit.Kriging} for detail usages.
}
\examples{
# one dimensional example 
f <- function(x) {
  x <- 0.5 + 2*x
  y <- sin(10*pi*x)/(2*x) + (x-1)^4
  return (y)
}

set.seed(1234)
# train set
n <- 30
p <- 1
X <- matrix(runif(n),ncol=p)
y <- apply(X, 1, f)
newX <- matrix(seq(0,1,length=1001), ncol=p)

# approach 1
kriging <- Limit.Kriging(X, y, fit=TRUE, kernel.parameters=list(type="Gaussian"))
pred <- Predict.Kriging(kriging, newX)
plot(newX, f(newX), "l")
points(X, y, pch=16, col="blue")
lines(newX, pred$mean, col="red", lty=2)
lines(newX, pred$mean-2*pred$sd, col="red", lty=3)
lines(newX, pred$mean+2*pred$sd, col="red", lty=3)
Get.Kriging.Parameters(kriging)

# approach 2
kriging <- Fit.Kriging(X, y, interpolation=TRUE, fit=TRUE, model="LK",
                       kernel.parameters=list(type="Gaussian"))
pred <- Predict.Kriging(kriging, newX)
plot(newX, f(newX), "l")
points(X, y, pch=16, col="blue")
lines(newX, pred$mean, col="red", lty=2)
lines(newX, pred$mean-2*pred$sd, col="red", lty=3)
lines(newX, pred$mean+2*pred$sd, col="red", lty=3)
Get.Kriging.Parameters(kriging)

}
\references{
Joseph, V. R. (2006). \emph{Limit kriging}. Technometrics, 48(4), 458-466.
}
\seealso{
\link{Fit.Kriging}, \link{Predict.Kriging}, \link{Get.Kriging.Parameters}.
}
\author{
Chaofan Huang and V. Roshan Joseph
}
