% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/RcppExports.R
\name{zigzag_logit_ss}
\alias{zigzag_logit_ss}
\title{zigzag_logit_ss}
\usage{
zigzag_logit_ss(
  maxTime,
  dataX,
  datay,
  prior_sigma2,
  x0,
  theta0,
  cvref,
  rj_val = 0.6,
  ppi = 0.5,
  nmax = 1000000L,
  burn = -1L
)
}
\arguments{
\item{maxTime}{Maximum runtime (in Seconds) of the algorithm; will terminate the code after a given computation time or nmax iterations of the algorithm is reached.}

\item{dataX}{Matrix of all covariates where the i-th row corresponds to all p covariates x_{i,1}, ..., x_{i,p} of the i-th observation.}

\item{datay}{Vector of n observations of a {0, 1}-valued variable y.}

\item{prior_sigma2}{Double for the prior variance for included variables.}

\item{x0}{Initial position of the regression parameter}

\item{theta0}{Initial velocity for the sampler (Default has 1s on all components). This should be chosen with unit velocities on each component (regardless of sign).}

\item{cvref}{Control variate vector of dimension p for subsampling. If no control variate set to a vector of zeros.}

\item{rj_val}{Reversible jump parameter for the PDMP method. This value is fixed over all models and is interpreted as the probability to jump to a reduced model when a parameter hits zero.}

\item{ppi}{Double for the prior probability of inclusion (ppi) for each parameter.}

\item{nmax}{Maximum number of iterations (simulated events) of the algorithm; will stop the algorithm when this number of iterations of the method have occured. Default value is 1e6, lower values should be chosen for memory constraints if less iterations are desired.}

\item{burn}{Optional number of iterations to use for burnin. These are not stored so can be useful in memory intensive problems.}
}
\value{
Returns a list with the following objects:

\code{times}: Vector of event times where ZigZag process switchs velocity or jumps models.

\code{positions}: Matrix of positions at which event times occur, these are not samples from the PDMP.

\code{theta}: Matrix of new velocities at event times.
}
\description{
Applies the Reversible Jump ZigZag Sampler with subsampling to a Logistic regression with dirac spike and slab distribution, as detailed in Reversible Jump PDMP Samplers for Variable Selection, 2020.
For included variables an independent Gaussian prior is assumed with variance \code{prior_sigma2} and mean zero, variables are given prior probabilities of inclusion \code{ppi}.
}
\examples{

generate.logistic.data <- function(beta, n.obs, Sig) {
p <- length(beta)
dataX <- MASS::mvrnorm(n=n.obs,mu=rep(0,p),Sigma=Sig)
vals <- dataX \%*\% as.vector(beta)
generateY <- function(p) { rbinom(1, 1, p)}
dataY <- sapply(1/(1 + exp(-vals)), generateY)
return(list(dataX = dataX, dataY = dataY))
}

n <- 15
p <- 25
beta <- c(1, rep(0, p-1))
Siginv <- diag(1,p,p)
Siginv[1,2] <- Siginv[2,1] <- 0.9
set.seed(1)
data <- generate.logistic.data(beta, n, solve(Siginv))
ppi <- 2/p

\dontrun{
zigzag_fit <- zigzag_logit(maxTime = 1, dataX = data$dataX,
                           datay = data$dataY, prior_sigma2 = 10,
                           theta0 = rep(0, p), x0 = rep(0, p), rj_val = 0.6,
                           ppi = ppi)

zigzag_fit_s <- zigzag_logit_ss(maxTime = 1, dataX = data$dataX,
                                datay = data$dataY,prior_sigma2 = 10,
                                theta0 = rep(0, p), x0 = rep(0, p),
                                rj_val = 0.6, cvref = c(1,rep(0,p-1)),
                                ppi = ppi)

gibbs_fit <- gibbs_logit(maxTime = 1, dataX = data$dataX, datay =data$dataY,
                         prior_sigma2 = 10,beta = rep(0,p), gamma =rep(0,p),
                         ppi = ppi)

plot_pdmp_multiple(list(zigzag_fit,zigzag_fit_s), coords = 1:2, burn = .1,
                   inds = 1:1e2, nsamples = 1e4,
                   mcmc_samples = t(gibbs_fit$beta*gibbs_fit$gamma))
}
}
