% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/ee_download.R
\name{ee_image_to_asset}
\alias{ee_image_to_asset}
\title{Creates a task to export an EE Image to their EE Assets.}
\usage{
ee_image_to_asset(
  image,
  description = "myExportImageTask",
  assetId = NULL,
  overwrite = FALSE,
  pyramidingPolicy = NULL,
  dimensions = NULL,
  region = NULL,
  scale = NULL,
  crs = NULL,
  crsTransform = NULL,
  maxPixels = NULL
)
}
\arguments{
\item{image}{The image to be exported.}

\item{description}{Human-readable name of the task.}

\item{assetId}{The destination asset ID.}

\item{overwrite}{Logical. If TRUE, the assetId will be overwritten if
it exists.}

\item{pyramidingPolicy}{The pyramiding policy to apply to each band
in the image, a dictionary keyed by band name. Values must be one
of: "mean", "sample", "min", "max", or "mode". Defaults to "mean".
A special key, ".default", may be used to change the default for all bands.}

\item{dimensions}{The dimensions of the exported image. It takes either a
single positive integer as the maximum dimension or "WIDTHxHEIGHT" where
WIDTH and HEIGHT are each positive integers.}

\item{region}{The lon,lat coordinates for a LinearRing or Polygon
specifying the region to export. It can be specified as nested lists
of numbers or a serialized string. Defaults to the image's region.}

\item{scale}{The resolution in meters per pixel. Defaults to the native
resolution of the image asset unless a crsTransform is specified.}

\item{crs}{The coordinate reference system of the exported image's
projection. Defaults to the image's default projection.}

\item{crsTransform}{A comma-separated string of 6 numbers describing
the affine transform of the coordinate reference system of the exported
image's projection, in the order:
xScale, xShearing, xTranslation, yShearing, yScale, and yTranslation.
Defaults to the image's native CRS transform.}

\item{maxPixels}{The maximum allowed number of pixels in the exported
image. The task will fail if the exported region covers more pixels
in the specified projection. Defaults to 100,000,000. **kwargs: Holds
other keyword arguments that may have been deprecated, such
as 'crs_transform'.}
}
\value{
An unstarted task
}
\description{
Creates a task to export an EE Image to their EE Assets.
This function is a wrapper around \code{ee$batch$Export$image$toAsset(...)}.
}
\examples{
\dontrun{
library(rgee)
library(stars)
library(sf)

ee_users()
ee_Initialize()

# Define study area (local -> earth engine)
# Communal Reserve Amarakaeri - Peru
rlist <- list(xmin = -71.13, xmax = -70.95,ymin = -12.89, ymax = -12.73)
ROI <- c(rlist$xmin, rlist$ymin,
         rlist$xmax, rlist$ymin,
         rlist$xmax, rlist$ymax,
         rlist$xmin, rlist$ymax,
         rlist$xmin, rlist$ymin)
ee_ROI <- matrix(ROI, ncol = 2, byrow = TRUE) \%>\%
  list() \%>\%
  st_polygon() \%>\%
  st_sfc() \%>\%
  st_set_crs(4326) \%>\%
  sf_as_ee()


# Get the mean annual NDVI for 2011
cloudMaskL457 <- function(image) {
  qa <- image$select("pixel_qa")
  cloud <- qa$bitwiseAnd(32L)$
    And(qa$bitwiseAnd(128L))$
    Or(qa$bitwiseAnd(8L))
  mask2 <- image$mask()$reduce(ee$Reducer$min())
  image <- image$updateMask(cloud$Not())$updateMask(mask2)
  image$normalizedDifference(list("B4", "B3"))
}

ic_l5 <- ee$ImageCollection("LANDSAT/LT05/C01/T1_SR")$
  filterBounds(ee$FeatureCollection(ee_ROI))$
  filterDate("2011-01-01", "2011-12-31")$
  map(cloudMaskL457)

# Create simple composite
mean_l5 <- ic_l5$mean()$rename("NDVI")
mean_l5 <- mean_l5$reproject(crs = "EPSG:4326", scale = 500)
mean_l5_Amarakaeri <- mean_l5$clip(ee_ROI)

# Move results from Earth Engine to Drive
assetid <- paste0(ee_get_assethome(), '/l5_Amarakaeri')
task_img <- ee_image_to_asset(
  image = mean_l5_Amarakaeri,
  assetId = assetid,
  overwrite = TRUE,
  scale = 500,
  region = ee_ROI
)

task_img$start()
ee_monitoring(task_img)

ee_l5 <- ee$Image(assetid)
Map$centerObject(ee_l5)
Map$addLayer(ee_l5)
}
}
\seealso{
Other image export task creator: 
\code{\link{ee_image_to_drive}()},
\code{\link{ee_image_to_gcs}()}
}
\concept{image export task creator}
