#' @export profileCombinations
#' @title Compute drug combination response signatures
#' @description  Returns the average of every possible two response profile non-redundant combination from the input matrix
#' @param responseProfiles A matrix of response profiles for drugs (rows are genes, columns are drugs)
#' @return A matrix with genes in the rows and profiles combination in the columns. The values represent the expected transcriptional response of the cells (used as input of retriever) to a drug combination.
#' @examples
#' # Loading robust profiles across different breast cancer cell lines generated by retriever.
#' # BRCA <- retriever(cellLines = c('MDAMB231', 'MCF7', 'SKBR3', 'HS578T', 'BT20'))
#'
#'   BRCA <- read.csv(system.file("BRCA.csv",package="retriever"), row.names = 1)
#'   dim(BRCA)
#'   # [1] 1001  125
#'
#' # Computing response profiles combinations
#'   combinationsBRCA <- profileCombinations(BRCA)
#'   dim(combinationsBRCA)
#'   # [1] 1001 7750

profileCombinations <- function(responseProfiles){
  pCombination <- t(utils::combn(colnames(responseProfiles),2))
  combinationNames <- apply(pCombination,1,function(X){paste0(X,collapse = ' + ')})
  combinationProfiles <- pbapply::pbapply(pCombination,1,function(X){rowMeans(responseProfiles[,X])})
  colnames(combinationProfiles) <- combinationNames
  return(combinationProfiles)
}
