% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/createRedcapProject.R
\name{createRedcapProject}
\alias{createRedcapProject}
\alias{createRedcapProject.redcapApiConnection}
\title{Create REDCap Project}
\usage{
createRedcapProject(
  rcon,
  project_title,
  purpose = REDCAP_PROJECT_PURPOSE,
  purpose_other = NULL,
  is_longitudinal = FALSE,
  surveys_enabled = FALSE,
  record_autonumbering_enabled = FALSE,
  xml = NULL,
  ...
)

\method{createRedcapProject}{redcapApiConnection}(
  rcon,
  project_title,
  purpose = REDCAP_PROJECT_PURPOSE,
  purpose_other = NULL,
  is_longitudinal = FALSE,
  surveys_enabled = FALSE,
  record_autonumbering_enabled = FALSE,
  xml = NULL,
  ...
)
}
\arguments{
\item{rcon}{A \code{redcapConnection} object.}

\item{project_title}{\code{character(1)}. Title for the new project.}

\item{purpose}{\code{character}, one of
\code{c("Practice/just for fun", "Other", "Research", "Quality Improvement", "Operational Support")}}

\item{purpose_other}{\code{character(1)} or \code{NULL}. Ignored unless
\code{purpose = "Other"}, in which case this becomes a required argument.}

\item{is_longitudinal}{\code{logical(1)}. When \code{TRUE} the project will
be set as a longitudinal project.}

\item{surveys_enabled}{\code{logical(1)}. When \code{TRUE} surveys are enabled
for the project. (This will not add any survey instruments, only
enable them).}

\item{record_autonumbering_enabled}{\code{logical(1)}. When \code{TRUE} if
auto numbering will be enabled in the project.}

\item{xml}{\code{character(1)} or \code{NULL} an XML string in CDISC ODM XML format that
contains project metadata (fields, forms, events, arms) and might
optionally contain data to be imported as well. When not \code{NULL},
all other arguments are ignored. See Details.}

\item{...}{Arguments to pass to other methods}
}
\value{
Returns a \code{character(1)} the 32-character, project level API token
assigned to the user that created the project. This token is intended
to be used for further project configuration using the API.
}
\description{
These methods enable a user with a 64-character Super API
token to create a new REDCap project.
}
\details{
The user creating the project will automatically be added to the project as a
user with full user privileges and a project-level API token,
which could then be used for subsequent project-level API requests.

When the project is created, it will automatically be given all the
project-level defaults just as if it had been created via the web user
interface, such as automatically creating a single data collection
instrument seeded with a single Record ID field and Form Status field,
as well as (for longitudinal projects) one arm with one event.

If the user intends to populate the project with arms and events
immediately after creating the project, it is recommended that
\code{override = TRUE} be used in \code{importArms} and \code{importEvents} so that the
default arm and event are removed.

The \code{xml} argument must be in CDISC ODM XML format. It may come from a
REDCap Project XML export  file from REDCap itself
(see \code{\link[=exportProjectXml]{exportProjectXml()}}), or may come from another system that is capable of
exporting projects and data in CDISC ODM format. If the \code{xml} argument
is used in the API request, it will use the XML to import its
contents into the newly created project. This will not only
create the project with the API request, but also to import all fields,
forms, and project attributes (and events and arms, if longitudinal) as
well as record data all at the same time.

Only users with a 64-character Super API Tokens can utilize this method
(the standard API token is 32 characters). Users can only be granted a
super token by a REDCap administrator (using the API Tokens page in the
REDCap Control Center). Please be advised that users with a Super API
Token can create new REDCap projects via the API without any approval
needed by a REDCap administrator.
}
\examples{
\dontrun{
# The token must be a 64-character token
super_token <- redcapConnection(url = "your_redcap_url", 
                                token = "[64-character-super-api-token]")
         
# Create a new project    
createRedcapProject(super_token, 
                    project_title = "New Project Name", 
                    purpose = "Quality Improvement", 
                    is_longitudinal = FALSE, 
                    surveys_enabled = TRUE)
                    
                    
                    
# Copy an existing project into a new project
unlockREDCap(connections = c(rcon = "token_alias"), 
             url = "your_redcap_url", 
             keyring = "API_KEYs", 
             envir = globalenv())
             
xml_file <- tempfile(file.ext = ".xml")
exportProjectXml(rcon, 
                 file = xml_file)
 
xml_text <- paste0(readLines(xml_file), collapse = " ")
createRedcapProject(super_token, 
                    xml = xml_text)
}

}
\seealso{
\code{\link[=exportProjectXml]{exportProjectXml()}}
}
