% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/IPWE_MADopt.R
\name{IPWE_MADopt}
\alias{IPWE_MADopt}
\title{Estimation of the Optimal Treatment Regime defined as Minimizing
Gini's Mean Differences}
\usage{
IPWE_MADopt(data, regimeClass, moPropen = "BinaryRandom", s.tol, it.num = 8,
  hard_limit = FALSE, cl.setup = 1, p_level = 1, pop.size = 3000)
}
\arguments{
\item{data}{a data frame, containing variables in the \code{moPropen} and \code{RegimeClass} and 
a component \code{y} as the response.}

\item{regimeClass}{a formula specifying the class of treatment regimes to search,
e.g. if \code{regimeClass = a~x1+x2}, and then this function will search the class of treatment regimes
of the form 
     \deqn{d(x)=I\left(\beta_0 +\beta_1  x_1 + \beta_2  x_2 > 0\right).
       }{d(x)=I(\beta_0 +\beta_1 * x1  + \beta_2 * x2 > 0).}
       Polynomial arguments are also supported.
         See also 'Details'.}

\item{moPropen}{The propensity score model for the probability of receiving 
treatment level 1.
When \code{moPropen} equals the string "BinaryRandom",  the proportion of observations
receiving treatment level 1 in the sample will be employed
as a good estimate of the probability for each observation.
Otherwise, this argument should be a formula/string, based on which this function
will fit a logistic regression on the treatment level.  e.g. \code{a1~x1}.}

\item{s.tol}{This is the tolerance level used by \code{genoud}. 
Default is \eqn{10^{-5}} times the difference between
the largest and the smallest value in the observed responses.
This is particularly important when it comes to evaluating \code{it.num}.}

\item{it.num}{integer > 1. This argument will be used in \code{rgeound::geound} function.
    If there is no improvement in the objective function in this number of generations,
\code{rgenoud::genoud} will think that it has found the optimum.}

\item{hard_limit}{logical. When it is true the maximum number of generations
         in  \code{rgeound::geound} cannot exceed 100. Otherwise, in this function, only
\code{it.num} softly controls when \code{genoud} stops. Default is \code{FALSE}.}

\item{cl.setup}{the number of nodes. >1 indicates choosing parallel computing option in 
\code{rgenoud::genoud}. Default is 1.}

\item{p_level}{choose between 0,1,2,3 to indicate different levels of output
from the genetic function. Specifically, 0 (minimal printing),
  1 (normal), 2 (detailed), and 3 (debug.)}

\item{pop.size}{an integer with the default set to be 3000. This is the population number for the first generation
in the genetic algorithm (\code{rgenoud::genoud}).}
}
\value{
This function returns an object with 6 objects. Both \code{coefficients}
and \code{coef.orgn.scale}  were normalized to have unit euclidean norm.
\describe{
 \item{\code{coefficients}}{the parameters indexing the estimated 
 MAD-optimal treatment regime for 
 standardized covariates.}
 \item{\code{coef.orgn.scale}}{the parameter indexing the estimated 
 MAD-optimal treatment regime for the original input covariates.}
 \item{\code{hat_MAD}}{the estimated MAD when a treatment regime indexed by
        \code{coef.orgn.scale} is applied on everyone. See the 'details' for
         connection between \code{coef.orgn.scale} and
          \code{coefficient}.}
 \item{\code{call}}{the user's call.}
 \item{\code{moPropen}}{the user specified propensity score model}
 \item{\code{regimeClass}}{the user specified class of treatment regimes}
 }
}
\description{
\code{IPWE_MADopt} seeks to estimated the treatment regime which \bold{minimizes}
 the Gini's Mean difference defined below.

Besides mean and quantile criterion, in some applications
people seek minimization of dispersion in the outcome, which, for example, can
be described by Gini's mean difference. Formally, it is defined as the absolute
differences of two random variables \eqn{Y_1} and \eqn{Y_2} drawn independently
from the same distribution: \deqn{MAD:= E(|Y_1-Y_2|).}

Given a treatment regime \eqn{d}, define the potential outcome of a subject
following the treatment recommended by \code{d} as
\eqn{Y^{*}(d)}. When \eqn{d} is followed by everyone in the target population,
 the Gini's mean absolute difference is
 \deqn{MAD(d):= E(| Y_1^{*}(d)-Y_2^{*}(d) |).}
}
\details{
Note that all estimation functions in this package use the same type
of standardization on covariates. Doing so would allow us to provide a bounded 
domain of parameters for searching in the genetic algorithm.
 
This estimated parameters indexing the MAD-optimal treatment regime are returned 
\emph{in two scales:}
\enumerate{
   \item The returned \code{coefficients} is the set of parameters after covariates \eqn{X} 
   are standardized to be in the interval [0, 1]. To be exact, every covariate is 
   subtracted by the smallest observed value and divided by the difference between 
   the largest and the smallest value.  Next, we carried out the algorithm in Wang 2016 to get the estimated
   regime parameters, \code{coefficients}, based on the standardized data. 
   For the identifiability issue, we force the Euclidean norm of \code{coefficients}
   to be 1.

   \item In contrast, \code{coef.orgn.scale} corresponds to the original covariates,
    so the associated decision rule can be applied directly to novel observations. 
    In other words, let \eqn{\beta} denote the estimated parameter in the original 
   scale, then the estimated treatment regime is:  
       \deqn{ d(x)= I\{\hat{\beta}_0 + \hat{\beta}_1 x_1 + ... + \hat{\beta}_k x_k > 0\}.}{
        d(x)= I{\beta_0 + \beta_1*x_1 + ... + \beta_k*x_k > 0}.}
   The estimated \eqn{\bm{\hat{\beta}}}{\beta} is returned as \code{coef.orgn.scale}.
   The same as \code{coefficients}, we force the Euclidean norm of \code{coef.orgn.scale}
   to be 1.
}
    If, for every input covariate, the smallest observed value is exactly 0 and the range 
   (i.e. the largest number minus the smallest number) is exactly 1, then the estimated 
   \code{coefficients} and \code{coef.orgn.scale} will render identical.
}
\examples{
GenerateData.MAD <- function(n)
{
  x1 <- runif(n)
  x2 <- runif(n)
  tp <- exp(-1+1*(x1+x2))/(1+exp(-1+1*(x1+x2)))
  a<-rbinom(n = n, size = 1, prob=tp)
  error <- rnorm(length(x1))
  y <- (1 + a*0.6*(-1+x1+x2<0) +  a*-0.6*(-1+x1+x2>0)) * error
  return(data.frame(x1=x1,x2=x2,a=a,y=y))
}
# The true MAD optimal treatment regime for this generative model
# can be deduced trivially, and it is:  c( -0.5773503,  0.5773503,  0.5773503).
\dontshow{
  set.seed(1103)
  testData <- GenerateData.MAD(30)
  fit0 <- IPWE_MADopt(data = testData, regimeClass = a~x1+x2,
             moPropen=a~x1+x2, s.tol=0.2,
             pop.size=300, it.num=2)
}

# With correctly specified propensity model   ####
\donttest{
n <- 400
testData <- GenerateData.MAD(n)
fit1 <- IPWE_MADopt(data = testData, regimeClass = a~x1+x2,
                    moPropen=a~x1+x2, cl.setup=2)
fit1
}
             


# With incorrectly specified propensity model ####
\donttest{
fit2 <- IPWE_MADopt(data = testData, regimeClass = a~x1+x2,
                    moPropen="BinaryRandom", cl.setup=2)
fit2
}

}
\references{
\insertRef{wang2016quant}{quantoptr}
}
