#' Read a CSV file exported from Qualtrics
#'
#' Reads comma separated CSV files generated by Qualtrics
#' software. The second line containing the variable labels is imported.
#' Repetitive introductions to matrix questions are automatically removed.
#' Variable labels are stored as attributes.
#' @param ... All arguments for \code{\link[qualtRics]{read_survey}}
#'
#' @export
readSurvey <- function(...) {
  read_survey(...)
}


#' Read a CSV file exported from Qualtrics
#'
#' Reads comma separated CSV files generated by Qualtrics
#' software. The second line containing the variable labels is imported.
#' Repetitive introductions to matrix questions are automatically removed.
#' Variable labels are stored as attributes.
#'
#' @param file_name String. A CSV data file.
#' @param strip_html Logical. If TRUE, then remove HTML tags. Defaults to TRUE.
#' @param legacy Logical. If TRUE, then import "legacy" format CSV files
#' (as of 2017). Defaults to FALSE.
#'
#' @importFrom utils read.csv
#' @importFrom sjlabelled set_label
#' @importFrom jsonlite fromJSON
#' @importFrom stringr str_match
#' @importFrom readr read_csv
#' @return A data frame. Variable labels are stored as attributes. They are not
#' printed on the console but are visibile in the RStudio viewer.
#' @export
#' @examples
#' \dontrun{
#' # Generic use of read_survey()
#' df <- read_survey("<YOUR-PATH-TO-CSV-FILE>")
#' }
#' # Example using current data format
#' file <- system.file("extdata", "sample.csv", package = "qualtRics")
#' df <- read_survey(file)
#'
#' # Example using legacy data format
#' file <- system.file("extdata", "sample_legacy.csv", package = "qualtRics")
#' df <- read_survey(file, legacy = TRUE)
#'
#'
read_survey <- function(file_name,
                        strip_html = TRUE,
                        legacy = FALSE) {

  # START UP: CHECK ARGUMENTS PASSED BY USER ----

  # check if file exists
  assert_surveyFile_exists(file_name)
  # skip 2 rows if legacy format, else 3 when loading the data
  skipNr <- ifelse(legacy, 2, 3)

  # READ DATA ----

  # import data including variable names (row 1) and variable labels (row 2)
  rawdata <- suppressMessages(readr::read_csv(
    file = file_name,
    col_names = FALSE,
    skip = skipNr,
    na = c("")
  ))
  # Need contingency when 0 rows
  assertthat::assert_that(nrow(rawdata) > 0,
    msg = "The survey you are trying to import has no responses."
  ) # nolint
  # Load headers
  header <- suppressMessages(readr::read_csv(
    file = file_name,
    col_names = TRUE,
    n_max = 1
  ))

  # MANIPULATE DATA ----

  # make them data.frame's, else the factor conversion
  # in `infer_data_types` crashes
  # rawdata <- as.data.frame(rawdata)
  # header <- as.data.frame(header)
  # Add names
  names(rawdata) <- names(header)

  # If Qualtrics adds an empty column at the end, remove it
  if (grepl(",$", readLines(file_name, n = 1))) {
    header <- header[, 1:(ncol(header) - 1)]
    rawdata <- rawdata[, 1:(ncol(rawdata) - 1)]
  }
  # extract second row, remove it from df
  secondrow <- unlist(header)
  row.names(rawdata) <- NULL

  # Clean variable labels
  if (strip_html) {
    # weird regex to strip HTML tags, leaving only content
    # https://www.r-bloggers.com/htmltotext-extracting-text-from-html-via-xpath/ # nolint
    pattern <- "</?\\w+((\\s+\\w+(\\s*=\\s*(?:\".*?\"|'.*?'|[^'\">\\s]+))?)+\\s*|\\s*)/?>" # nolint
    secondrow <- gsub(pattern, "\\4", secondrow)
  }

  # Scale Question with subquestion:
  # If it matches one of ".?!" followed by "-", take subsequent part
  subquestions <- stringr::str_match(secondrow, ".*[:punct:]\\s*-(.*)")[, 2]

  # Else if subquestion returns NA, use whole string
  subquestions[is.na(subquestions)] <- unlist(secondrow[is.na(subquestions)])

  # Remaining NAs default to 'empty string'
  subquestions[is.na(subquestions)] <- ""

  # Add labels to data
  rawdata <- sjlabelled::set_label(rawdata, unlist(subquestions))

  # RETURN ----

  return(rawdata)
}
