% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/two_sided.R
\name{get_level_from_bounds_two_sided}
\alias{get_level_from_bounds_two_sided}
\title{Calculates Global Significance Level From Simultaneous Two-Sided Bounds for Rejection Region}
\usage{
get_level_from_bounds_two_sided(lower_bounds, upper_bounds)
}
\arguments{
\item{lower_bounds}{Numeric vector where the ith component is the lower bound for the acceptance interval
for the ith order statistic. The components must lie in [0, 1], and each component must be greater than
or equal to the previous one.}

\item{upper_bounds}{Numeric vector of the same length as \code{lower_bounds} where the ith component is the upper bound
for the acceptance interval for the ith order statistic. The components must lie in [0, 1], and each component must be
greater than or equal to the previous one. In addition,
the ith component of \code{upper_bounds} must be greater than or equal to the ith component of \code{lower_bounds}.}
}
\value{
Global Significance Level \eqn{\alpha}
}
\description{
For a test of uniformity of i.i.d. observations on the unit interval, this function will determine the significance
level as a function of the rejection region. Suppose \eqn{n} observations are drawn i.i.d. from some CDF F(x) on the unit interval,
and it is desired to test the null hypothesis that F(x) = x for all x in (0, 1) against a two-sided alternative.
Suppose the acceptance region for the test is described by a set of intervals, one for each order statistic.
Given the bounds for these intervals, this function calculates the significance level of the test where the
null hypothesis is rejected if at least one of the order statistics is outside its corresponding interval.
}
\details{
Uses the method of Moscovich and Nadler (2016) "Fast calculation of boundary crossing probabilities
for Poisson processes."
}
\examples{
# For X1, X2 iid unif(0,1), calculate 1 - P(.1 < min(X1, X2) < .6 and .5 < max(X1, X2) < .9)
get_level_from_bounds_two_sided(lower_bounds = c(.1, .5), upper_bounds = c(.6, .9))

# Finds the global significance level corresponding to the local level eta.
# Suppose we reject the null hypothesis that X1, ..., Xn are iid unif(0, 1) if and only if at least
# one of the order statistics X(i) is significantly different from
# its null distribution based on a level-eta
# two-sided test, i.e. we reject if and only if X(i) is outside the interval
# (qbeta(eta/2, i, n - i + 1), qbeta(1 - eta/2, i, n - i + 1)) for at least one i.
# The lines of code below calculate the global significance level of
# the test (which is necessarily larger than eta if n > 1).
n <- 100
eta <- .05
lb <- qbeta(eta / 2, c(1:n), c(n:1))
ub <- qbeta(1 - eta / 2, c(1:n), c(n:1))
get_level_from_bounds_two_sided(lower_bounds = lb, upper_bounds = ub)

}
\references{
\itemize{
\item{\href{https://www.sciencedirect.com/science/article/abs/pii/S0167715216302802}{
Moscovich, Amit, and Boaz Nadler. "Fast calculation of boundary crossing probabilities for Poisson processes."
Statistics & Probability Letters 123 (2017): 177-182.}}
}
}
