\name{ratiobatch}
\alias{ratiobatch}

\title{Calculation of ratios in a batch format for multiple genes/samples}

\description{For multiple qPCR data from type 'pcrbatch', this function calculates ratios between samples, using normalization against one or more reference gene(s), if supplied. By default, multiple reference genes are averaged according to Vandesompele \emph{et al}. (2002). The input can be single qPCR data or (more likely) data containing replicates. This is essentially a version of \code{\link{ratiocalc}} that can handle multiple reference genes and genes-of-interest with multiple (replicated) samples as found in large-scale qPCR runs such as 96- or 384-Well plates. The results are automatically stored as a file or copied into the clipboard. A boxplot representation for all Monte-Carlo simulations, permutations and error propagations including 95\% confidence intervals is also given.
}

\usage{
ratiobatch(data, group = NULL, plot = TRUE, 
           combs = c("same", "across", "all"), 
           type.eff = "mean.single", which.cp = "cpD2", 
           which.eff = "sli", refmean = TRUE,
           dataout = "clip", verbose = TRUE, ...)
}

\arguments{
  \item{data}{multiple qPCR data generated by \code{\link{pcrbatch}}.}
  \item{group}{a character vector defining the replicates (if any) of control/treatment samples and reference genes/genes-of-interest. See 'Details'}.
  \item{plot}{logical. If \code{TRUE}, plots are displayed for the diagnostics and analysis.}
  \item{combs}{type of combinations between different samples (i.e. r1s1:g1s2). See 'Details'.}
  \item{type.eff}{type of efficiency averaging used. Same as in \code{\link{ratiocalc}}.}
  \item{which.eff}{efficiency obtained from which method. Same as in \code{\link{ratiocalc}}.}
  \item{which.cp}{threshold cycle obtained from which method. Same as in \code{\link{ratiocalc}}.}   
  \item{dataout}{where to store the result dataframe. Either it is copied to the clipboard (default) or any path + filename such as \code{c:\\temp\\result.txt}.}   
  \item{refmean}{logical. If \code{TRUE}, multiple reference are averaged before calculating the ratios. See 'Details'.}
  \item{verbose}{logical. If \code{TRUE}, the steps of analysis are shown in the console window}
  \item{...}{other parameters to be passed to \code{\link{ratiocalc}}.}
}

\details{
Similar to \code{\link{ratiocalc}}, the replicates of the 'pcrbatch' data columns are to be defined as a character vector with the following abbreviations:

"g1s1":   gene-of-interest #1 in treatment sample #1\cr
"g1c1":   gene-of-interest #1 in control sample #1\cr
"r1s1":   reference gene #1 in treatment sample #1\cr
"r1c1":   reference gene #1 in control sample #1

There is no distinction between the different technical replicates so that three different runs of gene-of-interest #1 in treatment sample #2 are defined as c("g1s2", "g1s2", "g1s2"). 

Example:\cr
1 control sample with 2 genes-of-interest (2 technical replicates), 2 treatment samples with 2 genes-of-interest (2 technical replicates):\cr
"g1c1", "g1c1", "g2c1", "g2c1", "g1s1", "g1s1", "g1s2", "g1s2", "g2s1", "g2s1", "g2s2", "g2s2"

The ratios are calculated for all pairwise 'rc:gc' and 'rs:gs' combinations according to:\cr
For all control samples \eqn{i = 1 \ldots I} and treatment samples \eqn{j = 1 \ldots J}, reference genes \eqn{k = 1 \ldots K} and genes-of-interest \eqn{l = 1 \ldots L}, calculate\cr

Without reference genes: \deqn{\frac{E_{g_is_j}^{cp_{g_is_j}}}{E_{g_kc_l}^{cp_{g_kc_l}}}}
With reference genes: \deqn{\frac{E_{g_is_j}^{cp_{g_is_j}}}{E_{g_kc_l}^{cp_{g_kc_l}}}/\frac{E_{r_ms_n}^{cp_{r_ms_n}}}{E_{r_oc_p}^{cp_{r_oc_p}}}}  

For the mechanistic model \code{mak3} the following is calculated:\cr

Without reference genes: \deqn{\frac{D0_{g_is_j}}{D0_{g_kc_l}}}
With reference genes: \deqn{\frac{D0_{g_is_j}}{D0_{g_kc_l}}/\frac{D0_{r_ms_n}}{D0_{r_oc_p}}}

Efficiencies can be taken from the individual curves or averaged from the replicates as described in the documentation to \code{\link{ratiocalc}}. It is also possible to give external efficiencies (i.e. acquired by some calibration curve) to the function. See 'Examples'. The different combinations of \code{type.eff}, \code{which.eff} and \code{which.cp} can yield very different results in ratio calculation. We observed a relatively stable setup which minimizes the overall variance using the combination
  
\code{type.eff = "mean.single"}     # averaging efficiency across replicates\cr
\code{which.eff = "sli"}            # taking efficiency from the sliding window method\cr
\code{which.cp = "sig"}             # using the second derivative maximum of a sigmoidal fit 

This is also the default setup in the function. The lowest variance can be obtained for the threshold cycles if the asymmetric 5-parameter \code{l5} model is used in the \code{\link{pcrbatch}} function. 

There are three different combination setups possible when calculating the pairwise ratios:\cr
\code{combs = "same"}: reference genes, genes-of-interest, control and treatment samples are the \code{same}, i.e. \eqn{i = k, m = o, j = n, l = p}.\cr
\code{combs = "across"}: control and treatment samples are the same, while the genes are combinated, i.e. \eqn{i \neq k, m \neq o, j = n, l = p, }.\cr
\code{combs = "all"}: reference genes, genes-of-interest, control and treatment samples are all combinated, i.e. \eqn{i \neq k, m \neq o, j \neq n, l \neq p}.

The last setting rarely makes sense and is very time-intensive. \code{combs = "same"} is the most common setting, but \code{combs = "across"} also makes sense if different genes-of-interest and reference gene combinations should be calculated for the same samples.

From version 1.3-6, \code{ratiobatch} has the option of averaging several reference genes, as described in Vandesompele \emph{et al.} (2002). Threshold cycles and efficiency values for any \eqn{i} reference genes with \eqn{j} replicates are averaged before calculating the ratios using the averaged value \eqn{\mu_r} for all reference genes in a control/treatment sample. The overall error \eqn{\sigma_r} is obtained by error propagation. The whole procedure is accomplished by function \code{\link{refmean}}, which can be used as a stand-alone function, but is most conveniently used inside \code{ratiobatch} setting \code{refmean = TRUE}. See also Example #2 in 'Examples'. For details about reference gene averaging by \code{\link{refmean}}, see there. The default setting is \code{refmean = TRUE}, so that the number of reference genes is checked by \code{\link{refmean}}. If none or only one per sample is found, the data is analyzed without using reference gene averaging/error propagation.
}

\value{
A list with the following components:
\item{resList}{a list with the results from the combinations as list items.}
\item{resDat}{a dataframe with the results in colums.}
Both \code{resList} and \code{resDat} have as names the combinations used for the ratio calculation.
If \code{plot = TRUE}, a boxplot matrix from the Monte-Carlo simulations, permutations and error propagations is given including 95\% confidence intervals as coloured horizontal lines.
}

\author{
Andrej-Nikolai Spiess
}

\note{
This function can be used quite conveniently when the raw fluorescence data from the 96- or 384-well runs come from Excel with 'Cycles' in the first column and run descriptions as explained above in the remaining column descriptions (such as 'r1c6'). Examples for a proper format can be found under \url{http://www.dr-spiess.de//qpcR//datasets.html}. This data may then be imported into \R by \code{dat <- pcrimport()}.
}

\references{
Accurate normalization of real-time quantitative RT-PCR data by geometric averaging of multiple internal control genes.\cr
Vandesompele J, De Preter K, Pattyn F, Poppe B, Van Roy N, De Paepe A, Speleman F.\cr
\emph{Genome Biol} (2002), \bold{3}: research0034-research0034.11.\cr
}

\examples{
\dontrun{
## One control sample, two treatment samples, 
## one gene-of-interest, two reference genes, 
## two replicates each. Replicates are averaged,
## but reference genes not, so that we have 4 ratios.
DAT1 <- pcrbatch(reps, fluo = 2:19, model = l5)
GROUP1 <- c("r1c1", "r1c1", "r2c1", "r2c1", "g1c1", "g1c1",
           "r1s1", "r1s1", "r1s2", "r1s2", "r2s1", "r2s1",
           "r2s2", "r2s2", "g1s1", "g1s1", "g1s2", "g1s2") 
RES1 <- ratiobatch(DAT1, GROUP1, refmean = FALSE)    

## Same as above, but now we average the two
## reference genes, so that we have 2 ratios
RES2 <- ratiobatch(DAT1, GROUP1, refmean = TRUE)

## Two control samples, one treatment sample, 
## one gene-of-interest, one reference gene, 
## no replicates. Use same efficiency E = 2.         
DAT3 <- pcrbatch(reps, fluo = 2:7, model = l5)
GROUP3 <- c("r1c1", "r1c2", "g1c1", "g1c2", 
            "r1s1", "g1s1") 
RES3 <- ratiobatch(DAT3, GROUP3, which.eff = 2) 
                   
## One control sample, one treatment sample, 
## three genes-of-interest, no reference gene, 
## three replicates. Using efficiency from sigmoidal model. 
DAT4 <- pcrbatch(reps, fluo = 2:19, model = l5)
GROUP4 <- c("g1c1", "g1c1", "g1c1", "g2c1", "g2c1", "g2c1", "g3c1", "g3c1", "g3c1",
            "g1s1", "g1s1", "g1s1", "g2s1", "g2s1", "g2s1", "g3s1", "g3s1", "g3s1")
RES4 <- ratiobatch(DAT4, GROUP4, which.eff = "sig")

## Using a mechanistic model (mak3).
## BEWARE: type.eff must be "individual"!
DAT5 <- pcrbatch(reps, fluo = 2:19, do.mak = "mak3")
GROUP5 <- c("r1c1", "r1c1", "r2c1", "r2c1", "g1c1", "g1c1",
           "r1s1", "r1s1", "r1s2", "r1s2", "r2s1", "r2s1",
           "r2s2", "r2s2", "g1s1", "g1s1", "g1s2", "g1s2")
RES5 <- ratiobatch(DAT5, GROUP5, which.eff = "mak", 
                  type.eff = "individual")

## Using external efficiencies from a 
## calibration curve. Can be supplied by the
## user from external calibration (or likewise),
## but in this example acquired by function 'calib'.
ml1 <- modlist(reps, model = l5)
DIL <- rep(10^(6:0), each = 4) 
EFF <- calib(refcurve = ml1, dil = DIL)$eff   
pba <- pcrbatch(ml1)
GROUP6 <- c(rep("g1s1", 4), rep("g1s2", 4),
            rep("g1s3", 4), rep("g1s4", 4), 
            rep("g1s5", 4), rep("g1s6", 4), 
            rep("g1c1", 4)) 
RES6 <- ratiobatch(pba, GROUP6, which.eff = EFF)
}        
}



\keyword{nonlinear}

