% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/wdd.R
\name{wdd}
\alias{wdd}
\title{Estimates the WDD Test}
\usage{
wdd(
  control,
  treated,
  disp_control = c(0, 0),
  disp_treated = c(0, 0),
  time_weights = c(1, 1),
  area_weights = c(1, 1, 1, 1),
  alpha = 0.1,
  silent = FALSE
)
}
\arguments{
\item{control}{vector with counts in pre,post for control area}

\item{treated}{vector with counts in pre,post for treated area}

\item{disp_control}{vector with counts in pre,post for displacement control area (default \code{c(0,0)})}

\item{disp_treated}{vector with counts in pre,post for displacement treated area (default \code{c(0,0)})}

\item{time_weights}{vector with weights for time periods for pre/post (default \code{c(1,1)})}

\item{area_weights}{vector with weights for different sized areas, order is c(control,treated,disp_control,disp_treated) (default \code{c(1,1,1,1)})}

\item{alpha}{scaler alpha level for confidence interval (default \code{0.1})}

\item{silent}{boolean set to TRUE if you do not want anything printed out (default FALSE)}
}
\value{
A length 9 vector with names:
\itemize{
\item \code{Est_Local} and \code{SE_Local}, the WDD and its standard error for the local estimates
\item \code{Est_Displace} and \code{SE_Displace}, the WDD and its standard error for the displacement areas
\item \code{Est_Total} and \code{SE_Total}, the WDD and its standard error for the combined local/displacement areas
\item \code{Z}, the Z-score for the total estimate
\item and the lower and upper confidence intervals, \code{LowCI} and \code{HighCI}, for whatever alpha level you specified for the total estimate.
}
}
\description{
Estimates the weighted displacement difference test from \href{https://crimesciencejournal.biomedcentral.com/articles/10.1186/s40163-018-0085-5}{Wheeler & Ratcliffe, \emph{A simple weighted displacement difference test to evaluate place based crime interventions}, Crime Science}
}
\details{
The wdd (weighted displacement difference) test is an extensions to differences-in-differences when observed count data pre/post in treated control areas. The test statistic (ignoring displacement areas and weights) is:

\deqn{WDD = \Delta T - \Delta Ct}{WDD = \Delta T - \Delta Ct}
where \eqn{\Delta T = T_1 - T_0}{\Delta T = T1 - T0}, the post time period count minus the pre time period count for the treated areas. And ditto for the control areas, Ct. The variance is calculated as:

\deqn{T_1 + T_0 + Ct_1 + Ct_0}{T1 + T0 + Ct1 + Ct0}

that is this test uses the normal approximation to the Poisson distribution to calculate the standard error for the WDD. So beware if using very tiny counts -- this approximation is less likely to be applicable (or count data that is Poisson, e.g. very overdispersed).

This function also incorporates weights for different examples, such as differing \href{https://andrewpwheeler.com/2021/01/09/the-wdd-test-with-different-pre-post-time-periods/}{pre/post time periods} (e.g. 2 years in pre and 1 year in post), or \href{https://andrewpwheeler.com/2021/02/23/the-wdd-test-with-different-area-sizes/}{different area sizes} (e.g. a one square mile area vs a two square mile area). The subsequent test statistic can then be interpreted as changes per unit time or changes per unit area (e.g. density) or both per time and density.
}
\examples{
# No weights and no displacement
cont <- c(20,20); treat <- c(20,10)
wdd(cont,treat)

# With Displacement stats
disptreat <- c(30,20); dispcont <- c(30,30)
wdd(cont,treat,dispcont,disptreat)

# With different time periods for pre/post
wdd(cont,treat,time_weights=c(2,1))

# With different area sizes
wdd(cont,treat,dispcont,disptreat,area_weights=c(2,1.5,3,3.2))

# You can technically use this even without pre (so just normal based approximation)
# just put in 0's for the pre data (so does not factor into variance)
res_test <- wdd(c(0,20),c(0,10))
twotail_p <- pnorm(res_test['Z'])*2
print(twotail_p) #~0.068
# e-test is very similar
e_test(20,10) #~0.069
}
\references{
Wheeler, A. P., & Ratcliffe, J. H. (2018). A simple weighted displacement difference test to evaluate place based crime interventions. \emph{Crime Science}, 7(1), 1-9.
}
\seealso{
\code{\link[=wdd_harm]{wdd_harm()}} for aggregating multiple WDD tests into one metric (e.g. based on crime harm weights)
\code{\link[=e_test]{e_test()}} for checking the difference in two Poisson means
}
