#' Conjoint Triad Descriptor
#'
#' Conjoint Triad Descriptor
#' 
#' This function calculates the Conjoint Triad descriptor (Dim: 343).
#' 
#' @param x A character vector, as the input protein sequence. 
#'
#' @return A length 343 named vector
#' 
#' @keywords extract CTriad extractCTriad Conjoint Triad
#'
#' @aliases extractCTriad
#' 
#' @author Nan Xiao <\url{http://www.road2stat.com}>
#' 
#' @export extractCTriad
#' 
#' @references
#' J.W. Shen, J. Zhang, X.M. Luo, W.L. Zhu, 
#' K.Q. Yu, K.X. Chen, Y.X. Li, H.L. Jiang. 
#' Predicting Protein-protein Interactions Based Only on Sequences Information. 
#' \emph{Proceedings of the National Academy of Sciences}. 007, 104, 4337-4341.
#' 
#' @examples
#' x = readFASTA(system.file('protseq/P00750.fasta', package = 'protr'))[[1]]
#' extractCTriad(x)
#' 

extractCTriad = function (x) {
  
  if (protcheck(x) == FALSE) stop('x has unrecognized amino acid type')
  
  # To achieve maximum performance, here we use dictionary directly
  # CTDict could also be generated with
  #
  # classes = vector('list', 7)
  # classes[[1]] = c('A', 'G', 'V')
  # classes[[2]] = c('I', 'L', 'F', 'P')
  # classes[[3]] = c('Y', 'M', 'T', 'S')
  # classes[[4]] = c('H', 'N', 'Q', 'W')
  # classes[[5]] = c('R', 'K')
  # classes[[6]] = c('D', 'E')
  # classes[[7]] = c('C')
  #   
  # vspace = expand.grid(1:7, 1:7, 1:7)
  #   
  # CTDict = vector('list', 343)
  #   
  # for (i in 1:343) {
  #   tmp = as.vector(outer(classes[[vspace[i, 1]]], 
  #                         classes[[vspace[i, 2]]], paste, sep = ''))
  #   CTDict[[i]] = as.vector(outer(tmp, 
  #                                 classes[[vspace[i, 3]]], paste, sep = ''))
  # }
  # 
  # CTDict = unlist(CTDict)
  
  CTDict = c('AAA', 'GAA', 'VAA', 'AGA', 'GGA', 'VGA', 'AVA', 'GVA', 'VVA', 
             'AAG', 'GAG', 'VAG', 'AGG', 'GGG', 'VGG', 'AVG', 'GVG', 'VVG', 
             'AAV', 'GAV', 'VAV', 'AGV', 'GGV', 'VGV', 'AVV', 'GVV', 'VVV', 
             'IAA', 'LAA', 'FAA', 'PAA', 'IGA', 'LGA', 'FGA', 'PGA', 'IVA', 
             'LVA', 'FVA', 'PVA', 'IAG', 'LAG', 'FAG', 'PAG', 'IGG', 'LGG', 
             'FGG', 'PGG', 'IVG', 'LVG', 'FVG', 'PVG', 'IAV', 'LAV', 'FAV', 
             'PAV', 'IGV', 'LGV', 'FGV', 'PGV', 'IVV', 'LVV', 'FVV', 'PVV', 
             'YAA', 'MAA', 'TAA', 'SAA', 'YGA', 'MGA', 'TGA', 'SGA', 'YVA', 
             'MVA', 'TVA', 'SVA', 'YAG', 'MAG', 'TAG', 'SAG', 'YGG', 'MGG', 
             'TGG', 'SGG', 'YVG', 'MVG', 'TVG', 'SVG', 'YAV', 'MAV', 'TAV', 
             'SAV', 'YGV', 'MGV', 'TGV', 'SGV', 'YVV', 'MVV', 'TVV', 'SVV', 
             'HAA', 'NAA', 'QAA', 'WAA', 'HGA', 'NGA', 'QGA', 'WGA', 'HVA', 
             'NVA', 'QVA', 'WVA', 'HAG', 'NAG', 'QAG', 'WAG', 'HGG', 'NGG', 
             'QGG', 'WGG', 'HVG', 'NVG', 'QVG', 'WVG', 'HAV', 'NAV', 'QAV', 
             'WAV', 'HGV', 'NGV', 'QGV', 'WGV', 'HVV', 'NVV', 'QVV', 'WVV', 
             'RAA', 'KAA', 'RGA', 'KGA', 'RVA', 'KVA', 'RAG', 'KAG', 'RGG', 
             'KGG', 'RVG', 'KVG', 'RAV', 'KAV', 'RGV', 'KGV', 'RVV', 'KVV', 
             'DAA', 'EAA', 'DGA', 'EGA', 'DVA', 'EVA', 'DAG', 'EAG', 'DGG', 
             'EGG', 'DVG', 'EVG', 'DAV', 'EAV', 'DGV', 'EGV', 'DVV', 'EVV', 
             'CAA', 'CGA', 'CVA', 'CAG', 'CGG', 'CVG', 'CAV', 'CGV', 'CVV', 
             'AIA', 'GIA', 'VIA', 'ALA', 'GLA', 'VLA', 'AFA', 'GFA', 'VFA', 
             'APA', 'GPA', 'VPA', 'AIG', 'GIG', 'VIG', 'ALG', 'GLG', 'VLG', 
             'AFG', 'GFG', 'VFG', 'APG', 'GPG', 'VPG', 'AIV', 'GIV', 'VIV', 
             'ALV', 'GLV', 'VLV', 'AFV', 'GFV', 'VFV', 'APV', 'GPV', 'VPV', 
             'IIA', 'LIA', 'FIA', 'PIA', 'ILA', 'LLA', 'FLA', 'PLA', 'IFA', 
             'LFA', 'FFA', 'PFA', 'IPA', 'LPA', 'FPA', 'PPA', 'IIG', 'LIG', 
             'FIG', 'PIG', 'ILG', 'LLG', 'FLG', 'PLG', 'IFG', 'LFG', 'FFG', 
             'PFG', 'IPG', 'LPG', 'FPG', 'PPG', 'IIV', 'LIV', 'FIV', 'PIV', 
             'ILV', 'LLV', 'FLV', 'PLV', 'IFV', 'LFV', 'FFV', 'PFV', 'IPV', 
             'LPV', 'FPV', 'PPV', 'YIA', 'MIA', 'TIA', 'SIA', 'YLA', 'MLA', 
             'TLA', 'SLA', 'YFA', 'MFA', 'TFA', 'SFA', 'YPA', 'MPA', 'TPA', 
             'SPA', 'YIG', 'MIG', 'TIG', 'SIG', 'YLG', 'MLG', 'TLG', 'SLG', 
             'YFG', 'MFG', 'TFG', 'SFG', 'YPG', 'MPG', 'TPG', 'SPG', 'YIV', 
             'MIV', 'TIV', 'SIV', 'YLV', 'MLV', 'TLV', 'SLV', 'YFV', 'MFV', 
             'TFV', 'SFV', 'YPV', 'MPV', 'TPV', 'SPV', 'HIA', 'NIA', 'QIA', 
             'WIA', 'HLA', 'NLA', 'QLA', 'WLA', 'HFA', 'NFA', 'QFA', 'WFA', 
             'HPA', 'NPA', 'QPA', 'WPA', 'HIG', 'NIG', 'QIG', 'WIG', 'HLG', 
             'NLG', 'QLG', 'WLG', 'HFG', 'NFG', 'QFG', 'WFG', 'HPG', 'NPG', 
             'QPG', 'WPG', 'HIV', 'NIV', 'QIV', 'WIV', 'HLV', 'NLV', 'QLV', 
             'WLV', 'HFV', 'NFV', 'QFV', 'WFV', 'HPV', 'NPV', 'QPV', 'WPV', 
             'RIA', 'KIA', 'RLA', 'KLA', 'RFA', 'KFA', 'RPA', 'KPA', 'RIG', 
             'KIG', 'RLG', 'KLG', 'RFG', 'KFG', 'RPG', 'KPG', 'RIV', 'KIV', 
             'RLV', 'KLV', 'RFV', 'KFV', 'RPV', 'KPV', 'DIA', 'EIA', 'DLA', 
             'ELA', 'DFA', 'EFA', 'DPA', 'EPA', 'DIG', 'EIG', 'DLG', 'ELG', 
             'DFG', 'EFG', 'DPG', 'EPG', 'DIV', 'EIV', 'DLV', 'ELV', 'DFV', 
             'EFV', 'DPV', 'EPV', 'CIA', 'CLA', 'CFA', 'CPA', 'CIG', 'CLG', 
             'CFG', 'CPG', 'CIV', 'CLV', 'CFV', 'CPV', 'AYA', 'GYA', 'VYA', 
             'AMA', 'GMA', 'VMA', 'ATA', 'GTA', 'VTA', 'ASA', 'GSA', 'VSA', 
             'AYG', 'GYG', 'VYG', 'AMG', 'GMG', 'VMG', 'ATG', 'GTG', 'VTG', 
             'ASG', 'GSG', 'VSG', 'AYV', 'GYV', 'VYV', 'AMV', 'GMV', 'VMV', 
             'ATV', 'GTV', 'VTV', 'ASV', 'GSV', 'VSV', 'IYA', 'LYA', 'FYA', 
             'PYA', 'IMA', 'LMA', 'FMA', 'PMA', 'ITA', 'LTA', 'FTA', 'PTA', 
             'ISA', 'LSA', 'FSA', 'PSA', 'IYG', 'LYG', 'FYG', 'PYG', 'IMG', 
             'LMG', 'FMG', 'PMG', 'ITG', 'LTG', 'FTG', 'PTG', 'ISG', 'LSG', 
             'FSG', 'PSG', 'IYV', 'LYV', 'FYV', 'PYV', 'IMV', 'LMV', 'FMV', 
             'PMV', 'ITV', 'LTV', 'FTV', 'PTV', 'ISV', 'LSV', 'FSV', 'PSV', 
             'YYA', 'MYA', 'TYA', 'SYA', 'YMA', 'MMA', 'TMA', 'SMA', 'YTA', 
             'MTA', 'TTA', 'STA', 'YSA', 'MSA', 'TSA', 'SSA', 'YYG', 'MYG', 
             'TYG', 'SYG', 'YMG', 'MMG', 'TMG', 'SMG', 'YTG', 'MTG', 'TTG', 
             'STG', 'YSG', 'MSG', 'TSG', 'SSG', 'YYV', 'MYV', 'TYV', 'SYV', 
             'YMV', 'MMV', 'TMV', 'SMV', 'YTV', 'MTV', 'TTV', 'STV', 'YSV', 
             'MSV', 'TSV', 'SSV', 'HYA', 'NYA', 'QYA', 'WYA', 'HMA', 'NMA', 
             'QMA', 'WMA', 'HTA', 'NTA', 'QTA', 'WTA', 'HSA', 'NSA', 'QSA', 
             'WSA', 'HYG', 'NYG', 'QYG', 'WYG', 'HMG', 'NMG', 'QMG', 'WMG', 
             'HTG', 'NTG', 'QTG', 'WTG', 'HSG', 'NSG', 'QSG', 'WSG', 'HYV', 
             'NYV', 'QYV', 'WYV', 'HMV', 'NMV', 'QMV', 'WMV', 'HTV', 'NTV', 
             'QTV', 'WTV', 'HSV', 'NSV', 'QSV', 'WSV', 'RYA', 'KYA', 'RMA', 
             'KMA', 'RTA', 'KTA', 'RSA', 'KSA', 'RYG', 'KYG', 'RMG', 'KMG', 
             'RTG', 'KTG', 'RSG', 'KSG', 'RYV', 'KYV', 'RMV', 'KMV', 'RTV', 
             'KTV', 'RSV', 'KSV', 'DYA', 'EYA', 'DMA', 'EMA', 'DTA', 'ETA', 
             'DSA', 'ESA', 'DYG', 'EYG', 'DMG', 'EMG', 'DTG', 'ETG', 'DSG', 
             'ESG', 'DYV', 'EYV', 'DMV', 'EMV', 'DTV', 'ETV', 'DSV', 'ESV', 
             'CYA', 'CMA', 'CTA', 'CSA', 'CYG', 'CMG', 'CTG', 'CSG', 'CYV', 
             'CMV', 'CTV', 'CSV', 'AHA', 'GHA', 'VHA', 'ANA', 'GNA', 'VNA', 
             'AQA', 'GQA', 'VQA', 'AWA', 'GWA', 'VWA', 'AHG', 'GHG', 'VHG', 
             'ANG', 'GNG', 'VNG', 'AQG', 'GQG', 'VQG', 'AWG', 'GWG', 'VWG', 
             'AHV', 'GHV', 'VHV', 'ANV', 'GNV', 'VNV', 'AQV', 'GQV', 'VQV', 
             'AWV', 'GWV', 'VWV', 'IHA', 'LHA', 'FHA', 'PHA', 'INA', 'LNA', 
             'FNA', 'PNA', 'IQA', 'LQA', 'FQA', 'PQA', 'IWA', 'LWA', 'FWA', 
             'PWA', 'IHG', 'LHG', 'FHG', 'PHG', 'ING', 'LNG', 'FNG', 'PNG', 
             'IQG', 'LQG', 'FQG', 'PQG', 'IWG', 'LWG', 'FWG', 'PWG', 'IHV', 
             'LHV', 'FHV', 'PHV', 'INV', 'LNV', 'FNV', 'PNV', 'IQV', 'LQV', 
             'FQV', 'PQV', 'IWV', 'LWV', 'FWV', 'PWV', 'YHA', 'MHA', 'THA', 
             'SHA', 'YNA', 'MNA', 'TNA', 'SNA', 'YQA', 'MQA', 'TQA', 'SQA', 
             'YWA', 'MWA', 'TWA', 'SWA', 'YHG', 'MHG', 'THG', 'SHG', 'YNG', 
             'MNG', 'TNG', 'SNG', 'YQG', 'MQG', 'TQG', 'SQG', 'YWG', 'MWG', 
             'TWG', 'SWG', 'YHV', 'MHV', 'THV', 'SHV', 'YNV', 'MNV', 'TNV', 
             'SNV', 'YQV', 'MQV', 'TQV', 'SQV', 'YWV', 'MWV', 'TWV', 'SWV', 
             'HHA', 'NHA', 'QHA', 'WHA', 'HNA', 'NNA', 'QNA', 'WNA', 'HQA', 
             'NQA', 'QQA', 'WQA', 'HWA', 'NWA', 'QWA', 'WWA', 'HHG', 'NHG', 
             'QHG', 'WHG', 'HNG', 'NNG', 'QNG', 'WNG', 'HQG', 'NQG', 'QQG', 
             'WQG', 'HWG', 'NWG', 'QWG', 'WWG', 'HHV', 'NHV', 'QHV', 'WHV', 
             'HNV', 'NNV', 'QNV', 'WNV', 'HQV', 'NQV', 'QQV', 'WQV', 'HWV', 
             'NWV', 'QWV', 'WWV', 'RHA', 'KHA', 'RNA', 'KNA', 'RQA', 'KQA', 
             'RWA', 'KWA', 'RHG', 'KHG', 'RNG', 'KNG', 'RQG', 'KQG', 'RWG', 
             'KWG', 'RHV', 'KHV', 'RNV', 'KNV', 'RQV', 'KQV', 'RWV', 'KWV', 
             'DHA', 'EHA', 'DNA', 'ENA', 'DQA', 'EQA', 'DWA', 'EWA', 'DHG', 
             'EHG', 'DNG', 'ENG', 'DQG', 'EQG', 'DWG', 'EWG', 'DHV', 'EHV', 
             'DNV', 'ENV', 'DQV', 'EQV', 'DWV', 'EWV', 'CHA', 'CNA', 'CQA', 
             'CWA', 'CHG', 'CNG', 'CQG', 'CWG', 'CHV', 'CNV', 'CQV', 'CWV', 
             'ARA', 'GRA', 'VRA', 'AKA', 'GKA', 'VKA', 'ARG', 'GRG', 'VRG', 
             'AKG', 'GKG', 'VKG', 'ARV', 'GRV', 'VRV', 'AKV', 'GKV', 'VKV', 
             'IRA', 'LRA', 'FRA', 'PRA', 'IKA', 'LKA', 'FKA', 'PKA', 'IRG', 
             'LRG', 'FRG', 'PRG', 'IKG', 'LKG', 'FKG', 'PKG', 'IRV', 'LRV', 
             'FRV', 'PRV', 'IKV', 'LKV', 'FKV', 'PKV', 'YRA', 'MRA', 'TRA', 
             'SRA', 'YKA', 'MKA', 'TKA', 'SKA', 'YRG', 'MRG', 'TRG', 'SRG', 
             'YKG', 'MKG', 'TKG', 'SKG', 'YRV', 'MRV', 'TRV', 'SRV', 'YKV', 
             'MKV', 'TKV', 'SKV', 'HRA', 'NRA', 'QRA', 'WRA', 'HKA', 'NKA', 
             'QKA', 'WKA', 'HRG', 'NRG', 'QRG', 'WRG', 'HKG', 'NKG', 'QKG', 
             'WKG', 'HRV', 'NRV', 'QRV', 'WRV', 'HKV', 'NKV', 'QKV', 'WKV', 
             'RRA', 'KRA', 'RKA', 'KKA', 'RRG', 'KRG', 'RKG', 'KKG', 'RRV', 
             'KRV', 'RKV', 'KKV', 'DRA', 'ERA', 'DKA', 'EKA', 'DRG', 'ERG', 
             'DKG', 'EKG', 'DRV', 'ERV', 'DKV', 'EKV', 'CRA', 'CKA', 'CRG', 
             'CKG', 'CRV', 'CKV', 'ADA', 'GDA', 'VDA', 'AEA', 'GEA', 'VEA', 
             'ADG', 'GDG', 'VDG', 'AEG', 'GEG', 'VEG', 'ADV', 'GDV', 'VDV', 
             'AEV', 'GEV', 'VEV', 'IDA', 'LDA', 'FDA', 'PDA', 'IEA', 'LEA', 
             'FEA', 'PEA', 'IDG', 'LDG', 'FDG', 'PDG', 'IEG', 'LEG', 'FEG', 
             'PEG', 'IDV', 'LDV', 'FDV', 'PDV', 'IEV', 'LEV', 'FEV', 'PEV', 
             'YDA', 'MDA', 'TDA', 'SDA', 'YEA', 'MEA', 'TEA', 'SEA', 'YDG', 
             'MDG', 'TDG', 'SDG', 'YEG', 'MEG', 'TEG', 'SEG', 'YDV', 'MDV', 
             'TDV', 'SDV', 'YEV', 'MEV', 'TEV', 'SEV', 'HDA', 'NDA', 'QDA', 
             'WDA', 'HEA', 'NEA', 'QEA', 'WEA', 'HDG', 'NDG', 'QDG', 'WDG', 
             'HEG', 'NEG', 'QEG', 'WEG', 'HDV', 'NDV', 'QDV', 'WDV', 'HEV', 
             'NEV', 'QEV', 'WEV', 'RDA', 'KDA', 'REA', 'KEA', 'RDG', 'KDG', 
             'REG', 'KEG', 'RDV', 'KDV', 'REV', 'KEV', 'DDA', 'EDA', 'DEA', 
             'EEA', 'DDG', 'EDG', 'DEG', 'EEG', 'DDV', 'EDV', 'DEV', 'EEV', 
             'CDA', 'CEA', 'CDG', 'CEG', 'CDV', 'CEV', 'ACA', 'GCA', 'VCA', 
             'ACG', 'GCG', 'VCG', 'ACV', 'GCV', 'VCV', 'ICA', 'LCA', 'FCA', 
             'PCA', 'ICG', 'LCG', 'FCG', 'PCG', 'ICV', 'LCV', 'FCV', 'PCV', 
             'YCA', 'MCA', 'TCA', 'SCA', 'YCG', 'MCG', 'TCG', 'SCG', 'YCV', 
             'MCV', 'TCV', 'SCV', 'HCA', 'NCA', 'QCA', 'WCA', 'HCG', 'NCG', 
             'QCG', 'WCG', 'HCV', 'NCV', 'QCV', 'WCV', 'RCA', 'KCA', 'RCG', 
             'KCG', 'RCV', 'KCV', 'DCA', 'ECA', 'DCG', 'ECG', 'DCV', 'ECV', 
             'CCA', 'CCG', 'CCV', 'AAI', 'GAI', 'VAI', 'AGI', 'GGI', 'VGI', 
             'AVI', 'GVI', 'VVI', 'AAL', 'GAL', 'VAL', 'AGL', 'GGL', 'VGL', 
             'AVL', 'GVL', 'VVL', 'AAF', 'GAF', 'VAF', 'AGF', 'GGF', 'VGF', 
             'AVF', 'GVF', 'VVF', 'AAP', 'GAP', 'VAP', 'AGP', 'GGP', 'VGP', 
             'AVP', 'GVP', 'VVP', 'IAI', 'LAI', 'FAI', 'PAI', 'IGI', 'LGI', 
             'FGI', 'PGI', 'IVI', 'LVI', 'FVI', 'PVI', 'IAL', 'LAL', 'FAL', 
             'PAL', 'IGL', 'LGL', 'FGL', 'PGL', 'IVL', 'LVL', 'FVL', 'PVL', 
             'IAF', 'LAF', 'FAF', 'PAF', 'IGF', 'LGF', 'FGF', 'PGF', 'IVF', 
             'LVF', 'FVF', 'PVF', 'IAP', 'LAP', 'FAP', 'PAP', 'IGP', 'LGP', 
             'FGP', 'PGP', 'IVP', 'LVP', 'FVP', 'PVP', 'YAI', 'MAI', 'TAI', 
             'SAI', 'YGI', 'MGI', 'TGI', 'SGI', 'YVI', 'MVI', 'TVI', 'SVI', 
             'YAL', 'MAL', 'TAL', 'SAL', 'YGL', 'MGL', 'TGL', 'SGL', 'YVL', 
             'MVL', 'TVL', 'SVL', 'YAF', 'MAF', 'TAF', 'SAF', 'YGF', 'MGF', 
             'TGF', 'SGF', 'YVF', 'MVF', 'TVF', 'SVF', 'YAP', 'MAP', 'TAP', 
             'SAP', 'YGP', 'MGP', 'TGP', 'SGP', 'YVP', 'MVP', 'TVP', 'SVP', 
             'HAI', 'NAI', 'QAI', 'WAI', 'HGI', 'NGI', 'QGI', 'WGI', 'HVI', 
             'NVI', 'QVI', 'WVI', 'HAL', 'NAL', 'QAL', 'WAL', 'HGL', 'NGL', 
             'QGL', 'WGL', 'HVL', 'NVL', 'QVL', 'WVL', 'HAF', 'NAF', 'QAF', 
             'WAF', 'HGF', 'NGF', 'QGF', 'WGF', 'HVF', 'NVF', 'QVF', 'WVF', 
             'HAP', 'NAP', 'QAP', 'WAP', 'HGP', 'NGP', 'QGP', 'WGP', 'HVP', 
             'NVP', 'QVP', 'WVP', 'RAI', 'KAI', 'RGI', 'KGI', 'RVI', 'KVI', 
             'RAL', 'KAL', 'RGL', 'KGL', 'RVL', 'KVL', 'RAF', 'KAF', 'RGF', 
             'KGF', 'RVF', 'KVF', 'RAP', 'KAP', 'RGP', 'KGP', 'RVP', 'KVP', 
             'DAI', 'EAI', 'DGI', 'EGI', 'DVI', 'EVI', 'DAL', 'EAL', 'DGL', 
             'EGL', 'DVL', 'EVL', 'DAF', 'EAF', 'DGF', 'EGF', 'DVF', 'EVF', 
             'DAP', 'EAP', 'DGP', 'EGP', 'DVP', 'EVP', 'CAI', 'CGI', 'CVI', 
             'CAL', 'CGL', 'CVL', 'CAF', 'CGF', 'CVF', 'CAP', 'CGP', 'CVP', 
             'AII', 'GII', 'VII', 'ALI', 'GLI', 'VLI', 'AFI', 'GFI', 'VFI', 
             'API', 'GPI', 'VPI', 'AIL', 'GIL', 'VIL', 'ALL', 'GLL', 'VLL', 
             'AFL', 'GFL', 'VFL', 'APL', 'GPL', 'VPL', 'AIF', 'GIF', 'VIF', 
             'ALF', 'GLF', 'VLF', 'AFF', 'GFF', 'VFF', 'APF', 'GPF', 'VPF', 
             'AIP', 'GIP', 'VIP', 'ALP', 'GLP', 'VLP', 'AFP', 'GFP', 'VFP', 
             'APP', 'GPP', 'VPP', 'III', 'LII', 'FII', 'PII', 'ILI', 'LLI', 
             'FLI', 'PLI', 'IFI', 'LFI', 'FFI', 'PFI', 'IPI', 'LPI', 'FPI', 
             'PPI', 'IIL', 'LIL', 'FIL', 'PIL', 'ILL', 'LLL', 'FLL', 'PLL', 
             'IFL', 'LFL', 'FFL', 'PFL', 'IPL', 'LPL', 'FPL', 'PPL', 'IIF', 
             'LIF', 'FIF', 'PIF', 'ILF', 'LLF', 'FLF', 'PLF', 'IFF', 'LFF', 
             'FFF', 'PFF', 'IPF', 'LPF', 'FPF', 'PPF', 'IIP', 'LIP', 'FIP', 
             'PIP', 'ILP', 'LLP', 'FLP', 'PLP', 'IFP', 'LFP', 'FFP', 'PFP', 
             'IPP', 'LPP', 'FPP', 'PPP', 'YII', 'MII', 'TII', 'SII', 'YLI', 
             'MLI', 'TLI', 'SLI', 'YFI', 'MFI', 'TFI', 'SFI', 'YPI', 'MPI', 
             'TPI', 'SPI', 'YIL', 'MIL', 'TIL', 'SIL', 'YLL', 'MLL', 'TLL', 
             'SLL', 'YFL', 'MFL', 'TFL', 'SFL', 'YPL', 'MPL', 'TPL', 'SPL', 
             'YIF', 'MIF', 'TIF', 'SIF', 'YLF', 'MLF', 'TLF', 'SLF', 'YFF', 
             'MFF', 'TFF', 'SFF', 'YPF', 'MPF', 'TPF', 'SPF', 'YIP', 'MIP', 
             'TIP', 'SIP', 'YLP', 'MLP', 'TLP', 'SLP', 'YFP', 'MFP', 'TFP', 
             'SFP', 'YPP', 'MPP', 'TPP', 'SPP', 'HII', 'NII', 'QII', 'WII', 
             'HLI', 'NLI', 'QLI', 'WLI', 'HFI', 'NFI', 'QFI', 'WFI', 'HPI', 
             'NPI', 'QPI', 'WPI', 'HIL', 'NIL', 'QIL', 'WIL', 'HLL', 'NLL', 
             'QLL', 'WLL', 'HFL', 'NFL', 'QFL', 'WFL', 'HPL', 'NPL', 'QPL', 
             'WPL', 'HIF', 'NIF', 'QIF', 'WIF', 'HLF', 'NLF', 'QLF', 'WLF', 
             'HFF', 'NFF', 'QFF', 'WFF', 'HPF', 'NPF', 'QPF', 'WPF', 'HIP', 
             'NIP', 'QIP', 'WIP', 'HLP', 'NLP', 'QLP', 'WLP', 'HFP', 'NFP', 
             'QFP', 'WFP', 'HPP', 'NPP', 'QPP', 'WPP', 'RII', 'KII', 'RLI', 
             'KLI', 'RFI', 'KFI', 'RPI', 'KPI', 'RIL', 'KIL', 'RLL', 'KLL', 
             'RFL', 'KFL', 'RPL', 'KPL', 'RIF', 'KIF', 'RLF', 'KLF', 'RFF', 
             'KFF', 'RPF', 'KPF', 'RIP', 'KIP', 'RLP', 'KLP', 'RFP', 'KFP', 
             'RPP', 'KPP', 'DII', 'EII', 'DLI', 'ELI', 'DFI', 'EFI', 'DPI', 
             'EPI', 'DIL', 'EIL', 'DLL', 'ELL', 'DFL', 'EFL', 'DPL', 'EPL', 
             'DIF', 'EIF', 'DLF', 'ELF', 'DFF', 'EFF', 'DPF', 'EPF', 'DIP', 
             'EIP', 'DLP', 'ELP', 'DFP', 'EFP', 'DPP', 'EPP', 'CII', 'CLI', 
             'CFI', 'CPI', 'CIL', 'CLL', 'CFL', 'CPL', 'CIF', 'CLF', 'CFF', 
             'CPF', 'CIP', 'CLP', 'CFP', 'CPP', 'AYI', 'GYI', 'VYI', 'AMI', 
             'GMI', 'VMI', 'ATI', 'GTI', 'VTI', 'ASI', 'GSI', 'VSI', 'AYL', 
             'GYL', 'VYL', 'AML', 'GML', 'VML', 'ATL', 'GTL', 'VTL', 'ASL', 
             'GSL', 'VSL', 'AYF', 'GYF', 'VYF', 'AMF', 'GMF', 'VMF', 'ATF', 
             'GTF', 'VTF', 'ASF', 'GSF', 'VSF', 'AYP', 'GYP', 'VYP', 'AMP', 
             'GMP', 'VMP', 'ATP', 'GTP', 'VTP', 'ASP', 'GSP', 'VSP', 'IYI', 
             'LYI', 'FYI', 'PYI', 'IMI', 'LMI', 'FMI', 'PMI', 'ITI', 'LTI', 
             'FTI', 'PTI', 'ISI', 'LSI', 'FSI', 'PSI', 'IYL', 'LYL', 'FYL', 
             'PYL', 'IML', 'LML', 'FML', 'PML', 'ITL', 'LTL', 'FTL', 'PTL', 
             'ISL', 'LSL', 'FSL', 'PSL', 'IYF', 'LYF', 'FYF', 'PYF', 'IMF', 
             'LMF', 'FMF', 'PMF', 'ITF', 'LTF', 'FTF', 'PTF', 'ISF', 'LSF', 
             'FSF', 'PSF', 'IYP', 'LYP', 'FYP', 'PYP', 'IMP', 'LMP', 'FMP', 
             'PMP', 'ITP', 'LTP', 'FTP', 'PTP', 'ISP', 'LSP', 'FSP', 'PSP', 
             'YYI', 'MYI', 'TYI', 'SYI', 'YMI', 'MMI', 'TMI', 'SMI', 'YTI', 
             'MTI', 'TTI', 'STI', 'YSI', 'MSI', 'TSI', 'SSI', 'YYL', 'MYL', 
             'TYL', 'SYL', 'YML', 'MML', 'TML', 'SML', 'YTL', 'MTL', 'TTL', 
             'STL', 'YSL', 'MSL', 'TSL', 'SSL', 'YYF', 'MYF', 'TYF', 'SYF', 
             'YMF', 'MMF', 'TMF', 'SMF', 'YTF', 'MTF', 'TTF', 'STF', 'YSF', 
             'MSF', 'TSF', 'SSF', 'YYP', 'MYP', 'TYP', 'SYP', 'YMP', 'MMP', 
             'TMP', 'SMP', 'YTP', 'MTP', 'TTP', 'STP', 'YSP', 'MSP', 'TSP', 
             'SSP', 'HYI', 'NYI', 'QYI', 'WYI', 'HMI', 'NMI', 'QMI', 'WMI', 
             'HTI', 'NTI', 'QTI', 'WTI', 'HSI', 'NSI', 'QSI', 'WSI', 'HYL', 
             'NYL', 'QYL', 'WYL', 'HML', 'NML', 'QML', 'WML', 'HTL', 'NTL', 
             'QTL', 'WTL', 'HSL', 'NSL', 'QSL', 'WSL', 'HYF', 'NYF', 'QYF', 
             'WYF', 'HMF', 'NMF', 'QMF', 'WMF', 'HTF', 'NTF', 'QTF', 'WTF', 
             'HSF', 'NSF', 'QSF', 'WSF', 'HYP', 'NYP', 'QYP', 'WYP', 'HMP', 
             'NMP', 'QMP', 'WMP', 'HTP', 'NTP', 'QTP', 'WTP', 'HSP', 'NSP', 
             'QSP', 'WSP', 'RYI', 'KYI', 'RMI', 'KMI', 'RTI', 'KTI', 'RSI', 
             'KSI', 'RYL', 'KYL', 'RML', 'KML', 'RTL', 'KTL', 'RSL', 'KSL', 
             'RYF', 'KYF', 'RMF', 'KMF', 'RTF', 'KTF', 'RSF', 'KSF', 'RYP', 
             'KYP', 'RMP', 'KMP', 'RTP', 'KTP', 'RSP', 'KSP', 'DYI', 'EYI', 
             'DMI', 'EMI', 'DTI', 'ETI', 'DSI', 'ESI', 'DYL', 'EYL', 'DML', 
             'EML', 'DTL', 'ETL', 'DSL', 'ESL', 'DYF', 'EYF', 'DMF', 'EMF', 
             'DTF', 'ETF', 'DSF', 'ESF', 'DYP', 'EYP', 'DMP', 'EMP', 'DTP', 
             'ETP', 'DSP', 'ESP', 'CYI', 'CMI', 'CTI', 'CSI', 'CYL', 'CML', 
             'CTL', 'CSL', 'CYF', 'CMF', 'CTF', 'CSF', 'CYP', 'CMP', 'CTP', 
             'CSP', 'AHI', 'GHI', 'VHI', 'ANI', 'GNI', 'VNI', 'AQI', 'GQI', 
             'VQI', 'AWI', 'GWI', 'VWI', 'AHL', 'GHL', 'VHL', 'ANL', 'GNL', 
             'VNL', 'AQL', 'GQL', 'VQL', 'AWL', 'GWL', 'VWL', 'AHF', 'GHF', 
             'VHF', 'ANF', 'GNF', 'VNF', 'AQF', 'GQF', 'VQF', 'AWF', 'GWF', 
             'VWF', 'AHP', 'GHP', 'VHP', 'ANP', 'GNP', 'VNP', 'AQP', 'GQP', 
             'VQP', 'AWP', 'GWP', 'VWP', 'IHI', 'LHI', 'FHI', 'PHI', 'INI', 
             'LNI', 'FNI', 'PNI', 'IQI', 'LQI', 'FQI', 'PQI', 'IWI', 'LWI', 
             'FWI', 'PWI', 'IHL', 'LHL', 'FHL', 'PHL', 'INL', 'LNL', 'FNL', 
             'PNL', 'IQL', 'LQL', 'FQL', 'PQL', 'IWL', 'LWL', 'FWL', 'PWL', 
             'IHF', 'LHF', 'FHF', 'PHF', 'INF', 'LNF', 'FNF', 'PNF', 'IQF', 
             'LQF', 'FQF', 'PQF', 'IWF', 'LWF', 'FWF', 'PWF', 'IHP', 'LHP', 
             'FHP', 'PHP', 'INP', 'LNP', 'FNP', 'PNP', 'IQP', 'LQP', 'FQP', 
             'PQP', 'IWP', 'LWP', 'FWP', 'PWP', 'YHI', 'MHI', 'THI', 'SHI', 
             'YNI', 'MNI', 'TNI', 'SNI', 'YQI', 'MQI', 'TQI', 'SQI', 'YWI', 
             'MWI', 'TWI', 'SWI', 'YHL', 'MHL', 'THL', 'SHL', 'YNL', 'MNL', 
             'TNL', 'SNL', 'YQL', 'MQL', 'TQL', 'SQL', 'YWL', 'MWL', 'TWL', 
             'SWL', 'YHF', 'MHF', 'THF', 'SHF', 'YNF', 'MNF', 'TNF', 'SNF', 
             'YQF', 'MQF', 'TQF', 'SQF', 'YWF', 'MWF', 'TWF', 'SWF', 'YHP', 
             'MHP', 'THP', 'SHP', 'YNP', 'MNP', 'TNP', 'SNP', 'YQP', 'MQP', 
             'TQP', 'SQP', 'YWP', 'MWP', 'TWP', 'SWP', 'HHI', 'NHI', 'QHI', 
             'WHI', 'HNI', 'NNI', 'QNI', 'WNI', 'HQI', 'NQI', 'QQI', 'WQI', 
             'HWI', 'NWI', 'QWI', 'WWI', 'HHL', 'NHL', 'QHL', 'WHL', 'HNL', 
             'NNL', 'QNL', 'WNL', 'HQL', 'NQL', 'QQL', 'WQL', 'HWL', 'NWL', 
             'QWL', 'WWL', 'HHF', 'NHF', 'QHF', 'WHF', 'HNF', 'NNF', 'QNF', 
             'WNF', 'HQF', 'NQF', 'QQF', 'WQF', 'HWF', 'NWF', 'QWF', 'WWF', 
             'HHP', 'NHP', 'QHP', 'WHP', 'HNP', 'NNP', 'QNP', 'WNP', 'HQP', 
             'NQP', 'QQP', 'WQP', 'HWP', 'NWP', 'QWP', 'WWP', 'RHI', 'KHI', 
             'RNI', 'KNI', 'RQI', 'KQI', 'RWI', 'KWI', 'RHL', 'KHL', 'RNL', 
             'KNL', 'RQL', 'KQL', 'RWL', 'KWL', 'RHF', 'KHF', 'RNF', 'KNF', 
             'RQF', 'KQF', 'RWF', 'KWF', 'RHP', 'KHP', 'RNP', 'KNP', 'RQP', 
             'KQP', 'RWP', 'KWP', 'DHI', 'EHI', 'DNI', 'ENI', 'DQI', 'EQI', 
             'DWI', 'EWI', 'DHL', 'EHL', 'DNL', 'ENL', 'DQL', 'EQL', 'DWL', 
             'EWL', 'DHF', 'EHF', 'DNF', 'ENF', 'DQF', 'EQF', 'DWF', 'EWF', 
             'DHP', 'EHP', 'DNP', 'ENP', 'DQP', 'EQP', 'DWP', 'EWP', 'CHI', 
             'CNI', 'CQI', 'CWI', 'CHL', 'CNL', 'CQL', 'CWL', 'CHF', 'CNF', 
             'CQF', 'CWF', 'CHP', 'CNP', 'CQP', 'CWP', 'ARI', 'GRI', 'VRI', 
             'AKI', 'GKI', 'VKI', 'ARL', 'GRL', 'VRL', 'AKL', 'GKL', 'VKL', 
             'ARF', 'GRF', 'VRF', 'AKF', 'GKF', 'VKF', 'ARP', 'GRP', 'VRP', 
             'AKP', 'GKP', 'VKP', 'IRI', 'LRI', 'FRI', 'PRI', 'IKI', 'LKI', 
             'FKI', 'PKI', 'IRL', 'LRL', 'FRL', 'PRL', 'IKL', 'LKL', 'FKL', 
             'PKL', 'IRF', 'LRF', 'FRF', 'PRF', 'IKF', 'LKF', 'FKF', 'PKF', 
             'IRP', 'LRP', 'FRP', 'PRP', 'IKP', 'LKP', 'FKP', 'PKP', 'YRI', 
             'MRI', 'TRI', 'SRI', 'YKI', 'MKI', 'TKI', 'SKI', 'YRL', 'MRL', 
             'TRL', 'SRL', 'YKL', 'MKL', 'TKL', 'SKL', 'YRF', 'MRF', 'TRF', 
             'SRF', 'YKF', 'MKF', 'TKF', 'SKF', 'YRP', 'MRP', 'TRP', 'SRP', 
             'YKP', 'MKP', 'TKP', 'SKP', 'HRI', 'NRI', 'QRI', 'WRI', 'HKI', 
             'NKI', 'QKI', 'WKI', 'HRL', 'NRL', 'QRL', 'WRL', 'HKL', 'NKL', 
             'QKL', 'WKL', 'HRF', 'NRF', 'QRF', 'WRF', 'HKF', 'NKF', 'QKF', 
             'WKF', 'HRP', 'NRP', 'QRP', 'WRP', 'HKP', 'NKP', 'QKP', 'WKP', 
             'RRI', 'KRI', 'RKI', 'KKI', 'RRL', 'KRL', 'RKL', 'KKL', 'RRF', 
             'KRF', 'RKF', 'KKF', 'RRP', 'KRP', 'RKP', 'KKP', 'DRI', 'ERI', 
             'DKI', 'EKI', 'DRL', 'ERL', 'DKL', 'EKL', 'DRF', 'ERF', 'DKF', 
             'EKF', 'DRP', 'ERP', 'DKP', 'EKP', 'CRI', 'CKI', 'CRL', 'CKL', 
             'CRF', 'CKF', 'CRP', 'CKP', 'ADI', 'GDI', 'VDI', 'AEI', 'GEI', 
             'VEI', 'ADL', 'GDL', 'VDL', 'AEL', 'GEL', 'VEL', 'ADF', 'GDF', 
             'VDF', 'AEF', 'GEF', 'VEF', 'ADP', 'GDP', 'VDP', 'AEP', 'GEP', 
             'VEP', 'IDI', 'LDI', 'FDI', 'PDI', 'IEI', 'LEI', 'FEI', 'PEI', 
             'IDL', 'LDL', 'FDL', 'PDL', 'IEL', 'LEL', 'FEL', 'PEL', 'IDF', 
             'LDF', 'FDF', 'PDF', 'IEF', 'LEF', 'FEF', 'PEF', 'IDP', 'LDP', 
             'FDP', 'PDP', 'IEP', 'LEP', 'FEP', 'PEP', 'YDI', 'MDI', 'TDI', 
             'SDI', 'YEI', 'MEI', 'TEI', 'SEI', 'YDL', 'MDL', 'TDL', 'SDL', 
             'YEL', 'MEL', 'TEL', 'SEL', 'YDF', 'MDF', 'TDF', 'SDF', 'YEF', 
             'MEF', 'TEF', 'SEF', 'YDP', 'MDP', 'TDP', 'SDP', 'YEP', 'MEP', 
             'TEP', 'SEP', 'HDI', 'NDI', 'QDI', 'WDI', 'HEI', 'NEI', 'QEI', 
             'WEI', 'HDL', 'NDL', 'QDL', 'WDL', 'HEL', 'NEL', 'QEL', 'WEL', 
             'HDF', 'NDF', 'QDF', 'WDF', 'HEF', 'NEF', 'QEF', 'WEF', 'HDP', 
             'NDP', 'QDP', 'WDP', 'HEP', 'NEP', 'QEP', 'WEP', 'RDI', 'KDI', 
             'REI', 'KEI', 'RDL', 'KDL', 'REL', 'KEL', 'RDF', 'KDF', 'REF', 
             'KEF', 'RDP', 'KDP', 'REP', 'KEP', 'DDI', 'EDI', 'DEI', 'EEI', 
             'DDL', 'EDL', 'DEL', 'EEL', 'DDF', 'EDF', 'DEF', 'EEF', 'DDP', 
             'EDP', 'DEP', 'EEP', 'CDI', 'CEI', 'CDL', 'CEL', 'CDF', 'CEF', 
             'CDP', 'CEP', 'ACI', 'GCI', 'VCI', 'ACL', 'GCL', 'VCL', 'ACF', 
             'GCF', 'VCF', 'ACP', 'GCP', 'VCP', 'ICI', 'LCI', 'FCI', 'PCI', 
             'ICL', 'LCL', 'FCL', 'PCL', 'ICF', 'LCF', 'FCF', 'PCF', 'ICP', 
             'LCP', 'FCP', 'PCP', 'YCI', 'MCI', 'TCI', 'SCI', 'YCL', 'MCL', 
             'TCL', 'SCL', 'YCF', 'MCF', 'TCF', 'SCF', 'YCP', 'MCP', 'TCP', 
             'SCP', 'HCI', 'NCI', 'QCI', 'WCI', 'HCL', 'NCL', 'QCL', 'WCL', 
             'HCF', 'NCF', 'QCF', 'WCF', 'HCP', 'NCP', 'QCP', 'WCP', 'RCI', 
             'KCI', 'RCL', 'KCL', 'RCF', 'KCF', 'RCP', 'KCP', 'DCI', 'ECI', 
             'DCL', 'ECL', 'DCF', 'ECF', 'DCP', 'ECP', 'CCI', 'CCL', 'CCF', 
             'CCP', 'AAY', 'GAY', 'VAY', 'AGY', 'GGY', 'VGY', 'AVY', 'GVY', 
             'VVY', 'AAM', 'GAM', 'VAM', 'AGM', 'GGM', 'VGM', 'AVM', 'GVM', 
             'VVM', 'AAT', 'GAT', 'VAT', 'AGT', 'GGT', 'VGT', 'AVT', 'GVT', 
             'VVT', 'AAS', 'GAS', 'VAS', 'AGS', 'GGS', 'VGS', 'AVS', 'GVS', 
             'VVS', 'IAY', 'LAY', 'FAY', 'PAY', 'IGY', 'LGY', 'FGY', 'PGY', 
             'IVY', 'LVY', 'FVY', 'PVY', 'IAM', 'LAM', 'FAM', 'PAM', 'IGM', 
             'LGM', 'FGM', 'PGM', 'IVM', 'LVM', 'FVM', 'PVM', 'IAT', 'LAT', 
             'FAT', 'PAT', 'IGT', 'LGT', 'FGT', 'PGT', 'IVT', 'LVT', 'FVT', 
             'PVT', 'IAS', 'LAS', 'FAS', 'PAS', 'IGS', 'LGS', 'FGS', 'PGS', 
             'IVS', 'LVS', 'FVS', 'PVS', 'YAY', 'MAY', 'TAY', 'SAY', 'YGY', 
             'MGY', 'TGY', 'SGY', 'YVY', 'MVY', 'TVY', 'SVY', 'YAM', 'MAM', 
             'TAM', 'SAM', 'YGM', 'MGM', 'TGM', 'SGM', 'YVM', 'MVM', 'TVM', 
             'SVM', 'YAT', 'MAT', 'TAT', 'SAT', 'YGT', 'MGT', 'TGT', 'SGT', 
             'YVT', 'MVT', 'TVT', 'SVT', 'YAS', 'MAS', 'TAS', 'SAS', 'YGS', 
             'MGS', 'TGS', 'SGS', 'YVS', 'MVS', 'TVS', 'SVS', 'HAY', 'NAY', 
             'QAY', 'WAY', 'HGY', 'NGY', 'QGY', 'WGY', 'HVY', 'NVY', 'QVY', 
             'WVY', 'HAM', 'NAM', 'QAM', 'WAM', 'HGM', 'NGM', 'QGM', 'WGM', 
             'HVM', 'NVM', 'QVM', 'WVM', 'HAT', 'NAT', 'QAT', 'WAT', 'HGT', 
             'NGT', 'QGT', 'WGT', 'HVT', 'NVT', 'QVT', 'WVT', 'HAS', 'NAS', 
             'QAS', 'WAS', 'HGS', 'NGS', 'QGS', 'WGS', 'HVS', 'NVS', 'QVS', 
             'WVS', 'RAY', 'KAY', 'RGY', 'KGY', 'RVY', 'KVY', 'RAM', 'KAM', 
             'RGM', 'KGM', 'RVM', 'KVM', 'RAT', 'KAT', 'RGT', 'KGT', 'RVT', 
             'KVT', 'RAS', 'KAS', 'RGS', 'KGS', 'RVS', 'KVS', 'DAY', 'EAY', 
             'DGY', 'EGY', 'DVY', 'EVY', 'DAM', 'EAM', 'DGM', 'EGM', 'DVM', 
             'EVM', 'DAT', 'EAT', 'DGT', 'EGT', 'DVT', 'EVT', 'DAS', 'EAS', 
             'DGS', 'EGS', 'DVS', 'EVS', 'CAY', 'CGY', 'CVY', 'CAM', 'CGM', 
             'CVM', 'CAT', 'CGT', 'CVT', 'CAS', 'CGS', 'CVS', 'AIY', 'GIY', 
             'VIY', 'ALY', 'GLY', 'VLY', 'AFY', 'GFY', 'VFY', 'APY', 'GPY', 
             'VPY', 'AIM', 'GIM', 'VIM', 'ALM', 'GLM', 'VLM', 'AFM', 'GFM', 
             'VFM', 'APM', 'GPM', 'VPM', 'AIT', 'GIT', 'VIT', 'ALT', 'GLT', 
             'VLT', 'AFT', 'GFT', 'VFT', 'APT', 'GPT', 'VPT', 'AIS', 'GIS', 
             'VIS', 'ALS', 'GLS', 'VLS', 'AFS', 'GFS', 'VFS', 'APS', 'GPS', 
             'VPS', 'IIY', 'LIY', 'FIY', 'PIY', 'ILY', 'LLY', 'FLY', 'PLY', 
             'IFY', 'LFY', 'FFY', 'PFY', 'IPY', 'LPY', 'FPY', 'PPY', 'IIM', 
             'LIM', 'FIM', 'PIM', 'ILM', 'LLM', 'FLM', 'PLM', 'IFM', 'LFM', 
             'FFM', 'PFM', 'IPM', 'LPM', 'FPM', 'PPM', 'IIT', 'LIT', 'FIT', 
             'PIT', 'ILT', 'LLT', 'FLT', 'PLT', 'IFT', 'LFT', 'FFT', 'PFT', 
             'IPT', 'LPT', 'FPT', 'PPT', 'IIS', 'LIS', 'FIS', 'PIS', 'ILS', 
             'LLS', 'FLS', 'PLS', 'IFS', 'LFS', 'FFS', 'PFS', 'IPS', 'LPS', 
             'FPS', 'PPS', 'YIY', 'MIY', 'TIY', 'SIY', 'YLY', 'MLY', 'TLY', 
             'SLY', 'YFY', 'MFY', 'TFY', 'SFY', 'YPY', 'MPY', 'TPY', 'SPY', 
             'YIM', 'MIM', 'TIM', 'SIM', 'YLM', 'MLM', 'TLM', 'SLM', 'YFM', 
             'MFM', 'TFM', 'SFM', 'YPM', 'MPM', 'TPM', 'SPM', 'YIT', 'MIT', 
             'TIT', 'SIT', 'YLT', 'MLT', 'TLT', 'SLT', 'YFT', 'MFT', 'TFT', 
             'SFT', 'YPT', 'MPT', 'TPT', 'SPT', 'YIS', 'MIS', 'TIS', 'SIS', 
             'YLS', 'MLS', 'TLS', 'SLS', 'YFS', 'MFS', 'TFS', 'SFS', 'YPS', 
             'MPS', 'TPS', 'SPS', 'HIY', 'NIY', 'QIY', 'WIY', 'HLY', 'NLY', 
             'QLY', 'WLY', 'HFY', 'NFY', 'QFY', 'WFY', 'HPY', 'NPY', 'QPY', 
             'WPY', 'HIM', 'NIM', 'QIM', 'WIM', 'HLM', 'NLM', 'QLM', 'WLM', 
             'HFM', 'NFM', 'QFM', 'WFM', 'HPM', 'NPM', 'QPM', 'WPM', 'HIT', 
             'NIT', 'QIT', 'WIT', 'HLT', 'NLT', 'QLT', 'WLT', 'HFT', 'NFT', 
             'QFT', 'WFT', 'HPT', 'NPT', 'QPT', 'WPT', 'HIS', 'NIS', 'QIS', 
             'WIS', 'HLS', 'NLS', 'QLS', 'WLS', 'HFS', 'NFS', 'QFS', 'WFS', 
             'HPS', 'NPS', 'QPS', 'WPS', 'RIY', 'KIY', 'RLY', 'KLY', 'RFY', 
             'KFY', 'RPY', 'KPY', 'RIM', 'KIM', 'RLM', 'KLM', 'RFM', 'KFM', 
             'RPM', 'KPM', 'RIT', 'KIT', 'RLT', 'KLT', 'RFT', 'KFT', 'RPT', 
             'KPT', 'RIS', 'KIS', 'RLS', 'KLS', 'RFS', 'KFS', 'RPS', 'KPS', 
             'DIY', 'EIY', 'DLY', 'ELY', 'DFY', 'EFY', 'DPY', 'EPY', 'DIM', 
             'EIM', 'DLM', 'ELM', 'DFM', 'EFM', 'DPM', 'EPM', 'DIT', 'EIT', 
             'DLT', 'ELT', 'DFT', 'EFT', 'DPT', 'EPT', 'DIS', 'EIS', 'DLS', 
             'ELS', 'DFS', 'EFS', 'DPS', 'EPS', 'CIY', 'CLY', 'CFY', 'CPY', 
             'CIM', 'CLM', 'CFM', 'CPM', 'CIT', 'CLT', 'CFT', 'CPT', 'CIS', 
             'CLS', 'CFS', 'CPS', 'AYY', 'GYY', 'VYY', 'AMY', 'GMY', 'VMY', 
             'ATY', 'GTY', 'VTY', 'ASY', 'GSY', 'VSY', 'AYM', 'GYM', 'VYM', 
             'AMM', 'GMM', 'VMM', 'ATM', 'GTM', 'VTM', 'ASM', 'GSM', 'VSM', 
             'AYT', 'GYT', 'VYT', 'AMT', 'GMT', 'VMT', 'ATT', 'GTT', 'VTT', 
             'AST', 'GST', 'VST', 'AYS', 'GYS', 'VYS', 'AMS', 'GMS', 'VMS', 
             'ATS', 'GTS', 'VTS', 'ASS', 'GSS', 'VSS', 'IYY', 'LYY', 'FYY', 
             'PYY', 'IMY', 'LMY', 'FMY', 'PMY', 'ITY', 'LTY', 'FTY', 'PTY', 
             'ISY', 'LSY', 'FSY', 'PSY', 'IYM', 'LYM', 'FYM', 'PYM', 'IMM', 
             'LMM', 'FMM', 'PMM', 'ITM', 'LTM', 'FTM', 'PTM', 'ISM', 'LSM', 
             'FSM', 'PSM', 'IYT', 'LYT', 'FYT', 'PYT', 'IMT', 'LMT', 'FMT', 
             'PMT', 'ITT', 'LTT', 'FTT', 'PTT', 'IST', 'LST', 'FST', 'PST', 
             'IYS', 'LYS', 'FYS', 'PYS', 'IMS', 'LMS', 'FMS', 'PMS', 'ITS', 
             'LTS', 'FTS', 'PTS', 'ISS', 'LSS', 'FSS', 'PSS', 'YYY', 'MYY', 
             'TYY', 'SYY', 'YMY', 'MMY', 'TMY', 'SMY', 'YTY', 'MTY', 'TTY', 
             'STY', 'YSY', 'MSY', 'TSY', 'SSY', 'YYM', 'MYM', 'TYM', 'SYM', 
             'YMM', 'MMM', 'TMM', 'SMM', 'YTM', 'MTM', 'TTM', 'STM', 'YSM', 
             'MSM', 'TSM', 'SSM', 'YYT', 'MYT', 'TYT', 'SYT', 'YMT', 'MMT', 
             'TMT', 'SMT', 'YTT', 'MTT', 'TTT', 'STT', 'YST', 'MST', 'TST', 
             'SST', 'YYS', 'MYS', 'TYS', 'SYS', 'YMS', 'MMS', 'TMS', 'SMS', 
             'YTS', 'MTS', 'TTS', 'STS', 'YSS', 'MSS', 'TSS', 'SSS', 'HYY', 
             'NYY', 'QYY', 'WYY', 'HMY', 'NMY', 'QMY', 'WMY', 'HTY', 'NTY', 
             'QTY', 'WTY', 'HSY', 'NSY', 'QSY', 'WSY', 'HYM', 'NYM', 'QYM', 
             'WYM', 'HMM', 'NMM', 'QMM', 'WMM', 'HTM', 'NTM', 'QTM', 'WTM', 
             'HSM', 'NSM', 'QSM', 'WSM', 'HYT', 'NYT', 'QYT', 'WYT', 'HMT', 
             'NMT', 'QMT', 'WMT', 'HTT', 'NTT', 'QTT', 'WTT', 'HST', 'NST', 
             'QST', 'WST', 'HYS', 'NYS', 'QYS', 'WYS', 'HMS', 'NMS', 'QMS', 
             'WMS', 'HTS', 'NTS', 'QTS', 'WTS', 'HSS', 'NSS', 'QSS', 'WSS', 
             'RYY', 'KYY', 'RMY', 'KMY', 'RTY', 'KTY', 'RSY', 'KSY', 'RYM', 
             'KYM', 'RMM', 'KMM', 'RTM', 'KTM', 'RSM', 'KSM', 'RYT', 'KYT', 
             'RMT', 'KMT', 'RTT', 'KTT', 'RST', 'KST', 'RYS', 'KYS', 'RMS', 
             'KMS', 'RTS', 'KTS', 'RSS', 'KSS', 'DYY', 'EYY', 'DMY', 'EMY', 
             'DTY', 'ETY', 'DSY', 'ESY', 'DYM', 'EYM', 'DMM', 'EMM', 'DTM', 
             'ETM', 'DSM', 'ESM', 'DYT', 'EYT', 'DMT', 'EMT', 'DTT', 'ETT', 
             'DST', 'EST', 'DYS', 'EYS', 'DMS', 'EMS', 'DTS', 'ETS', 'DSS', 
             'ESS', 'CYY', 'CMY', 'CTY', 'CSY', 'CYM', 'CMM', 'CTM', 'CSM', 
             'CYT', 'CMT', 'CTT', 'CST', 'CYS', 'CMS', 'CTS', 'CSS', 'AHY', 
             'GHY', 'VHY', 'ANY', 'GNY', 'VNY', 'AQY', 'GQY', 'VQY', 'AWY', 
             'GWY', 'VWY', 'AHM', 'GHM', 'VHM', 'ANM', 'GNM', 'VNM', 'AQM', 
             'GQM', 'VQM', 'AWM', 'GWM', 'VWM', 'AHT', 'GHT', 'VHT', 'ANT', 
             'GNT', 'VNT', 'AQT', 'GQT', 'VQT', 'AWT', 'GWT', 'VWT', 'AHS', 
             'GHS', 'VHS', 'ANS', 'GNS', 'VNS', 'AQS', 'GQS', 'VQS', 'AWS', 
             'GWS', 'VWS', 'IHY', 'LHY', 'FHY', 'PHY', 'INY', 'LNY', 'FNY', 
             'PNY', 'IQY', 'LQY', 'FQY', 'PQY', 'IWY', 'LWY', 'FWY', 'PWY', 
             'IHM', 'LHM', 'FHM', 'PHM', 'INM', 'LNM', 'FNM', 'PNM', 'IQM', 
             'LQM', 'FQM', 'PQM', 'IWM', 'LWM', 'FWM', 'PWM', 'IHT', 'LHT', 
             'FHT', 'PHT', 'INT', 'LNT', 'FNT', 'PNT', 'IQT', 'LQT', 'FQT', 
             'PQT', 'IWT', 'LWT', 'FWT', 'PWT', 'IHS', 'LHS', 'FHS', 'PHS', 
             'INS', 'LNS', 'FNS', 'PNS', 'IQS', 'LQS', 'FQS', 'PQS', 'IWS', 
             'LWS', 'FWS', 'PWS', 'YHY', 'MHY', 'THY', 'SHY', 'YNY', 'MNY', 
             'TNY', 'SNY', 'YQY', 'MQY', 'TQY', 'SQY', 'YWY', 'MWY', 'TWY', 
             'SWY', 'YHM', 'MHM', 'THM', 'SHM', 'YNM', 'MNM', 'TNM', 'SNM', 
             'YQM', 'MQM', 'TQM', 'SQM', 'YWM', 'MWM', 'TWM', 'SWM', 'YHT', 
             'MHT', 'THT', 'SHT', 'YNT', 'MNT', 'TNT', 'SNT', 'YQT', 'MQT', 
             'TQT', 'SQT', 'YWT', 'MWT', 'TWT', 'SWT', 'YHS', 'MHS', 'THS', 
             'SHS', 'YNS', 'MNS', 'TNS', 'SNS', 'YQS', 'MQS', 'TQS', 'SQS', 
             'YWS', 'MWS', 'TWS', 'SWS', 'HHY', 'NHY', 'QHY', 'WHY', 'HNY', 
             'NNY', 'QNY', 'WNY', 'HQY', 'NQY', 'QQY', 'WQY', 'HWY', 'NWY', 
             'QWY', 'WWY', 'HHM', 'NHM', 'QHM', 'WHM', 'HNM', 'NNM', 'QNM', 
             'WNM', 'HQM', 'NQM', 'QQM', 'WQM', 'HWM', 'NWM', 'QWM', 'WWM', 
             'HHT', 'NHT', 'QHT', 'WHT', 'HNT', 'NNT', 'QNT', 'WNT', 'HQT', 
             'NQT', 'QQT', 'WQT', 'HWT', 'NWT', 'QWT', 'WWT', 'HHS', 'NHS', 
             'QHS', 'WHS', 'HNS', 'NNS', 'QNS', 'WNS', 'HQS', 'NQS', 'QQS', 
             'WQS', 'HWS', 'NWS', 'QWS', 'WWS', 'RHY', 'KHY', 'RNY', 'KNY', 
             'RQY', 'KQY', 'RWY', 'KWY', 'RHM', 'KHM', 'RNM', 'KNM', 'RQM', 
             'KQM', 'RWM', 'KWM', 'RHT', 'KHT', 'RNT', 'KNT', 'RQT', 'KQT', 
             'RWT', 'KWT', 'RHS', 'KHS', 'RNS', 'KNS', 'RQS', 'KQS', 'RWS', 
             'KWS', 'DHY', 'EHY', 'DNY', 'ENY', 'DQY', 'EQY', 'DWY', 'EWY', 
             'DHM', 'EHM', 'DNM', 'ENM', 'DQM', 'EQM', 'DWM', 'EWM', 'DHT', 
             'EHT', 'DNT', 'ENT', 'DQT', 'EQT', 'DWT', 'EWT', 'DHS', 'EHS', 
             'DNS', 'ENS', 'DQS', 'EQS', 'DWS', 'EWS', 'CHY', 'CNY', 'CQY', 
             'CWY', 'CHM', 'CNM', 'CQM', 'CWM', 'CHT', 'CNT', 'CQT', 'CWT', 
             'CHS', 'CNS', 'CQS', 'CWS', 'ARY', 'GRY', 'VRY', 'AKY', 'GKY', 
             'VKY', 'ARM', 'GRM', 'VRM', 'AKM', 'GKM', 'VKM', 'ART', 'GRT', 
             'VRT', 'AKT', 'GKT', 'VKT', 'ARS', 'GRS', 'VRS', 'AKS', 'GKS', 
             'VKS', 'IRY', 'LRY', 'FRY', 'PRY', 'IKY', 'LKY', 'FKY', 'PKY', 
             'IRM', 'LRM', 'FRM', 'PRM', 'IKM', 'LKM', 'FKM', 'PKM', 'IRT', 
             'LRT', 'FRT', 'PRT', 'IKT', 'LKT', 'FKT', 'PKT', 'IRS', 'LRS', 
             'FRS', 'PRS', 'IKS', 'LKS', 'FKS', 'PKS', 'YRY', 'MRY', 'TRY', 
             'SRY', 'YKY', 'MKY', 'TKY', 'SKY', 'YRM', 'MRM', 'TRM', 'SRM', 
             'YKM', 'MKM', 'TKM', 'SKM', 'YRT', 'MRT', 'TRT', 'SRT', 'YKT', 
             'MKT', 'TKT', 'SKT', 'YRS', 'MRS', 'TRS', 'SRS', 'YKS', 'MKS', 
             'TKS', 'SKS', 'HRY', 'NRY', 'QRY', 'WRY', 'HKY', 'NKY', 'QKY', 
             'WKY', 'HRM', 'NRM', 'QRM', 'WRM', 'HKM', 'NKM', 'QKM', 'WKM', 
             'HRT', 'NRT', 'QRT', 'WRT', 'HKT', 'NKT', 'QKT', 'WKT', 'HRS', 
             'NRS', 'QRS', 'WRS', 'HKS', 'NKS', 'QKS', 'WKS', 'RRY', 'KRY', 
             'RKY', 'KKY', 'RRM', 'KRM', 'RKM', 'KKM', 'RRT', 'KRT', 'RKT', 
             'KKT', 'RRS', 'KRS', 'RKS', 'KKS', 'DRY', 'ERY', 'DKY', 'EKY', 
             'DRM', 'ERM', 'DKM', 'EKM', 'DRT', 'ERT', 'DKT', 'EKT', 'DRS', 
             'ERS', 'DKS', 'EKS', 'CRY', 'CKY', 'CRM', 'CKM', 'CRT', 'CKT', 
             'CRS', 'CKS', 'ADY', 'GDY', 'VDY', 'AEY', 'GEY', 'VEY', 'ADM', 
             'GDM', 'VDM', 'AEM', 'GEM', 'VEM', 'ADT', 'GDT', 'VDT', 'AET', 
             'GET', 'VET', 'ADS', 'GDS', 'VDS', 'AES', 'GES', 'VES', 'IDY', 
             'LDY', 'FDY', 'PDY', 'IEY', 'LEY', 'FEY', 'PEY', 'IDM', 'LDM', 
             'FDM', 'PDM', 'IEM', 'LEM', 'FEM', 'PEM', 'IDT', 'LDT', 'FDT', 
             'PDT', 'IET', 'LET', 'FET', 'PET', 'IDS', 'LDS', 'FDS', 'PDS', 
             'IES', 'LES', 'FES', 'PES', 'YDY', 'MDY', 'TDY', 'SDY', 'YEY', 
             'MEY', 'TEY', 'SEY', 'YDM', 'MDM', 'TDM', 'SDM', 'YEM', 'MEM', 
             'TEM', 'SEM', 'YDT', 'MDT', 'TDT', 'SDT', 'YET', 'MET', 'TET', 
             'SET', 'YDS', 'MDS', 'TDS', 'SDS', 'YES', 'MES', 'TES', 'SES', 
             'HDY', 'NDY', 'QDY', 'WDY', 'HEY', 'NEY', 'QEY', 'WEY', 'HDM', 
             'NDM', 'QDM', 'WDM', 'HEM', 'NEM', 'QEM', 'WEM', 'HDT', 'NDT', 
             'QDT', 'WDT', 'HET', 'NET', 'QET', 'WET', 'HDS', 'NDS', 'QDS', 
             'WDS', 'HES', 'NES', 'QES', 'WES', 'RDY', 'KDY', 'REY', 'KEY', 
             'RDM', 'KDM', 'REM', 'KEM', 'RDT', 'KDT', 'RET', 'KET', 'RDS', 
             'KDS', 'RES', 'KES', 'DDY', 'EDY', 'DEY', 'EEY', 'DDM', 'EDM', 
             'DEM', 'EEM', 'DDT', 'EDT', 'DET', 'EET', 'DDS', 'EDS', 'DES', 
             'EES', 'CDY', 'CEY', 'CDM', 'CEM', 'CDT', 'CET', 'CDS', 'CES', 
             'ACY', 'GCY', 'VCY', 'ACM', 'GCM', 'VCM', 'ACT', 'GCT', 'VCT', 
             'ACS', 'GCS', 'VCS', 'ICY', 'LCY', 'FCY', 'PCY', 'ICM', 'LCM', 
             'FCM', 'PCM', 'ICT', 'LCT', 'FCT', 'PCT', 'ICS', 'LCS', 'FCS', 
             'PCS', 'YCY', 'MCY', 'TCY', 'SCY', 'YCM', 'MCM', 'TCM', 'SCM', 
             'YCT', 'MCT', 'TCT', 'SCT', 'YCS', 'MCS', 'TCS', 'SCS', 'HCY', 
             'NCY', 'QCY', 'WCY', 'HCM', 'NCM', 'QCM', 'WCM', 'HCT', 'NCT', 
             'QCT', 'WCT', 'HCS', 'NCS', 'QCS', 'WCS', 'RCY', 'KCY', 'RCM', 
             'KCM', 'RCT', 'KCT', 'RCS', 'KCS', 'DCY', 'ECY', 'DCM', 'ECM', 
             'DCT', 'ECT', 'DCS', 'ECS', 'CCY', 'CCM', 'CCT', 'CCS', 'AAH', 
             'GAH', 'VAH', 'AGH', 'GGH', 'VGH', 'AVH', 'GVH', 'VVH', 'AAN', 
             'GAN', 'VAN', 'AGN', 'GGN', 'VGN', 'AVN', 'GVN', 'VVN', 'AAQ', 
             'GAQ', 'VAQ', 'AGQ', 'GGQ', 'VGQ', 'AVQ', 'GVQ', 'VVQ', 'AAW', 
             'GAW', 'VAW', 'AGW', 'GGW', 'VGW', 'AVW', 'GVW', 'VVW', 'IAH', 
             'LAH', 'FAH', 'PAH', 'IGH', 'LGH', 'FGH', 'PGH', 'IVH', 'LVH', 
             'FVH', 'PVH', 'IAN', 'LAN', 'FAN', 'PAN', 'IGN', 'LGN', 'FGN', 
             'PGN', 'IVN', 'LVN', 'FVN', 'PVN', 'IAQ', 'LAQ', 'FAQ', 'PAQ', 
             'IGQ', 'LGQ', 'FGQ', 'PGQ', 'IVQ', 'LVQ', 'FVQ', 'PVQ', 'IAW', 
             'LAW', 'FAW', 'PAW', 'IGW', 'LGW', 'FGW', 'PGW', 'IVW', 'LVW', 
             'FVW', 'PVW', 'YAH', 'MAH', 'TAH', 'SAH', 'YGH', 'MGH', 'TGH', 
             'SGH', 'YVH', 'MVH', 'TVH', 'SVH', 'YAN', 'MAN', 'TAN', 'SAN', 
             'YGN', 'MGN', 'TGN', 'SGN', 'YVN', 'MVN', 'TVN', 'SVN', 'YAQ', 
             'MAQ', 'TAQ', 'SAQ', 'YGQ', 'MGQ', 'TGQ', 'SGQ', 'YVQ', 'MVQ', 
             'TVQ', 'SVQ', 'YAW', 'MAW', 'TAW', 'SAW', 'YGW', 'MGW', 'TGW', 
             'SGW', 'YVW', 'MVW', 'TVW', 'SVW', 'HAH', 'NAH', 'QAH', 'WAH', 
             'HGH', 'NGH', 'QGH', 'WGH', 'HVH', 'NVH', 'QVH', 'WVH', 'HAN', 
             'NAN', 'QAN', 'WAN', 'HGN', 'NGN', 'QGN', 'WGN', 'HVN', 'NVN', 
             'QVN', 'WVN', 'HAQ', 'NAQ', 'QAQ', 'WAQ', 'HGQ', 'NGQ', 'QGQ', 
             'WGQ', 'HVQ', 'NVQ', 'QVQ', 'WVQ', 'HAW', 'NAW', 'QAW', 'WAW', 
             'HGW', 'NGW', 'QGW', 'WGW', 'HVW', 'NVW', 'QVW', 'WVW', 'RAH', 
             'KAH', 'RGH', 'KGH', 'RVH', 'KVH', 'RAN', 'KAN', 'RGN', 'KGN', 
             'RVN', 'KVN', 'RAQ', 'KAQ', 'RGQ', 'KGQ', 'RVQ', 'KVQ', 'RAW', 
             'KAW', 'RGW', 'KGW', 'RVW', 'KVW', 'DAH', 'EAH', 'DGH', 'EGH', 
             'DVH', 'EVH', 'DAN', 'EAN', 'DGN', 'EGN', 'DVN', 'EVN', 'DAQ', 
             'EAQ', 'DGQ', 'EGQ', 'DVQ', 'EVQ', 'DAW', 'EAW', 'DGW', 'EGW', 
             'DVW', 'EVW', 'CAH', 'CGH', 'CVH', 'CAN', 'CGN', 'CVN', 'CAQ', 
             'CGQ', 'CVQ', 'CAW', 'CGW', 'CVW', 'AIH', 'GIH', 'VIH', 'ALH', 
             'GLH', 'VLH', 'AFH', 'GFH', 'VFH', 'APH', 'GPH', 'VPH', 'AIN', 
             'GIN', 'VIN', 'ALN', 'GLN', 'VLN', 'AFN', 'GFN', 'VFN', 'APN', 
             'GPN', 'VPN', 'AIQ', 'GIQ', 'VIQ', 'ALQ', 'GLQ', 'VLQ', 'AFQ', 
             'GFQ', 'VFQ', 'APQ', 'GPQ', 'VPQ', 'AIW', 'GIW', 'VIW', 'ALW', 
             'GLW', 'VLW', 'AFW', 'GFW', 'VFW', 'APW', 'GPW', 'VPW', 'IIH', 
             'LIH', 'FIH', 'PIH', 'ILH', 'LLH', 'FLH', 'PLH', 'IFH', 'LFH', 
             'FFH', 'PFH', 'IPH', 'LPH', 'FPH', 'PPH', 'IIN', 'LIN', 'FIN', 
             'PIN', 'ILN', 'LLN', 'FLN', 'PLN', 'IFN', 'LFN', 'FFN', 'PFN', 
             'IPN', 'LPN', 'FPN', 'PPN', 'IIQ', 'LIQ', 'FIQ', 'PIQ', 'ILQ', 
             'LLQ', 'FLQ', 'PLQ', 'IFQ', 'LFQ', 'FFQ', 'PFQ', 'IPQ', 'LPQ', 
             'FPQ', 'PPQ', 'IIW', 'LIW', 'FIW', 'PIW', 'ILW', 'LLW', 'FLW', 
             'PLW', 'IFW', 'LFW', 'FFW', 'PFW', 'IPW', 'LPW', 'FPW', 'PPW', 
             'YIH', 'MIH', 'TIH', 'SIH', 'YLH', 'MLH', 'TLH', 'SLH', 'YFH', 
             'MFH', 'TFH', 'SFH', 'YPH', 'MPH', 'TPH', 'SPH', 'YIN', 'MIN', 
             'TIN', 'SIN', 'YLN', 'MLN', 'TLN', 'SLN', 'YFN', 'MFN', 'TFN', 
             'SFN', 'YPN', 'MPN', 'TPN', 'SPN', 'YIQ', 'MIQ', 'TIQ', 'SIQ', 
             'YLQ', 'MLQ', 'TLQ', 'SLQ', 'YFQ', 'MFQ', 'TFQ', 'SFQ', 'YPQ', 
             'MPQ', 'TPQ', 'SPQ', 'YIW', 'MIW', 'TIW', 'SIW', 'YLW', 'MLW', 
             'TLW', 'SLW', 'YFW', 'MFW', 'TFW', 'SFW', 'YPW', 'MPW', 'TPW', 
             'SPW', 'HIH', 'NIH', 'QIH', 'WIH', 'HLH', 'NLH', 'QLH', 'WLH', 
             'HFH', 'NFH', 'QFH', 'WFH', 'HPH', 'NPH', 'QPH', 'WPH', 'HIN', 
             'NIN', 'QIN', 'WIN', 'HLN', 'NLN', 'QLN', 'WLN', 'HFN', 'NFN', 
             'QFN', 'WFN', 'HPN', 'NPN', 'QPN', 'WPN', 'HIQ', 'NIQ', 'QIQ', 
             'WIQ', 'HLQ', 'NLQ', 'QLQ', 'WLQ', 'HFQ', 'NFQ', 'QFQ', 'WFQ', 
             'HPQ', 'NPQ', 'QPQ', 'WPQ', 'HIW', 'NIW', 'QIW', 'WIW', 'HLW', 
             'NLW', 'QLW', 'WLW', 'HFW', 'NFW', 'QFW', 'WFW', 'HPW', 'NPW', 
             'QPW', 'WPW', 'RIH', 'KIH', 'RLH', 'KLH', 'RFH', 'KFH', 'RPH', 
             'KPH', 'RIN', 'KIN', 'RLN', 'KLN', 'RFN', 'KFN', 'RPN', 'KPN', 
             'RIQ', 'KIQ', 'RLQ', 'KLQ', 'RFQ', 'KFQ', 'RPQ', 'KPQ', 'RIW', 
             'KIW', 'RLW', 'KLW', 'RFW', 'KFW', 'RPW', 'KPW', 'DIH', 'EIH', 
             'DLH', 'ELH', 'DFH', 'EFH', 'DPH', 'EPH', 'DIN', 'EIN', 'DLN', 
             'ELN', 'DFN', 'EFN', 'DPN', 'EPN', 'DIQ', 'EIQ', 'DLQ', 'ELQ', 
             'DFQ', 'EFQ', 'DPQ', 'EPQ', 'DIW', 'EIW', 'DLW', 'ELW', 'DFW', 
             'EFW', 'DPW', 'EPW', 'CIH', 'CLH', 'CFH', 'CPH', 'CIN', 'CLN', 
             'CFN', 'CPN', 'CIQ', 'CLQ', 'CFQ', 'CPQ', 'CIW', 'CLW', 'CFW', 
             'CPW', 'AYH', 'GYH', 'VYH', 'AMH', 'GMH', 'VMH', 'ATH', 'GTH', 
             'VTH', 'ASH', 'GSH', 'VSH', 'AYN', 'GYN', 'VYN', 'AMN', 'GMN', 
             'VMN', 'ATN', 'GTN', 'VTN', 'ASN', 'GSN', 'VSN', 'AYQ', 'GYQ', 
             'VYQ', 'AMQ', 'GMQ', 'VMQ', 'ATQ', 'GTQ', 'VTQ', 'ASQ', 'GSQ', 
             'VSQ', 'AYW', 'GYW', 'VYW', 'AMW', 'GMW', 'VMW', 'ATW', 'GTW', 
             'VTW', 'ASW', 'GSW', 'VSW', 'IYH', 'LYH', 'FYH', 'PYH', 'IMH', 
             'LMH', 'FMH', 'PMH', 'ITH', 'LTH', 'FTH', 'PTH', 'ISH', 'LSH', 
             'FSH', 'PSH', 'IYN', 'LYN', 'FYN', 'PYN', 'IMN', 'LMN', 'FMN', 
             'PMN', 'ITN', 'LTN', 'FTN', 'PTN', 'ISN', 'LSN', 'FSN', 'PSN', 
             'IYQ', 'LYQ', 'FYQ', 'PYQ', 'IMQ', 'LMQ', 'FMQ', 'PMQ', 'ITQ', 
             'LTQ', 'FTQ', 'PTQ', 'ISQ', 'LSQ', 'FSQ', 'PSQ', 'IYW', 'LYW', 
             'FYW', 'PYW', 'IMW', 'LMW', 'FMW', 'PMW', 'ITW', 'LTW', 'FTW', 
             'PTW', 'ISW', 'LSW', 'FSW', 'PSW', 'YYH', 'MYH', 'TYH', 'SYH', 
             'YMH', 'MMH', 'TMH', 'SMH', 'YTH', 'MTH', 'TTH', 'STH', 'YSH', 
             'MSH', 'TSH', 'SSH', 'YYN', 'MYN', 'TYN', 'SYN', 'YMN', 'MMN', 
             'TMN', 'SMN', 'YTN', 'MTN', 'TTN', 'STN', 'YSN', 'MSN', 'TSN', 
             'SSN', 'YYQ', 'MYQ', 'TYQ', 'SYQ', 'YMQ', 'MMQ', 'TMQ', 'SMQ', 
             'YTQ', 'MTQ', 'TTQ', 'STQ', 'YSQ', 'MSQ', 'TSQ', 'SSQ', 'YYW', 
             'MYW', 'TYW', 'SYW', 'YMW', 'MMW', 'TMW', 'SMW', 'YTW', 'MTW', 
             'TTW', 'STW', 'YSW', 'MSW', 'TSW', 'SSW', 'HYH', 'NYH', 'QYH', 
             'WYH', 'HMH', 'NMH', 'QMH', 'WMH', 'HTH', 'NTH', 'QTH', 'WTH', 
             'HSH', 'NSH', 'QSH', 'WSH', 'HYN', 'NYN', 'QYN', 'WYN', 'HMN', 
             'NMN', 'QMN', 'WMN', 'HTN', 'NTN', 'QTN', 'WTN', 'HSN', 'NSN', 
             'QSN', 'WSN', 'HYQ', 'NYQ', 'QYQ', 'WYQ', 'HMQ', 'NMQ', 'QMQ', 
             'WMQ', 'HTQ', 'NTQ', 'QTQ', 'WTQ', 'HSQ', 'NSQ', 'QSQ', 'WSQ', 
             'HYW', 'NYW', 'QYW', 'WYW', 'HMW', 'NMW', 'QMW', 'WMW', 'HTW', 
             'NTW', 'QTW', 'WTW', 'HSW', 'NSW', 'QSW', 'WSW', 'RYH', 'KYH', 
             'RMH', 'KMH', 'RTH', 'KTH', 'RSH', 'KSH', 'RYN', 'KYN', 'RMN', 
             'KMN', 'RTN', 'KTN', 'RSN', 'KSN', 'RYQ', 'KYQ', 'RMQ', 'KMQ', 
             'RTQ', 'KTQ', 'RSQ', 'KSQ', 'RYW', 'KYW', 'RMW', 'KMW', 'RTW', 
             'KTW', 'RSW', 'KSW', 'DYH', 'EYH', 'DMH', 'EMH', 'DTH', 'ETH', 
             'DSH', 'ESH', 'DYN', 'EYN', 'DMN', 'EMN', 'DTN', 'ETN', 'DSN', 
             'ESN', 'DYQ', 'EYQ', 'DMQ', 'EMQ', 'DTQ', 'ETQ', 'DSQ', 'ESQ', 
             'DYW', 'EYW', 'DMW', 'EMW', 'DTW', 'ETW', 'DSW', 'ESW', 'CYH', 
             'CMH', 'CTH', 'CSH', 'CYN', 'CMN', 'CTN', 'CSN', 'CYQ', 'CMQ', 
             'CTQ', 'CSQ', 'CYW', 'CMW', 'CTW', 'CSW', 'AHH', 'GHH', 'VHH', 
             'ANH', 'GNH', 'VNH', 'AQH', 'GQH', 'VQH', 'AWH', 'GWH', 'VWH', 
             'AHN', 'GHN', 'VHN', 'ANN', 'GNN', 'VNN', 'AQN', 'GQN', 'VQN', 
             'AWN', 'GWN', 'VWN', 'AHQ', 'GHQ', 'VHQ', 'ANQ', 'GNQ', 'VNQ', 
             'AQQ', 'GQQ', 'VQQ', 'AWQ', 'GWQ', 'VWQ', 'AHW', 'GHW', 'VHW', 
             'ANW', 'GNW', 'VNW', 'AQW', 'GQW', 'VQW', 'AWW', 'GWW', 'VWW', 
             'IHH', 'LHH', 'FHH', 'PHH', 'INH', 'LNH', 'FNH', 'PNH', 'IQH', 
             'LQH', 'FQH', 'PQH', 'IWH', 'LWH', 'FWH', 'PWH', 'IHN', 'LHN', 
             'FHN', 'PHN', 'INN', 'LNN', 'FNN', 'PNN', 'IQN', 'LQN', 'FQN', 
             'PQN', 'IWN', 'LWN', 'FWN', 'PWN', 'IHQ', 'LHQ', 'FHQ', 'PHQ', 
             'INQ', 'LNQ', 'FNQ', 'PNQ', 'IQQ', 'LQQ', 'FQQ', 'PQQ', 'IWQ', 
             'LWQ', 'FWQ', 'PWQ', 'IHW', 'LHW', 'FHW', 'PHW', 'INW', 'LNW', 
             'FNW', 'PNW', 'IQW', 'LQW', 'FQW', 'PQW', 'IWW', 'LWW', 'FWW', 
             'PWW', 'YHH', 'MHH', 'THH', 'SHH', 'YNH', 'MNH', 'TNH', 'SNH', 
             'YQH', 'MQH', 'TQH', 'SQH', 'YWH', 'MWH', 'TWH', 'SWH', 'YHN', 
             'MHN', 'THN', 'SHN', 'YNN', 'MNN', 'TNN', 'SNN', 'YQN', 'MQN', 
             'TQN', 'SQN', 'YWN', 'MWN', 'TWN', 'SWN', 'YHQ', 'MHQ', 'THQ', 
             'SHQ', 'YNQ', 'MNQ', 'TNQ', 'SNQ', 'YQQ', 'MQQ', 'TQQ', 'SQQ', 
             'YWQ', 'MWQ', 'TWQ', 'SWQ', 'YHW', 'MHW', 'THW', 'SHW', 'YNW', 
             'MNW', 'TNW', 'SNW', 'YQW', 'MQW', 'TQW', 'SQW', 'YWW', 'MWW', 
             'TWW', 'SWW', 'HHH', 'NHH', 'QHH', 'WHH', 'HNH', 'NNH', 'QNH', 
             'WNH', 'HQH', 'NQH', 'QQH', 'WQH', 'HWH', 'NWH', 'QWH', 'WWH', 
             'HHN', 'NHN', 'QHN', 'WHN', 'HNN', 'NNN', 'QNN', 'WNN', 'HQN', 
             'NQN', 'QQN', 'WQN', 'HWN', 'NWN', 'QWN', 'WWN', 'HHQ', 'NHQ', 
             'QHQ', 'WHQ', 'HNQ', 'NNQ', 'QNQ', 'WNQ', 'HQQ', 'NQQ', 'QQQ', 
             'WQQ', 'HWQ', 'NWQ', 'QWQ', 'WWQ', 'HHW', 'NHW', 'QHW', 'WHW', 
             'HNW', 'NNW', 'QNW', 'WNW', 'HQW', 'NQW', 'QQW', 'WQW', 'HWW', 
             'NWW', 'QWW', 'WWW', 'RHH', 'KHH', 'RNH', 'KNH', 'RQH', 'KQH', 
             'RWH', 'KWH', 'RHN', 'KHN', 'RNN', 'KNN', 'RQN', 'KQN', 'RWN', 
             'KWN', 'RHQ', 'KHQ', 'RNQ', 'KNQ', 'RQQ', 'KQQ', 'RWQ', 'KWQ', 
             'RHW', 'KHW', 'RNW', 'KNW', 'RQW', 'KQW', 'RWW', 'KWW', 'DHH', 
             'EHH', 'DNH', 'ENH', 'DQH', 'EQH', 'DWH', 'EWH', 'DHN', 'EHN', 
             'DNN', 'ENN', 'DQN', 'EQN', 'DWN', 'EWN', 'DHQ', 'EHQ', 'DNQ', 
             'ENQ', 'DQQ', 'EQQ', 'DWQ', 'EWQ', 'DHW', 'EHW', 'DNW', 'ENW', 
             'DQW', 'EQW', 'DWW', 'EWW', 'CHH', 'CNH', 'CQH', 'CWH', 'CHN', 
             'CNN', 'CQN', 'CWN', 'CHQ', 'CNQ', 'CQQ', 'CWQ', 'CHW', 'CNW', 
             'CQW', 'CWW', 'ARH', 'GRH', 'VRH', 'AKH', 'GKH', 'VKH', 'ARN', 
             'GRN', 'VRN', 'AKN', 'GKN', 'VKN', 'ARQ', 'GRQ', 'VRQ', 'AKQ', 
             'GKQ', 'VKQ', 'ARW', 'GRW', 'VRW', 'AKW', 'GKW', 'VKW', 'IRH', 
             'LRH', 'FRH', 'PRH', 'IKH', 'LKH', 'FKH', 'PKH', 'IRN', 'LRN', 
             'FRN', 'PRN', 'IKN', 'LKN', 'FKN', 'PKN', 'IRQ', 'LRQ', 'FRQ', 
             'PRQ', 'IKQ', 'LKQ', 'FKQ', 'PKQ', 'IRW', 'LRW', 'FRW', 'PRW', 
             'IKW', 'LKW', 'FKW', 'PKW', 'YRH', 'MRH', 'TRH', 'SRH', 'YKH', 
             'MKH', 'TKH', 'SKH', 'YRN', 'MRN', 'TRN', 'SRN', 'YKN', 'MKN', 
             'TKN', 'SKN', 'YRQ', 'MRQ', 'TRQ', 'SRQ', 'YKQ', 'MKQ', 'TKQ', 
             'SKQ', 'YRW', 'MRW', 'TRW', 'SRW', 'YKW', 'MKW', 'TKW', 'SKW', 
             'HRH', 'NRH', 'QRH', 'WRH', 'HKH', 'NKH', 'QKH', 'WKH', 'HRN', 
             'NRN', 'QRN', 'WRN', 'HKN', 'NKN', 'QKN', 'WKN', 'HRQ', 'NRQ', 
             'QRQ', 'WRQ', 'HKQ', 'NKQ', 'QKQ', 'WKQ', 'HRW', 'NRW', 'QRW', 
             'WRW', 'HKW', 'NKW', 'QKW', 'WKW', 'RRH', 'KRH', 'RKH', 'KKH', 
             'RRN', 'KRN', 'RKN', 'KKN', 'RRQ', 'KRQ', 'RKQ', 'KKQ', 'RRW', 
             'KRW', 'RKW', 'KKW', 'DRH', 'ERH', 'DKH', 'EKH', 'DRN', 'ERN', 
             'DKN', 'EKN', 'DRQ', 'ERQ', 'DKQ', 'EKQ', 'DRW', 'ERW', 'DKW', 
             'EKW', 'CRH', 'CKH', 'CRN', 'CKN', 'CRQ', 'CKQ', 'CRW', 'CKW', 
             'ADH', 'GDH', 'VDH', 'AEH', 'GEH', 'VEH', 'ADN', 'GDN', 'VDN', 
             'AEN', 'GEN', 'VEN', 'ADQ', 'GDQ', 'VDQ', 'AEQ', 'GEQ', 'VEQ', 
             'ADW', 'GDW', 'VDW', 'AEW', 'GEW', 'VEW', 'IDH', 'LDH', 'FDH', 
             'PDH', 'IEH', 'LEH', 'FEH', 'PEH', 'IDN', 'LDN', 'FDN', 'PDN', 
             'IEN', 'LEN', 'FEN', 'PEN', 'IDQ', 'LDQ', 'FDQ', 'PDQ', 'IEQ', 
             'LEQ', 'FEQ', 'PEQ', 'IDW', 'LDW', 'FDW', 'PDW', 'IEW', 'LEW', 
             'FEW', 'PEW', 'YDH', 'MDH', 'TDH', 'SDH', 'YEH', 'MEH', 'TEH', 
             'SEH', 'YDN', 'MDN', 'TDN', 'SDN', 'YEN', 'MEN', 'TEN', 'SEN', 
             'YDQ', 'MDQ', 'TDQ', 'SDQ', 'YEQ', 'MEQ', 'TEQ', 'SEQ', 'YDW', 
             'MDW', 'TDW', 'SDW', 'YEW', 'MEW', 'TEW', 'SEW', 'HDH', 'NDH', 
             'QDH', 'WDH', 'HEH', 'NEH', 'QEH', 'WEH', 'HDN', 'NDN', 'QDN', 
             'WDN', 'HEN', 'NEN', 'QEN', 'WEN', 'HDQ', 'NDQ', 'QDQ', 'WDQ', 
             'HEQ', 'NEQ', 'QEQ', 'WEQ', 'HDW', 'NDW', 'QDW', 'WDW', 'HEW', 
             'NEW', 'QEW', 'WEW', 'RDH', 'KDH', 'REH', 'KEH', 'RDN', 'KDN', 
             'REN', 'KEN', 'RDQ', 'KDQ', 'REQ', 'KEQ', 'RDW', 'KDW', 'REW', 
             'KEW', 'DDH', 'EDH', 'DEH', 'EEH', 'DDN', 'EDN', 'DEN', 'EEN', 
             'DDQ', 'EDQ', 'DEQ', 'EEQ', 'DDW', 'EDW', 'DEW', 'EEW', 'CDH', 
             'CEH', 'CDN', 'CEN', 'CDQ', 'CEQ', 'CDW', 'CEW', 'ACH', 'GCH', 
             'VCH', 'ACN', 'GCN', 'VCN', 'ACQ', 'GCQ', 'VCQ', 'ACW', 'GCW', 
             'VCW', 'ICH', 'LCH', 'FCH', 'PCH', 'ICN', 'LCN', 'FCN', 'PCN', 
             'ICQ', 'LCQ', 'FCQ', 'PCQ', 'ICW', 'LCW', 'FCW', 'PCW', 'YCH', 
             'MCH', 'TCH', 'SCH', 'YCN', 'MCN', 'TCN', 'SCN', 'YCQ', 'MCQ', 
             'TCQ', 'SCQ', 'YCW', 'MCW', 'TCW', 'SCW', 'HCH', 'NCH', 'QCH', 
             'WCH', 'HCN', 'NCN', 'QCN', 'WCN', 'HCQ', 'NCQ', 'QCQ', 'WCQ', 
             'HCW', 'NCW', 'QCW', 'WCW', 'RCH', 'KCH', 'RCN', 'KCN', 'RCQ', 
             'KCQ', 'RCW', 'KCW', 'DCH', 'ECH', 'DCN', 'ECN', 'DCQ', 'ECQ', 
             'DCW', 'ECW', 'CCH', 'CCN', 'CCQ', 'CCW', 'AAR', 'GAR', 'VAR', 
             'AGR', 'GGR', 'VGR', 'AVR', 'GVR', 'VVR', 'AAK', 'GAK', 'VAK', 
             'AGK', 'GGK', 'VGK', 'AVK', 'GVK', 'VVK', 'IAR', 'LAR', 'FAR', 
             'PAR', 'IGR', 'LGR', 'FGR', 'PGR', 'IVR', 'LVR', 'FVR', 'PVR', 
             'IAK', 'LAK', 'FAK', 'PAK', 'IGK', 'LGK', 'FGK', 'PGK', 'IVK', 
             'LVK', 'FVK', 'PVK', 'YAR', 'MAR', 'TAR', 'SAR', 'YGR', 'MGR', 
             'TGR', 'SGR', 'YVR', 'MVR', 'TVR', 'SVR', 'YAK', 'MAK', 'TAK', 
             'SAK', 'YGK', 'MGK', 'TGK', 'SGK', 'YVK', 'MVK', 'TVK', 'SVK', 
             'HAR', 'NAR', 'QAR', 'WAR', 'HGR', 'NGR', 'QGR', 'WGR', 'HVR', 
             'NVR', 'QVR', 'WVR', 'HAK', 'NAK', 'QAK', 'WAK', 'HGK', 'NGK', 
             'QGK', 'WGK', 'HVK', 'NVK', 'QVK', 'WVK', 'RAR', 'KAR', 'RGR', 
             'KGR', 'RVR', 'KVR', 'RAK', 'KAK', 'RGK', 'KGK', 'RVK', 'KVK', 
             'DAR', 'EAR', 'DGR', 'EGR', 'DVR', 'EVR', 'DAK', 'EAK', 'DGK', 
             'EGK', 'DVK', 'EVK', 'CAR', 'CGR', 'CVR', 'CAK', 'CGK', 'CVK', 
             'AIR', 'GIR', 'VIR', 'ALR', 'GLR', 'VLR', 'AFR', 'GFR', 'VFR', 
             'APR', 'GPR', 'VPR', 'AIK', 'GIK', 'VIK', 'ALK', 'GLK', 'VLK', 
             'AFK', 'GFK', 'VFK', 'APK', 'GPK', 'VPK', 'IIR', 'LIR', 'FIR', 
             'PIR', 'ILR', 'LLR', 'FLR', 'PLR', 'IFR', 'LFR', 'FFR', 'PFR', 
             'IPR', 'LPR', 'FPR', 'PPR', 'IIK', 'LIK', 'FIK', 'PIK', 'ILK', 
             'LLK', 'FLK', 'PLK', 'IFK', 'LFK', 'FFK', 'PFK', 'IPK', 'LPK', 
             'FPK', 'PPK', 'YIR', 'MIR', 'TIR', 'SIR', 'YLR', 'MLR', 'TLR', 
             'SLR', 'YFR', 'MFR', 'TFR', 'SFR', 'YPR', 'MPR', 'TPR', 'SPR', 
             'YIK', 'MIK', 'TIK', 'SIK', 'YLK', 'MLK', 'TLK', 'SLK', 'YFK', 
             'MFK', 'TFK', 'SFK', 'YPK', 'MPK', 'TPK', 'SPK', 'HIR', 'NIR', 
             'QIR', 'WIR', 'HLR', 'NLR', 'QLR', 'WLR', 'HFR', 'NFR', 'QFR', 
             'WFR', 'HPR', 'NPR', 'QPR', 'WPR', 'HIK', 'NIK', 'QIK', 'WIK', 
             'HLK', 'NLK', 'QLK', 'WLK', 'HFK', 'NFK', 'QFK', 'WFK', 'HPK', 
             'NPK', 'QPK', 'WPK', 'RIR', 'KIR', 'RLR', 'KLR', 'RFR', 'KFR', 
             'RPR', 'KPR', 'RIK', 'KIK', 'RLK', 'KLK', 'RFK', 'KFK', 'RPK', 
             'KPK', 'DIR', 'EIR', 'DLR', 'ELR', 'DFR', 'EFR', 'DPR', 'EPR', 
             'DIK', 'EIK', 'DLK', 'ELK', 'DFK', 'EFK', 'DPK', 'EPK', 'CIR', 
             'CLR', 'CFR', 'CPR', 'CIK', 'CLK', 'CFK', 'CPK', 'AYR', 'GYR', 
             'VYR', 'AMR', 'GMR', 'VMR', 'ATR', 'GTR', 'VTR', 'ASR', 'GSR', 
             'VSR', 'AYK', 'GYK', 'VYK', 'AMK', 'GMK', 'VMK', 'ATK', 'GTK', 
             'VTK', 'ASK', 'GSK', 'VSK', 'IYR', 'LYR', 'FYR', 'PYR', 'IMR', 
             'LMR', 'FMR', 'PMR', 'ITR', 'LTR', 'FTR', 'PTR', 'ISR', 'LSR', 
             'FSR', 'PSR', 'IYK', 'LYK', 'FYK', 'PYK', 'IMK', 'LMK', 'FMK', 
             'PMK', 'ITK', 'LTK', 'FTK', 'PTK', 'ISK', 'LSK', 'FSK', 'PSK', 
             'YYR', 'MYR', 'TYR', 'SYR', 'YMR', 'MMR', 'TMR', 'SMR', 'YTR', 
             'MTR', 'TTR', 'STR', 'YSR', 'MSR', 'TSR', 'SSR', 'YYK', 'MYK', 
             'TYK', 'SYK', 'YMK', 'MMK', 'TMK', 'SMK', 'YTK', 'MTK', 'TTK', 
             'STK', 'YSK', 'MSK', 'TSK', 'SSK', 'HYR', 'NYR', 'QYR', 'WYR', 
             'HMR', 'NMR', 'QMR', 'WMR', 'HTR', 'NTR', 'QTR', 'WTR', 'HSR', 
             'NSR', 'QSR', 'WSR', 'HYK', 'NYK', 'QYK', 'WYK', 'HMK', 'NMK', 
             'QMK', 'WMK', 'HTK', 'NTK', 'QTK', 'WTK', 'HSK', 'NSK', 'QSK', 
             'WSK', 'RYR', 'KYR', 'RMR', 'KMR', 'RTR', 'KTR', 'RSR', 'KSR', 
             'RYK', 'KYK', 'RMK', 'KMK', 'RTK', 'KTK', 'RSK', 'KSK', 'DYR', 
             'EYR', 'DMR', 'EMR', 'DTR', 'ETR', 'DSR', 'ESR', 'DYK', 'EYK', 
             'DMK', 'EMK', 'DTK', 'ETK', 'DSK', 'ESK', 'CYR', 'CMR', 'CTR', 
             'CSR', 'CYK', 'CMK', 'CTK', 'CSK', 'AHR', 'GHR', 'VHR', 'ANR', 
             'GNR', 'VNR', 'AQR', 'GQR', 'VQR', 'AWR', 'GWR', 'VWR', 'AHK', 
             'GHK', 'VHK', 'ANK', 'GNK', 'VNK', 'AQK', 'GQK', 'VQK', 'AWK', 
             'GWK', 'VWK', 'IHR', 'LHR', 'FHR', 'PHR', 'INR', 'LNR', 'FNR', 
             'PNR', 'IQR', 'LQR', 'FQR', 'PQR', 'IWR', 'LWR', 'FWR', 'PWR', 
             'IHK', 'LHK', 'FHK', 'PHK', 'INK', 'LNK', 'FNK', 'PNK', 'IQK', 
             'LQK', 'FQK', 'PQK', 'IWK', 'LWK', 'FWK', 'PWK', 'YHR', 'MHR', 
             'THR', 'SHR', 'YNR', 'MNR', 'TNR', 'SNR', 'YQR', 'MQR', 'TQR', 
             'SQR', 'YWR', 'MWR', 'TWR', 'SWR', 'YHK', 'MHK', 'THK', 'SHK', 
             'YNK', 'MNK', 'TNK', 'SNK', 'YQK', 'MQK', 'TQK', 'SQK', 'YWK', 
             'MWK', 'TWK', 'SWK', 'HHR', 'NHR', 'QHR', 'WHR', 'HNR', 'NNR', 
             'QNR', 'WNR', 'HQR', 'NQR', 'QQR', 'WQR', 'HWR', 'NWR', 'QWR', 
             'WWR', 'HHK', 'NHK', 'QHK', 'WHK', 'HNK', 'NNK', 'QNK', 'WNK', 
             'HQK', 'NQK', 'QQK', 'WQK', 'HWK', 'NWK', 'QWK', 'WWK', 'RHR', 
             'KHR', 'RNR', 'KNR', 'RQR', 'KQR', 'RWR', 'KWR', 'RHK', 'KHK', 
             'RNK', 'KNK', 'RQK', 'KQK', 'RWK', 'KWK', 'DHR', 'EHR', 'DNR', 
             'ENR', 'DQR', 'EQR', 'DWR', 'EWR', 'DHK', 'EHK', 'DNK', 'ENK', 
             'DQK', 'EQK', 'DWK', 'EWK', 'CHR', 'CNR', 'CQR', 'CWR', 'CHK', 
             'CNK', 'CQK', 'CWK', 'ARR', 'GRR', 'VRR', 'AKR', 'GKR', 'VKR', 
             'ARK', 'GRK', 'VRK', 'AKK', 'GKK', 'VKK', 'IRR', 'LRR', 'FRR', 
             'PRR', 'IKR', 'LKR', 'FKR', 'PKR', 'IRK', 'LRK', 'FRK', 'PRK', 
             'IKK', 'LKK', 'FKK', 'PKK', 'YRR', 'MRR', 'TRR', 'SRR', 'YKR', 
             'MKR', 'TKR', 'SKR', 'YRK', 'MRK', 'TRK', 'SRK', 'YKK', 'MKK', 
             'TKK', 'SKK', 'HRR', 'NRR', 'QRR', 'WRR', 'HKR', 'NKR', 'QKR', 
             'WKR', 'HRK', 'NRK', 'QRK', 'WRK', 'HKK', 'NKK', 'QKK', 'WKK', 
             'RRR', 'KRR', 'RKR', 'KKR', 'RRK', 'KRK', 'RKK', 'KKK', 'DRR', 
             'ERR', 'DKR', 'EKR', 'DRK', 'ERK', 'DKK', 'EKK', 'CRR', 'CKR', 
             'CRK', 'CKK', 'ADR', 'GDR', 'VDR', 'AER', 'GER', 'VER', 'ADK', 
             'GDK', 'VDK', 'AEK', 'GEK', 'VEK', 'IDR', 'LDR', 'FDR', 'PDR', 
             'IER', 'LER', 'FER', 'PER', 'IDK', 'LDK', 'FDK', 'PDK', 'IEK', 
             'LEK', 'FEK', 'PEK', 'YDR', 'MDR', 'TDR', 'SDR', 'YER', 'MER', 
             'TER', 'SER', 'YDK', 'MDK', 'TDK', 'SDK', 'YEK', 'MEK', 'TEK', 
             'SEK', 'HDR', 'NDR', 'QDR', 'WDR', 'HER', 'NER', 'QER', 'WER', 
             'HDK', 'NDK', 'QDK', 'WDK', 'HEK', 'NEK', 'QEK', 'WEK', 'RDR', 
             'KDR', 'RER', 'KER', 'RDK', 'KDK', 'REK', 'KEK', 'DDR', 'EDR', 
             'DER', 'EER', 'DDK', 'EDK', 'DEK', 'EEK', 'CDR', 'CER', 'CDK', 
             'CEK', 'ACR', 'GCR', 'VCR', 'ACK', 'GCK', 'VCK', 'ICR', 'LCR', 
             'FCR', 'PCR', 'ICK', 'LCK', 'FCK', 'PCK', 'YCR', 'MCR', 'TCR', 
             'SCR', 'YCK', 'MCK', 'TCK', 'SCK', 'HCR', 'NCR', 'QCR', 'WCR', 
             'HCK', 'NCK', 'QCK', 'WCK', 'RCR', 'KCR', 'RCK', 'KCK', 'DCR', 
             'ECR', 'DCK', 'ECK', 'CCR', 'CCK', 'AAD', 'GAD', 'VAD', 'AGD', 
             'GGD', 'VGD', 'AVD', 'GVD', 'VVD', 'AAE', 'GAE', 'VAE', 'AGE', 
             'GGE', 'VGE', 'AVE', 'GVE', 'VVE', 'IAD', 'LAD', 'FAD', 'PAD', 
             'IGD', 'LGD', 'FGD', 'PGD', 'IVD', 'LVD', 'FVD', 'PVD', 'IAE', 
             'LAE', 'FAE', 'PAE', 'IGE', 'LGE', 'FGE', 'PGE', 'IVE', 'LVE', 
             'FVE', 'PVE', 'YAD', 'MAD', 'TAD', 'SAD', 'YGD', 'MGD', 'TGD', 
             'SGD', 'YVD', 'MVD', 'TVD', 'SVD', 'YAE', 'MAE', 'TAE', 'SAE', 
             'YGE', 'MGE', 'TGE', 'SGE', 'YVE', 'MVE', 'TVE', 'SVE', 'HAD', 
             'NAD', 'QAD', 'WAD', 'HGD', 'NGD', 'QGD', 'WGD', 'HVD', 'NVD', 
             'QVD', 'WVD', 'HAE', 'NAE', 'QAE', 'WAE', 'HGE', 'NGE', 'QGE', 
             'WGE', 'HVE', 'NVE', 'QVE', 'WVE', 'RAD', 'KAD', 'RGD', 'KGD', 
             'RVD', 'KVD', 'RAE', 'KAE', 'RGE', 'KGE', 'RVE', 'KVE', 'DAD', 
             'EAD', 'DGD', 'EGD', 'DVD', 'EVD', 'DAE', 'EAE', 'DGE', 'EGE', 
             'DVE', 'EVE', 'CAD', 'CGD', 'CVD', 'CAE', 'CGE', 'CVE', 'AID', 
             'GID', 'VID', 'ALD', 'GLD', 'VLD', 'AFD', 'GFD', 'VFD', 'APD', 
             'GPD', 'VPD', 'AIE', 'GIE', 'VIE', 'ALE', 'GLE', 'VLE', 'AFE', 
             'GFE', 'VFE', 'APE', 'GPE', 'VPE', 'IID', 'LID', 'FID', 'PID', 
             'ILD', 'LLD', 'FLD', 'PLD', 'IFD', 'LFD', 'FFD', 'PFD', 'IPD', 
             'LPD', 'FPD', 'PPD', 'IIE', 'LIE', 'FIE', 'PIE', 'ILE', 'LLE', 
             'FLE', 'PLE', 'IFE', 'LFE', 'FFE', 'PFE', 'IPE', 'LPE', 'FPE', 
             'PPE', 'YID', 'MID', 'TID', 'SID', 'YLD', 'MLD', 'TLD', 'SLD', 
             'YFD', 'MFD', 'TFD', 'SFD', 'YPD', 'MPD', 'TPD', 'SPD', 'YIE', 
             'MIE', 'TIE', 'SIE', 'YLE', 'MLE', 'TLE', 'SLE', 'YFE', 'MFE', 
             'TFE', 'SFE', 'YPE', 'MPE', 'TPE', 'SPE', 'HID', 'NID', 'QID', 
             'WID', 'HLD', 'NLD', 'QLD', 'WLD', 'HFD', 'NFD', 'QFD', 'WFD', 
             'HPD', 'NPD', 'QPD', 'WPD', 'HIE', 'NIE', 'QIE', 'WIE', 'HLE', 
             'NLE', 'QLE', 'WLE', 'HFE', 'NFE', 'QFE', 'WFE', 'HPE', 'NPE', 
             'QPE', 'WPE', 'RID', 'KID', 'RLD', 'KLD', 'RFD', 'KFD', 'RPD', 
             'KPD', 'RIE', 'KIE', 'RLE', 'KLE', 'RFE', 'KFE', 'RPE', 'KPE', 
             'DID', 'EID', 'DLD', 'ELD', 'DFD', 'EFD', 'DPD', 'EPD', 'DIE', 
             'EIE', 'DLE', 'ELE', 'DFE', 'EFE', 'DPE', 'EPE', 'CID', 'CLD', 
             'CFD', 'CPD', 'CIE', 'CLE', 'CFE', 'CPE', 'AYD', 'GYD', 'VYD', 
             'AMD', 'GMD', 'VMD', 'ATD', 'GTD', 'VTD', 'ASD', 'GSD', 'VSD', 
             'AYE', 'GYE', 'VYE', 'AME', 'GME', 'VME', 'ATE', 'GTE', 'VTE', 
             'ASE', 'GSE', 'VSE', 'IYD', 'LYD', 'FYD', 'PYD', 'IMD', 'LMD', 
             'FMD', 'PMD', 'ITD', 'LTD', 'FTD', 'PTD', 'ISD', 'LSD', 'FSD', 
             'PSD', 'IYE', 'LYE', 'FYE', 'PYE', 'IME', 'LME', 'FME', 'PME', 
             'ITE', 'LTE', 'FTE', 'PTE', 'ISE', 'LSE', 'FSE', 'PSE', 'YYD', 
             'MYD', 'TYD', 'SYD', 'YMD', 'MMD', 'TMD', 'SMD', 'YTD', 'MTD', 
             'TTD', 'STD', 'YSD', 'MSD', 'TSD', 'SSD', 'YYE', 'MYE', 'TYE', 
             'SYE', 'YME', 'MME', 'TME', 'SME', 'YTE', 'MTE', 'TTE', 'STE', 
             'YSE', 'MSE', 'TSE', 'SSE', 'HYD', 'NYD', 'QYD', 'WYD', 'HMD', 
             'NMD', 'QMD', 'WMD', 'HTD', 'NTD', 'QTD', 'WTD', 'HSD', 'NSD', 
             'QSD', 'WSD', 'HYE', 'NYE', 'QYE', 'WYE', 'HME', 'NME', 'QME', 
             'WME', 'HTE', 'NTE', 'QTE', 'WTE', 'HSE', 'NSE', 'QSE', 'WSE', 
             'RYD', 'KYD', 'RMD', 'KMD', 'RTD', 'KTD', 'RSD', 'KSD', 'RYE', 
             'KYE', 'RME', 'KME', 'RTE', 'KTE', 'RSE', 'KSE', 'DYD', 'EYD', 
             'DMD', 'EMD', 'DTD', 'ETD', 'DSD', 'ESD', 'DYE', 'EYE', 'DME', 
             'EME', 'DTE', 'ETE', 'DSE', 'ESE', 'CYD', 'CMD', 'CTD', 'CSD', 
             'CYE', 'CME', 'CTE', 'CSE', 'AHD', 'GHD', 'VHD', 'AND', 'GND', 
             'VND', 'AQD', 'GQD', 'VQD', 'AWD', 'GWD', 'VWD', 'AHE', 'GHE', 
             'VHE', 'ANE', 'GNE', 'VNE', 'AQE', 'GQE', 'VQE', 'AWE', 'GWE', 
             'VWE', 'IHD', 'LHD', 'FHD', 'PHD', 'IND', 'LND', 'FND', 'PND', 
             'IQD', 'LQD', 'FQD', 'PQD', 'IWD', 'LWD', 'FWD', 'PWD', 'IHE', 
             'LHE', 'FHE', 'PHE', 'INE', 'LNE', 'FNE', 'PNE', 'IQE', 'LQE', 
             'FQE', 'PQE', 'IWE', 'LWE', 'FWE', 'PWE', 'YHD', 'MHD', 'THD', 
             'SHD', 'YND', 'MND', 'TND', 'SND', 'YQD', 'MQD', 'TQD', 'SQD', 
             'YWD', 'MWD', 'TWD', 'SWD', 'YHE', 'MHE', 'THE', 'SHE', 'YNE', 
             'MNE', 'TNE', 'SNE', 'YQE', 'MQE', 'TQE', 'SQE', 'YWE', 'MWE', 
             'TWE', 'SWE', 'HHD', 'NHD', 'QHD', 'WHD', 'HND', 'NND', 'QND', 
             'WND', 'HQD', 'NQD', 'QQD', 'WQD', 'HWD', 'NWD', 'QWD', 'WWD', 
             'HHE', 'NHE', 'QHE', 'WHE', 'HNE', 'NNE', 'QNE', 'WNE', 'HQE', 
             'NQE', 'QQE', 'WQE', 'HWE', 'NWE', 'QWE', 'WWE', 'RHD', 'KHD', 
             'RND', 'KND', 'RQD', 'KQD', 'RWD', 'KWD', 'RHE', 'KHE', 'RNE', 
             'KNE', 'RQE', 'KQE', 'RWE', 'KWE', 'DHD', 'EHD', 'DND', 'END', 
             'DQD', 'EQD', 'DWD', 'EWD', 'DHE', 'EHE', 'DNE', 'ENE', 'DQE', 
             'EQE', 'DWE', 'EWE', 'CHD', 'CND', 'CQD', 'CWD', 'CHE', 'CNE', 
             'CQE', 'CWE', 'ARD', 'GRD', 'VRD', 'AKD', 'GKD', 'VKD', 'ARE', 
             'GRE', 'VRE', 'AKE', 'GKE', 'VKE', 'IRD', 'LRD', 'FRD', 'PRD', 
             'IKD', 'LKD', 'FKD', 'PKD', 'IRE', 'LRE', 'FRE', 'PRE', 'IKE', 
             'LKE', 'FKE', 'PKE', 'YRD', 'MRD', 'TRD', 'SRD', 'YKD', 'MKD', 
             'TKD', 'SKD', 'YRE', 'MRE', 'TRE', 'SRE', 'YKE', 'MKE', 'TKE', 
             'SKE', 'HRD', 'NRD', 'QRD', 'WRD', 'HKD', 'NKD', 'QKD', 'WKD', 
             'HRE', 'NRE', 'QRE', 'WRE', 'HKE', 'NKE', 'QKE', 'WKE', 'RRD', 
             'KRD', 'RKD', 'KKD', 'RRE', 'KRE', 'RKE', 'KKE', 'DRD', 'ERD', 
             'DKD', 'EKD', 'DRE', 'ERE', 'DKE', 'EKE', 'CRD', 'CKD', 'CRE', 
             'CKE', 'ADD', 'GDD', 'VDD', 'AED', 'GED', 'VED', 'ADE', 'GDE', 
             'VDE', 'AEE', 'GEE', 'VEE', 'IDD', 'LDD', 'FDD', 'PDD', 'IED', 
             'LED', 'FED', 'PED', 'IDE', 'LDE', 'FDE', 'PDE', 'IEE', 'LEE', 
             'FEE', 'PEE', 'YDD', 'MDD', 'TDD', 'SDD', 'YED', 'MED', 'TED', 
             'SED', 'YDE', 'MDE', 'TDE', 'SDE', 'YEE', 'MEE', 'TEE', 'SEE', 
             'HDD', 'NDD', 'QDD', 'WDD', 'HED', 'NED', 'QED', 'WED', 'HDE', 
             'NDE', 'QDE', 'WDE', 'HEE', 'NEE', 'QEE', 'WEE', 'RDD', 'KDD', 
             'RED', 'KED', 'RDE', 'KDE', 'REE', 'KEE', 'DDD', 'EDD', 'DED', 
             'EED', 'DDE', 'EDE', 'DEE', 'EEE', 'CDD', 'CED', 'CDE', 'CEE', 
             'ACD', 'GCD', 'VCD', 'ACE', 'GCE', 'VCE', 'ICD', 'LCD', 'FCD', 
             'PCD', 'ICE', 'LCE', 'FCE', 'PCE', 'YCD', 'MCD', 'TCD', 'SCD', 
             'YCE', 'MCE', 'TCE', 'SCE', 'HCD', 'NCD', 'QCD', 'WCD', 'HCE', 
             'NCE', 'QCE', 'WCE', 'RCD', 'KCD', 'RCE', 'KCE', 'DCD', 'ECD', 
             'DCE', 'ECE', 'CCD', 'CCE', 'AAC', 'GAC', 'VAC', 'AGC', 'GGC', 
             'VGC', 'AVC', 'GVC', 'VVC', 'IAC', 'LAC', 'FAC', 'PAC', 'IGC', 
             'LGC', 'FGC', 'PGC', 'IVC', 'LVC', 'FVC', 'PVC', 'YAC', 'MAC', 
             'TAC', 'SAC', 'YGC', 'MGC', 'TGC', 'SGC', 'YVC', 'MVC', 'TVC', 
             'SVC', 'HAC', 'NAC', 'QAC', 'WAC', 'HGC', 'NGC', 'QGC', 'WGC', 
             'HVC', 'NVC', 'QVC', 'WVC', 'RAC', 'KAC', 'RGC', 'KGC', 'RVC', 
             'KVC', 'DAC', 'EAC', 'DGC', 'EGC', 'DVC', 'EVC', 'CAC', 'CGC', 
             'CVC', 'AIC', 'GIC', 'VIC', 'ALC', 'GLC', 'VLC', 'AFC', 'GFC', 
             'VFC', 'APC', 'GPC', 'VPC', 'IIC', 'LIC', 'FIC', 'PIC', 'ILC', 
             'LLC', 'FLC', 'PLC', 'IFC', 'LFC', 'FFC', 'PFC', 'IPC', 'LPC', 
             'FPC', 'PPC', 'YIC', 'MIC', 'TIC', 'SIC', 'YLC', 'MLC', 'TLC', 
             'SLC', 'YFC', 'MFC', 'TFC', 'SFC', 'YPC', 'MPC', 'TPC', 'SPC', 
             'HIC', 'NIC', 'QIC', 'WIC', 'HLC', 'NLC', 'QLC', 'WLC', 'HFC', 
             'NFC', 'QFC', 'WFC', 'HPC', 'NPC', 'QPC', 'WPC', 'RIC', 'KIC', 
             'RLC', 'KLC', 'RFC', 'KFC', 'RPC', 'KPC', 'DIC', 'EIC', 'DLC', 
             'ELC', 'DFC', 'EFC', 'DPC', 'EPC', 'CIC', 'CLC', 'CFC', 'CPC', 
             'AYC', 'GYC', 'VYC', 'AMC', 'GMC', 'VMC', 'ATC', 'GTC', 'VTC', 
             'ASC', 'GSC', 'VSC', 'IYC', 'LYC', 'FYC', 'PYC', 'IMC', 'LMC', 
             'FMC', 'PMC', 'ITC', 'LTC', 'FTC', 'PTC', 'ISC', 'LSC', 'FSC', 
             'PSC', 'YYC', 'MYC', 'TYC', 'SYC', 'YMC', 'MMC', 'TMC', 'SMC', 
             'YTC', 'MTC', 'TTC', 'STC', 'YSC', 'MSC', 'TSC', 'SSC', 'HYC', 
             'NYC', 'QYC', 'WYC', 'HMC', 'NMC', 'QMC', 'WMC', 'HTC', 'NTC', 
             'QTC', 'WTC', 'HSC', 'NSC', 'QSC', 'WSC', 'RYC', 'KYC', 'RMC', 
             'KMC', 'RTC', 'KTC', 'RSC', 'KSC', 'DYC', 'EYC', 'DMC', 'EMC', 
             'DTC', 'ETC', 'DSC', 'ESC', 'CYC', 'CMC', 'CTC', 'CSC', 'AHC', 
             'GHC', 'VHC', 'ANC', 'GNC', 'VNC', 'AQC', 'GQC', 'VQC', 'AWC', 
             'GWC', 'VWC', 'IHC', 'LHC', 'FHC', 'PHC', 'INC', 'LNC', 'FNC', 
             'PNC', 'IQC', 'LQC', 'FQC', 'PQC', 'IWC', 'LWC', 'FWC', 'PWC', 
             'YHC', 'MHC', 'THC', 'SHC', 'YNC', 'MNC', 'TNC', 'SNC', 'YQC', 
             'MQC', 'TQC', 'SQC', 'YWC', 'MWC', 'TWC', 'SWC', 'HHC', 'NHC', 
             'QHC', 'WHC', 'HNC', 'NNC', 'QNC', 'WNC', 'HQC', 'NQC', 'QQC', 
             'WQC', 'HWC', 'NWC', 'QWC', 'WWC', 'RHC', 'KHC', 'RNC', 'KNC', 
             'RQC', 'KQC', 'RWC', 'KWC', 'DHC', 'EHC', 'DNC', 'ENC', 'DQC', 
             'EQC', 'DWC', 'EWC', 'CHC', 'CNC', 'CQC', 'CWC', 'ARC', 'GRC', 
             'VRC', 'AKC', 'GKC', 'VKC', 'IRC', 'LRC', 'FRC', 'PRC', 'IKC', 
             'LKC', 'FKC', 'PKC', 'YRC', 'MRC', 'TRC', 'SRC', 'YKC', 'MKC', 
             'TKC', 'SKC', 'HRC', 'NRC', 'QRC', 'WRC', 'HKC', 'NKC', 'QKC', 
             'WKC', 'RRC', 'KRC', 'RKC', 'KKC', 'DRC', 'ERC', 'DKC', 'EKC', 
             'CRC', 'CKC', 'ADC', 'GDC', 'VDC', 'AEC', 'GEC', 'VEC', 'IDC', 
             'LDC', 'FDC', 'PDC', 'IEC', 'LEC', 'FEC', 'PEC', 'YDC', 'MDC', 
             'TDC', 'SDC', 'YEC', 'MEC', 'TEC', 'SEC', 'HDC', 'NDC', 'QDC', 
             'WDC', 'HEC', 'NEC', 'QEC', 'WEC', 'RDC', 'KDC', 'REC', 'KEC', 
             'DDC', 'EDC', 'DEC', 'EEC', 'CDC', 'CEC', 'ACC', 'GCC', 'VCC', 
             'ICC', 'LCC', 'FCC', 'PCC', 'YCC', 'MCC', 'TCC', 'SCC', 'HCC', 
             'NCC', 'QCC', 'WCC', 'RCC', 'KCC', 'DCC', 'ECC', 'CCC')
  
  CTIndex = c(1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 
              1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 
              1, 1, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 
              2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 
              2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 3, 
              3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 
              3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 
              3, 3, 3, 3, 3, 3, 3, 3, 3, 4, 4, 4, 4, 
              4, 4, 4, 4, 4, 4, 4, 4, 4, 4, 4, 4, 4, 
              4, 4, 4, 4, 4, 4, 4, 4, 4, 4, 4, 4, 4, 
              4, 4, 4, 4, 4, 4, 5, 5, 5, 5, 5, 5, 5, 
              5, 5, 5, 5, 5, 5, 5, 5, 5, 5, 5, 6, 6, 
              6, 6, 6, 6, 6, 6, 6, 6, 6, 6, 6, 6, 6, 
              6, 6, 6, 7, 7, 7, 7, 7, 7, 7, 7, 7, 8, 
              8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 
              8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 
              8, 8, 8, 8, 8, 8, 8, 8, 8, 9, 9, 9, 9, 
              9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 
              9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 
              9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 
              9, 9, 9, 9, 9,
              10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 
              10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 
              10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 
              10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 
              10, 10, 10, 10, 10, 11, 11, 11, 11, 11, 11, 
              11, 11, 11, 11, 11, 11, 11, 11, 11, 11, 11, 
              11, 11, 11, 11, 11, 11, 11, 11, 11, 11, 11, 
              11, 11, 11, 11, 11, 11, 11, 11, 11, 11, 11, 
              11, 11, 11, 11, 11, 11, 11, 11, 11, 12, 12, 
              12, 12, 12, 12, 12, 12, 12, 12, 12, 12, 12, 
              12, 12, 12, 12, 12, 12, 12, 12, 12, 12, 12, 
              13, 13, 13, 13, 13, 13, 13, 13, 13, 13, 13, 
              13, 13, 13, 13, 13, 13, 13, 13, 13, 13, 13, 
              13, 13, 14, 14, 14, 14, 14, 14, 14, 14, 14, 
              14, 14, 14, 15, 15, 15, 15, 15, 15, 15, 15, 
              15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 
              15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 
              15, 15, 15, 15, 15, 15, 16, 16, 16, 16, 16, 
              16, 16, 16, 16, 16, 16, 16, 16, 16, 16, 16, 
              16, 16, 16, 16, 16, 16, 16, 16, 16, 16, 16, 
              16, 16, 16, 16, 16, 16, 16, 16, 16, 16, 16, 
              16, 16, 16, 16, 16, 16, 16, 16, 16, 16, 17, 
              17, 17, 17, 17, 17, 17, 17, 17, 17, 17, 17, 
              17, 17, 17, 17, 17, 17, 17, 17, 17, 17, 17, 
              17, 17, 17, 17, 17, 17, 17, 17, 17, 17, 17, 
              17, 17, 17, 17, 17, 17, 17, 17, 17, 17, 17, 
              17, 17, 17, 18, 18, 18, 18, 18, 18, 18, 18, 
              18, 18, 18, 18, 18, 18, 18, 18, 18, 18, 18, 
              18, 18, 18, 18, 18, 18, 18, 18, 18, 18, 18, 
              18, 18, 18, 18, 18, 18, 18, 18, 18, 18, 18, 
              18, 18, 18, 18, 18, 18, 18, 19, 19, 19, 19, 
              19, 19, 19, 19, 19, 19, 19, 19, 19, 19, 19, 
              19, 19, 19, 19, 19, 19, 19, 19, 19, 20, 20, 
              20, 20, 20, 20, 20, 20, 20, 20, 20, 20, 20, 
              20, 20, 20, 20, 20, 20, 20, 20, 20, 20, 20, 
              21, 21, 21, 21, 21, 21, 21, 21, 21, 21, 21, 
              21, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 
              22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 
              22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 
              22, 22, 22, 22, 23, 23, 23, 23, 23, 23, 23, 
              23, 23, 23, 23, 23, 23, 23, 23, 23, 23, 23, 
              23, 23, 23, 23, 23, 23, 23, 23, 23, 23, 23, 
              23, 23, 23, 23, 23, 23, 23, 23, 23, 23, 23, 
              23, 23, 23, 23, 23, 23, 23, 23, 24, 24, 24, 
              24, 24, 24, 24, 24, 24, 24, 24, 24, 24, 24, 
              24, 24, 24, 24, 24, 24, 24, 24, 24, 24, 24, 
              24, 24, 24, 24, 24, 24, 24, 24, 24, 24, 24, 
              24, 24, 24, 24, 24, 24, 24, 24, 24, 24, 24, 
              24, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 
              25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 
              25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 
              25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 
              25, 25, 25, 25, 25, 26, 26, 26, 26, 26, 26, 
              26, 26, 26, 26, 26, 26, 26, 26, 26, 26, 26, 
              26, 26, 26, 26, 26, 26, 26, 27, 27, 27, 27, 
              27, 27, 27, 27, 27, 27, 27, 27, 27, 27, 27, 
              27, 27, 27, 27, 27, 27, 27, 27, 27, 28, 28, 
              28, 28, 28, 28, 28, 28, 28, 28, 28, 28, 29, 
              29, 29, 29, 29, 29, 29, 29, 29, 29, 29, 29, 
              29, 29, 29, 29, 29, 29, 30, 30, 30, 30, 30, 
              30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 
              30, 30, 30, 30, 30, 30, 30, 30, 31, 31, 31, 
              31, 31, 31, 31, 31, 31, 31, 31, 31, 31, 31, 
              31, 31, 31, 31, 31, 31, 31, 31, 31, 31, 32, 
              32, 32, 32, 32, 32, 32, 32, 32, 32, 32, 32, 
              32, 32, 32, 32, 32, 32, 32, 32, 32, 32, 32, 
              32, 33, 33, 33, 33, 33, 33, 33, 33, 33, 33, 
              33, 33, 34, 34, 34, 34, 34, 34, 34, 34, 34, 
              34, 34, 34, 35, 35, 35, 35, 35, 35, 36, 36, 
              36, 36, 36, 36, 36, 36, 36, 36, 36, 36, 36, 
              36, 36, 36, 36, 36, 37, 37, 37, 37, 37, 37, 
              37, 37, 37, 37, 37, 37, 37, 37, 37, 37, 37, 
              37, 37, 37, 37, 37, 37, 37, 38, 38, 38, 38, 
              38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 
              38, 38, 38, 38, 38, 38, 38, 38, 38, 39, 39, 
              39, 39, 39, 39, 39, 39, 39, 39, 39, 39, 39, 
              39, 39, 39, 39, 39, 39, 39, 39, 39, 39, 39, 
              40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 
              40, 41, 41, 41, 41, 41, 41, 41, 41, 41, 41, 
              41, 41, 42, 42, 42, 42, 42, 42, 43, 43, 43, 
              43, 43, 43, 43, 43, 43, 44, 44, 44, 44, 44, 
              44, 44, 44, 44, 44, 44, 44, 45, 45, 45, 45, 
              45, 45, 45, 45, 45, 45, 45, 45, 46, 46, 46, 
              46, 46, 46, 46, 46, 46, 46, 46, 46, 47, 47, 
              47, 47, 47, 47, 48, 48, 48, 48, 48, 48, 49, 
              49, 49, 50, 50, 50, 50, 50, 50, 50, 50, 50, 
              50, 50, 50, 50, 50, 50, 50, 50, 50, 50, 50, 
              50, 50, 50, 50, 50, 50, 50, 50, 50, 50, 50, 
              50, 50, 50, 50, 50, 51, 51, 51, 51, 51, 51, 
              51, 51, 51, 51, 51, 51, 51, 51, 51, 51, 51, 
              51, 51, 51, 51, 51, 51, 51, 51, 51, 51, 51, 
              51, 51, 51, 51, 51, 51, 51, 51, 51, 51, 51, 
              51, 51, 51, 51, 51, 51, 51, 51, 51, 52, 52, 
              52, 52, 52, 52, 52, 52, 52, 52, 52, 52, 52, 
              52, 52, 52, 52, 52, 52, 52, 52, 52, 52, 52, 
              52, 52, 52, 52, 52, 52, 52, 52, 52, 52, 52, 
              52, 52, 52, 52, 52, 52, 52, 52, 52, 52, 52, 
              52, 52, 53, 53, 53, 53, 53, 53, 53, 53, 53, 
              53, 53, 53, 53, 53, 53, 53, 53, 53, 53, 53, 
              53, 53, 53, 53, 53, 53, 53, 53, 53, 53, 53, 
              53, 53, 53, 53, 53, 53, 53, 53, 53, 53, 53, 
              53, 53, 53, 53, 53, 53, 54, 54, 54, 54, 54, 
              54, 54, 54, 54, 54, 54, 54, 54, 54, 54, 54, 
              54, 54, 54, 54, 54, 54, 54, 54, 55, 55, 55, 
              55, 55, 55, 55, 55, 55, 55, 55, 55, 55, 55, 
              55, 55, 55, 55, 55, 55, 55, 55, 55, 55, 56, 
              56, 56, 56, 56, 56, 56, 56, 56, 56, 56, 56, 
              57, 57, 57, 57, 57, 57, 57, 57, 57, 57, 57, 
              57, 57, 57, 57, 57, 57, 57, 57, 57, 57, 57, 
              57, 57, 57, 57, 57, 57, 57, 57, 57, 57, 57, 
              57, 57, 57, 57, 57, 57, 57, 57, 57, 57, 57, 
              57, 57, 57, 57, 58, 58, 58, 58, 58, 58, 58, 
              58, 58, 58, 58, 58, 58, 58, 58, 58, 58, 58, 
              58, 58, 58, 58, 58, 58, 58, 58, 58, 58, 58, 
              58, 58, 58, 58, 58, 58, 58, 58, 58, 58, 58, 
              58, 58, 58, 58, 58, 58, 58, 58, 58, 58, 58, 
              58, 58, 58, 58, 58, 58, 58, 58, 58, 58, 58, 
              58, 58, 59, 59, 59, 59, 59, 59, 59, 59, 59, 
              59, 59, 59, 59, 59, 59, 59, 59, 59, 59, 59, 
              59, 59, 59, 59, 59, 59, 59, 59, 59, 59, 59, 
              59, 59, 59, 59, 59, 59, 59, 59, 59, 59, 59, 
              59, 59, 59, 59, 59, 59, 59, 59, 59, 59, 59, 
              59, 59, 59, 59, 59, 59, 59, 59, 59, 59, 59, 
              60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 
              60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 
              60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 
              60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 
              60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 
              60, 60, 60, 60, 60, 60, 60, 60, 60, 61, 61, 
              61, 61, 61, 61, 61, 61, 61, 61, 61, 61, 61, 
              61, 61, 61, 61, 61, 61, 61, 61, 61, 61, 61, 
              61, 61, 61, 61, 61, 61, 61, 61, 62, 62, 62, 
              62, 62, 62, 62, 62, 62, 62, 62, 62, 62, 62, 
              62, 62, 62, 62, 62, 62, 62, 62, 62, 62, 62, 
              62, 62, 62, 62, 62, 62, 62, 63, 63, 63, 63, 
              63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 
              63, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 
              64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 
              64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 
              64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 
              64, 64, 64, 64, 64, 65, 65, 65, 65, 65, 65, 
              65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 
              65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 
              65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 
              65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 
              65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 
              65, 65, 65, 66, 66, 66, 66, 66, 66, 66, 66, 
              66, 66, 66, 66, 66, 66, 66, 66, 66, 66, 66, 
              66, 66, 66, 66, 66, 66, 66, 66, 66, 66, 66, 
              66, 66, 66, 66, 66, 66, 66, 66, 66, 66, 66, 
              66, 66, 66, 66, 66, 66, 66, 66, 66, 66, 66, 
              66, 66, 66, 66, 66, 66, 66, 66, 66, 66, 66, 
              66, 67, 67, 67, 67, 67, 67, 67, 67, 67, 67, 
              67, 67, 67, 67, 67, 67, 67, 67, 67, 67, 67, 
              67, 67, 67, 67, 67, 67, 67, 67, 67, 67, 67, 
              67, 67, 67, 67, 67, 67, 67, 67, 67, 67, 67, 
              67, 67, 67, 67, 67, 67, 67, 67, 67, 67, 67, 
              67, 67, 67, 67, 67, 67, 67, 67, 67, 67, 68, 
              68, 68, 68, 68, 68, 68, 68, 68, 68, 68, 68, 
              68, 68, 68, 68, 68, 68, 68, 68, 68, 68, 68, 
              68, 68, 68, 68, 68, 68, 68, 68, 68, 69, 69, 
              69, 69, 69, 69, 69, 69, 69, 69, 69, 69, 69, 
              69, 69, 69, 69, 69, 69, 69, 69, 69, 69, 69, 
              69, 69, 69, 69, 69, 69, 69, 69, 70, 70, 70, 
              70, 70, 70, 70, 70, 70, 70, 70, 70, 70, 70, 
              70, 70, 71, 71, 71, 71, 71, 71, 71, 71, 71, 
              71, 71, 71, 71, 71, 71, 71, 71, 71, 71, 71, 
              71, 71, 71, 71, 71, 71, 71, 71, 71, 71, 71, 
              71, 71, 71, 71, 71, 71, 71, 71, 71, 71, 71, 
              71, 71, 71, 71, 71, 71, 72, 72, 72, 72, 72, 
              72, 72, 72, 72, 72, 72, 72, 72, 72, 72, 72, 
              72, 72, 72, 72, 72, 72, 72, 72, 72, 72, 72, 
              72, 72, 72, 72, 72, 72, 72, 72, 72, 72, 72, 
              72, 72, 72, 72, 72, 72, 72, 72, 72, 72, 72, 
              72, 72, 72, 72, 72, 72, 72, 72, 72, 72, 72, 
              72, 72, 72, 72, 73, 73, 73, 73, 73, 73, 73, 
              73, 73, 73, 73, 73, 73, 73, 73, 73, 73, 73, 
              73, 73, 73, 73, 73, 73, 73, 73, 73, 73, 73, 
              73, 73, 73, 73, 73, 73, 73, 73, 73, 73, 73, 
              73, 73, 73, 73, 73, 73, 73, 73, 73, 73, 73, 
              73, 73, 73, 73, 73, 73, 73, 73, 73, 73, 73, 
              73, 73, 74, 74, 74, 74, 74, 74, 74, 74, 74, 
              74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 
              74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 
              74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 
              74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 
              74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 
              75, 75, 75, 75, 75, 75, 75, 75, 75, 75, 75, 
              75, 75, 75, 75, 75, 75, 75, 75, 75, 75, 75, 
              75, 75, 75, 75, 75, 75, 75, 75, 75, 75, 76, 
              76, 76, 76, 76, 76, 76, 76, 76, 76, 76, 76, 
              76, 76, 76, 76, 76, 76, 76, 76, 76, 76, 76, 
              76, 76, 76, 76, 76, 76, 76, 76, 76, 77, 77, 
              77, 77, 77, 77, 77, 77, 77, 77, 77, 77, 77, 
              77, 77, 77, 78, 78, 78, 78, 78, 78, 78, 78, 
              78, 78, 78, 78, 78, 78, 78, 78, 78, 78, 78, 
              78, 78, 78, 78, 78, 79, 79, 79, 79, 79, 79, 
              79, 79, 79, 79, 79, 79, 79, 79, 79, 79, 79, 
              79, 79, 79, 79, 79, 79, 79, 79, 79, 79, 79, 
              79, 79, 79, 79, 80, 80, 80, 80, 80, 80, 80, 
              80, 80, 80, 80, 80, 80, 80, 80, 80, 80, 80, 
              80, 80, 80, 80, 80, 80, 80, 80, 80, 80, 80, 
              80, 80, 80, 81, 81, 81, 81, 81, 81, 81, 81, 
              81, 81, 81, 81, 81, 81, 81, 81, 81, 81, 81, 
              81, 81, 81, 81, 81, 81, 81, 81, 81, 81, 81, 
              81, 81, 82, 82, 82, 82, 82, 82, 82, 82, 82, 
              82, 82, 82, 82, 82, 82, 82, 83, 83, 83, 83, 
              83, 83, 83, 83, 83, 83, 83, 83, 83, 83, 83, 
              83, 84, 84, 84, 84, 84, 84, 84, 84, 85, 85, 
              85, 85, 85, 85, 85, 85, 85, 85, 85, 85, 85, 
              85, 85, 85, 85, 85, 85, 85, 85, 85, 85, 85, 
              86, 86, 86, 86, 86, 86, 86, 86, 86, 86, 86, 
              86, 86, 86, 86, 86, 86, 86, 86, 86, 86, 86, 
              86, 86, 86, 86, 86, 86, 86, 86, 86, 86, 87, 
              87, 87, 87, 87, 87, 87, 87, 87, 87, 87, 87, 
              87, 87, 87, 87, 87, 87, 87, 87, 87, 87, 87, 
              87, 87, 87, 87, 87, 87, 87, 87, 87, 88, 88, 
              88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 
              88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 
              88, 88, 88, 88, 88, 88, 88, 88, 89, 89, 89, 
              89, 89, 89, 89, 89, 89, 89, 89, 89, 89, 89, 
              89, 89, 90, 90, 90, 90, 90, 90, 90, 90, 90, 
              90, 90, 90, 90, 90, 90, 90, 91, 91, 91, 91, 
              91, 91, 91, 91, 92, 92, 92, 92, 92, 92, 92, 
              92, 92, 92, 92, 92, 93, 93, 93, 93, 93, 93, 
              93, 93, 93, 93, 93, 93, 93, 93, 93, 93, 94, 
              94, 94, 94, 94, 94, 94, 94, 94, 94, 94, 94, 
              94, 94, 94, 94, 95, 95, 95, 95, 95, 95, 95, 
              95, 95, 95, 95, 95, 95, 95, 95, 95, 96, 96, 
              96, 96, 96, 96, 96, 96, 97, 97, 97, 97, 97, 
              97, 97, 97, 98, 98, 98, 98, 99, 99, 99, 99, 
              99, 99, 99, 99, 99, 99, 99, 99, 99, 99, 99, 
              99, 99, 99, 99, 99, 99, 99, 99, 99, 99, 99, 
              99, 99, 99, 99, 99, 99, 99, 99, 99, 99,
              100, 100, 100, 100, 100, 100, 100, 100, 100, 
              100, 100, 100, 100, 100, 100, 100, 100, 100, 
              100, 100, 100, 100, 100, 100, 100, 100, 100, 
              100, 100, 100, 100, 100, 100, 100, 100, 100, 
              100, 100, 100, 100, 100, 100, 100, 100, 100, 
              100, 100, 100, 101, 101, 101, 101, 101, 101, 
              101, 101, 101, 101, 101, 101, 101, 101, 101, 
              101, 101, 101, 101, 101, 101, 101, 101, 101, 
              101, 101, 101, 101, 101, 101, 101, 101, 101, 
              101, 101, 101, 101, 101, 101, 101, 101, 101, 
              101, 101, 101, 101, 101, 101, 102, 102, 102, 
              102, 102, 102, 102, 102, 102, 102, 102, 102, 
              102, 102, 102, 102, 102, 102, 102, 102, 102, 
              102, 102, 102, 102, 102, 102, 102, 102, 102, 
              102, 102, 102, 102, 102, 102, 102, 102, 102, 
              102, 102, 102, 102, 102, 102, 102, 102, 102, 
              103, 103, 103, 103, 103, 103, 103, 103, 103, 
              103, 103, 103, 103, 103, 103, 103, 103, 103, 
              103, 103, 103, 103, 103, 103, 104, 104, 104, 
              104, 104, 104, 104, 104, 104, 104, 104, 104, 
              104, 104, 104, 104, 104, 104, 104, 104, 104, 
              104, 104, 104, 105, 105, 105, 105, 105, 105, 
              105, 105, 105, 105, 105, 105, 106, 106, 106, 
              106, 106, 106, 106, 106, 106, 106, 106, 106, 
              106, 106, 106, 106, 106, 106, 106, 106, 106, 
              106, 106, 106, 106, 106, 106, 106, 106, 106, 
              106, 106, 106, 106, 106, 106, 106, 106, 106, 
              106, 106, 106, 106, 106, 106, 106, 106, 106, 
              107, 107, 107, 107, 107, 107, 107, 107, 107, 
              107, 107, 107, 107, 107, 107, 107, 107, 107, 
              107, 107, 107, 107, 107, 107, 107, 107, 107, 
              107, 107, 107, 107, 107, 107, 107, 107, 107, 
              107, 107, 107, 107, 107, 107, 107, 107, 107, 
              107, 107, 107, 107, 107, 107, 107, 107, 107, 
              107, 107, 107, 107, 107, 107, 107, 107, 107, 
              107, 108, 108, 108, 108, 108, 108, 108, 108, 
              108, 108, 108, 108, 108, 108, 108, 108, 108, 
              108, 108, 108, 108, 108, 108, 108, 108, 108, 
              108, 108, 108, 108, 108, 108, 108, 108, 108, 
              108, 108, 108, 108, 108, 108, 108, 108, 108, 
              108, 108, 108, 108, 108, 108, 108, 108, 108, 
              108, 108, 108, 108, 108, 108, 108, 108, 108, 
              108, 108, 109, 109, 109, 109, 109, 109, 109, 
              109, 109, 109, 109, 109, 109, 109, 109, 109, 
              109, 109, 109, 109, 109, 109, 109, 109, 109, 
              109, 109, 109, 109, 109, 109, 109, 109, 109, 
              109, 109, 109, 109, 109, 109, 109, 109, 109, 
              109, 109, 109, 109, 109, 109, 109, 109, 109, 
              109, 109, 109, 109, 109, 109, 109, 109, 109, 
              109, 109, 109, 110, 110, 110, 110, 110, 110, 
              110, 110, 110, 110, 110, 110, 110, 110, 110, 
              110, 110, 110, 110, 110, 110, 110, 110, 110, 
              110, 110, 110, 110, 110, 110, 110, 110, 111, 
              111, 111, 111, 111, 111, 111, 111, 111, 111, 
              111, 111, 111, 111, 111, 111, 111, 111, 111, 
              111, 111, 111, 111, 111, 111, 111, 111, 111, 
              111, 111, 111, 111, 112, 112, 112, 112, 112, 
              112, 112, 112, 112, 112, 112, 112, 112, 112, 
              112, 112, 113, 113, 113, 113, 113, 113, 113, 
              113, 113, 113, 113, 113, 113, 113, 113, 113, 
              113, 113, 113, 113, 113, 113, 113, 113, 113, 
              113, 113, 113, 113, 113, 113, 113, 113, 113, 
              113, 113, 113, 113, 113, 113, 113, 113, 113, 
              113, 113, 113, 113, 113, 114, 114, 114, 114, 
              114, 114, 114, 114, 114, 114, 114, 114, 114, 
              114, 114, 114, 114, 114, 114, 114, 114, 114, 
              114, 114, 114, 114, 114, 114, 114, 114, 114, 
              114, 114, 114, 114, 114, 114, 114, 114, 114, 
              114, 114, 114, 114, 114, 114, 114, 114, 114, 
              114, 114, 114, 114, 114, 114, 114, 114, 114, 
              114, 114, 114, 114, 114, 114, 115, 115, 115, 
              115, 115, 115, 115, 115, 115, 115, 115, 115, 
              115, 115, 115, 115, 115, 115, 115, 115, 115, 
              115, 115, 115, 115, 115, 115, 115, 115, 115, 
              115, 115, 115, 115, 115, 115, 115, 115, 115, 
              115, 115, 115, 115, 115, 115, 115, 115, 115, 
              115, 115, 115, 115, 115, 115, 115, 115, 115, 
              115, 115, 115, 115, 115, 115, 115, 116, 116, 
              116, 116, 116, 116, 116, 116, 116, 116, 116, 
              116, 116, 116, 116, 116, 116, 116, 116, 116, 
              116, 116, 116, 116, 116, 116, 116, 116, 116, 
              116, 116, 116, 116, 116, 116, 116, 116, 116, 
              116, 116, 116, 116, 116, 116, 116, 116, 116, 
              116, 116, 116, 116, 116, 116, 116, 116, 116, 
              116, 116, 116, 116, 116, 116, 116, 116, 117, 
              117, 117, 117, 117, 117, 117, 117, 117, 117, 
              117, 117, 117, 117, 117, 117, 117, 117, 117, 
              117, 117, 117, 117, 117, 117, 117, 117, 117, 
              117, 117, 117, 117, 118, 118, 118, 118, 118, 
              118, 118, 118, 118, 118, 118, 118, 118, 118, 
              118, 118, 118, 118, 118, 118, 118, 118, 118, 
              118, 118, 118, 118, 118, 118, 118, 118, 118, 
              119, 119, 119, 119, 119, 119, 119, 119, 119, 
              119, 119, 119, 119, 119, 119, 119, 120, 120, 
              120, 120, 120, 120, 120, 120, 120, 120, 120, 
              120, 120, 120, 120, 120, 120, 120, 120, 120, 
              120, 120, 120, 120, 120, 120, 120, 120, 120, 
              120, 120, 120, 120, 120, 120, 120, 120, 120, 
              120, 120, 120, 120, 120, 120, 120, 120, 120, 
              120, 121, 121, 121, 121, 121, 121, 121, 121, 
              121, 121, 121, 121, 121, 121, 121, 121, 121, 
              121, 121, 121, 121, 121, 121, 121, 121, 121, 
              121, 121, 121, 121, 121, 121, 121, 121, 121, 
              121, 121, 121, 121, 121, 121, 121, 121, 121, 
              121, 121, 121, 121, 121, 121, 121, 121, 121, 
              121, 121, 121, 121, 121, 121, 121, 121, 121, 
              121, 121, 122, 122, 122, 122, 122, 122, 122, 
              122, 122, 122, 122, 122, 122, 122, 122, 122, 
              122, 122, 122, 122, 122, 122, 122, 122, 122, 
              122, 122, 122, 122, 122, 122, 122, 122, 122, 
              122, 122, 122, 122, 122, 122, 122, 122, 122, 
              122, 122, 122, 122, 122, 122, 122, 122, 122, 
              122, 122, 122, 122, 122, 122, 122, 122, 122, 
              122, 122, 122, 123, 123, 123, 123, 123, 123, 
              123, 123, 123, 123, 123, 123, 123, 123, 123, 
              123, 123, 123, 123, 123, 123, 123, 123, 123, 
              123, 123, 123, 123, 123, 123, 123, 123, 123, 
              123, 123, 123, 123, 123, 123, 123, 123, 123, 
              123, 123, 123, 123, 123, 123, 123, 123, 123, 
              123, 123, 123, 123, 123, 123, 123, 123, 123, 
              123, 123, 123, 123, 124, 124, 124, 124, 124, 
              124, 124, 124, 124, 124, 124, 124, 124, 124, 
              124, 124, 124, 124, 124, 124, 124, 124, 124, 
              124, 124, 124, 124, 124, 124, 124, 124, 124, 
              125, 125, 125, 125, 125, 125, 125, 125, 125, 
              125, 125, 125, 125, 125, 125, 125, 125, 125, 
              125, 125, 125, 125, 125, 125, 125, 125, 125, 
              125, 125, 125, 125, 125, 126, 126, 126, 126, 
              126, 126, 126, 126, 126, 126, 126, 126, 126, 
              126, 126, 126, 127, 127, 127, 127, 127, 127, 
              127, 127, 127, 127, 127, 127, 127, 127, 127, 
              127, 127, 127, 127, 127, 127, 127, 127, 127, 
              128, 128, 128, 128, 128, 128, 128, 128, 128, 
              128, 128, 128, 128, 128, 128, 128, 128, 128, 
              128, 128, 128, 128, 128, 128, 128, 128, 128, 
              128, 128, 128, 128, 128, 129, 129, 129, 129, 
              129, 129, 129, 129, 129, 129, 129, 129, 129, 
              129, 129, 129, 129, 129, 129, 129, 129, 129, 
              129, 129, 129, 129, 129, 129, 129, 129, 129, 
              129, 130, 130, 130, 130, 130, 130, 130, 130, 
              130, 130, 130, 130, 130, 130, 130, 130, 130, 
              130, 130, 130, 130, 130, 130, 130, 130, 130, 
              130, 130, 130, 130, 130, 130, 131, 131, 131, 
              131, 131, 131, 131, 131, 131, 131, 131, 131, 
              131, 131, 131, 131, 132, 132, 132, 132, 132, 
              132, 132, 132, 132, 132, 132, 132, 132, 132, 
              132, 132, 133, 133, 133, 133, 133, 133, 133, 
              133, 134, 134, 134, 134, 134, 134, 134, 134, 
              134, 134, 134, 134, 134, 134, 134, 134, 134, 
              134, 134, 134, 134, 134, 134, 134, 135, 135, 
              135, 135, 135, 135, 135, 135, 135, 135, 135, 
              135, 135, 135, 135, 135, 135, 135, 135, 135, 
              135, 135, 135, 135, 135, 135, 135, 135, 135, 
              135, 135, 135, 136, 136, 136, 136, 136, 136, 
              136, 136, 136, 136, 136, 136, 136, 136, 136, 
              136, 136, 136, 136, 136, 136, 136, 136, 136, 
              136, 136, 136, 136, 136, 136, 136, 136, 137, 
              137, 137, 137, 137, 137, 137, 137, 137, 137, 
              137, 137, 137, 137, 137, 137, 137, 137, 137, 
              137, 137, 137, 137, 137, 137, 137, 137, 137, 
              137, 137, 137, 137, 138, 138, 138, 138, 138, 
              138, 138, 138, 138, 138, 138, 138, 138, 138, 
              138, 138, 139, 139, 139, 139, 139, 139, 139, 
              139, 139, 139, 139, 139, 139, 139, 139, 139, 
              140, 140, 140, 140, 140, 140, 140, 140, 141, 
              141, 141, 141, 141, 141, 141, 141, 141, 141, 
              141, 141, 142, 142, 142, 142, 142, 142, 142, 
              142, 142, 142, 142, 142, 142, 142, 142, 142, 
              143, 143, 143, 143, 143, 143, 143, 143, 143, 
              143, 143, 143, 143, 143, 143, 143, 144, 144, 
              144, 144, 144, 144, 144, 144, 144, 144, 144, 
              144, 144, 144, 144, 144, 145, 145, 145, 145, 
              145, 145, 145, 145, 146, 146, 146, 146, 146, 
              146, 146, 146, 147, 147, 147, 147, 148, 148, 
              148, 148, 148, 148, 148, 148, 148, 148, 148, 
              148, 148, 148, 148, 148, 148, 148, 148, 148, 
              148, 148, 148, 148, 148, 148, 148, 148, 148, 
              148, 148, 148, 148, 148, 148, 148, 149, 149, 
              149, 149, 149, 149, 149, 149, 149, 149, 149, 
              149, 149, 149, 149, 149, 149, 149, 149, 149, 
              149, 149, 149, 149, 149, 149, 149, 149, 149, 
              149, 149, 149, 149, 149, 149, 149, 149, 149, 
              149, 149, 149, 149, 149, 149, 149, 149, 149, 
              149, 150, 150, 150, 150, 150, 150, 150, 150, 
              150, 150, 150, 150, 150, 150, 150, 150, 150, 
              150, 150, 150, 150, 150, 150, 150, 150, 150, 
              150, 150, 150, 150, 150, 150, 150, 150, 150, 
              150, 150, 150, 150, 150, 150, 150, 150, 150, 
              150, 150, 150, 150, 151, 151, 151, 151, 151, 
              151, 151, 151, 151, 151, 151, 151, 151, 151, 
              151, 151, 151, 151, 151, 151, 151, 151, 151, 
              151, 151, 151, 151, 151, 151, 151, 151, 151, 
              151, 151, 151, 151, 151, 151, 151, 151, 151, 
              151, 151, 151, 151, 151, 151, 151, 152, 152, 
              152, 152, 152, 152, 152, 152, 152, 152, 152, 
              152, 152, 152, 152, 152, 152, 152, 152, 152, 
              152, 152, 152, 152, 153, 153, 153, 153, 153, 
              153, 153, 153, 153, 153, 153, 153, 153, 153, 
              153, 153, 153, 153, 153, 153, 153, 153, 153, 
              153, 154, 154, 154, 154, 154, 154, 154, 154, 
              154, 154, 154, 154, 155, 155, 155, 155, 155, 
              155, 155, 155, 155, 155, 155, 155, 155, 155, 
              155, 155, 155, 155, 155, 155, 155, 155, 155, 
              155, 155, 155, 155, 155, 155, 155, 155, 155, 
              155, 155, 155, 155, 155, 155, 155, 155, 155, 
              155, 155, 155, 155, 155, 155, 155, 156, 156, 
              156, 156, 156, 156, 156, 156, 156, 156, 156, 
              156, 156, 156, 156, 156, 156, 156, 156, 156, 
              156, 156, 156, 156, 156, 156, 156, 156, 156, 
              156, 156, 156, 156, 156, 156, 156, 156, 156, 
              156, 156, 156, 156, 156, 156, 156, 156, 156, 
              156, 156, 156, 156, 156, 156, 156, 156, 156, 
              156, 156, 156, 156, 156, 156, 156, 156, 157, 
              157, 157, 157, 157, 157, 157, 157, 157, 157, 
              157, 157, 157, 157, 157, 157, 157, 157, 157, 
              157, 157, 157, 157, 157, 157, 157, 157, 157, 
              157, 157, 157, 157, 157, 157, 157, 157, 157, 
              157, 157, 157, 157, 157, 157, 157, 157, 157, 
              157, 157, 157, 157, 157, 157, 157, 157, 157, 
              157, 157, 157, 157, 157, 157, 157, 157, 157, 
              158, 158, 158, 158, 158, 158, 158, 158, 158, 
              158, 158, 158, 158, 158, 158, 158, 158, 158, 
              158, 158, 158, 158, 158, 158, 158, 158, 158, 
              158, 158, 158, 158, 158, 158, 158, 158, 158, 
              158, 158, 158, 158, 158, 158, 158, 158, 158, 
              158, 158, 158, 158, 158, 158, 158, 158, 158, 
              158, 158, 158, 158, 158, 158, 158, 158, 158, 
              158, 159, 159, 159, 159, 159, 159, 159, 159, 
              159, 159, 159, 159, 159, 159, 159, 159, 159, 
              159, 159, 159, 159, 159, 159, 159, 159, 159, 
              159, 159, 159, 159, 159, 159, 160, 160, 160, 
              160, 160, 160, 160, 160, 160, 160, 160, 160, 
              160, 160, 160, 160, 160, 160, 160, 160, 160, 
              160, 160, 160, 160, 160, 160, 160, 160, 160, 
              160, 160, 161, 161, 161, 161, 161, 161, 161, 
              161, 161, 161, 161, 161, 161, 161, 161, 161, 
              162, 162, 162, 162, 162, 162, 162, 162, 162, 
              162, 162, 162, 162, 162, 162, 162, 162, 162, 
              162, 162, 162, 162, 162, 162, 162, 162, 162, 
              162, 162, 162, 162, 162, 162, 162, 162, 162, 
              162, 162, 162, 162, 162, 162, 162, 162, 162, 
              162, 162, 162, 163, 163, 163, 163, 163, 163, 
              163, 163, 163, 163, 163, 163, 163, 163, 163, 
              163, 163, 163, 163, 163, 163, 163, 163, 163, 
              163, 163, 163, 163, 163, 163, 163, 163, 163, 
              163, 163, 163, 163, 163, 163, 163, 163, 163, 
              163, 163, 163, 163, 163, 163, 163, 163, 163, 
              163, 163, 163, 163, 163, 163, 163, 163, 163, 
              163, 163, 163, 163, 164, 164, 164, 164, 164, 
              164, 164, 164, 164, 164, 164, 164, 164, 164, 
              164, 164, 164, 164, 164, 164, 164, 164, 164, 
              164, 164, 164, 164, 164, 164, 164, 164, 164, 
              164, 164, 164, 164, 164, 164, 164, 164, 164, 
              164, 164, 164, 164, 164, 164, 164, 164, 164, 
              164, 164, 164, 164, 164, 164, 164, 164, 164, 
              164, 164, 164, 164, 164, 165, 165, 165, 165, 
              165, 165, 165, 165, 165, 165, 165, 165, 165, 
              165, 165, 165, 165, 165, 165, 165, 165, 165, 
              165, 165, 165, 165, 165, 165, 165, 165, 165, 
              165, 165, 165, 165, 165, 165, 165, 165, 165, 
              165, 165, 165, 165, 165, 165, 165, 165, 165, 
              165, 165, 165, 165, 165, 165, 165, 165, 165, 
              165, 165, 165, 165, 165, 165, 166, 166, 166, 
              166, 166, 166, 166, 166, 166, 166, 166, 166, 
              166, 166, 166, 166, 166, 166, 166, 166, 166, 
              166, 166, 166, 166, 166, 166, 166, 166, 166, 
              166, 166, 167, 167, 167, 167, 167, 167, 167, 
              167, 167, 167, 167, 167, 167, 167, 167, 167, 
              167, 167, 167, 167, 167, 167, 167, 167, 167, 
              167, 167, 167, 167, 167, 167, 167, 168, 168, 
              168, 168, 168, 168, 168, 168, 168, 168, 168, 
              168, 168, 168, 168, 168, 169, 169, 169, 169, 
              169, 169, 169, 169, 169, 169, 169, 169, 169, 
              169, 169, 169, 169, 169, 169, 169, 169, 169, 
              169, 169, 169, 169, 169, 169, 169, 169, 169, 
              169, 169, 169, 169, 169, 169, 169, 169, 169, 
              169, 169, 169, 169, 169, 169, 169, 169, 170, 
              170, 170, 170, 170, 170, 170, 170, 170, 170, 
              170, 170, 170, 170, 170, 170, 170, 170, 170, 
              170, 170, 170, 170, 170, 170, 170, 170, 170, 
              170, 170, 170, 170, 170, 170, 170, 170, 170, 
              170, 170, 170, 170, 170, 170, 170, 170, 170, 
              170, 170, 170, 170, 170, 170, 170, 170, 170, 
              170, 170, 170, 170, 170, 170, 170, 170, 170, 
              171, 171, 171, 171, 171, 171, 171, 171, 171, 
              171, 171, 171, 171, 171, 171, 171, 171, 171, 
              171, 171, 171, 171, 171, 171, 171, 171, 171, 
              171, 171, 171, 171, 171, 171, 171, 171, 171, 
              171, 171, 171, 171, 171, 171, 171, 171, 171, 
              171, 171, 171, 171, 171, 171, 171, 171, 171, 
              171, 171, 171, 171, 171, 171, 171, 171, 171, 
              171, 172, 172, 172, 172, 172, 172, 172, 172, 
              172, 172, 172, 172, 172, 172, 172, 172, 172, 
              172, 172, 172, 172, 172, 172, 172, 172, 172, 
              172, 172, 172, 172, 172, 172, 172, 172, 172, 
              172, 172, 172, 172, 172, 172, 172, 172, 172, 
              172, 172, 172, 172, 172, 172, 172, 172, 172, 
              172, 172, 172, 172, 172, 172, 172, 172, 172, 
              172, 172, 173, 173, 173, 173, 173, 173, 173, 
              173, 173, 173, 173, 173, 173, 173, 173, 173, 
              173, 173, 173, 173, 173, 173, 173, 173, 173, 
              173, 173, 173, 173, 173, 173, 173, 174, 174, 
              174, 174, 174, 174, 174, 174, 174, 174, 174, 
              174, 174, 174, 174, 174, 174, 174, 174, 174, 
              174, 174, 174, 174, 174, 174, 174, 174, 174, 
              174, 174, 174, 175, 175, 175, 175, 175, 175, 
              175, 175, 175, 175, 175, 175, 175, 175, 175, 
              175, 176, 176, 176, 176, 176, 176, 176, 176, 
              176, 176, 176, 176, 176, 176, 176, 176, 176, 
              176, 176, 176, 176, 176, 176, 176, 177, 177, 
              177, 177, 177, 177, 177, 177, 177, 177, 177, 
              177, 177, 177, 177, 177, 177, 177, 177, 177, 
              177, 177, 177, 177, 177, 177, 177, 177, 177, 
              177, 177, 177, 178, 178, 178, 178, 178, 178, 
              178, 178, 178, 178, 178, 178, 178, 178, 178, 
              178, 178, 178, 178, 178, 178, 178, 178, 178, 
              178, 178, 178, 178, 178, 178, 178, 178, 179, 
              179, 179, 179, 179, 179, 179, 179, 179, 179, 
              179, 179, 179, 179, 179, 179, 179, 179, 179, 
              179, 179, 179, 179, 179, 179, 179, 179, 179, 
              179, 179, 179, 179, 180, 180, 180, 180, 180, 
              180, 180, 180, 180, 180, 180, 180, 180, 180, 
              180, 180, 181, 181, 181, 181, 181, 181, 181, 
              181, 181, 181, 181, 181, 181, 181, 181, 181, 
              182, 182, 182, 182, 182, 182, 182, 182, 183, 
              183, 183, 183, 183, 183, 183, 183, 183, 183, 
              183, 183, 183, 183, 183, 183, 183, 183, 183, 
              183, 183, 183, 183, 183, 184, 184, 184, 184, 
              184, 184, 184, 184, 184, 184, 184, 184, 184, 
              184, 184, 184, 184, 184, 184, 184, 184, 184, 
              184, 184, 184, 184, 184, 184, 184, 184, 184, 
              184, 185, 185, 185, 185, 185, 185, 185, 185, 
              185, 185, 185, 185, 185, 185, 185, 185, 185, 
              185, 185, 185, 185, 185, 185, 185, 185, 185, 
              185, 185, 185, 185, 185, 185, 186, 186, 186, 
              186, 186, 186, 186, 186, 186, 186, 186, 186, 
              186, 186, 186, 186, 186, 186, 186, 186, 186, 
              186, 186, 186, 186, 186, 186, 186, 186, 186, 
              186, 186, 187, 187, 187, 187, 187, 187, 187, 
              187, 187, 187, 187, 187, 187, 187, 187, 187, 
              188, 188, 188, 188, 188, 188, 188, 188, 188, 
              188, 188, 188, 188, 188, 188, 188, 189, 189, 
              189, 189, 189, 189, 189, 189, 190, 190, 190, 
              190, 190, 190, 190, 190, 190, 190, 190, 190, 
              191, 191, 191, 191, 191, 191, 191, 191, 191, 
              191, 191, 191, 191, 191, 191, 191, 192, 192, 
              192, 192, 192, 192, 192, 192, 192, 192, 192, 
              192, 192, 192, 192, 192, 193, 193, 193, 193, 
              193, 193, 193, 193, 193, 193, 193, 193, 193, 
              193, 193, 193, 194, 194, 194, 194, 194, 194, 
              194, 194, 195, 195, 195, 195, 195, 195, 195, 
              195, 196, 196, 196, 196, 197, 197, 197, 197, 
              197, 197, 197, 197, 197, 197, 197, 197, 197, 
              197, 197, 197, 197, 197, 198, 198, 198, 198, 
              198, 198, 198, 198, 198, 198, 198, 198, 198, 
              198, 198, 198, 198, 198, 198, 198, 198, 198, 
              198, 198, 199, 199, 199, 199, 199, 199, 199, 
              199, 199, 199, 199, 199, 199, 199, 199, 199, 
              199, 199, 199, 199, 199, 199, 199, 199, 200, 
              200, 200, 200, 200, 200, 200, 200, 200, 200, 
              200, 200, 200, 200, 200, 200, 200, 200, 200, 
              200, 200, 200, 200, 200, 201, 201, 201, 201, 
              201, 201, 201, 201, 201, 201, 201, 201, 202, 
              202, 202, 202, 202, 202, 202, 202, 202, 202, 
              202, 202, 203, 203, 203, 203, 203, 203, 204, 
              204, 204, 204, 204, 204, 204, 204, 204, 204, 
              204, 204, 204, 204, 204, 204, 204, 204, 204, 
              204, 204, 204, 204, 204, 205, 205, 205, 205, 
              205, 205, 205, 205, 205, 205, 205, 205, 205, 
              205, 205, 205, 205, 205, 205, 205, 205, 205, 
              205, 205, 205, 205, 205, 205, 205, 205, 205, 
              205, 206, 206, 206, 206, 206, 206, 206, 206, 
              206, 206, 206, 206, 206, 206, 206, 206, 206, 
              206, 206, 206, 206, 206, 206, 206, 206, 206, 
              206, 206, 206, 206, 206, 206, 207, 207, 207, 
              207, 207, 207, 207, 207, 207, 207, 207, 207, 
              207, 207, 207, 207, 207, 207, 207, 207, 207, 
              207, 207, 207, 207, 207, 207, 207, 207, 207, 
              207, 207, 208, 208, 208, 208, 208, 208, 208, 
              208, 208, 208, 208, 208, 208, 208, 208, 208, 
              209, 209, 209, 209, 209, 209, 209, 209, 209, 
              209, 209, 209, 209, 209, 209, 209, 210, 210, 
              210, 210, 210, 210, 210, 210, 211, 211, 211, 
              211, 211, 211, 211, 211, 211, 211, 211, 211, 
              211, 211, 211, 211, 211, 211, 211, 211, 211, 
              211, 211, 211, 212, 212, 212, 212, 212, 212, 
              212, 212, 212, 212, 212, 212, 212, 212, 212, 
              212, 212, 212, 212, 212, 212, 212, 212, 212, 
              212, 212, 212, 212, 212, 212, 212, 212, 213, 
              213, 213, 213, 213, 213, 213, 213, 213, 213, 
              213, 213, 213, 213, 213, 213, 213, 213, 213, 
              213, 213, 213, 213, 213, 213, 213, 213, 213, 
              213, 213, 213, 213, 214, 214, 214, 214, 214, 
              214, 214, 214, 214, 214, 214, 214, 214, 214, 
              214, 214, 214, 214, 214, 214, 214, 214, 214, 
              214, 214, 214, 214, 214, 214, 214, 214, 214, 
              215, 215, 215, 215, 215, 215, 215, 215, 215, 
              215, 215, 215, 215, 215, 215, 215, 216, 216, 
              216, 216, 216, 216, 216, 216, 216, 216, 216, 
              216, 216, 216, 216, 216, 217, 217, 217, 217, 
              217, 217, 217, 217, 218, 218, 218, 218, 218, 
              218, 218, 218, 218, 218, 218, 218, 218, 218, 
              218, 218, 218, 218, 218, 218, 218, 218, 218, 
              218, 219, 219, 219, 219, 219, 219, 219, 219, 
              219, 219, 219, 219, 219, 219, 219, 219, 219, 
              219, 219, 219, 219, 219, 219, 219, 219, 219, 
              219, 219, 219, 219, 219, 219, 220, 220, 220, 
              220, 220, 220, 220, 220, 220, 220, 220, 220, 
              220, 220, 220, 220, 220, 220, 220, 220, 220, 
              220, 220, 220, 220, 220, 220, 220, 220, 220, 
              220, 220, 221, 221, 221, 221, 221, 221, 221, 
              221, 221, 221, 221, 221, 221, 221, 221, 221, 
              221, 221, 221, 221, 221, 221, 221, 221, 221, 
              221, 221, 221, 221, 221, 221, 221, 222, 222, 
              222, 222, 222, 222, 222, 222, 222, 222, 222, 
              222, 222, 222, 222, 222, 223, 223, 223, 223, 
              223, 223, 223, 223, 223, 223, 223, 223, 223, 
              223, 223, 223, 224, 224, 224, 224, 224, 224, 
              224, 224, 225, 225, 225, 225, 225, 225, 225, 
              225, 225, 225, 225, 225, 226, 226, 226, 226, 
              226, 226, 226, 226, 226, 226, 226, 226, 226, 
              226, 226, 226, 227, 227, 227, 227, 227, 227, 
              227, 227, 227, 227, 227, 227, 227, 227, 227, 
              227, 228, 228, 228, 228, 228, 228, 228, 228, 
              228, 228, 228, 228, 228, 228, 228, 228, 229, 
              229, 229, 229, 229, 229, 229, 229, 230, 230, 
              230, 230, 230, 230, 230, 230, 231, 231, 231, 
              231, 232, 232, 232, 232, 232, 232, 232, 232, 
              232, 232, 232, 232, 233, 233, 233, 233, 233, 
              233, 233, 233, 233, 233, 233, 233, 233, 233, 
              233, 233, 234, 234, 234, 234, 234, 234, 234, 
              234, 234, 234, 234, 234, 234, 234, 234, 234, 
              235, 235, 235, 235, 235, 235, 235, 235, 235, 
              235, 235, 235, 235, 235, 235, 235, 236, 236, 
              236, 236, 236, 236, 236, 236, 237, 237, 237, 
              237, 237, 237, 237, 237, 238, 238, 238, 238, 
              239, 239, 239, 239, 239, 239, 240, 240, 240, 
              240, 240, 240, 240, 240, 241, 241, 241, 241, 
              241, 241, 241, 241, 242, 242, 242, 242, 242, 
              242, 242, 242, 243, 243, 243, 243, 244, 244, 
              244, 244, 245, 245, 246, 246, 246, 246, 246, 
              246, 246, 246, 246, 246, 246, 246, 246, 246, 
              246, 246, 246, 246, 247, 247, 247, 247, 247, 
              247, 247, 247, 247, 247, 247, 247, 247, 247, 
              247, 247, 247, 247, 247, 247, 247, 247, 247, 
              247, 248, 248, 248, 248, 248, 248, 248, 248, 
              248, 248, 248, 248, 248, 248, 248, 248, 248, 
              248, 248, 248, 248, 248, 248, 248, 249, 249, 
              249, 249, 249, 249, 249, 249, 249, 249, 249, 
              249, 249, 249, 249, 249, 249, 249, 249, 249, 
              249, 249, 249, 249, 250, 250, 250, 250, 250, 
              250, 250, 250, 250, 250, 250, 250, 251, 251, 
              251, 251, 251, 251, 251, 251, 251, 251, 251, 
              251, 252, 252, 252, 252, 252, 252, 253, 253, 
              253, 253, 253, 253, 253, 253, 253, 253, 253, 
              253, 253, 253, 253, 253, 253, 253, 253, 253, 
              253, 253, 253, 253, 254, 254, 254, 254, 254, 
              254, 254, 254, 254, 254, 254, 254, 254, 254, 
              254, 254, 254, 254, 254, 254, 254, 254, 254, 
              254, 254, 254, 254, 254, 254, 254, 254, 254, 
              255, 255, 255, 255, 255, 255, 255, 255, 255, 
              255, 255, 255, 255, 255, 255, 255, 255, 255, 
              255, 255, 255, 255, 255, 255, 255, 255, 255, 
              255, 255, 255, 255, 255, 256, 256, 256, 256, 
              256, 256, 256, 256, 256, 256, 256, 256, 256, 
              256, 256, 256, 256, 256, 256, 256, 256, 256, 
              256, 256, 256, 256, 256, 256, 256, 256, 256, 
              256, 257, 257, 257, 257, 257, 257, 257, 257, 
              257, 257, 257, 257, 257, 257, 257, 257, 258, 
              258, 258, 258, 258, 258, 258, 258, 258, 258, 
              258, 258, 258, 258, 258, 258, 259, 259, 259, 
              259, 259, 259, 259, 259, 260, 260, 260, 260, 
              260, 260, 260, 260, 260, 260, 260, 260, 260, 
              260, 260, 260, 260, 260, 260, 260, 260, 260, 
              260, 260, 261, 261, 261, 261, 261, 261, 261, 
              261, 261, 261, 261, 261, 261, 261, 261, 261, 
              261, 261, 261, 261, 261, 261, 261, 261, 261, 
              261, 261, 261, 261, 261, 261, 261, 262, 262, 
              262, 262, 262, 262, 262, 262, 262, 262, 262, 
              262, 262, 262, 262, 262, 262, 262, 262, 262, 
              262, 262, 262, 262, 262, 262, 262, 262, 262, 
              262, 262, 262, 263, 263, 263, 263, 263, 263, 
              263, 263, 263, 263, 263, 263, 263, 263, 263, 
              263, 263, 263, 263, 263, 263, 263, 263, 263, 
              263, 263, 263, 263, 263, 263, 263, 263, 264, 
              264, 264, 264, 264, 264, 264, 264, 264, 264, 
              264, 264, 264, 264, 264, 264, 265, 265, 265, 
              265, 265, 265, 265, 265, 265, 265, 265, 265, 
              265, 265, 265, 265, 266, 266, 266, 266, 266, 
              266, 266, 266, 267, 267, 267, 267, 267, 267, 
              267, 267, 267, 267, 267, 267, 267, 267, 267, 
              267, 267, 267, 267, 267, 267, 267, 267, 267, 
              268, 268, 268, 268, 268, 268, 268, 268, 268, 
              268, 268, 268, 268, 268, 268, 268, 268, 268, 
              268, 268, 268, 268, 268, 268, 268, 268, 268, 
              268, 268, 268, 268, 268, 269, 269, 269, 269, 
              269, 269, 269, 269, 269, 269, 269, 269, 269, 
              269, 269, 269, 269, 269, 269, 269, 269, 269, 
              269, 269, 269, 269, 269, 269, 269, 269, 269, 
              269, 270, 270, 270, 270, 270, 270, 270, 270, 
              270, 270, 270, 270, 270, 270, 270, 270, 270, 
              270, 270, 270, 270, 270, 270, 270, 270, 270, 
              270, 270, 270, 270, 270, 270, 271, 271, 271, 
              271, 271, 271, 271, 271, 271, 271, 271, 271, 
              271, 271, 271, 271, 272, 272, 272, 272, 272, 
              272, 272, 272, 272, 272, 272, 272, 272, 272, 
              272, 272, 273, 273, 273, 273, 273, 273, 273, 
              273, 274, 274, 274, 274, 274, 274, 274, 274, 
              274, 274, 274, 274, 275, 275, 275, 275, 275, 
              275, 275, 275, 275, 275, 275, 275, 275, 275, 
              275, 275, 276, 276, 276, 276, 276, 276, 276, 
              276, 276, 276, 276, 276, 276, 276, 276, 276, 
              277, 277, 277, 277, 277, 277, 277, 277, 277, 
              277, 277, 277, 277, 277, 277, 277, 278, 278, 
              278, 278, 278, 278, 278, 278, 279, 279, 279, 
              279, 279, 279, 279, 279, 280, 280, 280, 280, 
              281, 281, 281, 281, 281, 281, 281, 281, 281, 
              281, 281, 281, 282, 282, 282, 282, 282, 282, 
              282, 282, 282, 282, 282, 282, 282, 282, 282, 
              282, 283, 283, 283, 283, 283, 283, 283, 283, 
              283, 283, 283, 283, 283, 283, 283, 283, 284, 
              284, 284, 284, 284, 284, 284, 284, 284, 284, 
              284, 284, 284, 284, 284, 284, 285, 285, 285, 
              285, 285, 285, 285, 285, 286, 286, 286, 286, 
              286, 286, 286, 286, 287, 287, 287, 287, 288, 
              288, 288, 288, 288, 288, 289, 289, 289, 289, 
              289, 289, 289, 289, 290, 290, 290, 290, 290, 
              290, 290, 290, 291, 291, 291, 291, 291, 291, 
              291, 291, 292, 292, 292, 292, 293, 293, 293, 
              293, 294, 294, 295, 295, 295, 295, 295, 295, 
              295, 295, 295, 296, 296, 296, 296, 296, 296, 
              296, 296, 296, 296, 296, 296, 297, 297, 297, 
              297, 297, 297, 297, 297, 297, 297, 297, 297, 
              298, 298, 298, 298, 298, 298, 298, 298, 298, 
              298, 298, 298, 299, 299, 299, 299, 299, 299, 
              300, 300, 300, 300, 300, 300, 301, 301, 301, 
              302, 302, 302, 302, 302, 302, 302, 302, 302, 
              302, 302, 302, 303, 303, 303, 303, 303, 303, 
              303, 303, 303, 303, 303, 303, 303, 303, 303, 
              303, 304, 304, 304, 304, 304, 304, 304, 304, 
              304, 304, 304, 304, 304, 304, 304, 304, 305, 
              305, 305, 305, 305, 305, 305, 305, 305, 305, 
              305, 305, 305, 305, 305, 305, 306, 306, 306, 
              306, 306, 306, 306, 306, 307, 307, 307, 307, 
              307, 307, 307, 307, 308, 308, 308, 308, 309, 
              309, 309, 309, 309, 309, 309, 309, 309, 309, 
              309, 309, 310, 310, 310, 310, 310, 310, 310, 
              310, 310, 310, 310, 310, 310, 310, 310, 310, 
              311, 311, 311, 311, 311, 311, 311, 311, 311, 
              311, 311, 311, 311, 311, 311, 311, 312, 312, 
              312, 312, 312, 312, 312, 312, 312, 312, 312, 
              312, 312, 312, 312, 312, 313, 313, 313, 313, 
              313, 313, 313, 313, 314, 314, 314, 314, 314, 
              314, 314, 314, 315, 315, 315, 315, 316, 316, 
              316, 316, 316, 316, 316, 316, 316, 316, 316, 
              316, 317, 317, 317, 317, 317, 317, 317, 317, 
              317, 317, 317, 317, 317, 317, 317, 317, 318, 
              318, 318, 318, 318, 318, 318, 318, 318, 318, 
              318, 318, 318, 318, 318, 318, 319, 319, 319, 
              319, 319, 319, 319, 319, 319, 319, 319, 319, 
              319, 319, 319, 319, 320, 320, 320, 320, 320, 
              320, 320, 320, 321, 321, 321, 321, 321, 321, 
              321, 321, 322, 322, 322, 322, 323, 323, 323, 
              323, 323, 323, 324, 324, 324, 324, 324, 324, 
              324, 324, 325, 325, 325, 325, 325, 325, 325, 
              325, 326, 326, 326, 326, 326, 326, 326, 326, 
              327, 327, 327, 327, 328, 328, 328, 328, 329, 
              329, 330, 330, 330, 330, 330, 330, 331, 331, 
              331, 331, 331, 331, 331, 331, 332, 332, 332, 
              332, 332, 332, 332, 332, 333, 333, 333, 333, 
              333, 333, 333, 333, 334, 334, 334, 334, 335, 
              335, 335, 335, 336, 336, 337, 337, 337, 338, 
              338, 338, 338, 339, 339, 339, 339, 340, 340, 
              340, 340, 341, 341, 342, 342, 343)
  
  xSplitted = strsplit(x, split = '')[[1]]
  n  = nchar(x)
  CTAll = summary(factor(paste(paste(xSplitted[-c(n, n-1)], 
                                     xSplitted[-c(1, n)], sep = ''), 
                               xSplitted[-c(1, 2)], sep = ''), 
                         levels = CTDict), maxsum = 8001)
  
  MatchedIndex = which(CTAll != 0)
  MatchedNames = names(CTAll[MatchedIndex])
  MatchedTimes = as.integer(CTAll[MatchedIndex])
  CTAll = rep(MatchedNames, times = MatchedTimes)
  
  CT = rep(0L, 343)
  
  for (i in 1:length(MatchedNames)) {
    idx = CTIndex[which(CTDict == MatchedNames[i])]
    CT[idx] = CT[idx] + MatchedTimes[i]
  }
  
  CT = (CT - min(CT))/max(CT)
  
  names(CT) = c('VS111', 'VS211', 'VS311', 'VS411', 'VS511', 'VS611', 'VS711', 
                'VS121', 'VS221', 'VS321', 'VS421', 'VS521', 'VS621', 'VS721', 
                'VS131', 'VS231', 'VS331', 'VS431', 'VS531', 'VS631', 'VS731', 
                'VS141', 'VS241', 'VS341', 'VS441', 'VS541', 'VS641', 'VS741', 
                'VS151', 'VS251', 'VS351', 'VS451', 'VS551', 'VS651', 'VS751', 
                'VS161', 'VS261', 'VS361', 'VS461', 'VS561', 'VS661', 'VS761', 
                'VS171', 'VS271', 'VS371', 'VS471', 'VS571', 'VS671', 'VS771', 
                'VS112', 'VS212', 'VS312', 'VS412', 'VS512', 'VS612', 'VS712', 
                'VS122', 'VS222', 'VS322', 'VS422', 'VS522', 'VS622', 'VS722', 
                'VS132', 'VS232', 'VS332', 'VS432', 'VS532', 'VS632', 'VS732', 
                'VS142', 'VS242', 'VS342', 'VS442', 'VS542', 'VS642', 'VS742', 
                'VS152', 'VS252', 'VS352', 'VS452', 'VS552', 'VS652', 'VS752', 
                'VS162', 'VS262', 'VS362', 'VS462', 'VS562', 'VS662', 'VS762', 
                'VS172', 'VS272', 'VS372', 'VS472', 'VS572', 'VS672', 'VS772', 
                'VS113', 'VS213', 'VS313', 'VS413', 'VS513', 'VS613', 'VS713', 
                'VS123', 'VS223', 'VS323', 'VS423', 'VS523', 'VS623', 'VS723', 
                'VS133', 'VS233', 'VS333', 'VS433', 'VS533', 'VS633', 'VS733', 
                'VS143', 'VS243', 'VS343', 'VS443', 'VS543', 'VS643', 'VS743', 
                'VS153', 'VS253', 'VS353', 'VS453', 'VS553', 'VS653', 'VS753', 
                'VS163', 'VS263', 'VS363', 'VS463', 'VS563', 'VS663', 'VS763', 
                'VS173', 'VS273', 'VS373', 'VS473', 'VS573', 'VS673', 'VS773', 
                'VS114', 'VS214', 'VS314', 'VS414', 'VS514', 'VS614', 'VS714', 
                'VS124', 'VS224', 'VS324', 'VS424', 'VS524', 'VS624', 'VS724', 
                'VS134', 'VS234', 'VS334', 'VS434', 'VS534', 'VS634', 'VS734', 
                'VS144', 'VS244', 'VS344', 'VS444', 'VS544', 'VS644', 'VS744', 
                'VS154', 'VS254', 'VS354', 'VS454', 'VS554', 'VS654', 'VS754', 
                'VS164', 'VS264', 'VS364', 'VS464', 'VS564', 'VS664', 'VS764', 
                'VS174', 'VS274', 'VS374', 'VS474', 'VS574', 'VS674', 'VS774', 
                'VS115', 'VS215', 'VS315', 'VS415', 'VS515', 'VS615', 'VS715', 
                'VS125', 'VS225', 'VS325', 'VS425', 'VS525', 'VS625', 'VS725', 
                'VS135', 'VS235', 'VS335', 'VS435', 'VS535', 'VS635', 'VS735', 
                'VS145', 'VS245', 'VS345', 'VS445', 'VS545', 'VS645', 'VS745', 
                'VS155', 'VS255', 'VS355', 'VS455', 'VS555', 'VS655', 'VS755', 
                'VS165', 'VS265', 'VS365', 'VS465', 'VS565', 'VS665', 'VS765', 
                'VS175', 'VS275', 'VS375', 'VS475', 'VS575', 'VS675', 'VS775', 
                'VS116', 'VS216', 'VS316', 'VS416', 'VS516', 'VS616', 'VS716', 
                'VS126', 'VS226', 'VS326', 'VS426', 'VS526', 'VS626', 'VS726', 
                'VS136', 'VS236', 'VS336', 'VS436', 'VS536', 'VS636', 'VS736', 
                'VS146', 'VS246', 'VS346', 'VS446', 'VS546', 'VS646', 'VS746', 
                'VS156', 'VS256', 'VS356', 'VS456', 'VS556', 'VS656', 'VS756', 
                'VS166', 'VS266', 'VS366', 'VS466', 'VS566', 'VS666', 'VS766', 
                'VS176', 'VS276', 'VS376', 'VS476', 'VS576', 'VS676', 'VS776', 
                'VS117', 'VS217', 'VS317', 'VS417', 'VS517', 'VS617', 'VS717', 
                'VS127', 'VS227', 'VS327', 'VS427', 'VS527', 'VS627', 'VS727', 
                'VS137', 'VS237', 'VS337', 'VS437', 'VS537', 'VS637', 'VS737', 
                'VS147', 'VS247', 'VS347', 'VS447', 'VS547', 'VS647', 'VS747', 
                'VS157', 'VS257', 'VS357', 'VS457', 'VS557', 'VS657', 'VS757', 
                'VS167', 'VS267', 'VS367', 'VS467', 'VS567', 'VS667', 'VS767', 
                'VS177', 'VS277', 'VS377', 'VS477', 'VS577', 'VS677', 'VS777')
  
  return(CT)
  
}

