#' @include internal.R ConservationProblem-proto.R zones.R
NULL

#' Conservation planning problem
#'
#' Create a systematic conservation planning problem. This function is used to
#' specify the basic data used in a spatial prioritization problem: the
#' spatial distribution of the planning units and their costs, as well as
#' the features (e.g., species, ecosystems) that need to be conserved. After
#' constructing this `ConservationProblem-class` object, it can be
#' customized to meet specific goals using [objectives],
#' [targets], [constraints], and
#' [penalties]. After building the problem, the
#' [solve()] function can be used to identify solutions.
#' **Note that problems require an objective, and failing to specify an
#' an objective will throw an error when attempting to solve it.**
#'
#' @param x [`Raster-class`],
#'   [sf::st_sf()],
#'   [`SpatialPolygonsDataFrame-class`],
#'   [`SpatialLinesDataFrame-class`],
#'   [`SpatialPointsDataFrame-class`],
#'   [data.frame()] object,
#'   [numeric()] vector, or
#'   [matrix()] specifying the planning units to use in the reserve
#'   design exercise and their corresponding cost. It may be desirable to
#'   exclude some planning units from the analysis, for example those outside
#'   the study area. To exclude planning units, set the cost for those raster
#'   cells to `NA`, or use the `add_locked_out_constraint` function.
#'
#' @param features The feature data can be specified in a variety of ways.
#'   The specific formats that can be used depend on the cost data format (i.e.,
#'   argument to `x`) and whether the problem should have a single zone or
#'   multiple zones. If the problem should have a single zone, then the feature
#'   data can be specified following:
#'   * [`x = RasterLayer-class`][raster::RasterLayer-class], or
#'     [`x = Spatial-class`][sp::Spatial-class], or
#'     [`x = sf::st_sf()`][sf::st_sf()]:
#'     [`y = Raster-class`][raster::Raster-class]
#'     object showing the distribution of conservation features. Missing
#'     values (i.e., `NA` values) can be used to indicate the absence of
#'     a feature in a particular cell instead of explicitly setting these
#'     cells to zero. Note that this argument type for `features` can
#'     only be used to specify data for problems involving a single zone.
#'   * [`x = Spatial-class`][sp::Spatial-class], or
#'     [`x = sf::st_sf()`][sf::st_sf()], or
#'     `x = data.frame`:
#'     `y = character` vector
#'     with column names that correspond to the abundance or occurrence of
#'     different features in each planning unit. Note that this argument
#'     type can only be used to create problems involving a single zone.
#'   * `x = data.frame`, or
#'     `x = numeric` vector, or
#'     `x = matrix`:
#'     `y = data.frame` object
#'     containing the names of the features. Note that if this
#'     type of argument is supplied to `features` then the argument
#'     `rij` or `rij_matrix` must also be supplied. This type of
#'     argument should follow the conventions used by *Marxan*, wherein
#'     each row corresponds to a different feature. It must also contain the
#'     following columns:
#'     \describe{
#'     \item{id}{`integer` unique identifier for each feature
#'       These identifiers are used in the argument to `rij`.}
#'     \item{name}{`character` name for each feature.}
#'     \item{prop}{`numeric` relative target for each feature
#'       (optional).}
#'     \item{amount}{`numeric` absolute target for each
#'       feature (optional).}
#'     }
#'
#'   If the problem should have multiple zones, then the feature
#'   data can be specified following:
#'   * [`x = RasterStack-class`][raster::RasterStack-class], or
#'     [`x = RasterBrick-class`][raster::RasterBrick-class], or
#'     [`x = Spatial-class`][sp::Spatial-class], or
#'     [`x = sf::st_sf()`][sf::st_sf()]:
#'     [`y = ZonesRaster`][zones()]:
#'     object showing the distribution of conservation features in multiple
#'     zones. As above, missing values (i.e., `NA` values) can be used to
#'     indicate the absence of a feature in a particular cell instead of
#'     explicitly setting these cells to zero.
#'   * [`x = Spatial-class`][sp::Spatial-class], or
#'     [`x = sf::st_sf()`][sf::st_sf()], or
#'     or `x = data.frame`:
#'     [`y = ZonesCharacter`][zones()]
#'     object with column names that correspond to the abundance or
#'     occurrence of different features in each planning unit in different
#'     zones.
#'
#' @param cost_column `character` name or `integer` indicating the
#'   column(s) with the cost data. This argument must be supplied when the
#'   argument to `x` is a [`Spatial-class`] or
#'   `data.frame` object. This argument should contain the name of each
#'   column containing cost data for each management zone when creating
#'   problems with multiple zones. To create a problem with a single zone, then
#'   set the argument to `cost_column` as a single column name.
#'
#' @param rij `data.frame` containing information on the amount of
#'    each feature in each planning unit assuming each management zone. Similar
#'    to `data.frame` arguments for `features`, the `data.frame`
#'    objects must follow the conventions used by *Marxan*. Note that the
#'    `"zone"` column is not needed for problems involving a single
#'    management zone. Specifically, the argument should contain the following
#'    columns:
#'    \describe{
#'    \item{pu}{`integer` planning unit identifier.}
#'    \item{species}{`integer` feature identifier.}
#'    \item{zone}{`integer` zone identifier (optional for
#'      problems involving a single zone).}
#'    \item{amount}{`numeric` amount of the feature in the
#'      planning unit.}

#'    }
#'
#' @param rij_matrix `list` of `matrix` or
#'    [`dgCMatrix-class`]
#'    objects specifying the amount of each feature (rows) within each planning
#'    unit (columns) for each zone. The `list` elements denote
#'    different zones, matrix rows denote features, and matrix columns denote
#'    planning units. For convenience, the argument to
#'    `rij_matrix` can be a single `matrix` or
#'    [`dgCMatrix-class`] when specifying a problem with a
#'    single management zone. This argument is only used when the argument
#'    to `x` is a `numeric` or `matrix` object.
#'
#' @param zones `data.frame` containing information on the zones. This
#'   argument is only used when argument to `x` and `y` are
#'   both `data.frame` objects and the problem being built contains
#'   multiple zones. Following conventions used in `MarZone`, this
#'   argument should contain the following columns:
#'   columns:
#'   \describe{
#'   \item{id}{`integer` zone identifier.}
#'   \item{name}{`character` zone name.}
#'   }
#'
#' @param run_checks `logical` flag indicating whether checks should be
#'   run to ensure the integrity of the input data. These checks are run by
#'   default; however, for large datasets they may increase run time. If it is
#'   taking a prohibitively long time to create the prioritization problem, it
#'   is suggested to try setting `run_checks` to `FALSE`.
#'
#' @param ... not used.
#'
#' @details
#' A systematic conservation planning exercise leverages data to help inform
#' conservation decision making. To help ensure that the
#' data -- and resulting prioritizations -- are relevant to the over-arching
#' goals of the exercise, you should decide on the management action
#' (or set of actions) that need be considered in the exercise.
#' For example, these actions could include establishing protected areas,
#' selecting land for conservation easements, restoring habitat,
#' planting trees for carbon sequestration, eradicating invasive
#' species, or some combination of the previous actions.
#' If the exercise involves multiple different actions, they can
#' be incorporated by using multiple zones
#' (see the Management Zones vignette for details). After deciding
#' on the management action(s), you can compile the following data.
#'
#' First, you will need to create a set of planning units
#' (i.e., discrete spatial areas) to inform decision making.
#' Planning units are often created by subdividing a study region
#' into a set square or hexagonal cells. They can also be created using
#' administrative boundaries (e.g., provinces), land management boundaries
#' (e.g., property boundaries derived from cadastral data), or
#' ecological boundaries (e.g., based on ecosystem classification data).
#' The size (i.e., spatial grain) of the planning units is often determined
#' based on a compromise between the scale needed to inform decision making, the
#' spatial accuracy (resolution) of available datasets, and
#' the computational resources available for generating prioritizations
#' (e.g., RAM and number of CPUs on your computer).
#'
#' Second, you will need data to quantify the cost of implementing
#' implementing each management action within each planning unit.
#' Critically, the cost data should reflect the management action(s)
#' considered in the exercise.
#' For example, costs are often specified using data that reflect economic
#' expenditure (e.g., land acquisition cost),
#' socioeconomic conditions (e.g., human population density),
#' opportunity costs of foregone commercial activities
#' (e.g., logging or agriculture), or
#' opportunity costs of foregone recreational activities
#' (e.g., recreational fishing) activities,
#' In some cases -- depending on the management action(s) considered --
#' it can make sense to use a constant cost value
#' (e.g., all planning units are assigned a cost value equal to one)
#' or use a cost value based on spatial extent
#' (e.g., each planning unit is assigned a cost value based on its total area).
#' Also, in most cases, you want to avoid negative cost values.
#' This because a negative value means that a place is *desirable*
#' for implementing a management action, and such places will almost
#' always be selected for prioritization even if they provide no benefit.
#'
#' Third, you will need data to quantify the benefits of implementing
#' management actions within planning units.
#' To achieve this, you will need to select a set of conservation features
#' that relate to the over-arching goals of the exercise.
#' For example, conservation features often include
#' species (e.g., Clouded Leopard), habitats (e.g., mangroves or
#' cloud forest), or ecosystems.
#' The benefit that each feature derives from a planning unit
#' can take a variety of forms, but is typically occupancy (i.e.,
#' presence or absence), area of occurrence within each planning unit
#' (e.g., based on species' geographic range data), or
#' a measure of habitat suitability (e.g., estimated using a statistical model).
#' After compiling these data, you have the minimal data need to generate
#' a prioritization.
#'
#' A systematic conservation planning exercise involves prioritizing a set of
#' management actions to be implemented within certain planning units.
#' Critically, this prioritization should ideally optimize the trade-off
#' between benefits and costs.
#' To accomplish this, the \pkg{prioritizr} package uses input data
#' to formulate optimization problems (see Optimization section for details).
#' Broadly speaking, the goal of an optimization problem is to minimize
#' (or maximize) an objective function over a set of
#' decision variables, subject to a series of constraints.
#' Here, an objective function specifies the metric for evaluating
#' conservation plans. The decision variables are what we control, and usually
#' there is one binary variable for each planning unit to specify whether that
#' unit is selected or not (but other approaches are available, see
#' [decisions]). The constraints can be thought of as rules that must be
#' followed. For example, constraints can be used to ensure a prioritization
#' must stay within a certain budget. These constraints can also leverage
#' additional data to help ensure that prioritizations meet the over-arching
#' goals of the exercise. For example, to account for existing conservation
#' efforts, you could obtain data delineating the extent of existing protected
#' areas and use constraints to lock in planning units that are covered by them
#' (see [add_locked_in_constraints]).
#'
#' @section Optimization:
#' The \pkg{prioritizr} package uses exact algorithms to solve reserve design
#' problems (see [solvers] for details).
#' To achieve this, it internally formulates mathematical optimization problems
#' using mixed integer linear programming (MILP). The general form of
#' such problems can be expressed in matrix notation using
#' the following equation.
#'
#' \deqn{\mathit{Minimize} \space \mathbf{c}^{\mathbf{T}}\mathbf{x} \space
#' \mathit{subject \space to} \space
#' \mathbf{Ax}\geq= or\leq \mathbf{b}}{Minimize (c^T)*x subject to Ax \ge, =,
#' or \le b}
#'
#' Here, \eqn{x} is a vector of decision variables, \eqn{c} and \eqn{b} are
#' vectors of known coefficients, and \eqn{A} is the constraint
#' matrix. The final term specifies a series of structural
#' constraints where relational operators for the constraint can be either
#' \eqn{\ge}, \eqn{=}, or \eqn{\le} the coefficients. For example, in the
#' minimum set cover problem, \eqn{c} would be a vector of costs for each
#' planning unit, \eqn{b} a vector of targets for each conservation feature,
#' the relational operator would be \eqn{\ge} for all features, and \eqn{A}
#' would be the representation matrix with \eqn{A_{ij}=r_{ij}}{Aij = rij}, the
#' representation level of feature \eqn{i} in planning unit \eqn{j}.
#' If you wish to see exactly how a conservation planning problem is
#' formulated as mixed integer linear programming problem, you can use
#' the [write_problem()] function to save the optimization problem
#' to a plain-text file on your computer and then view it using a standard
#' text editor (e.g., Notepad).
#'
#' Please note that this function internally computes the amount of each
#' feature in each planning unit when this data is not supplied (using the
#' [rij_matrix] function). As a consequence, it can take a while to
#' initialize large-scale conservation planning problems that involve
#' millions of planning units.
#'
#' @return [`ConservationProblem-class`] object containing
#'   data for a prioritization.
#'
#' @seealso
#' See [solve()] for details on solving a problem to generate solutions.
#' Also, see [objectives], [penalties], [targets], [constraints],
#' [decisions], [portfolios], [solvers] for information on customizing problems.
#' Additionally, see [summaries] and [importance] for information on
#' evaluating solutions.
#'
#' @aliases problem,Raster,Raster-method problem,Spatial,Raster-method problem,data.frame,data.frame-method problem,numeric,data.frame-method problem,data.frame,character-method problem,Spatial,character-method problem,Raster,ZonesRaster-method problem,Spatial,ZonesRaster-method problem,Spatial,ZonesCharacter-method problem,data.frame,ZonesCharacter-method problem,matrix,data.frame-method problem,sf,Raster-method problem,sf,ZonesCharacter-method problem,sf,character-method problem,sf,ZonesRaster-method
#'
#' @exportMethod problem
#'
#' @name problem
#'
#' @examples
#' # load data
#' data(sim_pu_raster, sim_pu_polygons, sim_pu_lines, sim_pu_points,
#'      sim_pu_sf, sim_features)
#'
#' # create problem using raster planning unit data
#' p1 <- problem(sim_pu_raster, sim_features) %>%
#'       add_min_set_objective() %>%
#'       add_relative_targets(0.2) %>%
#'       add_binary_decisions() %>%
#'       add_default_solver(verbose = FALSE)
#'
#' \dontrun{
#' # create problem using polygon (Spatial) planning unit data
#' p2 <- problem(sim_pu_polygons, sim_features, "cost") %>%
#'       add_min_set_objective() %>%
#'       add_relative_targets(0.2) %>%
#'       add_binary_decisions() %>%
#'       add_default_solver(verbose = FALSE)
#'
#' # create problem using line (Spatial) planning unit data
#' p3 <- problem(sim_pu_lines, sim_features, "cost") %>%
#'       add_min_set_objective() %>%
#'       add_relative_targets(0.2) %>%
#'       add_binary_decisions() %>%
#'       add_default_solver(verbose = FALSE)
#'
#' # create problem using point (Spatial) planning unit data
#' p4 <- problem(sim_pu_points, sim_features, "cost") %>%
#'       add_min_set_objective() %>%
#'       add_relative_targets(0.2) %>%
#'       add_binary_decisions() %>%
#'       add_default_solver(verbose = FALSE)
#'
#' # create problem using polygon (sf) planning unit data
#' p5 <- problem(sim_pu_sf, sim_features, "cost") %>%
#'       add_min_set_objective() %>%
#'       add_relative_targets(0.2) %>%
#'       add_binary_decisions() %>%
#'       add_default_solver(verbose = FALSE)
#'
#' # since geo-processing can be slow for large spatial vector datasets
#' # (e.g., polygons, lines, points), it can be worthwhile to pre-process the
#' # planning unit data so that it contains columns indicating the amount of
#' # each feature inside each planning unit
#' # (i.e., each column corresponds to a different feature)
#'
#' # calculate the amount of each species within each planning unit
#' # (i.e., SpatialPolygonsDataFrame object)
#' pre_proc_data <- rij_matrix(sim_pu_polygons, sim_features)
#'
#' # add extra columns to the polygon (Spatial) planning unit data
#' # to indicate the amount of each species within each planning unit
#' pre_proc_data <- as.data.frame(t(as.matrix(pre_proc_data)))
#' names(pre_proc_data) <- names(sim_features)
#' sim_pu_polygons@data <- cbind(sim_pu_polygons@data, pre_proc_data)
#'
#' # create problem using the polygon (Spatial) planning unit data
#' # with the pre-processed columns
#' p6 <- problem(sim_pu_polygons, features = names(pre_proc_data), "cost") %>%
#'       add_min_set_objective() %>%
#'       add_relative_targets(0.2) %>%
#'       add_binary_decisions() %>%
#'       add_default_solver(verbose = FALSE)
#'
#' # this strategy of pre-processing columns can be used for sf objects too
#' pre_proc_data2 <- rij_matrix(sim_pu_sf, sim_features)
#' pre_proc_data2 <- as.data.frame(t(as.matrix(pre_proc_data2)))
#' names(pre_proc_data2) <- names(sim_features)
#' sim_pu_sf <- cbind(sim_pu_sf, pre_proc_data2)
#'
#' # create problem using the polygon (sf) planning unit data
#' # with pre-processed columns
#' p7 <- problem(sim_pu_sf, features = names(pre_proc_data2), "cost") %>%
#'       add_min_set_objective() %>%
#'       add_relative_targets(0.2) %>%
#'       add_binary_decisions() %>%
#'       add_default_solver(verbose = FALSE)
#'
#' # in addition to spatially explicit data, pre-processed aspatial data
#' # can also be used to create a problem
#' # (e.g., data created using external spreadsheet software)
#' costs <- sim_pu_polygons$cost
#' features <- data.frame(id = seq_len(nlayers(sim_features)),
#'                        name = names(sim_features))
#' rij_mat <- rij_matrix(sim_pu_polygons, sim_features)
#' p8 <- problem(costs, features, rij_matrix = rij_mat) %>%
#'       add_min_set_objective() %>%
#'       add_relative_targets(0.2) %>%
#'       add_binary_decisions() %>%
#'       add_default_solver(verbose = FALSE)
#'
#' # solve problems
#' s1 <- solve(p1)
#' s2 <- solve(p2)
#' s3 <- solve(p3)
#' s4 <- solve(p4)
#' s5 <- solve(p5)
#' s6 <- solve(p6)
#' s7 <- solve(p7)
#' s8 <- solve(p8)
#'
#' # plot solutions for problems associated with spatial data
#' par(mfrow = c(3, 2), mar = c(0, 0, 4.1, 0))
#' plot(s1, main = "raster data", axes = FALSE, box = FALSE, legend = FALSE)
#'
#' plot(s2, main = "polygon data")
#' plot(s2[s2$solution_1 > 0.5, ], col = "darkgreen", add = TRUE)
#'
#' plot(s3, main = "line data")
#' lines(s3[s3$solution_1 > 0.5, ], col = "darkgreen", lwd = 2)
#'
#' plot(s4, main = "point data", pch = 19)
#' points(s4[s4$solution_1 > 0.5, ], col = "darkgreen", cex = 2, pch = 19)
#'
#' # note that as_Spatial() is for convenience to plot all solutions together
#' plot(sf::as_Spatial(s5), main = "sf (polygon) data", pch = 19)
#' plot(sf::as_Spatial(s5[s5$solution_1 > 0.5, ]),
#'      col = "darkgreen", add = TRUE)
#'
#' plot(s6, main = "preprocessed data (polygon data)", pch = 19)
#' plot(s6[s6$solution_1 > 0.5, ], col = "darkgreen", add = TRUE)
#'
#' # show solutions for problems associated with aspatial data
#' str(s8)
#' }
#' # create some problems with multiple zones
#'
#' # first, create a matrix containing the targets for multi-zone problems
#' # here each row corresponds to a different feature, each
#' # column corresponds to a different zone, and values correspond
#' # to the total (absolute) amount of a given feature that needs to be secured
#' # in a given zone
#' targets <- matrix(rpois(15, 1),
#'                   nrow = number_of_features(sim_features_zones),
#'                   ncol = number_of_zones(sim_features_zones),
#'                   dimnames = list(feature_names(sim_features_zones),
#'                                   zone_names(sim_features_zones)))
#'
#' # print targets
#' print(targets)
#'
#' # create a multi-zone problem with raster data
#' p8 <- problem(sim_pu_zones_stack, sim_features_zones) %>%
#'       add_min_set_objective() %>%
#'       add_absolute_targets(targets) %>%
#'       add_binary_decisions() %>%
#'       add_default_solver(verbose = FALSE)
#' \dontrun{
#' # solve problem
#' s8 <- solve(p8)
#'
#' # plot solution
#' # here, each layer/panel corresponds to a different zone and pixel values
#' # indicate if a given planning unit has been allocated to a given zone
#' par(mfrow = c(1, 1))
#' plot(s8, main = c("zone 1", "zone 2", "zone 3"), axes = FALSE, box = FALSE)
#'
#' # alternatively, the category_layer function can be used to create
#' # a new raster object containing the zone ids for each planning unit
#' # in the solution (note this only works for problems with binary decisions)
#' par(mfrow = c(1, 1))
#' plot(category_layer(s8), axes = FALSE, box = FALSE)
#'
#' # create a multi-zone problem with polygon data
#' p9 <- problem(sim_pu_zones_polygons, sim_features_zones,
#'               cost_column = c("cost_1", "cost_2", "cost_3")) %>%
#'       add_min_set_objective() %>%
#'       add_absolute_targets(targets) %>%
#'       add_binary_decisions() %>%
#'       add_default_solver(verbose = FALSE)
#'
#' # solve problem
#' s9 <- solve(p9)
#'
#' # create column containing the zone id for which each planning unit was
#' # allocated to in the solution
#' s9$solution <- category_vector(s9@data[, c("solution_1_zone_1",
#'                                            "solution_1_zone_2",
#'                                            "solution_1_zone_3")])
#' s9$solution <- factor(s9$solution)
#'
#' # plot solution
#' spplot(s9, zcol = "solution", main = "solution", axes = FALSE, box = FALSE)
#'
#' # create a multi-zone problem with polygon planning unit data
#' # and where fields (columns) in the attribute table correspond
#' # to feature abundances
#'
#' # first fields need to be added to the planning unit data
#' # which indicate the amount of each feature in each zone
#' # to do this, the fields will be populated with random counts
#' sim_pu_zones_polygons$spp1_z1 <- rpois(nrow(sim_pu_zones_polygons), 1)
#' sim_pu_zones_polygons$spp2_z1 <- rpois(nrow(sim_pu_zones_polygons), 1)
#' sim_pu_zones_polygons$spp3_z1 <- rpois(nrow(sim_pu_zones_polygons), 1)
#' sim_pu_zones_polygons$spp1_z2 <- rpois(nrow(sim_pu_zones_polygons), 1)
#' sim_pu_zones_polygons$spp2_z2 <- rpois(nrow(sim_pu_zones_polygons), 1)
#' sim_pu_zones_polygons$spp3_z2 <- rpois(nrow(sim_pu_zones_polygons), 1)
#'
#' # create problem with polygon planning unit data and use field names
#' # to indicate feature data
#' # additionally, to make this example slightly more interesting,
#' # the problem will have prfoportion-type decisions such that
#' # a proportion of each planning unit can be allocated to each of the
#' # two management zones
#' p10 <- problem(sim_pu_zones_polygons,
#'                zones(c("spp1_z1", "spp2_z1", "spp3_z1"),
#'                      c("spp1_z2", "spp2_z2", "spp3_z2"),
#                       feature_names = c("spp1", "spp2", "spp3"),
#'                      zone_names = c("z1", "z2")),
#'                cost_column = c("cost_1", "cost_2")) %>%
#'        add_min_set_objective() %>%
#'        add_absolute_targets(targets[1:3, 1:2]) %>%
#'        add_proportion_decisions() %>%
#'       add_default_solver(verbose = FALSE)
#'
#' # solve problem
#' s10 <- solve(p10)
#'
#' # plot solution
#' spplot(s10, zcol = c("solution_1_z1", "solution_1_z2"), main = "solution",
#'        axes = FALSE, box = FALSE)
#' }
#' @export
methods::setGeneric("problem",
                    signature = methods::signature("x", "features"),
                    function(x, features, ...) standardGeneric("problem"))

#' @name problem
#' @usage \S4method{problem}{Raster,Raster}(x, features, run_checks, ...)
#' @rdname problem
methods::setMethod(
  "problem",
  methods::signature(x = "Raster", features = "Raster"),
  function(x, features, run_checks = TRUE, ...) {
    assertthat::assert_that(inherits(x, "Raster"), raster::nlayers(x) == 1,
                            no_extra_arguments(...))
    problem(x, zones(features, zone_names = names(x),
                     feature_names = names(features)),
            run_checks = run_checks, ...)
})

#' @name problem
#' @usage \S4method{problem}{Raster,ZonesRaster}(x, features, run_checks, ...)
#' @rdname problem
methods::setMethod(
  "problem",
  methods::signature(x = "Raster", features = "ZonesRaster"),
  function(x, features, run_checks = TRUE, ...) {
    # assert that arguments are valid
    assertthat::assert_that(
      inherits(x, "Raster"),
      inherits(features, "ZonesRaster"),
      assertthat::is.flag(run_checks),
      no_extra_arguments(...),
      raster::nlayers(x) > 0,
      number_of_features(features) > 0,
      raster::nlayers(x) == number_of_zones(features),
      is_comparable_raster(x, features[[1]]))
    if (run_checks) {
      assertthat::assert_that(
        all(raster::cellStats(!is.na(x), "sum") > 0))
      verify_that(all(raster::cellStats(x, "min") >= 0))
      verify_that(all(raster::cellStats(raster::stack(as.list(features)),
                                        "min") >= 0))
    }
    # convert x to RasterLayer if has only one layer
    if (inherits(x, c("RasterStack", "RasterBrick")) &&
        raster::nlayers(x) == 1)
      x <- x[[1]]
    # create rij matrix
    rij <- lapply(as.list(features), function(f) rij_matrix(x, f))
    names(rij) <- zone_names(features)
    # calculate feature abundances in total units
    fatu <- vapply(features, raster::cellStats,
                   numeric(number_of_features(features)), "sum")
    if (!is.matrix(fatu))
      fatu <- matrix(fatu, ncol = number_of_zones(features),
                     nrow = number_of_features(features))
    colnames(fatu) <- zone_names(features)
    rownames(fatu) <- feature_names(features)
    # create ConservationProblem object
    pproto(NULL, ConservationProblem,
           constraints = pproto(NULL, Collection),
           penalties = pproto(NULL, Collection),
           data = list(cost = x, features = features, rij_matrix = rij,
                       feature_abundances_in_total_units = fatu))
})

#' @name problem
#' @usage \S4method{problem}{Spatial,Raster}(x, features, cost_column, run_checks, ...)
#' @rdname problem
methods::setMethod(
  "problem",
  methods::signature(x = "Spatial", features = "Raster"),
  function(x, features, cost_column, run_checks = TRUE, ...) {
    assertthat::assert_that(assertthat::is.string(cost_column))
    problem(x, zones(features, zone_names = cost_column,
                     feature_names = names(features)),
            cost_column = cost_column, run_checks = run_checks, ...)
})

#' @name problem
#' @usage \S4method{problem}{Spatial,ZonesRaster}(x, features, cost_column, run_checks, ...)
#' @rdname problem
methods::setMethod(
  "problem",
  methods::signature(x = "Spatial", features = "ZonesRaster"),
  function(x, features, cost_column, run_checks = TRUE, ...) {
    # assert that arguments are valid
    assertthat::assert_that(
      inherits(x, c("SpatialPolygonsDataFrame", "SpatialLinesDataFrame",
                    "SpatialPointsDataFrame")),
      no_extra_arguments(...),
      length(x) > 0, is.character(cost_column), !anyNA(cost_column),
      all(cost_column %in% names(x)),
      length(cost_column) == number_of_zones(features),
      all(vapply(x@data[, cost_column, drop = FALSE], is.numeric, logical(1))),
      assertthat::is.flag(run_checks))
    # further validation checks
    assertthat::assert_that(
      length(x) > 0,
      all(colSums(!is.na(as.matrix(x@data[, cost_column, drop = FALSE])),
                  na.rm = TRUE) > 0),
      sf::st_crs(x@proj4string) == sf::st_crs(features[[1]]@crs),
      intersecting_extents(x, features[[1]]))
      verify_that(
        all(colSums(as.matrix(x@data[, cost_column, drop = FALSE]) < 0,
                    na.rm = TRUE) == 0),
        msg = "argument to x has negative cost values")
    if (run_checks) {
      verify_that(
        all(raster::cellStats(raster::stack(as.list(features)), "min") >= 0))
    }
    # compute rij matrix including non-planning unit cells
    rij <- rij_matrix(x, raster::stack(as.list(features)))
    rij <- lapply(seq_len(number_of_zones(features)), function(i) {
      m <- rij[((i - 1) * number_of_features(features)) +
               seq_len(number_of_features(features)), ,
          drop = FALSE]
      rownames(m) <- feature_names(features)
      return(m)
    })
    # calculate feature abundances in total units
    fatu <- vapply(rij, rowSums, numeric(number_of_features(features)),
                   na.rm = TRUE)
    if (!is.matrix(fatu))
      fatu <- matrix(fatu, nrow = number_of_features(features),
                     ncol = number_of_zones(features))
    rownames(fatu) <- feature_names(features)
    colnames(fatu) <- zone_names(features)
    # create rij matrix
    pos <- which(rowSums(!is.na(as.matrix(
             x@data[, cost_column, drop = FALSE]))) > 0)
    rij <- lapply(rij, function(x) x[, pos, drop = FALSE])
    names(rij) <- zone_names(features)
    # create ConservationProblem object
    pproto(NULL, ConservationProblem,
      constraints = pproto(NULL, Collection),
      penalties = pproto(NULL, Collection),
      data = list(cost = x, features = features, cost_column = cost_column,
                  rij_matrix = rij, feature_abundances_in_total_units = fatu))
})

#' @name problem
#' @usage \S4method{problem}{Spatial,character}(x, features, cost_column, ...)
#' @rdname problem
methods::setMethod(
  "problem",
  methods::signature(x = "Spatial", features = "character"),
  function(x, features, cost_column, ...) {
    assertthat::assert_that(assertthat::is.string(cost_column))
    problem(x, zones(features, feature_names = features,
                     zone_names = cost_column),
           cost_column = cost_column, ...)
})

#' @name problem
#' @usage \S4method{problem}{Spatial,ZonesCharacter}(x, features, cost_column, ...)
#' @rdname problem
methods::setMethod(
  "problem",
  methods::signature(x = "Spatial", features = "ZonesCharacter"),
  function(x, features, cost_column, ...) {
    # assert that arguments are valid
    assertthat::assert_that(
      inherits(x, c("SpatialPolygonsDataFrame", "SpatialLinesDataFrame",
                    "SpatialPointsDataFrame")),
      inherits(features, "ZonesCharacter"),
      no_extra_arguments(...),
      length(x) > 0, is.character(cost_column), !anyNA(cost_column),
      all(cost_column %in% names(x)),
      number_of_zones(features) == length(cost_column))
    assertthat::assert_that(
      all(unlist(as.list(features), recursive = TRUE, use.names = FALSE) %in%
                 names(x)),
      msg = paste("argument to features contains column names that are",
                  "not present in the argument to x"))
    assertthat::assert_that(
      all(vapply(x@data[, cost_column, drop = FALSE], is.numeric, logical(1))),
      msg = "argument to x has non-numeric cost data")
    assertthat::assert_that(
      all(colSums(!is.na(as.matrix(x@data[, cost_column, drop = FALSE])),
        na.rm = TRUE) > 0),
      msg = "argument to x has missing (NA) cost values")
    verify_that(all(as.matrix(x@data[, cost_column, drop = FALSE]) >= 0,
                            na.rm = TRUE),
                msg = "argument to x has negative cost values")
    verify_that(all(as.matrix(x@data[, unlist(features), drop = FALSE]) >= 0,
                    na.rm = TRUE),
                msg = "argument to features correspond to negative values")
    # create rij matrix
    pos <- which(rowSums(!is.na(as.matrix(
             x@data[, cost_column, drop = FALSE]))) > 0)
    rij <- lapply(features, function(z) {
      r <- t(as.matrix(x@data[pos, z, drop = FALSE]))
      r[is.na(r)] <- 0
      rownames(r) <- feature_names(features)
      methods::as(r, "sparseMatrix")
    })
    names(rij) <- zone_names(features)
    # calculate feature abundances in total units
    fatu <- colSums(x@data[, unlist(as.list(features)), drop = FALSE],
                    na.rm = TRUE)
    fatu <- matrix(fatu, ncol = number_of_zones(features),
                   nrow = number_of_features(features),
                   dimnames = list(feature_names(features),
                                   zone_names(features)))
    # create ConservationProblem object
    pproto(NULL, ConservationProblem,
      constraints = pproto(NULL, Collection),
      penalties = pproto(NULL, Collection),
      data = list(cost = x, features = features, cost_column = cost_column,
                  rij_matrix = rij, feature_abundances_in_total_units = fatu))
})

#' @name problem
#' @usage \S4method{problem}{data.frame,character}(x, features, cost_column, ...)
#' @rdname problem
methods::setMethod(
  "problem",
  methods::signature(x = "data.frame", features = "character"),
  function(x, features, cost_column, ...) {
    assertthat::assert_that(assertthat::is.string(cost_column))
    problem(x, zones(features, zone_names = cost_column,
                     feature_names = features),
            cost_column = cost_column, ...)
})

#' @name problem
#' @usage \S4method{problem}{data.frame,ZonesCharacter}(x, features, cost_column, ...)
#' @rdname problem
methods::setMethod(
  "problem",
  methods::signature(x = "data.frame", features = "ZonesCharacter"),
  function(x, features, cost_column, ...) {
    # assert that arguments are valid
    assertthat::assert_that(
      inherits(x, "data.frame"),
      inherits(features, "ZonesCharacter"),
      no_extra_arguments(...),
      nrow(x) > 0, is.character(cost_column), !anyNA(cost_column),
      all(cost_column %in% names(x)),
      number_of_zones(features) == length(cost_column))
    assertthat::assert_that(
      all(unlist(as.list(features), recursive = TRUE, use.names = FALSE) %in%
                 names(x)),
      msg = paste("argument to features contains column names that are",
                  "not present in the argument to x"))
    assertthat::assert_that(
      all(vapply(x[, cost_column, drop = FALSE], is.numeric, logical(1))),
      msg = "argument to x has non-numeric cost data")
    assertthat::assert_that(
      all(colSums(!is.na(as.matrix(x[, cost_column, drop = FALSE])),
        na.rm = TRUE) > 0),
      msg = "argument to x has missing (NA) cost values")
    verify_that(all(as.matrix(x[, cost_column, drop = FALSE]) >= 0,
                    na.rm = TRUE),
                msg = "argument to x has negative cost values")
    verify_that(all(as.matrix(x[, unlist(features), drop = FALSE]) >= 0,
                    na.rm = TRUE),
                msg = "argument to features correspond to negative values")
    # create rij matrix
    pos <- which(rowSums(!is.na(as.matrix(x[, cost_column, drop = FALSE]))) > 0)
    rij <- lapply(as.list(features), function(z) {
      r <- t(as.matrix(x[pos, z, drop = FALSE]))
      r[is.na(r)] <- 0
      rownames(r) <- feature_names(features)
      methods::as(r, "sparseMatrix")
    })
    names(rij) <- zone_names(features)
    # calculate feature abundances in total units
    fatu <- colSums(x[, unlist(as.list(features)), drop = FALSE],
                    na.rm = TRUE)
    fatu <- matrix(fatu, ncol = number_of_zones(features),
                   nrow = number_of_features(features),
                   dimnames = list(feature_names(features),
                                   zone_names(features)))
    # create ConservationProblem object
    pproto(NULL, ConservationProblem,
      constraints = pproto(NULL, Collection),
      penalties = pproto(NULL, Collection),
      data = list(cost = x, features = features, cost_column = cost_column,
                  rij_matrix = rij, feature_abundances_in_total_units = fatu))
})

#' @name problem
#' @usage \S4method{problem}{data.frame,data.frame}(x, features, rij, cost_column, zones, ...)
#' @rdname problem
methods::setMethod(
  "problem",
  methods::signature(x = "data.frame", features = "data.frame"),
  function(x, features, rij, cost_column, zones = NULL, ...) {
    # assert that arguments are valid
    assertthat::assert_that(
      inherits(x, "data.frame"), inherits(features, "data.frame"),
      is.character(cost_column), !anyNA(cost_column),
      inherits(rij, "data.frame"),
      nrow(x) > 0, nrow(features) > 0, nrow(rij) > 0,
      no_extra_arguments(...),
      # x
      assertthat::has_name(x, "id"), is.numeric(x$id), all(is.finite(x$id)),
      anyDuplicated(x$id) == 0, all(cost_column %in% names(x)),
      all(vapply(x[, cost_column, drop = FALSE], is.numeric, logical(1))),
      all(colSums(!is.na(as.matrix(x[, cost_column, drop = FALSE])),
                  na.rm = TRUE) > 0),
      # features
      assertthat::has_name(features, "id"),
      assertthat::has_name(features, "name"),
      anyDuplicated(features$id) == 0,
      anyDuplicated(features$name) == 0,
      !anyNA(features$id), !anyNA(features$name),
      is.numeric(features$id),
      is.character(features$name) || is.factor(features$name),
      # rij
      assertthat::has_name(rij, "pu"),
      assertthat::has_name(rij, "species"),
      assertthat::has_name(rij, "amount"),
      !anyNA(rij$pu), !anyNA(rij$species), !anyNA(rij$amount),
      is.numeric(rij$pu), is.numeric(rij$species), is.numeric(rij$amount),
      all(rij$pu %in% x$id),
      all(rij$species %in% features$id))
    # verifications
    verify_that(all(rij$amount >= 0))
    verify_that(all(as.matrix(x[, cost_column, drop = FALSE]) >= 0,
                    na.rm = TRUE),
                msg = "argument to x has negative cost values")
    # validate zone data
    if (!"zone" %in% names(rij))
      rij$zone <- 1
    if (length(unique(rij$zone)) > 1 && is.null(zones))
      stop("argument to zone must be specified for problems with multiple ",
           "zones")
    if (is.null(zones))
      zones <- data.frame(id = 1, name = cost_column)
    assertthat::assert_that(
      is.numeric(rij$zone),
      is.numeric(zones$id), is.character(zones$name) || is.factor(zones$name),
      !anyNA(rij$zone), !anyNA(zones$id), !anyNA(zones$name),
      anyDuplicated(zones$id) == 0, anyDuplicated(zones$name) == 0,
      nrow(zones) > 0, all(rij$zone %in% zones$id),
      nrow(zones) == length(cost_column))
    # standardize zone and feature ids
    rij$species <- match(rij$species, features$id)
    rij$zone <- match(rij$zone, zones$id)
    # calculate feature abundances in total units
    fatu <- Matrix::sparseMatrix(x = rij$amount, i = rij$species, j = rij$zone,
                                 use.last.ij = FALSE,
                                 dims = c(nrow(features), nrow(zones)),
                                 dimnames = list(as.character(features$name),
                                                 as.character(zones$name)))
    fatu <- as.matrix(fatu)
    # standardize planning unit ids
    pos <- which(rowSums(!is.na(as.matrix(x[, cost_column, drop = FALSE]))) > 0)
    rij$pu <- match(rij$pu, x$id[pos])
    rij <- rij[!is.na(rij$pu), ]
    # create rij matrix
    rij <- lapply(seq_along(zones$id), function(z) {
      r <- rij[rij$zone == z, ]
      Matrix::sparseMatrix(i = r$species, j = r$pu,
                           x = r$amount,
                           index1 = TRUE, use.last.ij = FALSE,
                           dims = c(nrow(features), length(pos)),
                           dimnames = list(features$name, NULL))
    })
    names(rij) <- as.character(zones$name)
    # create ConservationProblem object
    pproto(NULL, ConservationProblem,
      constraints = pproto(NULL, Collection),
      penalties = pproto(NULL, Collection),
      data = list(cost = x, features = features, cost_column = cost_column,
                  rij_matrix = rij, feature_abundances_in_total_units = fatu))
})

#' @name problem
#' @usage \S4method{problem}{numeric,data.frame}(x, features, rij_matrix, ...)
#' @rdname problem
methods::setMethod(
  "problem",
  methods::signature(x = "numeric", features = "data.frame"),
  function(x, features, rij_matrix, ...) {
    if (!is.list(rij_matrix))
      rij_matrix <- list("1" = rij_matrix)
    problem(matrix(x, ncol = 1), features, rij_matrix = rij_matrix)
})

#' @name problem
#' @usage \S4method{problem}{matrix,data.frame}(x, features, rij_matrix, ...)
#' @rdname problem
methods::setMethod(
  "problem",
  methods::signature(x = "matrix", features = "data.frame"),
  function(x, features, rij_matrix, ...) {
    # assert that arguments are valid
    if (!inherits(rij_matrix, "list"))
      rij_matrix <- list(rij_matrix)
    assertthat::assert_that(
      inherits(x, "matrix"), inherits(features, "data.frame"),
      inherits(rij_matrix, "list"),
      nrow(x) > 0, ncol(x) > 0, nrow(features) > 0, length(rij_matrix) > 0,
      no_extra_arguments(...),
      # x
      all(colSums(is.finite(x)) > 0),
      all(colSums(!is.na(x)) > 0),
      # features
      assertthat::has_name(features, "id"),
      assertthat::has_name(features, "name"),
      anyDuplicated(features$id) == 0,
      anyDuplicated(features$name) == 0,
      !anyNA(features$id), !anyNA(features$name),
      is.numeric(features$id),
      is.character(features$name) || is.factor(features$name),
      # rij_matrix
      all(vapply(rij_matrix, inherits, logical(1), c("matrix", "dgCMatrix"))),
      # multiple arguments
      ncol(x) == length(rij_matrix),
      all(vapply(rij_matrix, ncol, numeric(1)) == nrow(x)),
      all(vapply(rij_matrix, nrow, numeric(1)) == nrow(features)))
    # verifications
    verify_that(all(vapply(rij_matrix, min, numeric(1), na.rm = TRUE) >= 0),
                msg = "argument to rij_matrix has negative feature data")
    verify_that(all(x > 0, na.rm = TRUE))
    assertthat::assert_that(
      all(vapply(rij_matrix, FUN.VALUE = logical(1), function(x) {
        all(is.finite(c(min(x, na.rm = TRUE), max(x, na.rm = TRUE))))
      })),
      msg = "argument to x contains missing (NA) or non-finite (Inf) values")
    # add names to rij_matrix if missing
    if (is.null(names(rij_matrix)))
      names(rij_matrix) <- as.character(seq_along(rij_matrix))
    # calculate feature abundances in total units
    fatu <- vapply(rij_matrix, rowSums, numeric(nrow(rij_matrix[[1]])),
                   na.rm = TRUE)
    if (!is.matrix(fatu))
      fatu <- matrix(fatu, nrow = nrow(features), ncol = length(rij_matrix))
    rownames(fatu) <- as.character(features$name)
    colnames(fatu) <- names(rij_matrix)
    # convert rij matrices to sparse format if needed
    pos <- which(rowSums(!is.na(x)) > 0)
    rij <- lapply(rij_matrix, function(z) {
      if (inherits(z, "dgCMatrix")) {
        z@x[which(is.na(z@x))] <- 0
      } else {
        z[which(is.na(z))] <- 0
      }
      rownames(z) <- as.character(features$name)
      Matrix::drop0(methods::as(z[, pos, drop = FALSE], "dgCMatrix"))
    })
    names(rij) <- names(rij_matrix)
    # create new problem object
    pproto(NULL, ConservationProblem,
           constraints = pproto(NULL, Collection),
           penalties = pproto(NULL, Collection),
           data = list(cost = x, features = features, rij_matrix = rij,
                       feature_abundances_in_total_units = fatu))
})

#' @name problem
#' @usage \S4method{problem}{sf,Raster}(x, features, cost_column, run_checks, ...)
#' @rdname problem
methods::setMethod(
  "problem",
  methods::signature(x = "sf", features = "Raster"),
  function(x, features, cost_column, run_checks = TRUE, ...) {
    assertthat::assert_that(assertthat::is.string(cost_column))
    problem(x, zones(features, zone_names = cost_column,
                     feature_names = names(features)),
            cost_column = cost_column, run_checks = run_checks, ...)
})

#' @name problem
#' @usage \S4method{problem}{sf,ZonesRaster}(x, features, cost_column, run_checks, ...)
#' @rdname problem
methods::setMethod(
  "problem",
  methods::signature(x = "sf", features = "ZonesRaster"),
  function(x, features, cost_column, run_checks = TRUE, ...) {
    # assert that arguments are valid
    assertthat::assert_that(
      inherits(x, "sf"),
      no_extra_arguments(...),
      nrow(x) > 0, is.character(cost_column), !anyNA(cost_column),
      all(cost_column %in% names(x)),
      length(cost_column) == number_of_zones(features),
      assertthat::is.flag(run_checks))
    assertthat::assert_that(all(!geometry_classes(x) %in%
                                c("GEOMETRYCOLLECTION", "MULTIPOINT")))
    # further validation checks
    x2 <- sf::st_drop_geometry(x)
    assertthat::assert_that(
      all(vapply(x2[, cost_column, drop = FALSE], is.numeric, logical(1))),
      all(colSums(!is.na(as.matrix(x2[, cost_column, drop = FALSE])),
                  na.rm = TRUE) > 0),
      sf::st_crs(x) == sf::st_crs(features[[1]]@crs),
      intersecting_extents(x, features[[1]]))
      verify_that(
        all(colSums(as.matrix(x2[, cost_column, drop = FALSE]) < 0,
                    na.rm = TRUE) == 0),
        msg = "argument to x has negative cost values")
    if (run_checks) {
      verify_that(
        all(raster::cellStats(raster::stack(as.list(features)), "min") >= 0))
    }
    # compute rij matrix including non-planning unit cells
    rij <- rij_matrix(x, raster::stack(as.list(features)))
    rij <- lapply(seq_len(number_of_zones(features)), function(i) {
      m <- rij[((i - 1) * number_of_features(features)) +
               seq_len(number_of_features(features)), ,
          drop = FALSE]
      rownames(m) <- feature_names(features)
      return(m)
    })
    # calculate feature abundances in total units
    fatu <- vapply(rij, rowSums, numeric(number_of_features(features)),
                   na.rm = TRUE)
    if (!is.matrix(fatu))
      fatu <- matrix(fatu, nrow = number_of_features(features),
                     ncol = number_of_zones(features))
    rownames(fatu) <- feature_names(features)
    colnames(fatu) <- zone_names(features)
    # create rij matrix
    pos <- which(rowSums(!is.na(as.matrix(
             x2[, cost_column, drop = FALSE]))) > 0)
    rij <- lapply(rij, function(x) x[, pos, drop = FALSE])
    names(rij) <- zone_names(features)
    # create ConservationProblem object
    pproto(NULL, ConservationProblem,
      constraints = pproto(NULL, Collection),
      penalties = pproto(NULL, Collection),
      data = list(cost = x, features = features, cost_column = cost_column,
                  rij_matrix = rij, feature_abundances_in_total_units = fatu))
})

#' @name problem
#' @usage \S4method{problem}{sf,character}(x, features, cost_column, ...)
#' @rdname problem
methods::setMethod(
  "problem",
  methods::signature(x = "sf", features = "character"),
  function(x, features, cost_column, ...) {
    assertthat::assert_that(assertthat::is.string(cost_column))
    problem(x, zones(features, feature_names = features,
                     zone_names = cost_column),
            cost_column = cost_column, ...)
})

#' @name problem
#' @usage \S4method{problem}{sf,ZonesCharacter}(x, features, cost_column, ...)
#' @rdname problem
methods::setMethod(
  "problem",
  methods::signature(x = "sf", features = "ZonesCharacter"),
  function(x, features, cost_column, ...) {
    # assert that arguments are valid
    assertthat::assert_that(
      inherits(x, "sf"),
      inherits(features, "ZonesCharacter"),
      no_extra_arguments(...),
      nrow(x) > 0, is.character(cost_column), !anyNA(cost_column),
      all(cost_column %in% names(x)),
      number_of_zones(features) == length(cost_column))
    assertthat::assert_that(
      all(!geometry_classes(x) %in% c("GEOMETRYCOLLECTION", "MULTIPOINT")),
      msg = paste("argument to x contains invalid geometry types",
                  "(i.e., GEOMETRYCOLLECTION or MULTIPOINT)"))
    x2 <- sf::st_drop_geometry(x)
    assertthat::assert_that(
      all(unlist(as.list(features), recursive = TRUE, use.names = FALSE) %in%
                 names(x2)),
      msg = paste("argument to features contains column names that are",
                  "not present in the argument to x"))
    assertthat::assert_that(
      all(vapply(x2[, cost_column, drop = FALSE], is.numeric, logical(1))),
      msg = "argument to x has non-numeric cost data")
    assertthat::assert_that(
      all(colSums(!is.na(as.matrix(x2[, cost_column, drop = FALSE])),
        na.rm = TRUE) > 0),
      msg = "argument to x has missing (NA) cost values")
    assertthat::assert_that(
      all(vapply(x2[, cost_column, drop = FALSE], is.numeric, logical(1))),
      all(colSums(!is.na(as.matrix(x2[, cost_column, drop = FALSE])),
                  na.rm = TRUE) > 0))
    verify_that(all(as.matrix(x2[, cost_column, drop = FALSE]) >= 0,
                            na.rm = TRUE),
                msg = "argument to x has negative cost values")
    verify_that(all(as.matrix(x2[, unlist(features), drop = FALSE]) >= 0,
                    na.rm = TRUE),
                msg = "argument to features correspond to negative values")
    # create rij matrix
    pos <- which(rowSums(!is.na(as.matrix(
            x2[, cost_column, drop = FALSE]))) > 0)
    rij <- lapply(features, function(z) {
      r <- t(as.matrix(x2[pos, z, drop = FALSE]))
      r[is.na(r)] <- 0
      rownames(r) <- feature_names(features)
      methods::as(r, "sparseMatrix")
    })
    names(rij) <- zone_names(features)
    # calculate feature abundances in total units
    fatu <- colSums(x2[, unlist(as.list(features)), drop = FALSE],
                    na.rm = TRUE)
    fatu <- matrix(fatu, ncol = number_of_zones(features),
                   nrow = number_of_features(features),
                   dimnames = list(feature_names(features),
                                   zone_names(features)))
    # create ConservationProblem object
    pproto(NULL, ConservationProblem,
      constraints = pproto(NULL, Collection),
      penalties = pproto(NULL, Collection),
      data = list(cost = x, features = features, cost_column = cost_column,
                  rij_matrix = rij, feature_abundances_in_total_units = fatu))
})
