% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/rescalePopkin.R
\name{rescalePopkin}
\alias{rescalePopkin}
\title{Rescale kinship matrix to set a given kinship value to zero.}
\usage{
rescalePopkin(Phi, subpops = NULL, phiMin = NA)
}
\arguments{
\item{Phi}{An \eqn{n \times n}{n-by-n} kinship matrix.}

\item{subpops}{The length-\eqn{n} vector of subpopulation assignments for each individual.}

\item{phiMin}{A scalar kinship value to define the new zero kinship.}
}
\value{
The rescaled \eqn{n \times n}{n-by-n} kinship matrix, with the desired level of relatedness set to zero.
}
\description{
Rescales the input kinship matrix \eqn{\Phi^T} so that the value \eqn{\phi_{\mbox{min}}^T}{\phi_min^T} in the original kinship matrix becomes zero, using the formula
\deqn{\Phi^{T'} = \frac{\Phi^T - \phi_{\mbox{min}}^T}{1 - \phi_{\mbox{min}}^T}.}{\Phi^T' = (\Phi^T - \phi_min^T)/(1 - \phi_min^T).}
This is equivalent to changing the ancestral population \eqn{T} into \eqn{T'} such that \eqn{\phi_{\mbox{min}}^{T'} = 0}{\phi_min^T' = 0}.
If subpopulation labels \code{subpops} are provided, they are used to estimate \eqn{\phi_{\mbox{min}}^T}{\phi_min^T}.
If both \code{subpops} and \code{phiMin} are provided, only \code{phiMin} is used.
If both \code{subpops} and \code{phiMin} are omitted, the adjustment is equivalent to \code{phiMin=min(Phi)}.
}
\examples{
## Construct toy data
X <- matrix(c(0,1,2,1,0,1,1,0,2), nrow=3, byrow=TRUE) # genotype matrix
subpops <- c(1,1,2) # subpopulation assignments for individuals
subpops2 <- 1:3 # alternate labels treat every individual as a different subpop

## NOTE: for BED-formatted input, use BEDMatrix!
## "file" is path to BED file (excluding .bed extension)
# library(BEDMatrix)
# X <- BEDMatrix(file) # load genotype matrix object

## suppose we first estimate kinship without subpopulations, which will be more biased
Phi <- popkin(X) # calculate kinship from genotypes, WITHOUT subpops
## then we visualize this matrix, figure out a reasonable subpopulation partition

## now we can adjust the kinship matrix!
Phi2 <- rescalePopkin(Phi, subpops)
## prev is faster but otherwise equivalent to re-estimating Phi from scratch with subpops:
## Phi2 <- popkin(X, subpops) 

## can also manually set the level of relatedness phiMin we want to be zero:
phiMin <- min(Phi) # a naive choice for example
Phi2 <- rescalePopkin(Phi, phiMin=phiMin)

## lastly, omiting both subpops and phiMin sets the minimum value in Phi to zero
Phi3 <- rescalePopkin(Phi2)
## equivalent to both of:
## Phi3 <- popkin(X)
## Phi3 <- rescalePopkin(Phi2, phiMin=min(Phi))

}
