% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/POMDP_accessors.R
\name{POMDP_accessors}
\alias{POMDP_accessors}
\alias{transition_matrix}
\alias{transition_val}
\alias{observation_matrix}
\alias{observation_val}
\alias{reward_matrix}
\alias{reward_val}
\alias{start_vector}
\alias{normalize_POMDP}
\alias{normalize_MDP}
\title{Access to Parts of the POMDP Description}
\usage{
transition_matrix(
  x,
  action = NULL,
  episode = NULL,
  epoch = NULL,
  sparse = TRUE
)

transition_val(x, action, start.state, end.state, episode = NULL, epoch = NULL)

observation_matrix(
  x,
  action = NULL,
  episode = NULL,
  epoch = NULL,
  sparse = TRUE
)

observation_val(
  x,
  action,
  end.state,
  observation,
  episode = NULL,
  epoch = NULL
)

reward_matrix(
  x,
  action = NULL,
  start.state = NULL,
  episode = NULL,
  epoch = NULL,
  sparse = FALSE
)

reward_val(
  x,
  action,
  start.state,
  end.state,
  observation,
  episode = NULL,
  epoch = NULL
)

start_vector(x)

normalize_POMDP(x, sparse = TRUE)

normalize_MDP(x, sparse = TRUE)
}
\arguments{
\item{x}{A \link{POMDP} or \link{MDP} object.}

\item{action}{name or index of an action.}

\item{episode, epoch}{Episode or epoch used for time-dependent POMDPs. Epoch are internally converted
to the episode using the model horizon.}

\item{sparse}{logical; use sparse matrices when the density is below 50\% . \code{NULL} returns the
representation stored in the problem description.}

\item{start.state, end.state}{name or index of the state.}

\item{observation}{name or index of observation.}
}
\value{
A list or a list of lists of matrices.
}
\description{
Functions to provide uniform access to different parts of the POMDP description.
}
\details{
Several parts of the POMDP description can be defined in different ways. In particular,
\code{transition_prob}, \code{observation_prob}, \code{reward}, and \code{start} can be defined using matrices, data frames or
keywords. See \link{POMDP} for details. The functions provided here, provide unified access to the data in these fields
to make writing code easier.
\itemize{
\item \code{start_vector()} translates the initial probability vector into a vector.
\item For \code{transition_prob}, \code{observation_prob}, \code{reward}, functions ending in \verb{_matrix()} and \verb{_val()} are
provided to access the data as lists of matrices, a matrix or a scalar value. For \verb{_matrix()},
matrices with a density below 50\% can be requested in sparse format (as a \link[Matrix:dgCMatrix-class]{Matrix::dgCMatrix}).
\item \code{normalize_POMDP()} returns a new POMDP definition where \code{transition_prob},
\code{observations_prob}, \code{reward}, and \code{start} are normalized to (lists of) matrices and vectors to
make direct access easy.  Also, \code{states}, \code{actions}, and \code{observations} are ordered as given in the problem
definition to make safe access using numerical indices possible. Normalized POMDP descriptions are used for
C++ based code (e.g., \code{\link[=simulate_POMDP]{simulate_POMDP()}}) and normalizing them once will save time if the code is
called repeatedly.
}
}
\examples{
data("Tiger")

# List of |A| transition matrices. One per action in the from states x states
Tiger$transition_prob
transition_matrix(Tiger)
transition_val(Tiger, action = "listen", start.state = "tiger-left", end.state = "tiger-left")

# List of |A| observation matrices. One per action in the from states x observations
Tiger$observation_prob
observation_matrix(Tiger)
observation_val(Tiger, action = "listen", end.state = "tiger-left", observation = "tiger-left")

# List of list of reward matrices. 1st level is action and second level is the
#  start state in the form end state x observation
Tiger$reward
reward_matrix(Tiger)
reward_val(Tiger, action = "open-right", start.state = "tiger-left", end.state = "tiger-left", 
  observation = "tiger-left")

# Translate the initial belief vector
Tiger$start
start_vector(Tiger)

# Normalize the whole model
Tiger_norm <- normalize_POMDP(Tiger)
Tiger_norm$transition_prob

## Visualize transition matrix for action 'open-left'
library("igraph")
g <- graph_from_adjacency_matrix(transition_matrix(Tiger, action = "open-left"), weighted = TRUE)
edge_attr(g, "label") <- edge_attr(g, "weight")

igraph.options("edge.curved" = TRUE)
plot(g, layout = layout_on_grid, main = "Transitions for action 'open=left'")

## Use a function for the Tiger transition model
trans <- function(action, end.state, start.state) {
  ## listen has an identity matrix
  if (action == 'listen')
    if (end.state == start.state) return(1)
    else return(0)

  # other actions have a uniform distribution
  return(1/2)
}

Tiger$transition_prob <- trans

# transition_matrix evaluates the function
transition_matrix(Tiger)
}
\seealso{
Other POMDP: 
\code{\link{POMDP}()},
\code{\link{plot_belief_space}()},
\code{\link{plot_value_function}()},
\code{\link{projection}()},
\code{\link{sample_belief_space}()},
\code{\link{simulate_POMDP}()},
\code{\link{solve_POMDP}()},
\code{\link{solve_SARSOP}()},
\code{\link{update_belief}()},
\code{\link{write_POMDP}()}

Other MDP: 
\code{\link{MDP}()},
\code{\link{simulate_MDP}()},
\code{\link{solve_MDP}()}
}
\author{
Michael Hahsler
}
\concept{MDP}
\concept{POMDP}
