% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/poismf.R
\name{poismf}
\alias{poismf}
\title{Factorization of Sparse Counts Matrices through Poisson Likelihood}
\usage{
poismf(X, k = 50, l1_reg = 0, l2_reg = 1e+09, niter = 10,
  nupd = 1, step_size = 1e-07, init_type = "gamma", seed = 1,
  nthreads = -1)
}
\arguments{
\item{X}{The matrix to factorize. Can be:
a) a `data.frame` with 3 columns, containing in this order:
row index (starting at 1), column index, count value (the indices can also be character type, in wich
case it will enumerate them internally, and will return those same characters from `predict`);
b) A sparse matrix in COO format from the `SparseM` package;
c) a full matrix (of class `matrix` or `Matrix::dgTMatrix`);
d) a sparse matrix from package `Matrix` in triplets format.}

\item{k}{Dimensionality of the factorization (a.k.a. number of latent factors).}

\item{l1_reg}{Strength of the l1 regularization}

\item{l2_reg}{Strength of the l2 regularization.}

\item{niter}{Number of iterations to run.}

\item{nupd}{Number of updates per iteration.}

\item{step_size}{Initial step size to use (proximal gradient only). Will be decreased by 1/2 after each iteration.}

\item{init_type}{One of "gamma" or "uniform" (How to intialize the factorizing matrices).}

\item{seed}{Random seed to use for starting the factorizing matrices.}

\item{nthreads}{Number of parallel threads to use. Passing a negative number will use
the maximum available number of threads}
}
\value{
An object of class `poismf` with the following fields of interest:
}
\description{
Creates a low-rank non-negative factorization of a sparse counts matrix by
maximizing Poisson likelihood with L1/L2 regularization, using optimization routines
based on proximal gradient iteration.
}
\section{Fields}{

\describe{
\item{\code{A}}{: the user/document/row-factor matrix (as a vector, has to be reshaped to (nrows, k)).}

\item{\code{B}}{: the item/word/column-factor matrix (as a vector, has to be reshaped to (ncols, k)).}
}}

\examples{
library(poismf)

### create a random sparse data frame in COO format
nrow <- 10 ** 2
ncol <- 10 ** 3
nnz  <- 10 ** 4
set.seed(1)
X <- data.frame(
    row_ix = as.integer(runif(nnz, min = 1, max = nrow)),
    col_ix = as.integer(runif(nnz, min = 1, max = ncol)),
    count = rpois(nnz, 1) + 1)
X <- X[!duplicated(X[, c("row_ix", "col_ix")]), ]

### factorize the randomly-generated sparse matrix
model <- poismf(X, nthreads = 1)

### predict functionality
predict(model, 1, 10) ## predict entry (1, 10)
predict(model, 1, topN = 10) ## predict top-10 entries "B" for row 1 of "A".
predict(model, c(1, 1, 1), c(4, 5, 6)) ## predict entries [1,4], [1,5], [1,6]
head(predict(model, 1)) ## predict the whole row 1

#all predictions for new row/user/doc
head(predict(model, data.frame(col_ix = c(1,2,3), count = c(4,5,6)) ))
}
\references{
Cortes, David. "Fast Non-Bayesian Poisson Factorization for Implicit-Feedback Recommendations." arXiv preprint arXiv:1811.01908 (2018).
}
\seealso{
\link{predict.poismf} \link{predict_all}
}
