% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/table_transformers.R
\name{tt_summary_stats}
\alias{tt_summary_stats}
\title{Table Transformer: obtain a summary stats table for numeric columns}
\usage{
tt_summary_stats(tbl)
}
\arguments{
\item{tbl}{A table object to be used as input for the transformation. This
can be a data frame, a tibble, a \code{tbl_dbi} object, or a \code{tbl_spark} object.}
}
\value{
A \code{tibble} object.
}
\description{
With any table object, you can produce a summary table that is scoped to the
numeric column values. The output summary table will have a leading column
called \code{".param."} with labels for each of the nine rows, each corresponding
to the following summary statistics:
\enumerate{
\item Minimum (\code{"min"})
\item 5th Percentile (\code{"p05"})
\item 1st Quartile (\code{"q_1"})
\item Median (\code{"med"})
\item 3rd Quartile (\code{"q_3"})
\item 95th Percentile (\code{"p95"})
\item Maximum (\code{"max"})
\item Interquartile Range (\code{"iqr"})
\item Range (\code{"range"})
}

Only numerical data from the input table will generate columns in the output
table. Column names from the input will be used in the output, preserving
order as well.
}
\section{Function ID}{

12-1
}

\examples{
# Get summary statistics for the
# `game_revenue` dataset that is
# included in the package
tt_summary_stats(game_revenue)

# Ensure that the maximum revenue
# for individual purchases in the
# `game_revenue` table is less than
# $150
tt_summary_stats(game_revenue) \%>\%
  col_vals_lt(
    columns = vars(item_revenue),
    value = 150,
    segments = .param. ~ "max"
  )

# For in-app purchases in the
# `game_revenue` table, check that
# median revenue is somewhere
# between $8 and $12
game_revenue \%>\% 
  dplyr::filter(item_type == "iap") \%>\%
  tt_summary_stats() \%>\%
  col_vals_between(
    columns = vars(item_revenue),
    left = 8, right = 12,
    segments = .param. ~ "med"
  )

# While performing validations of the
# `game_revenue` table with an agent
# we can include the same revenue
# check by using `tt_summary_stats()`
# in the `preconditions` argument (this
# will transform the target table for
# the validation step); we also need
# to get just a segment of that table
# (the row with the median values)
agent <- 
  create_agent(
    read_fn = ~ game_revenue,
    tbl_name = "game_revenue",
    label = "An example.",
    actions = action_levels(
      warn_at = 0.10,
      stop_at = 0.25,
      notify_at = 0.35
    )
  ) \%>\%
  rows_complete() \%>\%
  rows_distinct() \%>\%
  col_vals_between(
    columns = vars(item_revenue),
    left = 8, right = 12,
    preconditions = ~ . \%>\%
      dplyr::filter(item_type == "iap") \%>\%
      tt_summary_stats(),
    segments = .param. ~ "med"
  ) \%>\%
  interrogate()

# This should all pass but let's check:
all_passed(agent)

}
\seealso{
Other Table Transformers: 
\code{\link{get_tt_param}()},
\code{\link{tt_string_info}()},
\code{\link{tt_tbl_colnames}()},
\code{\link{tt_tbl_dims}()},
\code{\link{tt_time_shift}()},
\code{\link{tt_time_slice}()}
}
\concept{Table Transformers}
