% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/action_levels.R
\name{action_levels}
\alias{action_levels}
\title{Set action levels for reacting to exceeding thresholds}
\usage{
action_levels(warn_at = NULL, stop_at = NULL, notify_at = NULL, fns = NULL)
}
\arguments{
\item{warn_at, stop_at, notify_at}{The threshold number or fraction of
validation units that can provide a \emph{fail} result before entering the
\code{warn}, \code{stop}, or \code{notify} states.}

\item{fns}{A named list of functions that can be used with each action type.
The syntax for this list involves using names from the set of \code{warn},
\code{stop}, and \code{notify}. The functions corresponding to the states are
provided as formulas (e.g., \code{list(warn = ~ warning("Too many failures."))}.
A series of expressions for each named state can be used by enclosing the
set of statements with \code{{ }}.}
}
\description{
This helper function works with the \code{actions} argument that is present in
every validation step function. With it, we can provide threshold \emph{fail}
levels for any combination of \code{warn}, \code{stop}, or \code{notify} states. We can
react to any entrance of a state by supplying corresponding functions to the
\code{fns} argument. They will undergo evaluation at the time when the matching
state is entered.
}
\details{
The output of the \code{action_levels()} call in \code{actions} will be interpreted
slightly different if using an \emph{agent} or using a validation step function
directly on a data table. For convenience, when working directly on data
any values supplied to \code{warn_at} or \code{stop_at} will be automatically given a
stock \code{warning()} or \code{stop()} function. If you were to supply those manually
then the stock functions would be overridden. In the that interactive data
case there is no automatic reaction function given for the \code{notify} state (as
that state is less commonly used and should instead be intended for custom
reporting functions).

When using an \emph{agent}, we often opt to not use any functions in \code{fns} as the
\code{warn}, \code{stop}, and \code{notify} states will be reported on when using
\code{create_agent_report()} (and, usually that's sufficient).
}
\examples{
library(dplyr)

# Create a simple data frame with
# a column of numerical values
tbl <- tibble(a = c(5, 7, 8, 5))

# Create an `action_levels()` list
# with fractional values for the
# `warn`, `stop`, and `notify` states
al <- 
  action_levels(
    warn_at = 0.2,
    stop_at = 0.8,
    notify_at = 0.345
  )

# Validate that values in column
# `a` are always greater than 7 and
# apply the list of action levels
agent <-
  create_agent(tbl = tbl) \%>\%
  col_vals_gt(vars(a), 7, actions = al) \%>\%
  interrogate()

# The report from the agent will show
# that the `warn` state has been entered
# for the first and only validation step.
agent \%>\%
  get_agent_report(display_table = FALSE)
  
# In the context of using validation
# stop functions directly on data, their
# use is commonly to trigger warnings
# and raise errors. The following will
# provide a warning (but that's suppressed
# here).
suppressWarnings(
  tbl \%>\%
    col_vals_gt(vars(a), 5, actions = al)
)
  
}
