#' @name sprinkle_height
#' @title Adjust Table Cell Height
#' 
#' @description Customize the height of a cell in a table. This may be done
#'   to improve the appearance of cells with long text.
#'   
#' @param x An object of class \code{dust}
#' @param rows Either a numeric vector of rows in the tabular object to be 
#'   modified or an object of class \code{call}.  When a \code{call}, 
#'   generated by \code{quote(expression)}, the expression resolves to 
#'   a logical vector the same length as the number of rows in the table.
#'   Sprinkles are applied to where the expression resolves to \code{TRUE}.
#' @param cols Either a numeric vector of columns in the tabular object to
#'   be modified, or a character vector of column names. A mixture of 
#'   character and numeric indices is permissible.
#' @param height \code{numeric(1)}. Gives the height of the cell.
#' @param height_units \code{character(1)}. Gives the units for \code{height}.
#'   One of \code{c("pt", "px", "cm", "in", "\%")}
#' @param part A character string denoting which part of the table to modify.
#' @param fixed \code{logical(1)} indicating if the values in \code{rows} 
#'   and \code{cols} should be read as fixed coordinate pairs.  By default, 
#'   sprinkles are applied at the intersection of \code{rows} and \code{cols}, 
#'   meaning that the arguments do not have to share the same length.  
#'   When \code{fixed = TRUE}, they must share the same length.
#' @param recycle A \code{character} one that determines how sprinkles are 
#'   managed when the sprinkle input doesn't match the length of the region
#'   to be sprinkled.  By default, recycling is turned off.  Recycling 
#'   may be performed across rows first (left to right, top to bottom), 
#'   or down columns first (top to bottom, left to right).
#' @param ... Additional arguments to pass to other methods. Currently ignored.
#' 
#' @details This sprinkle is only recognized by HTML and LaTeX.  All of the 
#'   \code{height_units} values are recognized by HTML.  For LaTeX, \code{"px"}
#'   is converted to \code{"pt"}. 
#'   
#' @section Functional Requirements:
#' \enumerate{
#'  \item Correctly reassigns the appropriate elements of \code{height} 
#'    and \code{height_units} columns in the table part.
#'  \item Casts an error if \code{x} is not a \code{dust} object.
#'  \item Casts an error if \code{height} is not a \code{numeric(1)}
#'  \item Casts an error if \code{height_units} is not a \code{character(1)}
#'  \item Casts an error if \code{part} is not one of \code{"body"}, 
#'    \code{"head"}, \code{"foot"}, or \code{"interfoot"}
#'  \item Casts an error if \code{fixed} is not a \code{logical(1)}
#'  \item Casts an error if \code{recycle} is not one of \code{"none"},
#'    \code{"rows"}, or \code{"cols"}
#'  \item Cast an error if \code{recycle = "none"} and \code{height}
#'    does not have length 1.
#'  \item When \code{recycle = "none"}, quietly coerce \code{height_units}
#'    to just the first element given.
#' }
#' 
#' The functional behavior of the \code{fixed} and \code{recycle} arguments 
#' is not tested for this function. It is tested and validated in the
#' tests for \code{\link{index_to_sprinkle}}.
#' 
#' @seealso \code{\link{sprinkle}}, 
#'   \code{\link{index_to_sprinkle}}
#'   
#' @export

sprinkle_height <- function(x, rows = NULL, cols = NULL, 
                            height = NULL, height_units = NULL,
                            part = c("body", "head", "foot", "interfoot", "table"),
                            fixed = FALSE, 
                            recycle = c("none", "rows", "cols", "columns"),
                            ...)
{
  UseMethod("sprinkle_height")
}

#' @rdname sprinkle_height
#' @export

sprinkle_height.default <- function(x, rows = NULL, cols = NULL, 
                                 height = NULL, height_units = NULL,
                                 part = c("body", "head", "foot", "interfoot", "table"),
                                 fixed = FALSE, 
                                 recycle = c("none", "rows", "cols", "columns"),
                                 ...)
{
  coll <- checkmate::makeAssertCollection()
  
  checkmate::assert_class(x = x,
                          classes = "dust",
                          add = coll)
  
  indices <- index_to_sprinkle(x = x, 
                               rows = rows, 
                               cols = cols, 
                               fixed = fixed,
                               part = part,
                               recycle = recycle,
                               coll = coll)
  
  recycle <- recycle[1]
  
  sprinkle_height_index_assert(height = height, 
                               height_units = height_units, 
                               recycle = recycle,
                               coll = coll)
  
  checkmate::reportAssertions(coll)
  
  part <- part[1]
  
  sprinkle_height_index(x = x, 
                        indices = indices, 
                        height = height, 
                        height_units = height_units, 
                        part = part)
}

#' @rdname sprinkle_height
#' @export

sprinkle_height.dust_list <- function(x, rows = NULL, cols = NULL, 
                                      height = NULL, height_units = NULL,
                                      part = c("body", "head", "foot", "interfoot", "table"),
                                      fixed = FALSE, 
                                      recycle = c("none", "rows", "cols", "columns"),
                                      ...)
{
  structure(
    lapply(X = x,
           FUN = sprinkle_height.default,
           rows = rows,
           cols = cols,
           height = height,
           height_units = height_units,
           part = part,
           fixed = fixed,
           recycle = recycle,
           ...),
    class = "dust_list"
  )
}

# Unexported Utility ------------------------------------------------

# These functions are to be used inside of the general `sprinkle` call
# When used inside `sprinkle`, the indices are already determined, 
# the only the `height` and `height_units` arguments needs to be validated. 
# The assert function is kept separate so it may be called earlier
# without attempting to perform the assignment.

sprinkle_height_index_assert <- function(height = NULL, height_units = NULL, 
                                         recycle = "none", coll)
{
  if (!is.null(height))
  {
    checkmate::assert_numeric(x = height,
                              add = coll,
                              .var.name = "height")
    
    if (recycle == "none" && length(height) != 1)
      coll$push("When `recycle` = 'none', height must have length 1.")
  }
  
  if (!is.null(height_units))
  {
    checkmate::assert_subset(x = height_units,
                             choices = c("px", "pt", "in", "cm", "%"),
                             add = coll,
                             .var.name = "height_units")
    
    if (recycle == "none")
      height_units <- height_units[1]
  }
}

sprinkle_height_index <- function(x, indices, height = NULL, height_units = NULL, part)
{
  if (!is.null(height))
  {
    x[[part]][["height"]][indices] <- height
  }
  
  if (!is.null(height_units))
  {
    x[[part]][["height_units"]][indices] <- height_units
  }
  
  x
}