% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/b31gmodpf.R
\name{b31gmodpf}
\alias{b31gmodpf}
\title{ASME B31G. Failure pressure of the corroded pipe (modified)}
\usage{
b31gmodpf(d, wth, smys, depth, l)
}
\arguments{
\item{d}{nominal outside diameter of the pipe, [\emph{inch}]. Type: \code{\link{assert_double}}.}

\item{wth}{nominal wall thickness of the pipe, [\emph{inch}]. Type: \code{\link{assert_double}}.}

\item{smys}{specified minimum yield of stress (\emph{SMYS}) as a
characteristics of steel strength, [\emph{PSI}]. Type: \code{\link{assert_double}}.}

\item{depth}{measured maximum depth of the corroded area, [\emph{inch}]. Type: \code{\link{assert_double}}.}

\item{l}{measured maximum longitudinal length of corroded area, [\emph{inch}]. Type: \code{\link{assert_double}}.}
}
\value{
Estimated failure pressure of the corroded pipe, [\emph{PSI}]. Type: \code{\link{assert_double}}.
}
\description{
Calculate failure pressure of the corroded pipe
 according to \emph{Modified B31G},
 \emph{Level-1} algorithm listed
 in \href{https://www.asme.org/codes-standards/find-codes-standards/b31g-manual-determining-remaining-strength-corroded-pipelines}{ASME B31G-2012}.

 The next assumption of the corrosion shape is adopted by \emph{Modified B31G}:

 \figure{b31gmodpf.png}

 There d\emph{cor} represents argument \code{depth}.
}
\details{
Since the definition of flow stress, \emph{Sflow}, in
  \href{https://www.asme.org/codes-standards/find-codes-standards/b31g-manual-determining-remaining-strength-corroded-pipelines}{ASME B31G-2012}
  is recommended with \emph{Level 1} as follows:

  \deqn{Sflow =  1.1SMYS}

  no other possibilities of its evaluation are incorporated.

  For this code we avoid possible semantic optimization to
  preserve readability and correlation with original text description
  in \href{https://www.asme.org/codes-standards/find-codes-standards/b31g-manual-determining-remaining-strength-corroded-pipelines}{ASME B31G-2012}.
  At the same time source code for estimated failure pressure preserves
  maximum affinity with its semantic description in \href{https://www.asme.org/codes-standards/find-codes-standards/b31g-manual-determining-remaining-strength-corroded-pipelines}{ASME B31G-2012}.

  Numeric \code{NA}s may appear in case prescribed conditions of
  use are offended.
}
\examples{
 ## Example: maximum percentage disparity of original B31G
 ## algorithm and modified B31G showed on CRVL.BAS data
 with(b31gdata, {
   original  <- b31gpf(d, wth, smys, depth, l)
   modified  <- b31gmodpf(d, wth, smys, depth, l)
   round(max(100*abs(1 - original/modified), na.rm = TRUE), 4)
 })
 ## Output:
 #[1] 32.6666

 ## Example: plot disparity of original B31G algorithm and
 ## modified B31G showed on CRVL data
 with(b31gdata[-(6:7),], {
   b31g  <- b31gpf(depth, wth, smys, depth, l)
   b31gmod  <- b31gmodpf(depth, wth, smys, depth, l)
   axe_range <- range(c(b31g, b31gmod))
   plot(b31g, b31g, type = 'b', pch = 16,
        xlab = 'Pressure, [PSI]',
        ylab = 'Pressure, [PSI]',
        main = 'Failure pressure method comparison',
        xlim = axe_range, ylim = axe_range)
   inc <- order(b31g)
   lines(b31g[inc], b31gmod[inc], type = 'b', col = 'red')
   legend('topleft',
          legend = c('B31G Original',
                     'B31G Modified'),
          col = c('black', 'red'),
          lty = 'solid')
 })

}
\references{
\enumerate{
 \item \href{https://www.techstreet.com/standards/asme-b31g-2012-r2017?product_id=1842873}{ASME B31G-2012}.
   Manual for determining the remaining strength of corroded pipelines:
   supplement to \emph{B31 Code} for pressure piping.
 \item  S. Timashev and A. Bushinskaya, \emph{Diagnostics and Reliability
   of Pipeline Systems}, Topics in Safety, Risk, Reliability and Quality 30,
   \strong{DOI 10.1007/978-3-319-25307-7}
 }
}
\seealso{
Other fail pressure functions: \code{\link{b31gpf}}, \code{\link{dnvpf}},
  \code{\link{shell92pf}}, \code{\link{pcorrcpf}}

Other ASME B31G functions: 
\code{\link{b31crvl}()},
\code{\link{b31gacd}()},
\code{\link{b31gacl}()},
\code{\link{b31gafr}()},
\code{\link{b31gdep}()},
\code{\link{b31gops}()},
\code{\link{b31gpf}()},
\code{\link{b31gsap}()}
}
\concept{ASME B31G functions}
