% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/rsquared.R
\name{rsquared}
\alias{rsquared}
\title{R-squared for linear regression}
\usage{
rsquared(modelList, method = NULL)
}
\arguments{
\item{modelList}{a regression, or a list of structural equations.}

\item{method}{The method used to compute the R2 value (See Details)}
}
\value{
Returns a \code{data.frame} with the response, its family and link,
the method used to estimate R2, and the R2 value itself. Mixed models also
return marginal and conditional R2 values.
}
\description{
Returns (pseudo)-R^2 values for all linear, generalized linear, and
generalized linear mixed effects models.
}
\details{
For mixed models, marginal R2 considers only the variance by the fixed
effects, and the conditional R2 by both the fixed and random effects.

For generalized additive models fit to gaussian distribution, the function
returns ths adjusted-R2. For all other distributions, it returns the proportion
of deviance explained.

For GLMs (\code{glm}), supported methods include: \itemize{
\item\code{mcfadden} 1 - ratio of likelihoods of full vs. null models

\item\code{coxsnell} McFadden's R2 but raised to 2/N. Upper limit is < 1

\item\code{nagelkerke} Adjusts Cox-Snell R2 so that upper limit = 1. The
DEFAULT method

} For GLMERs fit to Poisson, Gamma, and negative binomial distributions
(\code{glmer}, \code{glmmPQL}, \code{glmer.nb}), supported methods include
\itemize{ \item\code{delta} Approximates the observation variance based on
second-order Taylor series expansion. Can be used with many families and
link functions

\item\code{lognormal} Observation variance is the variance of the log-normal
distribution

\item\code{trigamma} Provides most accurate estimate of the observation
variance but is limited to only the log link. The DEFAULT method

} For GLMERs fit to the binomial distribution (\code{glmer},
\code{glmmPQL}), supported methods include: \itemize{
\item\code{theoretical} Assumes observation variance is pi^2/3

\item\code{delta} Approximates the observation variance as above. The
DEFAULT method

}
}
\examples{

  \dontrun{
    # Create data
    dat <- data.frame(
      ynorm = rnorm(100),
      ypois = rpois(100, 100),
      x1 = rnorm(100),
      random = letters[1:5]
    )

    # Get R2 for linear model
    rsquared(lm(ynorm ~ x1, dat))

    # Get R2 for generalized linear model
    rsquared(glm(ypois ~ x1, "poisson", dat))

    rsquared(glm(ypois ~ x1, "poisson", dat), method = "mcfadden") # McFadden R2

    # Get R2 for generalized least-squares model
    rsquared(gls(ynorm ~ x1, dat))

    # Get R2 for linear mixed effects model (nlme)
    rsquared(nlme::lme(ynorm ~ x1, random = ~ 1 | random, dat))

    # Get R2 for linear mixed effects model (lme4)
    rsquared(lme4::lmer(ynorm ~ x1 + (1 | random), dat))

    # Get R2 for generalized linear mixed effects model (lme4)
    rsquared(lme4::glmer(ypois ~ x1 + (1 | random), family = poisson, dat))

    rsquared(lme4::glmer(ypois ~ x1 + (1 | random), family = poisson, dat), method = "delta")

    # Get R2 for generalized linear mixed effects model (glmmPQL)
    rsquared(MASS::glmmPQL(ypois ~ x1, random = ~ 1 | random, family = poisson, dat))
    
    # Get R2 for generalized additive models (gam)
    rsquared(mgcv::gam(ynorm ~ x1, dat))
  }

}
\references{
Nakagawa, Shinichi, Paul CD Johnson, and Holger Schielzeth. 
"The coefficient of determination R 2 and intra-class correlation coefficient 
from generalized linear mixed-effects models revisited and expanded." 
Journal of the Royal Society Interface 14.134 (2017): 20170213.
}
\author{
Jon Lefcheck <lefcheckj@si.edu>
}
