% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/water.vapour.R
\name{water_vp_sat}
\alias{water_vp_sat}
\alias{water_dp}
\alias{water_fp}
\alias{water_vp2mvc}
\alias{water_mvc2vp}
\alias{water_vp2RH}
\alias{water_RH2vp}
\title{Water vapour pressure}
\usage{
water_vp_sat(
  temperature,
  over.ice = FALSE,
  method = "tetens",
  check.range = TRUE
)

water_dp(water.vp, over.ice = FALSE, method = "tetens", check.range = TRUE)

water_fp(water.vp, over.ice = TRUE, method = "tetens", check.range = TRUE)

water_vp2mvc(water.vp, temperature)

water_mvc2vp(water.mvc, temperature)

water_vp2RH(
  water.vp,
  temperature,
  over.ice = FALSE,
  method = "tetens",
  pc = TRUE,
  check.range = TRUE
)

water_RH2vp(
  relative.humidity,
  temperature,
  over.ice = FALSE,
  method = "tetens",
  pc = TRUE,
  check.range = TRUE
)
}
\arguments{
\item{temperature}{numeric vector of air temperatures (C).}

\item{over.ice}{logical Is the estimate for equilibrium with liquid water or
with ice.}

\item{method}{character Currently "tetens", modified "magnus", "wexler" and
"goff.gratch" equations  are supported.}

\item{check.range}{logical Flag indicating whether to check or not that
arguments for temperature are within the range of validity of the
\code{method} used.}

\item{water.vp}{numeric vector of water vapour pressure in air (Pa).}

\item{water.mvc}{numeric vector of water vapour concnetration as mass per
volume (g m-3).}

\item{pc}{logical flag for result returned as percent or not.}

\item{relative.humidity}{numeric Relative humedity as % (default) or as a
fraction of 1.}
}
\value{
A numeric vector of partial pressures in pascal (P) for
  \code{water_vp_sat} and \code{water_mvc2vp}, a numeric vector of dew point
  temperatures (C) for \code{water_dp} and numeric vector of mass per volume
  concentrations (g m-3) for \code{water_vp2mvc}.
}
\description{
Approximate water pressure in air as a function of temperature, and its
inverse the calculation of dewpoint.
}
\details{
Function \code{water_vp_sat()} provides implementations of several
  well known equations for the estimation of saturation vapor pressure in
  air. Functions \code{water_dp()} and \code{water_fp()} use the inverse of
  these equations to compute the dew point or frost point from water vapour
  pressure in air. The inverse functions are either analytical solutions or
  fitted approximations. None of these functions are solved numerically by
  iteration.

  Method \code{"tetens"} implements Tetens' (1930) equation for the cases of
  equilibrium with a water and an ice surface. Method \code{"magnus"}
  implements the modified Magnus equations of Alduchov and Eskridge (1996,
  eqs. 21 and 23). Method \code{"wexler"} implements the equations proposed
  by Wexler (1976, 1977), and their inverse according to Hardy (1998). Method
  \code{"goff.gratch"} implements the equations of Groff and Gratch (1946)
  with the minor updates of Groff (1956).

  The equations are approximations, and in spite of their different names,
  Tetens' and Magnus' equations have the same form with the only difference
  in the values of the parameters. However, the modified Magnus equation is
  more accurate as Tetens equation suffers from some bias errors at extreme
  low temperatures (< -40 C). In contrast Magnus equations with recently
  fitted values for the parameters are usable for temperatures from -80 C to
  +50 C over water and -80 C to 0 C over ice. The Groff Gratch equation is
  more complex and is frequently used as a reference in comparison as it is
  considered reliable over a broad range of temperatures. Wexler's equations
  are computationally simpler and fitted to relatively recent data. There is
  little difference at temperatures in the range -20 C to +50 C, and
  differences become large at extreme temperatures. Temperatures outside the
  range where estimations are highly reliable for each equation return
  \code{NA}, unless extrapolation is enabled by passing \code{FALSE} as
  argument to parameter \code{check.range}.

  The switch between equations for ice or water cannot be based on
  air temperature, as it depends on the presence or not of a surface of
  liquid water. It must be set by passing an argument to parameter
  \code{over.ice} which defaults to \code{FALSE}.

  Tetens equation is still very frequently used, and is for example the
  one recommended by FAO for computing potential evapotranspiration. For this
  reason it is used as default here.
}
\note{
The inverse of the Groff Gratch equation has yet to be implemented.
}
\examples{
water_vp_sat(20) # C -> Pa
water_vp_sat(temperature = c(0, 10, 20, 30, 40)) # C -> Pa
water_vp_sat(temperature = -10) # over water!!
water_vp_sat(temperature = -10, over.ice = TRUE)
water_vp_sat(temperature = 20) / 100 # C -> mbar

water_vp_sat(temperature = 20, method = "magnus") # C -> Pa
water_vp_sat(temperature = 20, method = "tetens") # C -> Pa
water_vp_sat(temperature = 20, method = "wexler") # C -> Pa
water_vp_sat(temperature = 20, method = "goff.gratch") # C -> Pa

water_vp_sat(temperature = -20, over.ice = TRUE, method = "magnus") # C -> Pa
water_vp_sat(temperature = -20, over.ice = TRUE, method = "tetens") # C -> Pa
water_vp_sat(temperature = -20, over.ice = TRUE, method = "wexler") # C -> Pa
water_vp_sat(temperature = -20, over.ice = TRUE, method = "goff.gratch") # C -> Pa

water_dp(water.vp = 1000) # Pa -> C
water_dp(water.vp = 1000, method = "magnus") # Pa -> C
water_dp(water.vp = 1000, method = "wexler") # Pa -> C
water_dp(water.vp = 500, over.ice = TRUE) # Pa -> C
water_dp(water.vp = 500, method = "wexler", over.ice = TRUE) # Pa -> C

water_fp(water.vp = 300) # Pa -> C
water_dp(water.vp = 300, over.ice = TRUE) # Pa -> C

water_vp2RH(water.vp = 1500, temperature = 20) # Pa, C -> RH \%
water_vp2RH(water.vp = 1500, temperature = c(20, 30)) # Pa, C -> RH \%
water_vp2RH(water.vp = c(600, 1500), temperature = 20) # Pa, C -> RH \%

water_vp2mvc(water.vp = 1000, temperature = 20) # Pa -> g m-3

water_mvc2vp(water.mvc = 30, temperature = 40) # g m-3 -> Pa

water_dp(water.vp = water_mvc2vp(water.mvc = 10, temperature = 30)) # g m-3 -> C

}
\references{
Tetens, O., 1930. Uber einige meteorologische Begriffe.
  Zeitschrift fur Geophysik, Vol. 6:297.

  Goff, J. A., and S. Gratch (1946) Low-pressure properties of water from
  -160 to 212 F, in Transactions of the American Society of Heating and
  Ventilating Engineers, pp 95-122, presented at the 52nd annual meeting of
  the American Society of Heating and Ventilating Engineers, New York, 1946.

  Wexler, A. (1976) Vapor Pressure Formulation for Water in Range 0 to 100°C.
  A Revision, Journal of Research ofthe National Bureau of Standards: A.
  Physics and Chemistry, September-December 1976, Vol. 80A, Nos.5 and 6,
  775-785

  Wexler, A.,  Vapor Pressure Formulation for Ice, Journal of Research of the
  National Bureau of Standards - A. Physics and Chemistry, January - February
  1977, Vol. 81A, No. 1, 5-19

  Alduchov, O. A., Eskridge, R. E., 1996. Improved Magnus Form Approximation
  of Saturation Vapor Pressure. Journal of Applied Meteorology, 35: 601-609 .

  Hardy, Bob (1998) ITS-90 formulations for vapor pressure, frostpoint
  temperature, dewpoint temperature, andenhancement factors in the range -100
  TO +100 C. The Proceedings of the Third International Symposium on Humidity
  & Moisture, Teddington, London, England, April 1998.
  \url{http://www.decatur.de/javascript/dew/resources/its90formulas.pdf}

  Monteith, J., Unsworth, M. (2008) Principles of Environmental Physics.
  Academic Press, Amsterdam.

  [Equations describing the physical properties of moist
  air](http://www.conservationphysics.org/atmcalc/atmoclc2.pdf)
}
