#' Print \code{clusterlm} object.
#'
#' @description Display the corrected p-values for each effects. Results of the \code{"clustermass"} procedure.
#'
#' @param x A \code{clusterlm} object.
#' @param alternative A character string indicating the alternative hypothesis. Choose between \code{"two.sided"}, \code{"greater"}, \code{"less"}. Default is \code{"two.sided"}.
#' @param ... Further arguments pass to \code{print}.
#' @export
print.clusterlm <- function(x, alternative = "two.sided", ...){
  cat(
    "Cluster ", x$test,
    " test using ", x$method,
    " to handle nuisance variables \n with ", paste(np(x$P), sep= ", ", collapse = ", "),
    " permutations and ",x$fun_name," as mass function.\n\n", sep = "")
  cat("Alternative Hypothesis : ",alternative,".\n \n",sep = "")
  switch(alternative,
         "two.sided" = {
           print(x$cluster_table,...)},
         "greater" = {
           print(x$cluster_table_greater,...)},
         "less" = {
           print(x$cluster_table_less,...)})
  }


#' @export
print.cluster_table <- function(x, ...) {
  cat(attr(x,"effect_name"),", threshold = ",attr(x,"threshold"),".\n",sep="")
  print.data.frame(x)
  cat("\n")
}

#' @export
print.listof_cluster_table<- function(x,...){
  dotargs = list(...)
  ei = NULL
  if(!is.null(dotargs$effect)){
    ei = which(names(x)%in%dotargs$effect)}
  if(length(ei)>0){
    for(i in ei){print(x[[i]])}
    }else{
      for(i in 1:length(x)){
        print(x[[i]])}}
}

###########################summary

#' Summarize a \code{clusterlm} object.
#'
#' @description Display the clusters with the corrected p-values for each effects. Results of the \code{"clustermass"} procedure.
#'
#' @param object A \code{clusterlm} object.
#' @param alternative A character string indicating the alternative hypothesis. Choose between \code{"two.sided"}, \code{"greater"}, \code{"less"}. Default is \code{"two.sided"}.
#' @param ... Further arguments see details.
#' @return A table for each effect indicating the statistics and p-values of the clusters.
#' @details If the \code{multcomp} argument is a character string that matches the \code{multcomp} argument of the \code{clusterlm} object, this method returns a matrix with the corrected statistics and p-values in columns and multiple tests by rows.
#' @export
summary.clusterlm <- function(object, alternative = "two.sided",...){
  dotargs = list(...)
  if(is.null(dotargs$multcomp)){
    switch(alternative,
           "two.sided" = {
             return(object$cluster_table)},
           "greater" = {
             return(object$cluster_table_greater)},
           "less" = {
             return(object$cluster_table_less)})
    }else{summary_multcomp(object = object, multcomp = dotargs$multcomp, alternative = alternative)}
}

#' @export
summary.cluster_table <- function(object,...){
  object
}


summary_multcomp <- function(object, multcomp, alternative){
  if(!(multcomp %in% object$multcomp)){
    stop(paste("The choosen multiple comparisons procedure does not match with the ones computed in the object.
                  Choose between ", paste(object$multcomp,sep=" "),".", sep=""))
  }
  switch(alternative,
         "two.sided" = {mc = object$multiple_comparison},
         "greater" = {mc = object$multiple_comparison_greater},
         "less" = {mc = object$multiple_comparison_less})

  out = lapply(seq_along(mc),function(i){
    out = mc[[i]][[multcomp]]$main[,1:2,drop = F]
    colnames(out) = paste(names(mc)[i], colnames(out))
    out})
  out = do.call("cbind",out)
  return(out)
}



#'Plot cluster or parameters.
#'
#' @description Plot method for class \code{clusterlm}.
#'
#' @param x A \code{clusterlm} object.
#' @param effect A vector of character naming the effects to display. Default is \code{"all"}.
#' @param type A character string that specified the values to highlight. \code{"statistic"} or \code{"coef"} are available. Default is \code{"statistic"}.
#' @param multcomp A character string specifying the method use to correct the p-value. It should match the one computed in the object. Default is \code{"clustermass"}. See \link{clusterlm}.
#' @param alternative A character string specifying the alternative hypothesis for the t-test. The available options are \code{"greater"}, \code{"less"} and \code{"two.sided"}. Default is \code{"two.sided"}.
#' @param enhanced_stat A logical. Default is \code{FALSE}. If \code{TRUE}, the enhanced statistic will be plotted overwise it will plot the observed statistic. Change for the \code{"tfce"} or the \code{"clustermass"} multiple comparisons procedures.
#' @param nbbaselinepts An integer. Default is 0. If the origin of the x axis should be shifted to show the start of the time lock, provide the number of baseline time points.
#' @param nbptsperunit An integer. Default is 1. Modify this value to change the scale of the label from the number of points to the desired unit. If points are e.g. sampled at 1024Hz, set to 1024 to scale into seconds and to 1.024 to scale into milliseconds.
#' @param ... further argument pass to plot.
#' @importFrom graphics points axis
#' @export
plot.clusterlm <- function(x, effect = "all", type = "statistic", multcomp = "clustermass", alternative = "two.sided", enhanced_stat = FALSE,
                           nbbaselinepts=0, nbptsperunit=1, ...) {

  ##select effect
  if("all" %in% effect){effect = names(x$multiple_comparison)}
  else if(sum(names(x$multiple_comparison)%in%effect) == 0){
    warning(" the specified effects do not exist. Plot 'all' effects.")
    effect = names(x$multiple_comparison)
  }
  effect_sel <- names(x$multiple_comparison)%in%effect

  ###switch mult comp
switch(alternative,
       "two.sided" = {multiple_comparison = x$multiple_comparison[effect_sel]},
       "greater" = {multiple_comparison = x$multiple_comparison_greater[effect_sel]},
       "less" = {multiple_comparison = x$multiple_comparison_less[effect_sel]})

  pvalue = t(sapply(multiple_comparison,function(m){
    m[[multcomp]]$main[,2]}))

  statistic = t(sapply(multiple_comparison,function(m){
    m[["uncorrected"]]$main[,1]}))
  if(enhanced_stat){
    statistic = t(sapply(multiple_comparison,function(m){
      m[[multcomp]]$main[,1]}))
  }



  ##swich value
  switch(type,
         "coef"={
           data <- x$coef[effect_sel,]
           title <- "coefficients"
           hl <- NULL
         },
         "statistic" ={
           data <- statistic
           title <- paste(x$test, " statistic",sep="",collapse = "")
           if(multcomp=="clustermass"){
           switch(x$test,
                  "fisher"={hl <- x$threshold},
                  "t"={
                    switch (alternative,
                      "less" ={hl <- -c(abs(x$threshold))},
                      "greater" ={hl <- c(abs(x$threshold))},
                      "two.sided" ={hl <- c(abs(x$threshold))}
                    )})}
         })

  title =paste(title," : ", multcomp, " correction",sep="", collapse = "")


  #####
  p = sum(NROW(data))
  rnames = row.names(data)


  #####

  par0 <- list(mfcol = par()$mfcol,mar = par()$mar,oma = par()$oma)

  par(mfcol = c(p,1),mar = c(0,4,0,0),oma = c(4,0,4,1),...=...)
  for (i in 1:p) {
    if(i==p){xaxt = NULL}else{xaxt = "n"}
    plot((1:ncol(data)-nbbaselinepts)/nbptsperunit,
      data[i,],type = "l",xaxt = xaxt,xlab = "",ylab = rnames[i]
    )
    if(type == "statistic"){
      xi = which(pvalue[i,]< x$alpha)
      y = data[i,xi]
      col="red"
      #lines(x = x,y= y,lwd=par()$lwd*2,col=col)
      points(x = (xi-nbbaselinepts)/nbptsperunit, y = y, pch=par()$pch,col=col)
      if(multcomp=="clustermass"){
        abline(h=hl[i],lty=3)
        if(x$test=="t"&alternative=="two.sided"){
          abline(h=-hl[i],lty=3)
        }
        }
  }}
  title(title,outer = T,cex = 2)
  par(mfcol = par0$mfcol, mar = par0$mar, oma = par0$oma)
  }


