% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/icc.R
\name{icc}
\alias{icc}
\alias{icc.brmsfit}
\title{Intraclass Correlation Coefficient (ICC)}
\usage{
icc(model, ...)

\method{icc}{brmsfit}(model, re.form = NULL, robust = TRUE,
  ci = 0.95, ...)
}
\arguments{
\item{model}{A mixed effects model of class \code{merMod}, \code{glmmTMB},
\code{MixMod}, \code{lme}, \code{mixed}, \code{stanreg} or \code{brmsfit}.}

\item{...}{Currently not used.}

\item{re.form}{Formula containing group-level effects to be considered in
the prediction. If \code{NULL} (default), include all group-level effects.
Else, for instance for nested models, name a specific group-level effect
to calculate the variance decomposition for this group-level.}

\item{robust}{Logical, if \code{TRUE}, the median instead of mean is used to
calculate the central tendency of the variances.}

\item{ci}{The Credible Interval level.}
}
\value{
A list with two values, the adjusted and conditional ICC. For models
  of class \code{brmsfit}, a list with two values, the decomposed ICC as well
  as the credible intervals for this ICC.
}
\description{
This function calculates the intraclass-correlation coefficient
 (ICC) - sometimes also called \emph{variance partition coefficient}
 (VPC) - for mixed effects models. The ICC is calculated for \code{merMod}
 (\pkg{lme4}), \code{glmmTMB} (\pkg{glmmTMB}), \code{MixMod} (\pkg{GLMMadpative}),
 \code{lme} (\pkg{nlme}), \code{mixed} (\pkg{afex}), and \code{stanreg}
 (\pkg{rstanarm}) objects. For models fitted with the \pkg{brms}-package,
 a variance decomposition based on the posterior predictive distribution
 is calculated (see 'Details').
}
\details{
\subsection{Interpretation}{
 The ICC can be interpreted as \dQuote{the proportion of the variance explained
 by the grouping structure in the population}. This index goes from 0, if the
 grouping conveys no information, to 1, if all observations in a group are
 identical (Gelman \& Hill, 2007, p. 258). In other word, the ICC
 \dQuote{can also be interpreted as the expected correlation between two
 randomly drawn units that are in the same group} \cite{(Hox 2010: 15)},
 altough this definition might not apply to mixed models with more complex
 random effects structures.
 }
 \subsection{Calculation}{
 The ICC is calculated by dividing the random effect variance,
 \ifelse{html}{\out{&sigma;<sup>2</sup><sub>i</sub>}}{\eqn{\sigma^2_i}}, by
 the total variance, i.e. the sum of the random effect variance and the
 residual variance, \ifelse{html}{\out{&sigma;<sup>2</sup><sub>&epsilon;</sub>}}{\eqn{\sigma^2_\epsilon}}.
 }
 \subsection{Adjusted and conditional ICC}{
 \code{icc()} calculates an adjusted and conditional ICC, which both take
 all sources of uncertainty (i.e. of \emph{all random effects}) into account. While
 the \emph{adjusted ICC} only relates to the random effects, the \emph{conditional ICC}
 also takes the fixed effects variances into account (see \cite{Nakagawa et al. 2017}).
 Typically, the \emph{adjusted} ICC is of interest when the analysis of random
 effects is of interest. \code{icc()} returns a meaningful ICC also for more
 complex random effects structures, like models with random slopes or nested
 design (more than two levels) and is applicable for models with other distributions
 than Gaussian. For more details on the computation of the variances, see
 \code{\link[insight]{get_variance}}.
 }
 \subsection{ICC for unconditional and conditional models}{
 Usually, the ICC is calculated for the null model ("unconditional model").
 However, according to \cite{Raudenbush and Bryk (2002)} or
 \cite{Rabe-Hesketh and Skrondal (2012)} it is also feasible to compute the ICC
 for full models with covariates ("conditional models") and compare how
 much, e.g., a level-2 variable explains the portion of variation in the grouping
 structure (random intercept).
 }
 \subsection{ICC for specific group-levels}{
 The proportion of variance for specific levels related to  each other
 (e.g., similarity of level-1-units within level-2-units or level-2-units
 within level-3-units) must be calculated manually. Use \code{\link[insight]{get_variance}}
 to get the random intercept variances (between-group-variances) and residual
 variance of the model, and calculate the ICC for the various level correlations.
 \cr \cr
 For example, for the ICC between level 1 and 2: \cr
 \code{sum(insight::get_variance_intercept(model)) /} \cr
 \code{  (sum(insight::get_variance_intercept(model)) + insight::get_variance_residual(model))}
 \cr \cr
 For for the ICC between level 2 and 3: \cr
 \code{insight::get_variance_intercept(model)[2] /} \cr
 \code{  sum(insight::get_variance_intercept(model))}
 }
 \subsection{ICC for brms-models}{
 If \code{model} is of class \code{brmsfit}, \code{icc()} calculates a
 variance decomposition based on the posterior predictive distribution. In
 this case, first, the draws from the posterior predictive distribution
 \emph{not conditioned} on group-level terms (\code{posterior_predict(..., re.form = NA)})
 are calculated as well as draws from this distribution \emph{conditioned}
 on \emph{all random effects} (by default, unless specified else in \code{re.form})
 are taken. Then, second, the variances for each of these draws are calculated.
 The "ICC" is then the ratio between these two variances. This is the recommended
 way to analyse random-effect-variances for non-Gaussian models. It is then possible
 to compare variances across models, also by specifying different group-level
 terms via the \code{re.form}-argument.
 \cr \cr
 Sometimes, when the variance of the posterior predictive distribution is
 very large, the variance ratio in the output makes no sense, e.g. because
 it is negative. In such cases, it might help to use \code{robust = TRUE}.
 }
}
\examples{
library(lme4)
model <- lme4::lmer(Sepal.Length ~ Petal.Length + (1 | Species), data = iris)
icc(model)

}
\references{
\itemize{
 \item Hox, J. J. (2010). Multilevel analysis: techniques and applications (2nd ed). New York: Routledge.
 \item Nakagawa, S., Johnson, P. C. D., & Schielzeth, H. (2017). The coefficient of determination R2 and intra-class correlation coefficient from generalized linear mixed-effects models revisited and expanded. Journal of The Royal Society Interface, 14(134), 20170213. \doi{10.1098/rsif.2017.0213}
 \item Rabe-Hesketh, S., & Skrondal, A. (2012). Multilevel and longitudinal modeling using Stata (3rd ed). College Station, Tex: Stata Press Publication.
 \item Raudenbush, S. W., & Bryk, A. S. (2002). Hierarchical linear models: applications and data analysis methods (2nd ed). Thousand Oaks: Sage Publications.
 }
}
