% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/vegetation_indices.R
\name{pa_compute_vi}
\alias{pa_compute_vi}
\title{Compute vegetation indices from a zipped Sentinel
  2 file}
\usage{
pa_compute_vi(
  satellite.images,
  vi = c("ndvi", "ndre", "gcvi", "reci", "evi", "bsi", "other"),
  aoi = NULL,
  formula = NULL,
  check.clouds = FALSE,
  buffer.clouds = 100,
  downscale.to = NULL,
  pixel.res = c("default", "10m", "20m", "60m"),
  img.formats = c("jp2", "tif"),
  fun = function(x) mean(x, na.rm = TRUE),
  verbose = TRUE
)
}
\arguments{
\item{satellite.images}{list of file paths to the
Sentinel 2 zip files}

\item{vi}{the vegetation index to be computed}

\item{aoi}{NULL or an sf object used to crop the
vegetation index raster to an area of interest}

\item{formula}{an optional two-sided formula with the vegetation
index name on the left side and the relationship between the bands
on the right side. See example.}

\item{check.clouds}{whether to check for clouds over the
area of interest. If clouds are found, the function
will skip cloudy images.}

\item{buffer.clouds}{distance in meters around the area
of interest within a cloud would be considered to
interfere with the index calculation. This is useful to
eliminate the effect of cloud shading from the
analysis.}

\item{downscale.to}{the resolution in meters to downscale the 
resolution of the vegetation index raster layer}

\item{pixel.res}{pixel resolution used to compute the
vegetation index. Can be one of 10m, 20m, 30m}

\item{img.formats}{image formats to search for in the
zipped file}

\item{fun}{function to be applied to consolidate duplicated images}

\item{verbose}{whether to display information on the
progress of operations}
}
\value{
an object of class veg.index and stars
}
\description{
Compute vegetation indices from a zipped
  Sentinel 2 file.
}
\details{
This is script that unzips the Sentinel 2 zipped
  file into a temporary folder, searches for the
  index-relevant bands, and then computes the index. If
  no \sQuote{aoi} is provided, the script will compute
  the vegetation index for the area covered by the image.
  The pre-specified vegetation indices are computed as follows:
  \deqn{BSI = \frac{(SWIR + RED) - (NIR + BLUE)}{(SWIR + RED) + (NIR + BLUE)}}
  \deqn{EVI = \frac{2.5 \times (NIR - RED)}{(NIR + (6 \times RED) - (7.5 \times BLUE) - 1) }}
  \deqn{GCVI = \frac{(NIR)}{(GREEN)} - 1}
  \deqn{NDRE = \frac{(NIR - RED edge)}{(NIR + RED edge)}} 
  \deqn{NDVI = \frac{(NIR - RED)}{(NIR + RED)}} 
  \deqn{RECI = \frac{(NIR)}{(RED edge)} - 1}
  
  The user can also specify custom vegetation indices using the formula 
  argument. The formula should be two-sided, with the left side naming the
  vegetation index and the right side defining the mathematical operations
  used to calculate the vegetation index. The bands should be specified
  as B01, B02, ..., B12.

  An important detail of this function is that, if there are
  duplicated dates, the function will consolidate the data into 
  a single raster layer. The default behavior is to average the 
  layers that belong to the same date. This can be changed with the 
  'fun' argument.
}
\examples{
\donttest{
extd.dir <- system.file("extdata", package = "pacu")
## List of zipped Sentinel files in a directory
s2a.files <- list.files(extd.dir, '\\\\.zip', full.names = TRUE)
area.of.interest <- sf::st_read(file.path(extd.dir, 'cobs_a_aoi.shp'))

## computing ndvi
ndvi <- pa_compute_vi(satellite.images = s2a.files,
                            vi = 'ndvi',
                            aoi = area.of.interest,
                            check.clouds = TRUE)

## computing ndre
ndre <- pa_compute_vi(satellite.images = s2a.files,
                            vi = 'ndre',
                            aoi = area.of.interest,
                            check.clouds = TRUE)

## specifying a differente vegetation index, in this case, the 
## excess green index
egi <- pa_compute_vi(satellite.images = s2a.files,
                            vi = 'other',
                            formula = EGI ~ (2 * B03) - B02 - B04,
                            aoi = area.of.interest,
                            check.clouds = TRUE)
}

}
\author{
Caio dos Santos and Fernando Miguez
}
