% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/calc_attributes_sites_exact.R
\name{calc_attributes_sites_exact}
\alias{calc_attributes_sites_exact}
\title{Calculate attributes of the sites.}
\usage{
calc_attributes_sites_exact(
  sites_map = "sites",
  input_raster = NULL,
  stat_rast = NULL,
  attr_name_rast = NULL,
  input_vector = NULL,
  stat_vect = NULL,
  attr_name_vect = NULL,
  round_dig = 2,
  calc_basin_area = TRUE,
  keep_basins = FALSE,
  overwrite = FALSE
)
}
\arguments{
\item{sites_map}{character; name of the sites (observation or prediction) 
attributes shall be calculated for. "sites" (default) refers to the observation sites.}

\item{input_raster}{character vector (optional); name of additional raster
maps to calculate attributes from.}

\item{stat_rast}{character vector (optional); statistics to be calculated, one of:
"min", "max", "mean", "stddev", "variance", "sum", "median", "percent", "area" or "percentile_X" (where X
gives the desired percentile e.g. 25 for the first). Must be provided if 
\code{input_raster} are given.}

\item{attr_name_rast}{character vector (optional); column name for the attributes
to be calculated. Attribute names must not be longer than 10 characters.
Must be provided if \code{input_raster} are given.}

\item{input_vector}{character string vector (optional); name of additional vector
maps to calculate attributes from.}

\item{stat_vect}{character string vector (optional); statistics to be calculated, 
one of: "percent", "area" or "count." Must be provided if \code{input_vector} is given.}

\item{attr_name_vect}{character string vector (optional); column name(s) in 
the vector file provided to calculate the attributes from (if \code{input_vector}
is a polygon map and stat_vect is "percent") or giving the new name attributes
to calculate (if \code{input_vector} is a point map and stat_vect is "count".
Must be provided if \code{input_vector} is given.}

\item{round_dig}{integer; number of digits to round results to. Can be a vector
of different values or just one value for all attributes.}

\item{calc_basin_area}{boolean; shall the catchment area be calculated? (Useful
to set to FALSE if the function has been called before with \code{keep_basins = TRUE}.)}

\item{keep_basins}{boolean; shall raster and vector maps of all the watersheds be kept?
Defaults to FALSE.}

\item{overwrite}{boolean; shall existing columns be overwritten; defaults to FALSE}
}
\value{
Nothing. The function appends new columns to the \code{sites_map} attribute table
\itemize{
 \item{'H2OArea':} {Total watershed area of the watershed upstream of each site.}
 \item{attr_name_rast:} {Additional optional attributes calculated based on input_raster maps.}
 \item{attributes form vector maps:}{Additional optional attributes 
 calculated based on input_vector maps. The column names are based on the unique entries 
 of the column(s) given in \code{attr_name_vect}.} 
}
Please note that for sampling points that lie in the same DEM raster cell 
 along a stream identical values are calculated because identical watersheds
 are derived.
}
\description{
For each site (observation or prediction) the total catchment area is
calculated ('H2OArea'). Additionally, other attributes (predictor variables)
can be derived based on given raster or vector maps. This function calculates
exact values for catchments derived with
\href{https://grass.osgeo.org/grass78/manuals/addons/r.stream.basins.html}{r.stream.basins}
and can take considerable time if there are many sites.
Catchment raster maps can optionally be stored as "sitename_catchm_X" (X = locID).
}
\note{
\code{\link{import_data}}, \code{\link{derive_streams}},
  \code{\link{calc_edges}} and \code{\link{calc_sites}} or
  \code{\link{calc_prediction_sites}} must be run before.
  
If \code{calc_basin_area = F} but there are no raster maps called 'sitename_catchm_x' 
with x = locID of all sites the catchments (and their area) are derived.
}
\examples{
\donttest{
# Initiate and setup GRASS
dem_path <- system.file("extdata", "nc", "elev_ned_30m.tif", package = "openSTARS")
if(.Platform$OS.type == "windows"){
  grass_program_path = "c:/Program Files/GRASS GIS 7.6"
  } else {
  grass_program_path = "/usr/lib/grass78/"
  }

setup_grass_environment(dem = dem_path, 
                        gisBase = grass_program_path,      
                        remove_GISRC = TRUE,
                        override = TRUE
                        )
gmeta()
                        
# Load files into GRASS
dem_path <- system.file("extdata", "nc", "elev_ned_30m.tif", package = "openSTARS")
sites_path <- system.file("extdata", "nc", "sites_nc.shp", package = "openSTARS")
import_data(dem = dem_path, sites = sites_path)

# Derive streams from DEM
derive_streams(burn = 0, accum_threshold = 700, condition = TRUE, clean = TRUE)

# Prepare edges
calc_edges()

# caluclate slope as potential predictor
execGRASS("r.slope.aspect", flags = c("overwrite","quiet"),
          parameters = list(
            elevation = "dem",
            slope = "slope"
          ))
calc_attributes_edges(input_raster = "slope", stat_rast = "max", attr_name_rast = "maxSlo")

# Prepare sites
calc_sites()
calc_attributes_sites_approx(input_attr_name = "maxSlo", output_attr_name = "maxSloA", stat = "max")
calc_attributes_sites_exact(input_raster = "slope", attr_name_rast = "maxSloE", stat_rast = "max")

# Plot data
library(sp)
dem <- readRAST('dem', ignore.stderr = TRUE, plugin = FALSE)
edges <- readVECT('edges', ignore.stderr = TRUE)
sites <- readVECT('sites', ignore.stderr = TRUE)
plot(dem, col = gray(seq(0,1,length.out=20)))
mm <- range(c(edges$maxSlo_e, sites$maxSloA, sites$maxSloE))
b <- seq(from = mm[1], to = mm[2] + diff(mm) * 0.01, length.out = 10)
c_ramp <- colorRampPalette(c("white", "blue", "orange", "red"))
cols <- c_ramp(length(b))[as.numeric(cut(edges$maxSlo_e, breaks = b, right = FALSE))]
# plot stream edges, color depending on maxSlope of the edge
plot(edges,col = cols, lwd = 2, add = TRUE)
cols <- c_ramp(length(b))[as.numeric(cut(sites$maxSloA,breaks = b,right = FALSE))]
# plot sites as points with color corresponding to maxSlop approximate
plot(sites, pch = 19, col = cols, cex = 2, add = TRUE)
cols <- c_ramp(length(b))[as.numeric(cut(sites$maxSloE,breaks = b,right = FALSE))]
#' # plot sites as ring around points with color corresponding to maxSlop exact
plot(sites, pch = 21, bg = cols, cex = 1.1, add = TRUE)
# Some points in the lower centre of the map indicate a difference in max slope between
# approximate and exact calculation (different colors for inner and outer points). However,
# for most points the values are similar
}
}
\author{
Mira Kattwinkel, \email{mira.kattwinkel@gmx.net}, 
  Eduard Szoecs, \email{eduardszoecs@gmail.com}
}
