% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/argo.R
\name{read.argo}
\alias{read.argo}
\title{Read an Argo Data File}
\usage{
read.argo(file, debug = getOption("oceDebug"), processingLog, ...)
}
\arguments{
\item{file}{a character string giving the name of the file to load.}

\item{debug}{a flag that turns on debugging.  Set to 1 to get a moderate amount
of debugging information, or to 2 to get more.}

\item{processingLog}{if provided, the action item to be stored in the log.
(Typically only provided for internal calls; the default that it provides is
better for normal calls by a user.)}

\item{...}{additional arguments, passed to called routines.}
}
\value{
An object of \code{\link{argo-class}}.
}
\description{
\code{read.argo} is used to read an Argo file, producing an object of type
\code{argo}. The file must be in the ARGO-style NetCDF format described at
in the Argo documentation [2,3].
}
\details{
Items are inferred from the data file in a straightforward way, using
\code{\link[ncdf4]{ncvar_get}}, converting from one-column matrices
to vectors, and trimming leading and trailing blank space in character
values, using \code{\link{trimString}}.

Items are renamed from the argo ('snake case') forms to oce ('camel
case') forms with \code{\link{argoNames2oceNames}}. For example,
\code{FIRMWARE_VERSION} in the data file is renamed as
\code{firmwareVersion} in the return value.
Note that some files use upper-case for items, while other files
use lower-case for the same things; \code{read.argo} attempts
to ignore this variation.

See the Argo documentation [2,3] for some details on what files contain.
Many items listed in section 2.2.3 of [3] are read from the
file and stored in the \code{metadata} slot, with the exception of
\code{longitude} and \code{latitude}, which are stored in the
\code{data} slot, alongside hydrographic information.

The following global attributes stored within the netcdf file are stored in the
\code{metadata} slot: \code{title}, \code{institution}, \code{source},
\code{history}, \code{references}, \code{userManualVersion}, \code{conventions},
and \code{featureType}. These names are identical to those in the netcdf
file, except that \code{userManualVersion} is named
\code{user_manual_version} in the file, and \code{conventions} is
named \code{Conventions} in the file.

It is assumed that the profile data are as listed in the NetCDF variable
called \code{STATION_PARAMETERS}. Each item can have variants, as
described in Sections 2.3.4 of [3].
For example, if \code{"PRES"} is found in \code{STATION_PARAMETERS},
then \code{PRES} (pressure) data are sought in the file, along with
\code{PRES_QC}, \code{PRES_ADJUSTED}, \code{PRES_ADJUSTED_QC}, and
\code{PRES_ERROR}. The same pattern works for other profile data. The variables
are stored with different names within the resultant \code{\link{argo-class}}
object, to match with \code{oce} conventions. Thus, \code{PRES} gets renamed
\code{pressure}, while \code{PRES_ADJUSTED} gets renamed \code{pressureAdjusted},
and \code{PRES_ERROR} gets renamed \code{pressureError}; all of these are
stored in the \code{data} slot. Meanwhile, the quality-control flags
\code{PRES_QC} and \code{PRES_ADJUSTED_QC} are stored as \code{pressure}
and \code{pressureAdjusted} in the \code{metadata$flags} slot.

Once a predefined series of items are inferred and stored in either the
\code{metadata} or \code{data} slot, \code{read.argo} then reads all
non-empty variables in the file, storing them in the \code{metadata}
slot, using with the original ('snake case') name that is used in
the data file. (Note that the sample dataset accessed with \code{data(argo)}
lacks metadata items with names starting with \code{HISTORY_}, because
those are zero-length in the source file.)
}
\section{Data sources}{

Argo data are made available at several websites. A bit of detective
work can be required to track down the data.

Some servers provide  data for floats that surfaced in a given ocean
on a given day, the anonymous FTP server
\url{ftp://usgodae.org/pub/outgoing/argo/geo/} being an example.

Other servers provide data on a per-float basis. A complicating
factor is that these data tend to be categorized by "dac" (data
archiving centre), which makes it difficult to find a particular
float. For example,
\url{https://www.usgodae.org/ftp/outgoing/argo/} is the top level of
a such a repository. If the ID of a float is known but not the
"dac", then a first step is to download the text file
\url{http://www.usgodae.org/ftp/outgoing/argo/ar_index_global_meta.txt}
and search for the ID. The first few lines of that file are header,
and after that the format is simple, with columns separated by slash
(\code{/}). The dac is in the first such column and the float ID in the
second. A simple search will reveal the dac.
For example \code{data(argo)} is based on float 6900388, and the line
containing that token is
\code{bodc/6900388/6900388_meta.nc,846,BO,20120225005617}, from
which the dac is seen to be the British Oceanographic Data Centre
(\code{bodc}). Armed with that information, visit
\url{https://www.usgodae.org/ftp/outgoing/argo/dac/bodc/6900388}
and see a directory called `profiles` that contains a NetCDF
file for each profile the float made. These can be read with
\code{read.argo}. It is also possible, and probably more common,
to read a NetCDF file containing all the profiles together and for
that purpose the file
\url{https://www.usgodae.org/ftp/outgoing/argo/dac/bodc/6900388/6900388_prof.nc}
should be downloaded and provided as the \code{file} argument to
\code{read.argo}.  This can be automated as in Example 2,
although readers are cautioned that URL structures tend to change
over time.

Similar steps can be followed on other servers.
}

\examples{
\dontrun{
## Example 1: read from a local file
library(oce)
d <- read.argo("/data/OAR/6900388_prof.nc")
summary(d)
plot(d)

## Example 2: construct URL for download (brittle)
id <- "6900388"
url <- "https://www.usgodae.org/ftp/outgoing/argo"
if (!length(list.files(pattern="argo_index.txt")))
    download.file(paste(url, "ar_index_global_meta.txt", sep="/"), "argo_index.txt")
index <- readLines("argo_index.txt")
line <- grep(id, index)
if (0 == length(line)) stop("id ", id, " not found")
if (1 < length(line)) stop("id ", id, " found multiple times")
dac <- strsplit(index[line], "/")[[1]][1]
profile <- paste(id, "_prof.nc", sep="")
float <- paste(url, "dac", dac, id, profile, sep="/")
download.file(float, profile)
argo <- read.argo(profile)
summary(argo)
}


}
\references{
1. \url{http://www.argo.ucsd.edu/}

2. Argo User's Manual Version 3.2, Dec 29th, 2015, available at
\url{https://archimer.ifremer.fr/doc/00187/29825/40575.pdf}
(but note that this is a draft; newer versions may have
replaced this by now).

3. User's Manual (ar-um-02-01) 13 July 2010, available at
\url{http://www.argodatamgt.org/content/download/4729/34634/file/argo-dm-user-manual-version-2.3.pdf},
which is the main document describing argo data.
}
\seealso{
The documentation for \code{\link{argo-class}} explains the structure of argo
objects, and also outlines the other functions dealing with them.

Other things related to argo data: \code{\link{[[,argo-method}},
  \code{\link{[[<-,argo-method}}, \code{\link{argo-class}},
  \code{\link{argoGrid}}, \code{\link{argoNames2oceNames}},
  \code{\link{argo}}, \code{\link{as.argo}},
  \code{\link{handleFlags,argo-method}},
  \code{\link{plot,argo-method}},
  \code{\link{subset,argo-method}},
  \code{\link{summary,argo-method}}
}
\author{
Dan Kelley
}
\concept{things related to argo data}
