% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/argo.R
\docType{methods}
\name{handleFlags,argo-method}
\alias{handleFlags,argo-method}
\title{Handle Flags in ARGO Objects}
\usage{
\S4method{handleFlags}{argo}(object, flags = NULL, actions = NULL,
  debug = getOption("oceDebug"))
}
\arguments{
\item{object}{An object of \code{\link{argo-class}}.}

\item{flags}{A \code{\link{list}} specifying flag values upon which
actions will be taken. This can take two forms. In the first, the
list has named elements each containing a vector of integers. For example,
salinities flagged with values of 1 or 3 through 9 would be specified
by \code{flags=list(salinity=c(1,3:9))}. Several data items can be specified,
e.g. \code{flags=list(salinity=c(1,3:9), temperature=c(1,3:9))} indicates
that the actions are to take place for both salinity and temperature.
In the second form, \code{flags} is a list with unnamed vectors, and
this means to apply the actions to all the data entries; thus,
\code{flags=list(c(1,3:9))} means to apply not just to salinity and temperature,
but also to everything else that is in the \code{data} slot. If \code{flags}
is not provided, then \code{\link{defaultFlags}} is called, to try to
determine a conservative default.}

\item{actions}{An optional \code{\link{list}} that contains items with
names that match those in the \code{flags} argument.  If \code{actions}
is not supplied, the default will be to set all values identified by
\code{flags} to \code{NA}; this can also be specified by
specifying \code{actions=list("NA")}. It is also possible to specify
functions that calculate replacement values. These are provided
with \code{object} as the single argument, and must return a
replacement for the data item in question.
See \dQuote{Details} for the default that is used if \code{actions} is not supplied.}

\item{debug}{An optional integer specifying the degree of debugging, with
value 0 meaning to skip debugging and 1 or higher meaning to print some
information about the arguments and the data. It is usually a good idea to set
this to 1 for initial work with a dataset, to see which flags are being
handled for each data item. If not supplied, this defaults to the value of
\code{\link{getOption}("oceDebug")}.}
}
\description{
Data-quality flags are stored in the \code{metadata}
slot of \code{\link{oce-class}} objects in a
\code{\link{list}} named \code{flags}.
The present function (a generic that has specialized versions
for various data classes) provides a way to
manipulate the core data based on
the data-quality flags. For example, a common operation is to replace suspicious
or erroneous data with \code{NA}.

If \code{metadata$flags} in the object supplied as the first argument
is empty, then that object is returned, unaltered.
Otherwise, \code{handleFlags} analyses the data-quality flags within
the object, in relation to the \code{flags} argument, and interprets
the \code{action} argument to select an action to be applied to matched
data.
}
\examples{
library(oce)
data(argo)
# 1. Default: set to NA any data that is not flagged with
# code value 1 (meaning \\code{"passed_all_tests"})
argoNew <- handleFlags(argo, flags=c(0, 2:9))
# Demonstrate replacement, looking at the second profile
f <- argo[["salinityFlag"]][,2] # first column with a flag=4 entry
df <- data.frame(flag=f, orig=argo[["salinity"]][,2], new=argoNew[["salinity"]][,2])
df[11:15,] # notice line 13

# 2. A less restrictive case: focussing just on salinity,
# retain only data with flags 1 (meaning \\code{"passed_all_tests"})
# and 2 (\\code{"probably_good"}).
argoNew <- handleFlags(argo, flags=list(salinity=c(0, 3:9)))

}
\references{
1. \url{http://www.argo.ucsd.edu/Argo_date_guide.html#dmodedata}
}
\seealso{
Other functions relating to data-quality flags: \code{\link{defaultFlags}},
  \code{\link{handleFlags,adp-method}},
  \code{\link{handleFlags,ctd-method}},
  \code{\link{handleFlags,section-method}},
  \code{\link{handleFlags}},
  \code{\link{initializeFlagScheme,ctd-method}},
  \code{\link{initializeFlagScheme,oce-method}},
  \code{\link{initializeFlagScheme,section-method}},
  \code{\link{initializeFlagSchemeInternal}},
  \code{\link{initializeFlagScheme}},
  \code{\link{initializeFlags,adp-method}},
  \code{\link{initializeFlags,oce-method}},
  \code{\link{initializeFlagsInternal}},
  \code{\link{initializeFlags}},
  \code{\link{setFlags,adp-method}},
  \code{\link{setFlags,ctd-method}},
  \code{\link{setFlags,oce-method}}, \code{\link{setFlags}}

Other things related to argo data: \code{\link{[[,argo-method}},
  \code{\link{[[<-,argo-method}}, \code{\link{argo-class}},
  \code{\link{argoGrid}}, \code{\link{argoNames2oceNames}},
  \code{\link{argo}}, \code{\link{as.argo}},
  \code{\link{plot,argo-method}}, \code{\link{read.argo}},
  \code{\link{subset,argo-method}},
  \code{\link{summary,argo-method}}
}
\author{
Dan Kelley
}
\concept{functions relating to data-quality flags}
\concept{things related to argo data}
