% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/nlme_fit.R
\name{nlme_ode}
\alias{nlme_ode}
\title{Fit nlme-based mixed-effect model using ODE implementation}
\usage{
nlme_ode(dat.o, model, par_model, par_trans, response, response.scaler = NULL,
  transit_abs = FALSE, atol = 1e-08, rtol = 1e-08, debugODE = FALSE,
  mc.cores = 1, ...)
}
\arguments{
\item{dat.o}{data to be fitted}

\item{model}{a string containing the set of ordinary differential equations (ODE) and other expressions defining the changes in the dynamic  system. For details, see the sections \dQuote{Details} and  \dQuote{\code{RxODE Syntax}} below.}

\item{par_model}{list: model for fixed effects, randoms effects and initial values.}

\item{par_trans}{function: calculation of PK parameters}

\item{response}{names of the response variable}

\item{response.scaler}{optional response variable scaler. default is NULL}

\item{transit_abs}{a logical if transit absorption model is enabled}

\item{atol}{atol (absolute tolerance for ODE-solver)}

\item{rtol}{rtol (relative tolerance for ODE-solver)}

\item{debugODE}{a logical if debugging is enabled}

\item{mc.cores}{number of cores used in fitting (only for Linux)}

\item{...}{additional nlme options}
}
\description{
'nlme_ode' fits a mixed-effect model described using ordinary differential
equation (ODEs). The ODE-definition follows RxODE syntax.
Specification of fixed effects, random effects and intial values follows
the standard nlme notations.
}
\details{
The ODE-based model specification may be coded inside a character
   string or in a text file, see Section \emph{RxODE Syntax} below for
   coding details.  An internal \code{RxODE} compilation manager object
   translates the ODE system into C, compiles it, and dynamically loads the
   object code into the current R session.  The call to \code{RxODE}
   produces an object of class \code{RxODE} which consists of a list-like
   structure (closure) with various member functions (see Section
   \emph{Value} below).
}
\section{RxODE Syntax}{


   An \code{RxODE} model specification consists of one or more
   statements terminated by semi-colons, \sQuote{\code{;}}, and
   optional comments (comments are delimited by \code{#} and an
   end-of-line marker).  \strong{NB:} Comments are not allowed
   inside statements.

   A block of statements is a set of statements delimited by
   curly braces, \sQuote{\code{\{ ... \}}}.
   Statements can be either assignments or conditional \code{if}
   statements. Assignment statements can be either \dQuote{simple}
   assignments, where the left hand is an identifier (i.e., variable), or
   special \dQuote{time-derivative} assignments, where the left hand
   specifies the change of that variable with respect to time
   e.g., \code{d/dt(depot)}.

   Expressions in assignment and \sQuote{\code{if}} statements can be
   numeric or logical (no character expressions are currently supported).
   Numeric expressions can include the following numeric operators
   (\sQuote{\code{+}}, \sQuote{\code{-}}, \sQuote{\code{*}},
   \sQuote{\code{/}}, \sQuote{\code{^}}),   and
   those mathematical functions defined in the C or the
   R math libraries (e.g., \code{fabs}, \code{exp}, \code{log}, \code{sin}).
   (Note that the modulo operator \sQuote{\code{\%}} is currently
   not supported.)

   Identifiers in an \code{RxODE} model specification can refer to:
   \itemize{
      \item state variables in the dynamic system (e.g., compartments in a
      pharmacokinetic/pharmacodynamic model);
      \item implied input variable, \code{t} (time),
      \code{podo} (oral dose, for absorption models), and
      \code{tlast} (last time point);
      \item model parameters, (\code{ka} rate of absorption, \code{CL}
      clearance, etc.);
      \item others, as created by assignments as part of the model
      specification.
   }

   Identifiers consist of case-sensitive alphanumeric characters,
   plus the underscore \sQuote{_} character.  \strong{NB:} the
   dot \sQuote{.} character is \strong{not} a valid character
   identifier.

   The values of these variables at pre-specified time points are
   saved as part of the fitted/integrated/solved model (see
   \code{\link{eventTable}}, in particular its member function
   \code{add.sampling} that defines a set of time points at which
   to capture a snapshot of the system via the values of these variables).

   The ODE specification mini-language is parsed with the help of
   the open source tool \emph{DParser}, Plevyak (2015).
}

\examples{
\dontrun{
library(nlmixr)
ode <- "
d/dt(depot) =-KA*depot;
d/dt(centr) = KA*depot - KE*centr;
"
dat <- read.table(system.file("examples/theo_md.txt", package = "nlmixr"), head=TRUE)
mypar <- function(lKA, lKE, lCL)
{
    KA=exp(lKA)
    KE=exp(lKE)
    CL=exp(lCL)
    V = CL/KE
}

specs <- list(fixed=lKA+lKE+lCL~1, random = pdDiag(lKA+lCL~1),
	start=c(lKA=0.5, lKE=-2.5, lCL=-3.2))
fit <- nlme_ode(dat, model=ode, par_model=specs, par_trans=mypar,
	response="centr", response.scaler="V")

}
}
\author{
Wenping Wang
}
