#' Random diffnet network
#'
#' Simulates a diffusion network by creating a random dynamic network and
#' adoption threshold levels.
#'
#' @export
#' @param n Integer scalar. Number of vertices.
#' @param t Integer scalar. Time length.
#' @param seed.nodes Either a character scalar or a vector. Type of seed nodes (see details).
#' @param seed.p.adopt Numeric scalar. Proportion of early adopters.
#' @param seed.graph Baseline graph used for the simulation (see details).
#' @param rgraph.args List. Arguments to be passed to rgraph.
#' @param rewire Logical scalar. When TRUE, network slices are generated by rewiring
#' (see \code{\link{rewire_graph}}).
#' @param rewire.args List. Arguments to be passed to \code{\link{rewire_graph}}.
#' @param threshold.dist Either a function to be applied via \code{\link{sapply}},
#' or a vector/matrix with \eqn{n} elements. Sets the adoption threshold for each node.
#' @param exposure.args List. Arguments to be passed to \code{\link{exposure}}.
#' @return A random \code{\link{diffnet}} class object.
#' @family simulation functions
#' @details
#'
#' Instead of randomizing whether an individual adopts the innovation or not, this
#' toy model randomizes threshold levels, seed adopters and network structure, so
#' an individual adopts the innovation in time \eqn{T} iff his exposure is above or
#' equal to his threshold. The simulation is done in the following steps:
#'
#' \enumerate{
#'  \item Using \code{seed.graph}, a baseline graph is created.
#'  \item Given the baseline graph, the set of initial adopters is defined
#'  using \code{seed.nodes}.
#'  \item Afterwards, if \code{rewire=TRUE} \eqn{t-1} slices of the network are created
#'  by iteratively rewiring the baseline graph.
#'  \item The \code{threshold.dist} function is applied to each node in the graph.
#'  \item Simulation starts at \eqn{t=2} asigning adopters in each time period
#'  accordingly to each vertex's threshold and exposure.
#' }
#'
#' When \code{seed.nodes} is a character scalar it can be \code{"marginal"}, \code{"central"} or \code{"random"},
#' So each of these values sets the initial adopters using the vertices with lowest
#' degree, with highest degree or completely randomly. The number of early adoptes
#' is set as \code{seed.p.adopt * n}. Please note that when marginal nodes are
#' set as seed it may be the case that no diffusion process is attained as the
#' chosen set of first adopters can be isolated. Any other case will be considered
#' as an index (via \code{\link{[<-}} methods), hence the user can manually set the set of initial adopters, for example
#' if the user sets \code{seed.nodes=c(1, 4, 7)} then nodes 1, 4 and 7 will be
#' selected as initial adopters.
#'
#' The argument \code{seed.graph} can be either a function that generates a graph
#' (Any class of accepted graph format (see \code{\link{netdiffuseR-graphs}})), a
#' graph itself or a character scalar in which the user sets the algorithm used to
#' generate the first network (network in t=1), this can be either "scale-free"
#' (Barabasi-Albert model using the \code{\link{rgraph_ba}} function, the default),
#' \code{"bernoulli"} (Erdos-Renyi model using the \code{\link{rgraph_er}} function),
#' or \code{"small-world"} (Watts-Strogatz model using the \code{\link{rgraph_ws}}
#' function). The list \code{rgraph.args} passes arguments to the chosen algorithm.
#'
#' When \code{rewire=TRUE}, the networks that follow t=1 will be generated using the
#' \code{\link{rewire_graph}} function as \eqn{G(t) = R(G(t-1))}, where \eqn{R}
#' is the rewiring algorithm.
#'
#' The argument \code{threshold.dist} sets the threshold for each vertex in the graph.
#' It is applied using \code{sapply} as follows
#'
#' \preformatted{
#' sapply(1:n, threshold.dist)
#' }
#'
#' By default sets the threshold to be random for each node in the graph.
#'
#' If \code{sgraph} is provided, no random graph is generated and the simulation
#' is applied using that graph instead.
#'
#' @examples
#' # Asimple example -----------------------------------------------------------
#' set.seed(123)
#' z <- rdiffnet(100,10)
#' z
#' summary(z)
#'
#' # A more complex example: Adopt if at least one neighbor has adopted --------
#' y <- rdiffnet(100, 10, threshold.dist=function(x) 1,
#'     exposure.args=list(valued=FALSE, normalized=FALSE))
#'
#' # Re thinking the Adoption of Tetracycline ----------------------------------
#' newMI <- rdiffnet(seed.graph = medInnovationsDiffNet$graph,
#'  threshold.dist = threshold(medInnovationsDiffNet), rewire=FALSE)
#'
#' @author George G. Vega Yon
rdiffnet <- function(n, t,
                     seed.nodes="random", seed.p.adopt=0.05,
                     seed.graph="scale-free", rgraph.args=list(),
                     rewire=TRUE, rewire.args=list(p=.1, undirected=FALSE),
                     threshold.dist=function(x) runif(1),
                     exposure.args=list(
                       outgoing=TRUE, valued=getOption("diffnet.valued", FALSE),
                       normalized=TRUE
                       )
                     ) {

  # Step 0.0: Creating the network seed ----------------------------------------
  # Checking the class of the seed.graph
  test <- class(seed.graph)
  if ("function" %in% test) {
    # Does it returns a graph
    test <- seed.graph()
    # Coercing into appropiate type
    if (inherits(test, "dgCMatrix")) {
      sgraph <- test
    }
    else if (inherits(test, "matrix")) {
      sgraph <- methods::as(test, "dgCMatrix")
    }
    else if (inherits(test, "array")) {
      sgraph <- apply(test, 3, function(x) methods::as(x, "dgCMatrix"))
    }
    else if (inherits(test, "diffnet")) {
      sgraph <- test$graph
    }
    else if (inherits(test, "list")) {
      sgraph <- test
    }

  # In the case of calling a function
  } else if ("character" %in% test) {
    if (seed.graph == "scale-free") {
      if (!length(rgraph.args$m0))
        rgraph.args$t <- n-1L

      sgraph <- do.call(rgraph_ba, rgraph.args)
    } else if (seed.graph == "bernoulli") {
      rgraph.args$n <- n

      sgraph <- do.call(rgraph_er, rgraph.args)
    } else if (seed.graph == "small-world") {

      rgraph.args$n <- n
      if (!length(rgraph.args$k)) rgraph.args$k <- 2L
      if (!length(rgraph.args$p)) rgraph.args$p <- .1

      sgraph <- do.call(rgraph_ws, rgraph.args)
    } else
      stop("Invalid -seed.graph-. It should be either ",
           "'scale-free\', \'bernoulli\' or \'small-world\'.")

    # Creating t duplicates
    graph <- rep(list(sgraph), t)

  } else if (test %in% c("matrix", "dgCMatrix", "array")) {
    # If not dgCMatrix
    if ("array" %in% test) {
      sgraph <- apply(seed.graph, 3, function(x) methods::as(x, "dgCMatrix"))
    } else {
      sgraph <- methods::as(seed.graph, "dgCMatrix")
    }
  } else if ("list" %in% test) {
    sgraph <- seed.graph
  } else if ("diffnet" %in% test) {
    sgraph <- seed.graph$graph
  }else {
    stop("Invalid argument for -seed.graph-. No support for objects of class -",test,"-.")
  }

  # Checking baseline graph --------------------------------------------------
  meta <- classify_graph(sgraph)

  # Was n set?
  if (!missing(n) && n != meta$n) {
    warning("While the user set n=",n,", nnodes(seed.graph)=", meta$n,". The later will be used.")
    n <- meta$n
  }
  if (missing(n)) n <- meta$n

  # Checking dimmensions
  test <- class(sgraph)
  if ("list" %in% test) {
    # Checking if all the graphs follow the condition
    test2 <- sapply(sgraph, function(x) inherits(x, "dgCMatrix") && all(dim(x) == c(n,n)) )
    if (any(!test2)) stop("-seed.graph- should be a list of 'dgCMatrices' of size ", n, "x", n,".",
                          " The following elements are invalid:\n\t", paste0(which(!test2), collapse=", "), ".")
  }

  # If static, t must be provided, otherwise t should be missing
  if (meta$nper == 1) {
    if (missing(t)) {
      stop("When -seed.graph- is static, -t- must be provided.")
    } else {
      sgraph <- rep(list(sgraph), t)
    }
  } else {
    if (!missing(t)) warning("When -seed.graph- is dynamic, -t- shouldn't be provided.")
    t <- meta$nper
  }

  # Step 0.1: Rewiring or not ------------------------------------------------

  # Rewiring
  rewire.args$graph <- sgraph
  if (rewire) sgraph <- do.call(rewire_graph, rewire.args)

  # Number of initial adopters
  if (n*seed.p.adopt < 1) warning("Set of initial adopters set to 1.")
  n0 <- max(1, n*seed.p.adopt)

  # Step 0.1: Setting the seed nodes -------------------------------------------
  cumadopt <- matrix(0L, ncol=t, nrow=n)
  toa   <- matrix(NA, ncol=1, nrow= n)

  if (length(seed.nodes) == 1) {
    if (seed.nodes %in% c("central","marginal")) {
      d <- dgr(sgraph)[,1,drop=FALSE]
      decre <- ifelse(seed.nodes == "central", TRUE, FALSE)
      d <- rownames(d[order(d, decreasing = decre),,drop=FALSE])
      d <- d[1:floor(n0)]
      d <- as.numeric(d)
    } else if (seed.nodes == "random") {
      d <- sample.int(n, floor(n0))
    } else
      stop("Unsupported -seed.nodes- value. It must be either \"central\", \"marginal\", or \"random\"")
  } else if (!inherits(seed.nodes, "character")) {
    d <- seed.nodes
  } else stop("Unsupported -seed.nodes- value. See the manual for references.")

  # Setting seed nodes via vector
  toa[d] <- 1L
  cumadopt[d,] <- 1L

  # Step 3.0: Thresholds -------------------------------------------------------
  if (inherits(threshold.dist, "function")) thr <- sapply(1:n, threshold.dist)
  else {
    # Setting depending on class
    if (any(class(threshold.dist) %in% c("data.frame", "matrix"))) {
      thr <- as.vector(as.matrix(threshold.dist))

      # Must match the length of n
      if (length(thr) != n)
        stop("Incorrect length for -threshold.dist- (",length(threshold.dist),")",
             ". It should be a vector of length ",n,".")
    } else if (is.vector(threshold.dist)) {
      thr <- threshold.dist

      # Must match the length of n
      if (length(thr) != n)
        stop("Incorrect length for -threshold.dist- (",length(threshold.dist),")",
             ". It should be a vector of length ",n,".")
    } else {
      stop("-threshold.dist- must be either a vector of length -n- or a function.")
    }
  }

  # Processing times

  for (i in 2:t) {

    # Computing exposure
    exposure.args[c("graph", "cumadopt")] <- list(sgraph[i], cumadopt[,i,drop=FALSE])
    expo <- do.call(exposure, exposure.args)

    whoadopts <- which( (expo >= thr) & is.na(toa))
    toa[whoadopts] <- i
    cumadopt[whoadopts, i:t] <- 1L
  }
  reachedt <- max(toa, na.rm=TRUE)

  if (reachedt == 1) {
    stop("No diffusion in this network (Ups!) try changing the seed or the parameters.")
  }

  as_diffnet(sgraph, as.integer(toa), undirected=FALSE, t0=1, t1=t,
             vertex.static.attrs = data.frame(real_threshold=thr))
}

