% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/adjmat.R
\name{edgelist_to_adjmat}
\alias{adjmat_to_edgelist}
\alias{edgelist_to_adjmat}
\title{Conversion between adjacency matrix and edgelist}
\usage{
edgelist_to_adjmat(edgelist, weights = NULL, times = NULL, t = NULL,
  simplify = TRUE, undirected = getOption("diffnet.undirected"),
  self = getOption("diffnet.self"),
  multiple = getOption("diffnet.multiple"), use.incomplete = TRUE,
  recode.ids = TRUE)

adjmat_to_edgelist(graph, undirected = getOption("diffnet.undirected"))
}
\arguments{
\item{edgelist}{Two column matrix/data.frame in the form of ego -source- and
alter -target- (see details).}

\item{weights}{Numeric vector. Strength of ties (optional).}

\item{times}{Integer vector. Starting time of the ties (optional).}

\item{t}{Integer scalar. If \code{times} but want to repeat the network \code{t} times.}

\item{simplify}{Logical scalar. When TRUE and \code{times=NULL} it will return an adjacency
matrix, otherwise an array of adjacency matrices.}

\item{undirected}{Logical scalar. TRUE when the graph is undirected.}

\item{self}{Logical scalar. TRUE when self edges are excluded.}

\item{multiple}{Logical scalar. TRUE when multiple edges should not be included
(see details).}

\item{use.incomplete}{Logical scalar. When FALSE, rows with \code{NA/NULL} values
(isolated vertices unless have autolink) will be droped
and will not be considered in the graph, which may reduce the size of the
adjacency matrix (see
details).}

\item{recode.ids}{Logical scalar. When TRUE ids are recoded using \code{\link{as.factor}}
(see details).}

\item{graph}{Any class of accepted graph format (see \code{\link{netdiffuseR-graphs}}).}
}
\value{
In the case of \code{edgelist_to_adjmat} either an adjacency matrix
(if times is NULL) or an array of these (if times is not null). For
\code{adjmat_to_edgelist} the output is an edgelist.
}
\description{
Generates adjacency matrix from an edgelist and vice versa.
}
\details{
When converting from edglist to adjmat the function will \code{\link{recode}} the
edgelist before starting. The user can keep track after the recording by checking
the resulting adjacency matrices' \code{\link{row.names}}. In the case that the
user decides skipping the recoding (because wants to keep vertices index numbers,
implying that the resulting graph will have isolated vertices), he can override
this by setting \code{recode.ids=FALSE} (see example).

When multiple edges are included, \code{multiple=TRUE},each vertex between \eqn{\{i,j\}}{{i,j}} will be counted
as many times it appears in the edgelist. So if a vertex \eqn{\{i,j\}}{{i,j}} appears 2
times, the adjacency matrix element \code{(i,j)} will be 2.

Including incomplete cases, \code{use.incomplete=TRUE}, can lead to an adjacency matrix
with isolated vertices. Otherwise, when \code{use.incomplete=FALSE}, if all the
edges in which a vertex participates have incomplete information in any of the
variables (a NA, NULL or NaN value, see \code{\link{complete.cases}}), it
will be dropped from the graph, thus, reducing the size of the adjacency
matrix by not including isolated vertices. Notice that the \emph{best way of adding %
isolated vertices} is to include them in the
edgelist as connecting to themselves. The option \code{self=FALSE} will not
drop the isolated vertices (but the adjacency matrix will have a 0-diagonal)
but the algorithm will include them on the graph.

The function performs several checks before starting to create the adjacency
matrix. These are:
\itemize{
 \item{Dimensions of the inputs, such as number of columns and length of vectors}
 \item{Having complete cases. If anly edge has a non-numeric value such as NAs or
 NULL in any variable (including \code{times} and \code{weights}), it will be
 removed. A full list of such edges can be retrieved from the attribute
 \code{incomplete}}
 \item{Nodes and times ids coding}
}

\code{recode.ids=FALSE} is useful when the vertices ids have already been
coded. For example, after having use \code{adjmat_to_edgelist}, ids are
correctly encoded, so when going back (using \code{edgelist_to_adjmat})
\code{recode.ids} should be FALSE.
}
\examples{
# Base data
set.seed(123)
n <- 5
edgelist <- rgraph_er(n, as.edgelist=TRUE)
times <- sample.int(3, nrow(edgelist), replace=TRUE)
w <- abs(rnorm(nrow(edgelist)))

# Simple example
edgelist_to_adjmat(edgelist)
edgelist_to_adjmat(edgelist, undirected = TRUE)

# Using weights
edgelist_to_adjmat(edgelist, w)
edgelist_to_adjmat(edgelist, w, undirected = TRUE)

# Using times
edgelist_to_adjmat(edgelist, times = times)
edgelist_to_adjmat(edgelist, times = times, undirected = TRUE)

# Using times and weights
edgelist_to_adjmat(edgelist, times = times, weights = w)
edgelist_to_adjmat(edgelist, times = times, undirected = TRUE, weights = w)

# Not recoding ----------------------------------------------------
# Notice that vertices 3, 4 and 5 are not present in this graph.
graph <- matrix(c(
 1,2,6,
 6,6,7
), ncol=2)

# Generates an adjmat of size 4 x 4
edgelist_to_adjmat(graph)

# Generates an adjmat of size 7 x 7
edgelist_to_adjmat(graph, recode.ids=FALSE)
}
\author{
Vega Yon, Dyal, Hayes & Valente
}
\seealso{
Other data management functions: \code{\link{isolated}}
}
\keyword{manip}

