# Copyright (C) 2022-2023 Hibiki AI Limited <info@hibiki-ai.com>
#
# This file is part of nanonext.
#
# nanonext is free software: you can redistribute it and/or modify it under the
# terms of the GNU General Public License as published by the Free Software
# Foundation, either version 3 of the License, or (at your option) any later
# version.
#
# nanonext is distributed in the hope that it will be useful, but WITHOUT ANY
# WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR
# A PARTICULAR PURPOSE. See the GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License along with
# nanonext. If not, see <https://www.gnu.org/licenses/>.

# nanonext - Utilities ---------------------------------------------------------

#' NNG Library Version
#'
#' Returns the versions of the 'libnng' and 'libmbedtls' libraries used.
#'
#' @return A character vector of length 2.
#'
#' @examples
#' nng_version()
#'
#' @export
#'
nng_version <- function() .Call(rnng_version)

#' Translate Error Codes
#'
#' Translate integer exit codes generated by the NNG library. All package
#'     functions return an integer exit code on error rather than the expected
#'     return value. These are classed 'errorValue' and may be checked by
#'     \code{\link{is_error_value}}.
#'
#' @param xc integer exit code to translate.
#'
#' @return A character vector.
#'
#' @examples
#' nng_error(1L)
#'
#' @export
#'
nng_error <- function(xc) .Call(rnng_strerror, xc)

#' Clock Utility
#'
#' Provides the number of elapsed milliseconds since an arbitrary reference time
#'     in the past. The reference time will be the same for a given session, but
#'     may differ between sessions.
#'
#' @details A convenience function for building concurrent applications. The
#'     resolution of the clock depends on the underlying system timing facilities
#'     and may not be particularly fine-grained. This utility should however be
#'     faster than using \code{Sys.time()}.
#'
#' @return A double.
#'
#' @examples
#' time <- mclock(); msleep(100); mclock() - time
#'
#' @export
#'
mclock <- function() .Call(rnng_clock)

#' Sleep Utility
#'
#' Sleep function. May block for longer than requested, with the actual wait
#'     time determined by the capabilities of the underlying system.
#'
#' @param msec integer number of milliseconds to block the caller.
#'
#' @return Invisible NULL.
#'
#' @details If 'msec' is non-integer, it will be coerced to integer. Non-numeric
#'     input will be ignored and return immediately.
#'
#'     Note that unlike \code{\link{Sys.sleep}}, this function is not
#'     user-interruptible by sending SIGINT e.g. with ctrl + c.
#'
#' @examples
#' time <- mclock(); msleep(100); mclock() - time
#'
#' @export
#'
msleep <- function(msec) invisible(.Call(rnng_sleep, msec))

#' NNG Random Number Generator
#'
#' Strictly not for statistical analysis. Not reproducible. No ability to set
#'     a seed value. Provides random numbers suitable for system functions such
#'     as cryptographic key generation. Random values are obtained using
#'     platform-specific strong cryptographic random number facilities where
#'     available.
#'
#' @param n [default 1L] length of vector to return.
#'
#' @return A length 'n' vector of random positive doubles.
#'
#' @details If 'n' is non-integer, it will be coerced to integer; if a vector,
#'     only the first element will be used.
#'
#' @examples
#' random()
#' random(n = 3L)
#'
#' @export
#'
random <- function(n = 1L) .Call(rnng_random, n)

#' Parse URL
#'
#' Parses a character string containing an RFC 3986 compliant URL as per NNG.
#'
#' @param url character string containing a URL.
#'
#' @return A named character vector of length 10, comprising: \itemize{
#'     \item{\code{rawurl}} {- the unparsed URL string.}
#'     \item{\code{scheme}} {- the URL scheme, such as "http" or "inproc"
#'     (always lower case).}
#'     \item{\code{userinfo}} {- the username and password if supplied in the
#'     URL string.}
#'     \item{\code{host}} {- the full host part of the URL, including the port
#'     if present (separated by a colon).}
#'     \item{\code{hostname}} {- the name of the host.}
#'     \item{\code{port}} {- the port (if not specified, the default port if
#'     defined by the scheme).}
#'     \item{\code{path}} {- the path, typically used with HTTP or WebSocket.}
#'     \item{\code{query}} {- the query info (typically following ? in the URL).}
#'     \item{\code{fragment}} {- used for specifying an anchor, the part after #
#'     in a URL.}
#'     \item{\code{requri}} {- the full Request-URI (path[?query][#fragment]).}
#'     }
#'     Values that cannot be determined are represented by an empty string \code{''}.
#'
#' @examples
#' parse_url("https://user:password@w3.org:8080/type/path?q=info#intro")
#' parse_url("tcp://192.168.0.2:5555")
#'
#' @export
#'
parse_url <- function(url) .Call(rnng_url_parse, url)

#' Create Device
#'
#' Creates a device which is a socket forwarder or proxy. Provides for improved
#'     horizontal scalability, reliability, and isolation.
#'
#' @param s1 a raw mode Socket.
#' @param s2 a raw mode Socket.
#'
#' @return NULL. If the device was successfully created, this function does not
#'     return.
#'
#' @details Only raw mode sockets may be used with this function. Sockets s1 and
#'     s2 must be compatible with each other, i.e. be opposite halves of a two
#'     protocol pattern, or both the same protocol for a single protocol pattern.
#'
#' @section Usage:
#'
#'     Warning: this function is designed to be called in an isolated process
#'     with the two sockets. Once called, it will block with no ability to
#'     interrupt. To terminate the device, the process must be killed (in
#'     interactive sessions this may be done by sending SIGQUIT e.g. ctrl + \).
#'
#' @export
#'
device <- function(s1, s2) .Call(rnng_device, s1, s2)

#' Validators
#'
#' Validator functions for object types created by \{nanonext\}.
#'
#' @param x an object.
#'
#' @return Logical value TRUE or FALSE.
#'
#' @details Is the object an Aio (inheriting from class 'sendAio' or 'recvAio').
#'
#'     Is the object an object inheriting from class 'nano' i.e. a nanoSocket,
#'     nanoContext, nanoStream, nanoListener, nanoDialer, or nano Object.
#'
#' @examples
#' sock <- socket(listen = "inproc://isaio")
#' r <- recv_aio(sock)
#' s <- send_aio(sock, "test")
#' is_aio(r)
#' is_aio(s)
#' close(sock)
#'
#' @export
#'
is_aio <- function(x) inherits(x, c("recvAio", "sendAio"))

#' @examples
#' s <- socket()
#' is_nano(s)
#' n <- nano()
#' is_nano(n)
#' close(s)
#' n$close()
#'
#' @rdname is_aio
#' @export
#'
is_nano <- function(x) inherits(x, c("nano", "nanoObject"))

#' Error Validators
#'
#' Validator functions for error value types created by \{nanonext\}.
#'
#' @param x an object.
#'
#' @return Logical value TRUE or FALSE.
#'
#' @details Is the object an error value inheriting from class 'errorValue'. All
#'     non-success integer return values from NNG are classed as such to be
#'     distinguishable from integer message values. Includes error values
#'     returned after a timeout etc.
#'
#'     Is the object a nul byte.
#'
#' @examples
#' is_error_value(1L)
#'
#' @export
#'
is_error_value <- function(x) inherits(x, "errorValue")

#' @examples
#' is_nul_byte(as.raw(0L))
#' is_nul_byte(raw(length = 1L))
#' is_nul_byte(writeBin("", con = raw()))
#' is_nul_byte(0L)
#' is_nul_byte(NULL)
#' is_nul_byte(NA)
#'
#' @rdname is_error_value
#' @export
#'
is_nul_byte <- function(x) .Call(rnng_is_nul_byte, x)

#' Translate HTTP Status Codes
#'
#' Provides an explanation for HTTP response status codes (in the range 100 to
#'     599). If the status code is not defined as per RFC 9110, 'Non-standard
#'     Response' is returned, which may be a custom code used by the server.
#'
#' @param x numeric HTTP status code to translate.
#'
#' @return A character vector.
#'
#' @examples
#' status_code(200)
#' status_code(404)
#'
#' @export
#'
status_code <- function(x) .Call(rnng_status_code, x)

