% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/QTL_gen_effects.R
\name{QTL_gen_effects}
\alias{QTL_gen_effects}
\title{QTL genetic effects}
\usage{
QTL_gen_effects(
  mppData,
  trait = 1,
  QTL = NULL,
  Q.eff = "cr",
  ref.par = NULL,
  sum_zero = FALSE
)
}
\arguments{
\item{mppData}{An object of class \code{mppData}.}

\item{trait}{\code{Numerical} or \code{character} indicator to specify which
trait of the \code{mppData} object should be used. Default = 1.}

\item{QTL}{Object of class \code{QTLlist} representing a list of
selected position obtained with the function \code{\link{QTL_select}} or
vector of \code{character} marker positions names.
Default = NULL.}

\item{Q.eff}{\code{Character} expression indicating the assumption concerning
the QTL effects: 1) "cr" for cross-specific; 2) "par" for parental; 3) "anc"
for ancestral; 4) "biall" for a bi-allelic. For more details see
\code{\link{mpp_SIM}}. Default = "cr".}

\item{ref.par}{Optional \code{Character} expression defining the parental
allele that will be used as reference for the parental model. For the
ancestral model, the ancestral class containing the reference parent will be
set as reference. \strong{This option can only be used if the MPP design is
composed of a unique connected part}. Default = NULL.}

\item{sum_zero}{Optional \code{Logical} value specifying if the QTL effect of
a parental or an ancestral model should be calculated using the sum to zero
constraint. Default = FALSE.}
}
\value{
Return:

Object of class \code{QeffRes} containing the following elements:

\item{Qeff}{\code{List} of \code{data.frame} (one per QTL) containing the
following information:

\enumerate{

\item{QTL genetic effects per cross or parent.}
\item{Standard error of the QTL effects.}
\item{Test statistics of the effects (t-test or Wald statistic).}
\item{P-value of the test statistics.}
\item{Significance of the QTL effects.}
\item{For cross-specific model, parent with the positive additive effects.}
\item{For parental and ancestral model, indicator of connected part of the
design and reference.}
\item{Allele scores of the parents if \code{geno.par} is non NULL
in the \code{mppData} object.}

}

}

\item{tab.Qeff}{\code{data.frame} with one column per QTL giving the
QTL genetic effects per cross or per parent with its significance. The
first two rows indicate the chromosome and the position in cM of each
QTL.}
}
\description{
Computes a multi-QTL model with a list of QTL candidates (\code{QTL}) and
return the decomposed QTL effects per cross or per parents.
}
\details{
This function computes for each QTL position the genetic effects of the
cross, parental, ancestral or SNP allele components. For the cross-specific
model (\code{Q.eff = "cr"}), the genetics effects represent the substitution
effect of an single allele from the parent 2 (or B) with respect to an allele
coming from the parent 1 or A. All effects are given in absolute value with
the parent that carries the positive allele.

For the parental and the ancestral model (\code{Q.eff = "par" or "anc"}), it
is possible to estimate maximum n-1 parental or ancestral alleles per
interconnected part of the design. For these two models, one
parental (ancestral) allele is set as reference per interconnected part of the
design. Effects of the other alleles are estimated as deviation with respect
to the reference. Connected parts of the design can be determined using Weeks
and Williams (1964) method (\code{\link{design_connectivity}}). By default,
the reference allele is the most frequent one. The user can also specify a
parental allele that will be used as reference using the argument
\code{ref.par}. This option is only available if the MPP design is composed
of a unique connected part.

For the parental and ancestral model it is also possible to estimate the QTL
effects using a sum to zero constraint \code{sum_zero = TRUE}. In that case,
the effects of the different parental (ancestral) allele will represent the
deviation with respect to the average trait value.

For the bi-allelic model (\code{Q.eff = "biall"}), the genetic effects
represent the effects of a single allele copy of the least frequent allele.
}
\examples{

data(mppData)

# QTL candidates

SIM <- mpp_SIM(mppData)
QTL <- QTL_select(SIM)

# Cross-specific model

QTL.effects <- QTL_gen_effects(mppData = mppData, QTL = QTL, Q.eff = "cr")
summary(QTL.effects)

# Parental model

QTL.effects <- QTL_gen_effects(mppData = mppData, QTL = QTL, Q.eff = "par")
summary(QTL.effects)

# Ancestral model

QTL.effects <- QTL_gen_effects(mppData = mppData, QTL = QTL, Q.eff = "anc")
summary(QTL.effects)

# Bi-allelic model

QTL.effects <- QTL_gen_effects(mppData = mppData, QTL = QTL, Q.eff = "biall")
summary(QTL.effects)

}
\references{
Weeks, D. L., & Williams, D. R. (1964). A note on the determination of
connectedness in an N-way cross classification. Technometrics, 6(3), 319-324.
}
\seealso{
\code{\link{QTL_select}}, \code{\link{summary.QeffRes}}
}
\author{
Vincent Garin
}
