% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/modelsummary.R
\name{msummary}
\alias{msummary}
\title{Beautiful, customizable summaries of statistical models}
\usage{
msummary(
  models,
  output = "default",
  fmt = 3,
  estimate = "estimate",
  statistic = "std.error",
  vcov = NULL,
  conf_level = 0.95,
  stars = FALSE,
  coef_map = NULL,
  coef_omit = NULL,
  coef_rename = NULL,
  gof_map = NULL,
  gof_omit = NULL,
  add_rows = NULL,
  align = NULL,
  notes = NULL,
  title = NULL,
  ...
)
}
\arguments{
\item{models}{a model or (optionally named) list of models}

\item{output}{filename or object type (character string)
\itemize{
  \item Supported filename extensions: .html, .tex, .md, .txt, .png, .jpg.
  \item Supported object types: "default", "html", "markdown", "latex", "latex_tabular", "data.frame", "gt", "kableExtra", "huxtable", "flextable".
  \item Warning: the `output` argument \emph{cannot} be used when customizing tables with external packages. See the 'Details' section below.
}}

\item{fmt}{determines how to format numeric values
\itemize{
  \item integer: the number of digits to keep after the period `format(round(x, fmt), nsmall=fmt)`
  \item character: passed to the `sprintf` function (e.g., '\%.3f' keeps 3 digits with trailing zero). See `?sprintf`
  \item function: returns a formatted character string.
}}

\item{estimate}{string or `glue` string of the estimate to display (or a
vector with one string per model). Valid entries include any column name of
the data.frame produced by `get_estimates(model)`. Examples:
\itemize{
  \item "estimate"
  \item "\{estimate\} (\{std.error\})\{stars\}"
  \item "\{estimate\} [\{conf.low\}, \{conf.high\}]"
}}

\item{statistic}{vector of strings or `glue` strings which select uncertainty statistics to report vertically below the estimate. NULL omits all uncertainty statistics. 
\itemize{
  \item "conf.int", "std.error", "statistic", "p.value", "conf.low", "conf.high", or any column name produced by: `get_estimates(model)`
  \item `glue` package strings with braces, such as: 
  \itemize{
    \item "\{p.value\} [\{conf.low\}, \{conf.high\}]"
    \item "Std.Error: \{std.error\}"
  }
  \item Note: Parentheses are added automatically unless the string includes `glue` curly braces \{\}.
  \item Note: To report uncertainty statistics \emph{next} to coefficients, you can supply a `glue` string to the `estimate` argument.
}}

\item{vcov}{robust standard errors and other manual statistics. The `vcov` argument accepts five types of input (see the 'Details' and 'Examples' sections below):
\itemize{
  \item string, vector, or list of strings: "robust", "HC", "HC0", "HC1", "HC2", "HC3", "HC4", "HC4m", "HC5", "stata", or "classical" (alias "constant" or "iid").
  \item formula or list of formulas with the cluster variable(s) on the right-hand side (e.g., ~clusterid).
  \item function or list of functions which return variance-covariance matrices with row and column names equal to the names of your coefficient estimates (e.g., `stats::vcov`, `sandwich::vcovHC`).
  \item list of `length(models)` variance-covariance matrices with row and column names equal to the names of your coefficient estimates.
  \item a list of length(models) vectors with names equal to the names of your coefficient estimates. See 'Examples' section below. 
}}

\item{conf_level}{confidence level to use for confidence intervals}

\item{stars}{to indicate statistical significance
\itemize{
  \item FALSE (default): no significance stars.
  \item TRUE: *=.1, **=.05, ***=.01
  \item Named numeric vector for custom stars such as `c('*' = .1, '+' = .05)`
}}

\item{coef_map}{named character vector. Values refer to the variable names
that will appear in the table. Names refer to the original term names stored
in the model object, e.g. c("hp:mpg"="hp X mpg") for an interaction term.
Coefficients that are omitted from this vector will be omitted from the
table. The table will be ordered in the same order as this vector.}

\item{coef_omit}{string regular expression. Omits all matching coefficients
from the table using `grepl(perl=TRUE)`.}

\item{coef_rename}{named character vector. Values refer to the variable names
that will appear in the table. Names refer to the original term names stored
in the model object, e.g. c("hp:mpg"="hp X mpg") for an interaction term.}

\item{gof_map}{\itemize{
  \item NULL (default): the `modelsummary::gof_map` dictionary is used for formatting, and all unknown statistic are included.
  \item data.frame with 3 columns named "raw", "clean", "fmt". Unknown statistics are omitted. See the 'Examples' section below.
  \item list of lists, each of which includes 3 elements named "raw", "clean", "fmt". Unknown statistics are omitted. See the 'Examples section below'.
}}

\item{gof_omit}{string regular expression. Omits all matching gof statistics from
the table (using `grepl(perl=TRUE)`).}

\item{add_rows}{a data.frame (or tibble) with the same number of columns as
your main table. By default, rows are appended to the bottom of the table.
You can define a "position" attribute of integers to set the row positions.
See Examples section below.}

\item{align}{A character string of length equal to the number of columns in
the table.  "lcr" means that the first column will be left-aligned, the 2nd
column center-aligned, and the 3rd column right-aligned.}

\item{notes}{list or vector of notes to append to the bottom of the table.}

\item{title}{string}

\item{...}{all other arguments are passed through to the extractor and
table-making functions. This allows users to specify additional options such
as set `broom::tidy(exponentiate=TRUE)` to exponentiate logistic regression
coefficients or `kableExtra::kbl(escape=FALSE)` to avoid escaping math
characters in `kableExtra` tables.}
}
\value{
a regression table in a format determined by the `output` argument.
}
\description{
`msummary()` is a shortcut to `modelsummary()`
}
\details{
`output` argument:

When a file name is supplied to the `output` argument, the table is written
immediately to file. If you want to customize your table by post-processing
it with an external package, you need to choose a different output format
and saving mechanism. Unfortunately, the approach differs from package to
package:
\itemize{
  \item `gt`: set `output="gt"`, post-process your table, and use the `gt::gtsave` function.
  \item `kableExtra`: set `output` to your destination format (e.g., "latex", "html", "markdown"), post-process your table, and use `kableExtra::save_kable` function.
}

`vcov` argument:

To use a string such as "robust" or "HC0", your model must be supported
by the `sandwich` package. This includes objects such as: lm, glm,
survreg, coxph, mlogit, polr, hurdle, zeroinfl, and more.

"classical", "iid", and "constant" are aliases which do not modify
uncertainty estimates and simply report the default standard errors stored
in the model object.

One-sided formulas such as `~clusterid` are passed to the `sandwich::vcovCL`
function.

Matrices and functions producing variance-covariance matrices are first
passed to `lmtest`. If this does not work, `modelsummary` attempts to take
the square root of the diagonal to adjust "std.error", but the other
uncertainty estimates are not be adjusted.

Numeric vectors are formatted according to `fmt` and placed in brackets.
Character vectors printed as given, without parentheses. 

If your model type is supported by the `lmtest` package, the
`vcov` argument will try to use that package to adjust all the
uncertainty estimates, including "std.error", "statistic", "p.value", and
"conf.int". If your model is not supported by `lmtest`, only the "std.error"
will be adjusted by, for example, taking the square root of the matrix's
diagonal.
}
\examples{
\dontrun{

# The `modelsummary` website includes \emph{many} examples and tutorials:
# https://vincentarelbundock.github.io/modelsummary

library(modelsummary)

# load data and estimate models
data(trees)
models <- list()
models[['Bivariate']] <- lm(Girth ~ Height, data = trees)
models[['Multivariate']] <- lm(Girth ~ Height + Volume, data = trees)

# simple table
modelsummary(models)

# statistic
modelsummary(models, statistic = NULL)
modelsummary(models, statistic = 'p.value')
modelsummary(models, statistic = 'statistic')
modelsummary(models, statistic = 'conf.int', conf_level = 0.99)
modelsummary(models, statistic = c("t = {statistic}",
                                   "se = {std.error}",
                                   "conf.int"))

# estimate
modelsummary(models, 
  statistic = NULL,
  estimate = "{estimate} [{conf.low}, {conf.high}]")
modelsummary(models,
  estimate = c("{estimate}{stars}",
               "{estimate} ({std.error})"))

# vcov
modelsummary(models, vcov = "robust")
modelsummary(models, vcov = list("classical", "stata"))
modelsummary(models, vcov = sandwich::vcovHC)
modelsummary(models, 
  vcov = list(stats::vcov, sandwich::vcovHC))
modelsummary(models, 
  vcov = list(c("(Intercept)"="", "Height"="!"),
                            c("(Intercept)"="", "Height"="!", "Volume"="!!")))

# coef_rename
modelsummary(models, coef_map = c('Volume' = 'Large', 'Height' = 'Tall'))

# coef_map
modelsummary(models, coef_map = c('Volume' = 'Large', 'Height' = 'Tall'))

# title
modelsummary(models, title = 'This is the title')

# add_rows
rows <- tibble::tribble(~term, ~Bivariate, ~Multivariate,
  'Empty row', '-', '-',
  'Another empty row', '?', '?')
attr(rows, 'position') <- c(1, 3)
modelsummary(models, add_rows = rows)

# notes
modelsummary(models, notes = list('A first note', 'A second note'))

# gof_map: data.frame
gm <- modelsummary::gof_map
gof_custom$omit[gof_custom$raw == 'deviance'] <- FALSE
gof_custom$fmt[gof_custom$raw == 'r.squared'] <- "\%.5f"
modelsummary(models, gof_map = gof_custom)

# gof_map: list of lists
f1 <- function(x) format(round(x, 3), big.mark=",")
f2 <- function(x) format(round(x, 0), big.mark=",")
gm <- list(
  list("raw" = "nobs", "clean" = "N", "fmt" = f2),
  list("raw" = "AIC", "clean" = "aic", "fmt" = f1))
modelsummary(models,
  fmt = f1,
  gof_map = gm)

}

}
\keyword{internal}
