% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/modelStudio.R
\name{modelStudio}
\alias{modelStudio}
\alias{modelStudio.explainer}
\title{Interactive Studio for Explanatory Model Analysis}
\usage{
modelStudio(explainer, ...)

\method{modelStudio}{explainer}(
  explainer,
  new_observation = NULL,
  new_observation_y = NULL,
  new_observation_n = 3,
  facet_dim = c(2, 2),
  time = 500,
  max_features = 10,
  N = 300,
  N_fi = N * 10,
  N_sv = N * 3,
  B = 10,
  B_fi = B,
  eda = TRUE,
  show_info = TRUE,
  parallel = FALSE,
  options = ms_options(),
  viewer = "external",
  widget_id = NULL,
  license = NULL,
  telemetry = TRUE,
  max_vars = NULL,
  verbose = NULL,
  ...
)
}
\arguments{
\item{explainer}{An \code{explainer} created with \code{DALEX::explain()}.}

\item{...}{Other parameters.}

\item{new_observation}{New observations with columns that correspond to variables used in the model.}

\item{new_observation_y}{True label for \code{new_observation} (optional).}

\item{new_observation_n}{Number of observations to be taken from the \code{explainer$data} if \code{new_observation = NULL}.
See \href{https://modelstudio.drwhy.ai/articles/ms-perks-features.html#instance-explanations}{\bold{vignette}}}

\item{facet_dim}{Dimensions of the grid. Default is \code{c(2,2)}.}

\item{time}{Time in ms. Set the animation length. Default is \code{500}.}

\item{max_features}{Maximum number of features to be included in BD and SV plots.
Default is \code{10}.}

\item{N}{Number of observations used for the calculation of PD and AD. Default is \code{300}.
See \href{https://modelstudio.drwhy.ai/articles/ms-perks-features.html#more-calculations-means-more-time}{\bold{vignette}}}

\item{N_fi}{Number of observations used for the calculation of FI. Default is \code{10*N}.}

\item{N_sv}{Number of observations used for the calculation of SV. Default is \code{3*N}.}

\item{B}{Number of permutation rounds used for calculation of SV. Default is \code{10}.
See \href{https://modelstudio.drwhy.ai/articles/ms-perks-features.html#more-calculations-means-more-time}{\bold{vignette}}}

\item{B_fi}{Number of permutation rounds used for calculation of FI. Default is \code{B}.}

\item{eda}{Compute EDA plots and Residuals vs Feature plot, which adds the data to the dashboard. Default is \code{TRUE}.}

\item{show_info}{Verbose a progress on the console. Default is \code{TRUE}.}

\item{parallel}{Speed up the computation using \code{parallelMap::parallelMap()}.
See \href{https://modelstudio.drwhy.ai/articles/ms-perks-features.html#parallel-computation}{\bold{vignette}}.
This might interfere with showing progress using \code{show_info}.}

\item{options}{Customize \code{modelStudio}. See \code{\link{ms_options}} and
\href{https://modelstudio.drwhy.ai/articles/ms-perks-features.html#additional-options-1}{\bold{vignette}}.}

\item{viewer}{Default is \code{external} to display in an external RStudio window.
Use \code{browser} to display in an external browser or
\code{internal} to use the RStudio internal viewer pane for output.}

\item{widget_id}{Use an explicit element ID for the widget (rather than an automatically generated one).
Useful e.g. when using \code{modelStudio} with Shiny.
See \href{https://modelstudio.drwhy.ai/articles/ms-perks-features.html#shiny-1}{\bold{vignette}}.}

\item{license}{Path to the file containing the license (\code{con} parameter passed to \code{readLines()}).
It can be used e.g. to include the license for \code{explainer$data} as a comment in the source of \code{.html} output file.}

\item{telemetry}{The dashboard gathers useful, but not sensitive, information about how it is being used (e.g. computation length,
package version, dashboard dimensions). This is for the development purposes only and can be blocked by setting \code{telemetry} to \code{FALSE}.}

\item{max_vars}{An alias for \code{max_features}. If provided, it will override the value.}

\item{verbose}{An alias for \code{show_info}. If provided, it will override the value.}
}
\value{
An object of the \code{r2d3, htmlwidget, modelStudio} class.
}
\description{
This function computes various (instance and dataset level) model explanations and
produces a customisable dashboard, which consists of multiple panels for plots with their
short descriptions. Easily save the dashboard and share it with others. Tools for
\href{https://ema.drwhy.ai/}{Explanatory Model Analysis} unite with tools for
Exploratory Data Analysis to give a broad overview of the model behavior.

The extensive documentation covers:

\itemize{
  \item Function parameters description -
 \href{https://modelstudio.drwhy.ai/articles/ms-perks-features.html}{\bold{perks and features}}
  \item Framework and model compatibility -
 \href{https://modelstudio.drwhy.ai/articles/ms-r-python-examples.html}{\bold{R & Python examples}}
  \item Theoretical introduction to the plots -
 \href{https://ema.drwhy.ai/}{Explanatory Model Analysis: Explore, Explain, and Examine Predictive Models}
}

Displayed variable can be changed by clicking on the bars of plots or with the first dropdown list,
 and observation can be changed with the second dropdown list.
The dashboard gathers useful, but not sensitive, information about how it is being used (e.g. computation length,
 package version, dashboard dimensions). This is for the development purposes only and can be blocked
 by setting \code{telemetry} to \code{FALSE}.
}
\examples{
library("DALEX")
library("modelStudio")

#:# ex1 classification on 'titanic' data

# fit a model
model_titanic <- glm(survived ~., data = titanic_imputed, family = "binomial")

# create an explainer for the model
explainer_titanic <- explain(model_titanic,
                             data = titanic_imputed,
                             y = titanic_imputed$survived,
                             label = "Titanic GLM")

# pick observations
new_observations <- titanic_imputed[1:2,]
rownames(new_observations) <- c("Lucas","James")

# make a studio for the model
modelStudio(explainer_titanic,
            new_observations,
            N = 200,  B = 5) # faster example

\donttest{

#:# ex2 regression on 'apartments' data
if (requireNamespace("ranger", quietly=TRUE)) {
  library("ranger")
  model_apartments <- ranger(m2.price ~. ,data = apartments)

  explainer_apartments <- explain(model_apartments,
                                  data = apartments,
                                  y = apartments$m2.price)

  new_apartments <- apartments[1:2,]
  rownames(new_apartments) <- c("ap1","ap2")

  # change dashboard dimensions and animation length
  modelStudio(explainer_apartments,
              new_apartments,
              facet_dim = c(2, 3),
              time = 800)

  # add information about true labels
  modelStudio(explainer_apartments,
              new_apartments,
              new_observation_y = new_apartments$m2.price)

  # don't compute EDA plots
  modelStudio(explainer_apartments,
              eda = FALSE)
}

#:# ex3 xgboost model on 'HR' dataset
if (requireNamespace("xgboost", quietly=TRUE)) {
  library("xgboost")
  HR_matrix <- model.matrix(status == "fired" ~ . -1, HR)

  # fit a model
  xgb_matrix <- xgb.DMatrix(HR_matrix, label = HR$status == "fired")
  params <- list(max_depth = 3, objective = "binary:logistic", eval_metric = "auc")
  model_HR <- xgb.train(params, xgb_matrix, nrounds = 300)

  # create an explainer for the model
  explainer_HR <- explain(model_HR,
                          data = HR_matrix,
                          y = HR$status == "fired",
                          type = "classification",
                          label = "xgboost")

  # pick observations
  new_observation <- HR_matrix[1:2, , drop=FALSE]
  rownames(new_observation) <- c("id1", "id2")

  # make a studio for the model
  modelStudio(explainer_HR,
              new_observation)
}
}

}
\references{
\itemize{
  \item The input object is implemented in \href{https://modeloriented.github.io/DALEX/}{\bold{DALEX}}
  \item Feature Importance, Ceteris Paribus, Partial Dependence and Accumulated Dependence explanations
   are implemented in \href{https://modeloriented.github.io/ingredients/}{\bold{ingredients}}
  \item Break Down and Shapley Values explanations are implemented in
   \href{https://modeloriented.github.io/iBreakDown/}{\bold{iBreakDown}}
}
}
\seealso{
Vignettes: \href{https://modelstudio.drwhy.ai/articles/ms-r-python-examples.html}{\bold{modelStudio - R & Python examples}}
and \href{https://modelstudio.drwhy.ai/articles/ms-perks-features.html}{\bold{modelStudio - perks and features}}
}
