% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/mnread_curve.R
\name{mnreadCurve}
\alias{mnreadCurve}
\title{MNREAD curve plotting}
\usage{
mnreadCurve(data, print_size, viewing_distance, reading_time, errors,
  ... = NULL)
}
\arguments{
\item{data}{The name of your dataframe}

\item{print_size}{The variable that contains print size values for each sentence (print size uncorrected for viewing distance)}

\item{viewing_distance}{The variable that contains the viewing distance value used for testing}

\item{reading_time}{The variable that contains the reading time for each sentence}

\item{errors}{The variable that contains the number of errors for each sentence}

\item{...}{Optional grouping arguments}
}
\value{
The function returns a plot of reading speed (in words/min) as a function of print size (in logMAR). 
Reading Acuity is marked as a triangle, Maximum Reading Speed and Critical Print Size are shown with dashed lines. 
When using two grouping arguments, a colored diamond is added for clarification.
Highlighted data points represent the range of print sizes included in the Reading Accessibility Index calculation.
}
\description{
This function plots individual MNREAD curves, while showing the estimated MNREAD parameters:
 \itemize{
  \item Maximum Reading Speed (MRS)
  \item Critical Print Size (CPS)
  \item Reading Acuity (RA)
  }
}
\section{Notes}{

This function can take no more that two grouping arguments. 
The first grouping argument is used to draw sub-plots (using facet_wrap from ggplot2). 
The second grouping argument is color-coded.

This function performs print size correction for non-standard testing viewing distance before plotting the curve.

This function uses the original algorithm described in Legge (2007) to estimate Maximum Reading Speed (MRS) and Critical Print Size (CPS).
For more details on the parameters estimation, see \code{\link{curveParam_RT}}.
}

\section{Warning}{

For the function to run properly, one needs to make sure that the variables are of the class:
 \itemize{
  \item \strong{print_size} -> numeric
  \item \strong{viewing_distance} -> integer
  \item \strong{reading_time} -> numeric
  \item \strong{errors} -> integer
  }

In cases where only 3 or less sentences were read during a test,
MRS and CPS cannot be estimated and won't be displayed on the plot.
In such cases, the Reading Accessibility Index (ACC) can be used to estimate the MNREAD score instead (cf. \code{\link{accIndex}}).
}

\examples{
# inspect the structure of the dataframe
head(data_low_vision, 10)

#------

# restrict dataset to one MNREAD test only (subject s1, regular polarity)
data_s1_reg <- data_low_vision \%>\%
   filter (subject == "s1", polarity == "regular")

# plot the MNREAD curve 
\dontrun{ mnreadCurve(data_s1_reg, ps, vd, rt, err)  }

#------

# restrict dataset to one subject (s1) and plot the MNREAD curves using ONE GROUPING ARGUMENT 
# (ie. polarity) 
data_s1 <- data_low_vision \%>\%
   filter (subject == "s1")

# plot the MNREAD curve using ONE GROUPING ARGUMENT (ie. polarity)
 \dontrun{ mnreadCurve(data_s1, ps, vd, rt, err, polarity)  }

#------

# restrict dataset to two subject (s1 & s2) and plot the MNREAD curves using TWO GROUPING ARGUMENTS 
# (ie. subject and polarity) 
data_s2 <- data_low_vision \%>\%
   filter (subject == "s1" | subject == "s2")

 \dontrun{ mnreadCurve(data_s2, ps, vd, rt, err, subject, polarity)  }

#------

# Once created, the MNREAD curve can be customized as needed using ggplot2, 
# for ex., by adding the number of errors for each sentence on top of the curve

# plot the MNREAD curve 
my.plot <- mnreadCurve(data_s1, ps, vd, rt, err, polarity)

# displays my.plot
print(my.plot)

# calculates reading speed and perform print size correction
data_s1_new <- as.data.frame(
data_s1 \%>\%
    filter (err != "NA" & rt > 0) \%>\%
    mutate (errors10 = replace (err, err > 10, 10) ) \%>\%
    mutate (rs = 60 * (10 - errors10) / rt ) \%>\%
    mutate (correct_ps = ps + round(log10(40/(vd)), 2)) ) 

# add the number of errors for each sentence 
my.new.plot <- my.plot + geom_text(aes(x = correct_ps, y = rs + 5, label = errors10),
                                   alpha = 0.5,
                                   data = data_s1_new \%>\% filter (errors10 != 0) )

# displays my.new.plot                                                                        
print(my.new.plot)


}
\seealso{
\code{\link{curveParam_RT}} for MRS and CPS estimation using values of reading time (instead of reading speed)

\code{\link{readingAcuity}} for Reading Acuity calculation
}
