% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/mkinpredict.R
\name{mkinpredict}
\alias{mkinpredict}
\alias{mkinpredict.mkinmod}
\alias{mkinpredict.mkinfit}
\title{Produce predictions from a kinetic model using specific parameters}
\usage{
mkinpredict(x, odeparms, odeini, outtimes, ...)

\method{mkinpredict}{mkinmod}(
  x,
  odeparms = c(k_parent_sink = 0.1),
  odeini = c(parent = 100),
  outtimes = seq(0, 120, by = 0.1),
  solution_type = "deSolve",
  use_compiled = "auto",
  use_symbols = FALSE,
  method.ode = "lsoda",
  atol = 1e-08,
  rtol = 1e-10,
  maxsteps = 20000L,
  map_output = TRUE,
  na_stop = TRUE,
  ...
)

\method{mkinpredict}{mkinfit}(
  x,
  odeparms = x$bparms.ode,
  odeini = x$bparms.state,
  outtimes = seq(0, 120, by = 0.1),
  solution_type = "deSolve",
  use_compiled = "auto",
  method.ode = "lsoda",
  atol = 1e-08,
  rtol = 1e-10,
  map_output = TRUE,
  ...
)
}
\arguments{
\item{x}{A kinetic model as produced by \link{mkinmod}, or a kinetic fit as
fitted by \link{mkinfit}. In the latter case, the fitted parameters are used for
the prediction.}

\item{odeparms}{A numeric vector specifying the parameters used in the
kinetic model, which is generally defined as a set of ordinary differential
equations.}

\item{odeini}{A numeric vector containing the initial values of the state
variables of the model. Note that the state variables can differ from the
observed variables, for example in the case of the SFORB model.}

\item{outtimes}{A numeric vector specifying the time points for which model
predictions should be generated.}

\item{\dots}{Further arguments passed to the ode solver in case such a
solver is used.}

\item{solution_type}{The method that should be used for producing the
predictions. This should generally be "analytical" if there is only one
observed variable, and usually "deSolve" in the case of several observed
variables. The third possibility "eigen" is fast in comparison to uncompiled
ODE models, but not applicable to some models, e.g. using FOMC for the
parent compound.}

\item{use_compiled}{If set to \code{FALSE}, no compiled version of the
\link{mkinmod} model is used, even if is present.}

\item{use_symbols}{If set to \code{TRUE} (default), symbol info present in
the \link{mkinmod} object is used if available for accessing compiled code}

\item{method.ode}{The solution method passed via \link{mkinpredict} to \code{deSolve::ode()} in
case the solution type is "deSolve" and we are not using compiled code.
When using compiled code, only lsoda is supported.}

\item{atol}{Absolute error tolerance, passed to the ode solver.}

\item{rtol}{Absolute error tolerance, passed to the ode solver.}

\item{maxsteps}{Maximum number of steps, passed to the ode solver.}

\item{map_output}{Boolean to specify if the output should list values for
the observed variables (default) or for all state variables (if set to
FALSE). Setting this to FALSE has no effect for analytical solutions,
as these always return mapped output.}

\item{na_stop}{Should it be an error if \code{deSolve::ode()} returns NaN values}
}
\value{
A matrix with the numeric solution in wide format
}
\description{
This function produces a time series for all the observed variables in a
kinetic model as specified by \link{mkinmod}, using a specific set of
kinetic parameters and initial values for the state variables.
}
\examples{

SFO <- mkinmod(degradinol = mkinsub("SFO"))
# Compare solution types
mkinpredict(SFO, c(k_degradinol = 0.3), c(degradinol = 100), 0:20,
      solution_type = "analytical")
mkinpredict(SFO, c(k_degradinol = 0.3), c(degradinol = 100), 0:20,
      solution_type = "deSolve")
mkinpredict(SFO, c(k_degradinol = 0.3), c(degradinol = 100), 0:20,
      solution_type = "deSolve", use_compiled = FALSE)
mkinpredict(SFO, c(k_degradinol = 0.3), c(degradinol = 100), 0:20,
      solution_type = "eigen")

# Compare integration methods to analytical solution
mkinpredict(SFO, c(k_degradinol = 0.3), c(degradinol = 100), 0:20,
      solution_type = "analytical")[21,]
mkinpredict(SFO, c(k_degradinol = 0.3), c(degradinol = 100), 0:20,
      method = "lsoda", use_compiled = FALSE)[21,]
mkinpredict(SFO, c(k_degradinol = 0.3), c(degradinol = 100), 0:20,
      method = "ode45", use_compiled = FALSE)[21,]
mkinpredict(SFO, c(k_degradinol = 0.3), c(degradinol = 100), 0:20,
      method = "rk4", use_compiled = FALSE)[21,]
# rk4 is not as precise here

# The number of output times used to make a lot of difference until the
# default for atol was adjusted
mkinpredict(SFO, c(k_degradinol = 0.3), c(degradinol = 100),
      seq(0, 20, by = 0.1))[201,]
mkinpredict(SFO, c(k_degradinol = 0.3), c(degradinol = 100),
      seq(0, 20, by = 0.01))[2001,]

# Comparison of the performance of solution types
SFO_SFO = mkinmod(parent = list(type = "SFO", to = "m1"),
                  m1 = list(type = "SFO"), use_of_ff = "max")
if(require(rbenchmark)) {
  benchmark(replications = 10, order = "relative", columns = c("test", "relative", "elapsed"),
    eigen = mkinpredict(SFO_SFO,
      c(k_parent = 0.15, f_parent_to_m1 = 0.5, k_m1 = 0.01),
      c(parent = 100, m1 = 0), seq(0, 20, by = 0.1),
      solution_type = "eigen")[201,],
    deSolve_compiled = mkinpredict(SFO_SFO,
      c(k_parent = 0.15, f_parent_to_m1 = 0.5, k_m1 = 0.01),
      c(parent = 100, m1 = 0), seq(0, 20, by = 0.1),
      solution_type = "deSolve")[201,],
    deSolve = mkinpredict(SFO_SFO,
      c(k_parent = 0.15, f_parent_to_m1 = 0.5, k_m1 = 0.01),
      c(parent = 100, m1 = 0), seq(0, 20, by = 0.1),
      solution_type = "deSolve", use_compiled = FALSE)[201,],
    analytical = mkinpredict(SFO_SFO,
      c(k_parent = 0.15, f_parent_to_m1 = 0.5, k_m1 = 0.01),
      c(parent = 100, m1 = 0), seq(0, 20, by = 0.1),
      solution_type = "analytical", use_compiled = FALSE)[201,])
}

\dontrun{
  # Predict from a fitted model
  f <- mkinfit(SFO_SFO, FOCUS_2006_C, quiet = TRUE)
  f <- mkinfit(SFO_SFO, FOCUS_2006_C, quiet = TRUE, solution_type = "deSolve")
  head(mkinpredict(f))
}

}
\author{
Johannes Ranke
}
