#' @param xdim dimensions of the grid.
#' @param ydim dimensions of the grid.
#' @param topo choose between a hexagonal or rectangular topology.
#' @param neighbourhood.fct choose between bubble and gaussian neighbourhoods when training a SOM.
#' @param toroidal logical, whether the grid is toroidal or not. If not provided to the \code{unit.distances} function, the information in 
#' the \code{grid} object will be used.
#' @return Function \code{somgrid} returns an object of class "somgrid", with elements \code{pts}, and the input arguments to the function.
#' @rdname unit.distances
#'
#' @export
somgrid <- function (xdim = 8, ydim = 6,
                     topo = c("rectangular", "hexagonal"),
                     neighbourhood.fct = c("bubble", "gaussian"),
                     toroidal = FALSE) 
{
  topo <- match.arg(topo)
  x <- 1L:xdim
  y <- 1L:ydim
  pts <- as.matrix(expand.grid(x = x, y = y))

  if (topo == "hexagonal") {
    pts[, 1L] <- pts[, 1L] + 0.5 * (pts[, 2L]%%2)
    pts[, 2L] <- sqrt(3)/2 * pts[, 2L]
  }

  ## Check neighbourhood function
  neighbourhood.fct <- match.arg(neighbourhood.fct)
  neighbourhood.fct <- factor(neighbourhood.fct,
                              levels = c("bubble", "gaussian"))

  res <- list(pts = pts, xdim = xdim, ydim = ydim, topo = topo,
              neighbourhood.fct = neighbourhood.fct,
              toroidal = toroidal)
  class(res) <- "somgrid"

  res
}

## some users succeed in supplying a somgrid generated by the class
## library function. This misses two elements. If that is the case,
## defaults are added and a warning is given.
check.somgrid <- function(grd) {
  mywarn <- FALSE
  if (is.null(grd$toroidal)) {
    mywarn <- TRUE
    grd$toroidal <- FALSE # default
  }
  if (is.null(grd$neighbourhood.fct)) {
    mywarn <- TRUE
    grd$neighbourhood.fct  <- 
      factor("bubble", levels = c("bubble", "gaussian")) # default
  }

  if (mywarn)
    warning("Added defaults for somgrid object - ",
            "you are probably using the somgrid function ",
            "from the class library...")
    
  grd
}
