% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/SingleGroup-methods.R
\docType{methods}
\name{residuals-method}
\alias{residuals-method}
\alias{residuals,SingleGroupClass-method}
\alias{residuals,MixtureClass-method}
\alias{residuals,MultipleGroupClass-method}
\alias{residuals,DiscreteClass-method}
\title{Compute model residuals}
\usage{
\S4method{residuals}{SingleGroupClass}(
  object,
  type = "LD",
  df.p = FALSE,
  full.scores = FALSE,
  QMC = FALSE,
  printvalue = NULL,
  tables = FALSE,
  verbose = TRUE,
  Theta = NULL,
  suppress = 1,
  theta_lim = c(-6, 6),
  quadpts = NULL,
  fold = TRUE,
  technical = list(),
  ...
)
}
\arguments{
\item{object}{an object of class \code{SingleGroupClass} or
\code{MultipleGroupClass}. Bifactor models are automatically detected and utilized for
better accuracy}

\item{type}{type of residuals to be displayed.
Can be either \code{'LD'} or \code{'LDG2'} for a local dependence matrix based on the
X2 or G2 statistics (Chen & Thissen, 1997), \code{'Q3'} for the statistic proposed by
Yen (1984), \code{'JSI'} for the jack-knife statistic proposed Edwards et al. (2018),
\code{'exp'} for the expected values for the frequencies of every response pattern,
and \code{'expfull'} for the expected values for every theoretically observable response pattern.
For the 'LD' and 'LDG2' types, the upper diagonal elements represent the standardized
residuals in the form of signed Cramers V coefficients}

\item{df.p}{logical; print the degrees of freedom and p-values?}

\item{full.scores}{logical; compute relevant statistics
for each subject in the original data?}

\item{QMC}{logical; use quasi-Monte Carlo integration? If \code{quadpts} is omitted the
default number of nodes is 5000}

\item{printvalue}{a numeric value to be specified when using the \code{res='exp'}
option. Only prints patterns that have standardized residuals greater than
\code{abs(printvalue)}. The default (NULL) prints all response patterns}

\item{tables}{logical; for LD type, return the observed, expected, and standardized residual
tables for each item combination?}

\item{verbose}{logical; allow information to be printed to the console?}

\item{Theta}{a matrix of factor scores used for statistics that require empirical estimates (i.e., Q3).
If supplied, arguments typically passed to \code{fscores()} will be ignored and these values will
be used instead}

\item{suppress}{a numeric value indicating which parameter local dependency combinations
to flag as being too high. Absolute values for the standardized estimates greater than
this value will be returned, while all values less than this value will be set to NA}

\item{theta_lim}{range for the integration grid}

\item{quadpts}{number of quadrature nodes to use. The default is extracted from model (if available)
or generated automatically if not available}

\item{fold}{logical; apply the sum 'folding' described by Edwards et al. (2018) for the JSI statistic?}

\item{technical}{list of technical arguments when models are re-estimated (see \code{\link{mirt}}
for details)}

\item{...}{additional arguments to be passed to \code{fscores()}}
}
\description{
Return model implied residuals for linear dependencies between items or at the person level.
If the latent trait density was approximated (e.g., Davidian curves, Empirical histograms, etc)
then passing \code{use_dentype_estimate = TRUE} will use the internally saved quadrature and
density components (where applicable).
}
\examples{

\dontrun{

x <- mirt(Science, 1)
residuals(x)
residuals(x, tables = TRUE)
residuals(x, type = 'exp')
residuals(x, suppress = .15)
residuals(x, df.p = TRUE)

# Pearson's X2 estimate for goodness-of-fit
full_table <- residuals(x, type = 'expfull')
head(full_table)
X2 <- with(full_table, sum((freq - exp)^2 / exp))
df <- nrow(full_table) - extract.mirt(x, 'nest') - 1
p <- pchisq(X2, df = df, lower.tail=FALSE)
data.frame(X2, df, p, row.names='Pearson-X2')

# above FOG test as a function
PearsonX2 <- function(x){
   full_table <- residuals(x, type = 'expfull')
   X2 <- with(full_table, sum((freq - exp)^2 / exp))
   df <- nrow(full_table) - extract.mirt(x, 'nest') - 1
   p <- pchisq(X2, df = df, lower.tail=FALSE)
   data.frame(X2, df, p, row.names='Pearson-X2')
}
PearsonX2(x)


# extract results manually
out <- residuals(x, df.p = TRUE, verbose=FALSE)
str(out)
out$df.p[1,2]

# with and without supplied factor scores
Theta <- fscores(x)
residuals(x, type = 'Q3', Theta=Theta)
residuals(x, type = 'Q3', method = 'ML')

# Edwards et al. (2018) JSI statistic
N <- 250
a <- rnorm(10, 1.7, 0.3)
d <- rnorm(10)
dat <- simdata(a, d, N=250, itemtype = '2PL')

mod <- mirt(dat, 1)
residuals(mod, type = 'JSI')
residuals(mod, type = 'JSI', fold=FALSE) # unfolded

# LD between items 1-2
aLD <- numeric(10)
aLD[1:2] <- rnorm(2, 2.55, 0.15)
a2 <- cbind(a, aLD)
dat <- simdata(a2, d, N=250, itemtype = '2PL')

mod <- mirt(dat, 1)

# JSI executed in parallel over multiple cores
mirtCluster()
residuals(mod, type = 'JSI')

}
}
\references{
Chalmers, R., P. (2012). mirt: A Multidimensional Item Response Theory
Package for the R Environment. \emph{Journal of Statistical Software, 48}(6), 1-29.
\doi{10.18637/jss.v048.i06}

Chen, W. H. & Thissen, D. (1997). Local dependence indices for item pairs using item
response theory. \emph{Journal of Educational and Behavioral Statistics, 22}, 265-289.

Edwards, M. C., Houts, C. R. & Cai, L. (2018). A Diagnostic Procedure to Detect Departures
From Local Independence in Item Response Theory Models.
\emph{Psychological Methods, 23}, 138-149.

Yen, W. (1984). Effects of local item dependence on the fit and equating performance of the three
parameter logistic model. \emph{Applied Psychological Measurement, 8}, 125-145.
}
