# Copyright (C) 2022-2024 Hibiki AI Limited <info@hibiki-ai.com>
#
# This file is part of mirai.
#
# mirai is free software: you can redistribute it and/or modify it under the
# terms of the GNU General Public License as published by the Free Software
# Foundation, either version 3 of the License, or (at your option) any later
# version.
#
# mirai is distributed in the hope that it will be useful, but WITHOUT ANY
# WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR
# A PARTICULAR PURPOSE. See the GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License along with
# mirai. If not, see <https://www.gnu.org/licenses/>.

# mirai ------------------------------------------------------------------------

#' Daemons (Set Persistent Processes)
#'
#' Set \sQuote{daemons} or persistent background processes to receive
#'     \code{\link{mirai}} requests. Specify \sQuote{n} to create daemons on the
#'     local machine. Specify \sQuote{url} for receiving connections from remote
#'     daemons (for distributed computing across the network). Specify
#'     \sQuote{remote} to optionally launch remote daemons via a remote
#'     configuration. By default, dispatcher ensures optimal scheduling.
#'
#' @inheritParams mirai
#' @inheritParams dispatcher
#' @param n integer number of daemons to set.
#' @param url [default NULL] if specified, the character URL or vector of URLs
#'     on the host for remote daemons to dial into, including a port accepting
#'     incoming connections (and optionally for websockets, a path), e.g.
#'     'tcp://hostname:5555' or 'ws://10.75.32.70:5555/path'. Specify a URL
#'     starting 'tls+tcp://' or 'wss://' to use secure TLS connections.
#'     Auxiliary function \code{\link{host_url}} may be used to construct a
#'     valid host URL.
#' @param remote [default NULL] required only for launching remote daemons, a
#'     configuration generated by \code{\link{remote_config}} or
#'     \code{\link{ssh_config}}.
#' @param dispatcher [default TRUE] logical value whether to use dispatcher.
#'     Dispatcher is a local background process that connects to daemons on
#'     behalf of the host and ensures FIFO scheduling (see Dispatcher section
#'     below).
#' @param ... (optional) additional arguments passed through to
#'     \code{\link{dispatcher}} if using dispatcher and/or \code{\link{daemon}}
#'     if launching daemons. These include \sQuote{retry} and \sQuote{token} at
#'     dispatcher and \sQuote{autoexit}, \sQuote{cleanup}, \sQuote{output},
#'     \sQuote{maxtasks}, \sQuote{idletime}, \sQuote{walltime} and
#'     \sQuote{timerstart} at daemon.
#' @param seed [default NULL] (optional) supply a random seed (single value,
#'     interpreted as an integer). This is used to inititalise the L'Ecuyer-CMRG
#'     RNG streams sent to each daemon. Note that reproducible results can be
#'     expected only for \code{dispatcher = FALSE}, as the unpredictable timing
#'     of task completions would otherwise influence the tasks sent to each
#'     daemon. Even for \code{dispatcher = FALSE}, reproducibility is not
#'     guaranteed if the order in which tasks are sent is not deterministic.
#' @param tls [default NULL] (optional for secure TLS connections) if not
#'     supplied, zero-configuration single-use keys and certificates are
#'     automatically generated. If supplied, \strong{either} the character path
#'     to a file containing the PEM-encoded TLS certificate and associated
#'     private key (may contain additional certificates leading to a validation
#'     chain, with the TLS certificate first), \strong{or} a length 2 character
#'     vector comprising [i] the TLS certificate (optionally certificate chain)
#'     and [ii] the associated private key.
#'
#' @return Depending on the arguments supplied:
#'
#'     \itemize{
#'     \item using dispatcher: integer number of daemons set.
#'     \item or else launching local daemons: integer number of daemons launched.
#'     \item otherwise: the character host URL.
#'     }
#'
#' @details Use \code{daemons(0)} to reset daemon connections:
#'     \itemize{
#'     \item All connected daemons and/or dispatchers exit automatically.
#'     \item \pkg{mirai} reverts to the default behaviour of creating a new
#'     background process for each request.
#'     \item Any unresolved \sQuote{mirai} will return an \sQuote{errorValue} 19
#'     (Connection reset) after a reset.
#'     \item Calling \code{daemons} with revised (or even the same) settings for
#'     the same compute profile implicitly resets daemons before applying the
#'     new settings.
#'     }
#'
#'     If the host session ends, all connected dispatcher and daemon processes
#'     automatically exit as soon as their connections are dropped (unless
#'     the daemons were started with \code{autoexit = FALSE}). If a daemon is
#'     processing a task, it will exit as soon as the task is complete.
#'
#'     To reset persistent daemons started with \code{autoexit = FALSE}, use
#'     \code{daemons(NULL)} instead, which also sends exit instructions to all
#'     connected daemons prior to resetting.
#'
#'     For historical reasons, \code{daemons()} with no arguments returns the
#'     value of \code{\link{status}}.
#'
#' @section Local Daemons:
#'
#'     Daemons provide a potentially more efficient solution for asynchronous
#'     operations as new processes no longer need to be created on an \emph{ad
#'     hoc} basis.
#'
#'     Supply the argument \sQuote{n} to set the number of daemons. New
#'     background \code{\link{daemon}} processes are automatically created on
#'     the local machine connecting back to the host process, either directly or
#'     via dispatcher.
#'
#' @section Dispatcher:
#'
#'     By default \code{dispatcher = TRUE}. This launches a background process
#'     running \code{\link{dispatcher}}. Dispatcher connects to daemons on
#'     behalf of the host and ensures FIFO scheduling of tasks. Dispatcher uses
#'     synchronisation primitives from \pkg{nanonext}, waiting rather than
#'     polling for tasks, which is both efficient (no resource usage) and fully
#'     event-driven (having no latency).
#'
#'     By specifying \code{dispatcher = FALSE}, daemons connect to the host
#'     directly rather than through dispatcher. The host sends tasks to
#'     connected daemons immediately in a round-robin fashion. Optimal
#'     scheduling is not guaranteed as the duration of tasks cannot be known
#'     \emph{a priori}, hence tasks can be queued at one daemon while other
#'     daemons remain idle. However, this provides a resource-light approach
#'     suited to working with similar-length tasks, or where concurrent tasks
#'     typically do not exceed available daemons.
#'
#' @section Distributed Computing:
#'
#'     Specifying \sQuote{url} allows tasks to be distributed across the network.
#'     This should be a character string such as \sQuote{tcp://10.75.32.70:5555}
#'     at which daemon processes should connect to. Switching the URL scheme to
#'     \sQuote{tls+tcp://} or \sQuote{wss://} automatically upgrades the
#'     connection to use TLS. The auxiliary function \code{\link{host_url}} may
#'     be used to automatically construct a valid host URL based on the
#'     computer's hostname.
#'
#'     Specify \sQuote{remote} with a call to \code{\link{remote_config}} or
#'     \code{\link{ssh_config}} to launch daemons on remote machines. Otherwise,
#'     \code{\link{launch_remote}} may be used to generate the shell commands to
#'     deploy daemons manually on remote resources.
#'
#'     IPv6 addresses are also supported and must be enclosed in square brackets
#'     [ ] to avoid confusion with the final colon separating the port. For
#'     example, port 5555 on the IPv6 loopback address ::1 would be specified
#'     as \sQuote{tcp://[::1]:5555}.
#'
#'     Specifying the wildcard value zero for the port number e.g.
#'     \sQuote{tcp://[::1]:0} or \sQuote{ws://[::1]:0} will automatically assign
#'     a free ephemeral port. Use \code{\link{status}} to inspect the actual
#'     assigned port at any time.
#'
#'     \strong{With Dispatcher}
#'
#'     When using dispatcher, it is recommended to use a websocket URL rather
#'     than TCP, as this requires only one port to connect to all daemons: a
#'     websocket URL supports a path after the port number, which can be made
#'     unique for each daemon.
#'
#'     Specifying a single host URL such as \sQuote{ws://10.75.32.70:5555} with
#'     \code{n = 6} will automatically append a sequence to the path, listening
#'     to the URLs \sQuote{ws://10.75.32.70:5555/1} through
#'     \sQuote{ws://110.75.32.70:5555/6}.
#'
#'     Alternatively, specify a vector of URLs to listen to arbitrary port
#'     numbers / paths. In this case it is optional to supply \sQuote{n} as this
#'     can be inferred by the length of vector supplied.
#'
#'     Individual daemons then dial in to each of these host URLs. At most one
#'     daemon can be dialled into each URL at any given time.
#'
#'     Dispatcher automatically adjusts to the number of daemons actually
#'     connected. Hence it is possible to dynamically scale up or down the
#'     number of daemons as required, subject to the maximum number initially
#'     specified.
#'
#'     Alternatively, supplying a single TCP URL will listen at a block of URLs
#'     with ports starting from the supplied port number and incrementing by one
#'     for \sQuote{n} specified e.g. the host URL
#'     \sQuote{tcp://10.75.32.70:5555} with \code{n = 6} listens to the
#'     contiguous block of ports 5555 through 5560.
#'
#'     \strong{Without Dispatcher}
#'
#'     A TCP URL may be used in this case as the host listens at only one
#'     address, utilising a single port.
#'
#'     The network topology is such that daemons (started with
#'     \code{\link{daemon}}) or indeed dispatchers (started with
#'     \code{\link{dispatcher}}) dial into the same host URL.
#'
#'     \sQuote{n} is not required in this case, and disregarded if supplied, as
#'     network resources may be added or removed at any time. The host
#'     automatically distributes tasks to all connected daemons and dispatchers
#'     in a round-robin fashion.
#'
#' @section Compute Profiles:
#'
#'     By default, the \sQuote{default} compute profile is used. Providing a
#'     character value for \sQuote{.compute} creates a new compute profile with
#'     the name specified. Each compute profile retains its own daemons
#'     settings, and may be operated independently of each other. Some usage
#'     examples follow:
#'
#'     \strong{local / remote} daemons may be set with a host URL and specifying
#'     \sQuote{.compute} as \sQuote{remote}, which creates a new compute
#'     profile. Subsequent \code{\link{mirai}} calls may then be sent for local
#'     computation by not specifying the \sQuote{.compute} argument, or for
#'     remote computation to connected daemons by specifying the
#'     \sQuote{.compute} argument as \sQuote{remote}.
#'
#'     \strong{cpu / gpu} some tasks may require access to different types of
#'     daemon, such as those with GPUs. In this case, \code{daemons()} may be
#'     called twice to set up host URLs for CPU-only daemons and for those
#'     with GPUs, specifying the \sQuote{.compute} argument as \sQuote{cpu} and
#'     \sQuote{gpu} respectively. By supplying the \sQuote{.compute} argument to
#'     subsequent \code{\link{mirai}} calls, tasks may be sent to either
#'     \sQuote{cpu} or \sQuote{gpu} daemons as appropriate.
#'
#'     Note: further actions such as resetting daemons via \code{daemons(0)}
#'     should be carried out with the desired \sQuote{.compute} argument
#'     specified.
#'
#' @examples
#' if (interactive()) {
#' # Only run examples in interactive R sessions
#'
#' # Create 2 local daemons (using dispatcher)
#' daemons(2)
#' status()
#' # Reset to zero
#' daemons(0)
#'
#' # Create 2 local daemons (not using dispatcher)
#' daemons(2, dispatcher = FALSE)
#' status()
#' # Reset to zero
#' daemons(0)
#'
#' # 2 remote daemons via dispatcher using WebSockets
#' daemons(2, url = host_url(ws = TRUE))
#' status()
#' # Reset to zero
#' daemons(0)
#'
#' # Set host URL for remote daemons to dial into
#' daemons(url = host_url(), dispatcher = FALSE)
#' status()
#' # Reset to zero
#' daemons(0)
#'
#' # Use with() to evaluate with daemons for the duration of the expression
#' with(
#'   daemons(2),
#'   {
#'     m1 <- mirai(Sys.getpid())
#'     m2 <- mirai(Sys.getpid())
#'     cat(call_mirai(m1)$data, call_mirai(m2)$data, "\n")
#'   }
#' )
#'
#' }
#'
#' \dontrun{
#' # Launch 2 daemons on remotes 'nodeone' and 'nodetwo' using SSH
#' # connecting back directly to the host URL over a TLS connection:
#'
#' daemons(url = host_url(tls = TRUE),
#'         remote = ssh_config(c('ssh://nodeone', 'ssh://nodetwo')),
#'         dispatcher = FALSE)
#'
#' # Launch 4 daemons on the remote machine 10.75.32.90 using SSH tunnelling
#' # over port 5555 ('url' hostname must be 'localhost' or '127.0.0.1'):
#'
#' daemons(n = 4,
#'         url = 'ws://localhost:5555',
#'         remote = ssh_config('ssh://10.75.32.90', tunnel = TRUE))
#'
#' }
#'
#' @export
#'
daemons <- function(n, url = NULL, remote = NULL, dispatcher = TRUE, ...,
                    seed = NULL, tls = NULL, pass = NULL, .compute = "default") {

  missing(n) && missing(url) && return(status(.compute))

  envir <- ..[[.compute]]

  if (is.character(url)) {

    if (is.null(envir)) {
      envir <- new.env(hash = FALSE, parent = ..)
      tls <- check_create_tls(url = url, tls = tls, envir = envir)
      create_stream(n = n, seed = seed, envir = envir)
      if (dispatcher) {
        n <- if (missing(n)) length(url) else if (is.numeric(n) && n >= 1L) as.integer(n) else stop(._[["n_one"]])
        if (length(tls)) tls_config(server = tls, pass = pass)
        cv <- cv()
        dots <- parse_dots(...)
        output <- attr(dots, "output")
        urld <- local_url()
        urlc <- sprintf("%s%s", urld, "c")
        sock <- req_socket(urld)
        sockc <- req_socket(urlc)
        launch_and_sync_daemon(sock, wa5(urld, dots, n, urlc, url), output, tls, pass) || stop(._[["sync_timeout"]])
        init_monitor(sockc = sockc, envir = envir)
        `[[<-`(envir, "cv", cv)
      } else {
        sock <- req_socket(url, tls = if (length(tls)) tls_config(server = tls, pass = pass))
        store_urls(sock = sock, envir = envir)
        n <- 0L
      }
      `[[<-`(.., .compute, `[[<-`(`[[<-`(envir, "sock", sock), "n", n))
      remotes <- substitute(remote)
      if (!is.symbol(remotes)) remote <- remotes
      if (length(remote))
        launch_remote(url = envir[["urls"]], remote = remote, tls = envir[["tls"]], ..., .compute = .compute)
      serialization_refhook()
    } else {
      daemons(n = 0L, .compute = .compute)
      return(daemons(n = n, url = url, remote = remote, dispatcher = dispatcher, ..., seed = seed, tls = tls, pass = pass, .compute = .compute))
    }

  } else {

    signal <- is.null(n)
    if (signal) n <- 0L
    is.numeric(n) || stop(._[["numeric_n"]])
    n <- as.integer(n)

    if (n == 0L) {
      is.null(envir) && return(0L)

      if (signal) send_signal(envir = envir)
      reap(envir[["sock"]])
      is.null(envir[["sockc"]]) || reap(envir[["sockc"]])
      ..[[.compute]] <- NULL -> envir

    } else if (is.null(envir)) {

      n > 0L || stop(._[["n_zero"]])
      envir <- new.env(hash = FALSE, parent = ..)
      urld <- local_url()
      create_stream(n = n, seed = seed, envir = envir)
      dots <- parse_dots(...)
      output <- attr(dots, "output")
      if (dispatcher) {
        cv <- cv()
        sock <- req_socket(urld)
        urlc <- sprintf("%s%s", urld, "c")
        sockc <- req_socket(urlc)
        launch_and_sync_daemon(sock, wa4(urld, dots, envir[["stream"]], n, urlc), output) || stop(._[["sync_timeout"]])
        for (i in seq_len(n)) next_stream(envir)
        init_monitor(sockc = sockc, envir = envir)
        `[[<-`(envir, "cv", cv)
      } else {
        sock <- req_socket(urld)
        for (i in seq_len(n))
          launch_and_sync_daemon(sock, wa3(urld, dots, next_stream(envir)), output)
        `[[<-`(envir, "urls", urld)
      }
      `[[<-`(.., .compute, `[[<-`(`[[<-`(envir, "sock", sock), "n", n))
      serialization_refhook()
    } else {
      daemons(n = 0L, .compute = .compute)
      return(daemons(n = n, url = url, remote = remote, dispatcher = dispatcher, ..., seed = seed, tls = tls, pass = pass, .compute = .compute))
    }

  }

  if (is.null(envir)) 0L else `class<-`(
    if (envir[["n"]]) envir[["n"]] else envir[["urls"]],
    c("miraiDaemons", .compute)
  )

}

#' @export
#'
print.miraiDaemons <- function(x, ...) print(unclass(x))

#' With Mirai Daemons
#'
#' Evaluate an expression with daemons that last for the duration of the
#'     expression.
#'
#' @param data a call to \code{\link{daemons}}.
#' @param expr an expression to evaluate.
#' @param ... not used.
#'
#' @return The return value of 'expr'.
#'
#' @details This function is an S3 method for the generic \code{with} for
#'     class 'miraiDaemons'.
#'
#' @examples
#' if (interactive()) {
#' # Only run examples in interactive R sessions
#'
#' with(
#'   daemons(2),
#'   {
#'     m1 <- mirai(Sys.getpid())
#'     m2 <- mirai(Sys.getpid())
#'     cat(call_mirai(m1)$data, call_mirai(m2)$data, "\n")
#'   }
#' )
#'
#' status()
#'
#' }
#'
#' @export
#'
with.miraiDaemons <- function(data, expr, ...) {

  on.exit(daemons(0L, .compute = class(data)[2L]))
  expr

}

#' Status Information
#'
#' Retrieve status information for the specified compute profile, comprising
#'     current connections and daemons status.
#'
#' @param .compute [default 'default'] character compute profile (each compute
#'     profile has its own set of daemons for connecting to different resources).
#'
#'     \strong{or} a \sQuote{miraiCluster} to obtain its status.
#'
#' @return A named list comprising:
#'     \itemize{
#'     \item \strong{connections} - integer number of active connections.
#'     \cr Using dispatcher: Always 1L as there is a single connection to
#'     dispatcher, which connects to the daemons in turn.
#'     \item \strong{daemons} - of variable type.
#'     \cr Using dispatcher: a status matrix (see Status Matrix section below),
#'     or else an integer \sQuote{errorValue} if communication with dispatcher
#'     failed.
#'     \cr Not using dispatcher: the character host URL.
#'     \cr Not set: 0L.
#'     }
#'
#' @section Status Matrix:
#'
#'     When using dispatcher, \code{$daemons} comprises an integer matrix with
#'     the following columns:
#'     \itemize{
#'     \item \strong{i} - integer index number.
#'     \item \strong{online} - shows as 1 when there is an active connection,
#'     or else 0 if a daemon has yet to connect or has disconnected.
#'     \item \strong{instance} - increments by 1 every time there is a new
#'     connection at a URL. This counter is designed to track new daemon
#'     instances connecting after previous ones have ended (due to time-outs
#'     etc.). The count becomes negative immediately after a URL is regenerated
#'     by \code{\link{saisei}}, but increments again once a new daemon connects.
#'     \item \strong{assigned} - shows the cumulative number of tasks assigned
#'     to the daemon.
#'     \item \strong{complete} - shows the cumulative number of tasks
#'     completed by the daemon.
#'     }
#'     The dispatcher URLs are stored as row names to the matrix.
#'
#' @examples
#' if (interactive()) {
#' # Only run examples in interactive R sessions
#'
#' status()
#' daemons(n = 2L, url = "wss://[::1]:0")
#' status()
#' daemons(0)
#'
#' }
#'
#' @export
#'
status <- function(.compute = "default") {

  is.list(.compute) && return(status(attr(.compute, "id")))
  envir <- ..[[.compute]]
  is.null(envir) && return(list(connections = 0L, daemons = 0L))
  list(connections = as.integer(stat(envir[["sock"]], "pipes")),
       daemons = if (is.null(envir[["sockc"]])) envir[["urls"]] else query_status(envir))

}

#' Custom Serialization Functions
#'
#' Registers custom serialization and unserialization functions for sending and
#'     receiving external pointer reference objects.
#'
#' @param refhook \strong{either} a list or pairlist of two functions: the
#'     signature for the first must accept a reference object inheriting from
#'     'class' (or a list of such objects) and return a raw vector, and the
#'     second must accept a raw vector and return reference objects (or a list
#'     of such objects), \cr \strong{or else} NULL to reset.
#' @param class [default ""] a character string representing the class of object
#'     that these serialization function will be applied to, e.g. 'ArrowTabular'
#'     or 'torch_tensor'.
#' @param vec [default FALSE] the serialization functions accept and return
#'     reference object individually e.g. \code{arrow::write_to_raw} and
#'     \code{arrow::read_ipc_stream}. If TRUE, the serialization functions are
#'     vectorized and accept and return a list of reference objects, e.g.
#'     \code{torch::torch_serialize} and \code{torch::torch_load}.
#'
#' @return Invisibly, the pairlist of currently-registered 'refhook' functions.
#'     A message is printed to the console when functions are successfully
#'     registered or reset.
#'
#' @details Calling without any arguments returns the pairlist of
#'     currently-registered 'refhook' functions.
#'
#'     This function may be called prior to or after setting daemons, with the
#'     registered functions applying across all compute profiles.
#'
#' @examples
#' r <- serialization(list(function(x) serialize(x, NULL), unserialize))
#' print(serialization())
#' serialization(r)
#'
#' serialization(NULL)
#' print(serialization())
#'
#' @export
#'
serialization <- function(refhook = list(), class = "", vec = FALSE) {

  register <- !missing(refhook)
  cfg <- next_config(refhook = refhook, class = class, vec = vec)

  if (register) {
    if (is.list(refhook) && length(refhook) == 2L && is.function(refhook[[1L]]) && is.function(refhook[[2L]]))
      cat("mirai serialization functions registered\n", file = stderr()) else
        if (is.null(refhook))
          cat("mirai serialization functions cancelled\n", file = stderr()) else
            stop(._[["refhook_invalid"]])
    `[[<-`(., "refhook", list(refhook, class, vec))
    register_everywhere(refhook = refhook, class = class, vec = vec)
  }

  invisible(cfg)

}

# internals --------------------------------------------------------------------

check_create_tls <- function(url, tls, envir) {
  purl <- parse_url(url)
  sch <- substr(purl[["scheme"]], 1L, 3L)
  if ((sch == "wss" || sch == "tls") && is.null(tls)) {
    cert <- write_cert(cn = purl[["hostname"]])
    `[[<-`(envir, "tls", cert[["client"]])
    tls <- cert[["server"]]
  }
  tls
}

create_stream <- function(n, seed, envir) {
  rexp(1L)
  oseed <- .GlobalEnv[[".Random.seed"]]
  RNGkind("L'Ecuyer-CMRG")
  if (length(seed)) set.seed(seed)
  `[[<-`(envir, "stream", .GlobalEnv[[".Random.seed"]])
  `[[<-`(.GlobalEnv, ".Random.seed", oseed)
}

tokenized_url <- function(url) sprintf("%s/%s", url, random(12L))

req_socket <- function(url, tls = NULL, resend = 0L)
  `opt<-`(socket(protocol = "req", listen = url, tls = tls), "req:resend-time", resend)

parse_dots <- function(...) {
  missing(...) && return("")
  dots <- list(...)
  for (dot in dots)
    is.numeric(dot) || is.logical(dot) || stop(._[["wrong_dots"]])
  dnames <- names(dots)
  out <- sprintf(",%s", paste(dnames, dots, sep = "=", collapse = ","))
  pos <- dnames == "output"
  any(pos) && as.logical(dots[pos])[1L] && return(`attr<-`(out, "output", ""))
  out
}

parse_tls <- function(tls)
  switch(length(tls) + 1L, "", sprintf(",tls='%s'", tls), sprintf(",tls=c('%s','%s')", tls[1L], tls[2L]))

libp <- function(lp = .libPaths()) lp[file.exists(file.path(lp, "mirai"))][1L]

wa2 <- function(url, dots, tls = NULL)
  shQuote(sprintf("mirai::daemon('%s'%s%s)", url, dots, parse_tls(tls)))

wa3 <- function(url, dots, rs, tls = NULL)
  shQuote(sprintf("mirai::daemon('%s'%s%s,rs=c(%s))", url, dots, parse_tls(tls), paste0(rs, collapse = ",")))

wa4 <- function(urld, dots, rs, n, urlc)
  shQuote(sprintf(".libPaths(c('%s',.libPaths()));mirai::dispatcher('%s',n=%d,rs=c(%s),monitor='%s'%s)", libp(), urld, n, paste0(rs, collapse= ","), urlc, dots))

wa5 <- function(urld, dots, n, urlc, url)
  shQuote(sprintf(".libPaths(c('%s',.libPaths()));mirai::dispatcher('%s',c('%s'),n=%d,monitor='%s'%s)", libp(), urld, paste0(url, collapse = "','"), n, urlc, dots))

launch_daemon <- function(args, output)
  system2(command = .command, args = c("-e", args), stdout = output, stderr = output, wait = FALSE)

launch_and_sync_daemon <- function(sock, args, output, tls = NULL, pass = NULL) {
  cv <- cv()
  pipe_notify(sock, cv = cv, add = TRUE)
  if (is.character(tls)) {
    switch(
      length(tls),
      {
        on.exit(Sys.unsetenv("MIRAI_TEMP_FIELD1"))
        Sys.setenv(MIRAI_TEMP_FIELD1 = tls)
        Sys.unsetenv("MIRAI_TEMP_FIELD2")
      },
      {
        on.exit(Sys.unsetenv(c("MIRAI_TEMP_FIELD1", "MIRAI_TEMP_FIELD2")))
        Sys.setenv(MIRAI_TEMP_FIELD1 = tls[1L])
        Sys.setenv(MIRAI_TEMP_FIELD2 = tls[2L])
      }
    )
    if (is.character(pass)) {
      on.exit(Sys.unsetenv("MIRAI_TEMP_VAR"), add = TRUE)
      Sys.setenv(MIRAI_TEMP_VAR = pass)
    }
  }
  launch_daemon(args, output)
  res <- until(cv, .limit_long)
  pipe_notify(sock, cv = NULL, add = TRUE)
  res
}

init_monitor <- function(sockc, envir) {
  res <- query_dispatcher(sockc, command = FALSE, mode = 2L, block = .limit_long)
  is.object(res) && stop(._[["sync_timeout"]])
  `[[<-`(`[[<-`(`[[<-`(envir, "sockc", sockc), "urls", res[-1L]), "pid", as.integer(res[1L]))
}

store_urls <- function(sock, envir) {
  listener <- attr(sock, "listener")[[1L]]
  urls <- opt(listener, "url")
  if (parse_url(urls)[["port"]] == "0")
    urls <- sub_real_port(port = opt(listener, "tcp-bound-port"), url = urls)
  `[[<-`(envir, "urls", urls)
}

send_signal <- function(envir) {
  signals <- max(length(envir[["urls"]]), stat(envir[["sock"]], "pipes"))
  for (i in seq_len(signals)) {
    send(envir[["sock"]], data = ._scm_., mode = 2L)
    msleep(10L)
  }
}

query_status <- function(envir) {
  res <- query_dispatcher(sock = envir[["sockc"]], command = 0L, mode = 5L)
  is.object(res) && return(res)
  `attributes<-`(res, list(dim = c(envir[["n"]], 5L),
                           dimnames = list(envir[["urls"]], c("i", "online", "instance", "assigned", "complete"))))
}

register_everywhere <- function(refhook, class, vec)
  for (.compute in names(..))
    everywhere(mirai::serialization(refhook = refhook, class = class, vec = vec),
               refhook = refhook, class = class, vec = vec, .compute = .compute)

serialization_refhook <- function(refhook = .[["refhook"]])
  if (length(refhook[[1L]]))
    register_everywhere(refhook = refhook[[1L]], class = refhook[[2L]], vec = refhook[[3L]])

._scm_. <- as.raw(c(0x07, 0x00, 0x00, 0x00, 0x42, 0x0a, 0x03, 0x00, 0x00, 0x00, 0x02, 0x03, 0x04, 0x00, 0x00, 0x05, 0x03, 0x00, 0x05, 0x00, 0x00, 0x00, 0x55, 0x54, 0x46, 0x2d, 0x38, 0xfc, 0x00, 0x00, 0x00))
