#' Identify individuals contained within a plot
#'
#' Given a spatially explicit data frame of individual locations in a simulated arena,
#' and the bounds of a series of plots, identifies the contents of each plot.
#'
#' @param arena Data frame of three columns: "individuals", "X", and "Y"
#' @param plotPlacer.results The results of a call to plotPlacer. A list with two
#' elementers. The first is a matrix of X Y coordinates of plots. The second is a
#' symmetrical distance matrix summarizing the distances among these plots.
#' 
#' @details Takes a data frame like that returned from filteringArena(), and a matrix
#' like that returned from plotPlacer(), and returns the resulting community data
#' matrix such as might be generated by someone surveying a forest plot. Plots with < 2
#' species are excluded from the CDM.
#'
#' @return A matrix with species as rows and plots as columns.
#'
#' @export
#'
#' @importFrom colorRamps blue2green2red
#'
#' @references Miller, E. T., D. R. Farine, and C. H. Trisos. 2015. Phylogenetic community
#' structure metrics and null models: a review with new methods and software.
#' bioRxiv 025726.
#'
#' @examples
#' tree <- geiger::sim.bdtree(b=0.1, d=0, stop="taxa", n=50)
#'
#' temp <- evolveTraits(tree)
#'
#' phydistmatrix <- ape::cophenetic.phylo(temp[[1]])
#'
#' #define a color for each species
#' cols <- colorRamps::blue2green2red(nrow(phydistmatrix))
#'
#' #prep the data for the simulation
#' prepped <- prepSimulations(tree, arena.length=300, mean.log.individuals=2, 
#' length.parameter=5000, sd.parameter=50, max.distance=20, proportion.killed=0.2,
#' competition.iterations=3)
#'
#' singleArena <- filteringArena(prepped)
#'
#' #plot the arena. don't close the window
#' plot(singleArena$arena$X, singleArena$arena$Y, pch=20, cex=0.5, xlim=c(0,300), 
#'	ylim=c(0,300), col=cols[singleArena$arena$individuals])
#'
#' boundResults <- plotPlacer(no.plots=10, arena.length=300, plot.length=50)
#'
#' plotPlotter(boundResults$plot.bounds)
#'
#' #return a CDM in picante format
#' cdm <- plotContents(singleArena$arena, boundResults)

plotContents <- function(arena, plotPlacer.results)
{
	species <- unique(arena$individuals)
	com.results <- matrix(0, ncol=dim(plotPlacer.results$plot.bounds)[1], 
		nrow=length(species))
	rownames(com.results) <- species

	for (i in c(1:dim(plotPlacer.results$plot.bounds)[1])) 
	{
		in_plot <- arena$individuals[arena$X >= 
			plotPlacer.results$plot.bounds[i,1] & 
			arena$X <= plotPlacer.results$plot.bounds[i,2] & 
			arena$Y >= plotPlacer.results$plot.bounds[i,3] & 
			arena$Y <= plotPlacer.results$plot.bounds[i,4]]

		for (j in c(1:length(species)))
		{
			com.results[j,i] <- sum(in_plot == species[j])
		}
	}

	#CDM was NOT IN PICANTE FORMAT. Transpose to picante
	com.results <- t(com.results)

	#now give the same temporary names as plotPlacer.results to CDM so you can see what
	#needs to get cut from the distance matrix
	row.names(com.results) <- row.names(plotPlacer.results$dists)
	
	#identify plots that have fewer than 2 species. pull their tempPlot ID.
	toCut <- row.names(com.results)[apply(com.results, 1, lengthNonZeros) < 2]

	#add a stop here that if the number of plots to cut is equal to the number of plots
	#total, it stops and warns what happened
	if(length(toCut) == dim(plotPlacer.results$plot.bounds)[1])
	{
		stop("No plots contained >2 spp. Try increasing density of individuals in arena.",
			call.=FALSE)
	}
	
	#pull these plots out of the CDM and out of the dists object
	com.results <- com.results[!(row.names(com.results) %in% toCut),]
	#first rows of distance object
	plotPlacer.results$dists <-
		plotPlacer.results$dists[!(row.names(plotPlacer.results$dists) %in% toCut),]

	#now columns of distance object
	plotPlacer.results$dists <-
		plotPlacer.results$dists[,!(colnames(plotPlacer.results$dists) %in% toCut)]	
		
	#assign names to plots (note that this happens after removing plots with < 2 spp
	#so i believe the regional null should not have any trouble accidentally creating 
	#plots with different names. need to confirm and should make regional null require
	#existing plot names to avoid any conflicts
	
	plot <- paste("plot",1:dim(com.results)[1], sep="")
	dimnames(com.results)[[1]] <- plot

	#assign the same new names to the distance object
	row.names(plotPlacer.results$dists) <- plot
	colnames(plotPlacer.results$dists) <- plot
	
	results <- list("picante.cdm"=com.results, "dists"=plotPlacer.results$dists)
	return(results)
}
