#' Weighted Average of Absolute Scores
#'
#' Compute the Weighted Average of Absolute Scores (Olivoto et al., 2019) for
#' quantifying the stability of \emph{g} genotypes conducted in \emph{e}
#' environments using linear mixed-effect models.
#'
#' The weighted average of absolute scores is computed considering all
#' Interaction Principal Component Axis (IPCA) from the Singular Value Decomposition (SVD) of the
#' matrix of genotype-environment interaction (GEI) effects generated by a linear
#' mixed-effect model, as follows:
#'  \deqn{WAASB_i = \sum_{k = 1}^{p} |IPCA_{ik} \times EP_k|/ \sum_{k = 1}^{p}EP_k}
#'
#' where \eqn{WAASB_i} is the weighted average of absolute scores of the
#' \emph{i}th genotype; \eqn{IPCA_{ik}} is the score of the \emph{i}th genotype
#' in the \emph{k}th Interaction Principal Component Axis (IPCA); and \eqn{EP_k} is the explained variance of the *k*th
#' IPCA for \emph{k = 1,2,..,p}, considering \eqn{p = min(g - 1; e - 1)}.
#'
#' The nature of the effects in the model is
#' chosen with the argument \code{random}. By default, the experimental design
#' considered in each environment is a randomized complete block design. If
#' \code{block} is informed, a resolvable alpha-lattice design (Patterson and
#' Williams, 1976) is implemented. The following six models can be fitted
#' depending on the values of \code{random} and \code{block} arguments.
#'   *  \strong{Model 1:} \code{block = NULL} and \code{random = "gen"} (The
#'   default option). This model considers a Randomized Complete Block Design in
#'   each environment assuming genotype and genotype-environment interaction as
#'   random effects. Environments and blocks nested within environments are
#'   assumed to fixed factors.
#'
#'   *  \strong{Model 2:} \code{block = NULL} and \code{random = "env"}. This
#'   model considers a Randomized Complete Block Design in each environment
#'   treating environment, genotype-environment interaction, and blocks nested
#'   within environments as random factors. Genotypes are assumed to be fixed
#'   factors.
#'
#'   *  \strong{Model 3:} \code{block = NULL} and \code{random = "all"}. This
#'   model considers a Randomized Complete Block Design in each environment
#'   assuming a random-effect model, i.e., all effects (genotypes, environments,
#'   genotype-vs-environment interaction and blocks nested within environments)
#'   are assumed to be random factors.
#'
#'   *  \strong{Model 4:} \code{block} is not \code{NULL} and \code{random =
#'   "gen"}. This model considers an alpha-lattice design in each environment
#'   assuming genotype, genotype-environment interaction, and incomplete blocks
#'   nested within complete replicates as random to make use of inter-block
#'   information (Mohring et al., 2015). Complete replicates nested within
#'   environments and environments are assumed to be fixed factors.
#'
#'   *  \strong{Model 5:} \code{block} is not \code{NULL} and \code{random =
#'   "env"}. This model considers an alpha-lattice design in each environment
#'   assuming genotype as fixed. All other sources of variation (environment,
#'   genotype-environment interaction, complete replicates nested within
#'   environments, and incomplete blocks nested within replicates) are assumed
#'   to be random factors.
#'
#'   *  \strong{Model 6:} \code{block} is not \code{NULL} and \code{random =
#'   "all"}. This model considers an alpha-lattice design in each environment
#'   assuming all effects, except the intercept, as random factors.
#'
#' @param .data The dataset containing the columns related to Environments,
#'   Genotypes, replication/block and response variable(s).
#' @param env The name of the column that contains the levels of the
#'   environments.
#' @param gen The name of the column that contains the levels of the genotypes.
#' @param rep The name of the column that contains the levels of the
#'   replications/blocks.
#' @param resp The response variable(s). To analyze multiple variables in a
#'   single procedure a vector of variables may be used. For example \code{resp
#'   = c(var1, var2, var3)}.
#' @param block Defaults to \code{NULL}. In this case, a randomized complete
#'   block design is considered. If block is informed, then an alpha-lattice
#'   design is employed considering block as random to make use of inter-block
#'   information, whereas the complete replicate effect is always taken as
#'   fixed, as no inter-replicate information was to be recovered (Mohring et
#'   al., 2015).
#' @param mresp  The new maximum value after rescaling the response variable. By
#'   default, all variables in \code{resp} are rescaled so that de maximum value
#'   is 100 and the minimum value is 0 (i.e., \code{mresp = 100}). It must be a
#'   numeric vector of the same length of \code{resp} if rescaling is assumed to
#'   be different across variables, e.g., if for the first variable smaller
#'   values are better and for the second one, higher values are better, then
#'   \code{mresp = c(0, 100)} must be used. Numeric value of length 1 will be
#'   recycled with a warning message.
#' @param wresp The weight for the response variable(s) for computing the WAASBY
#'   index. By default, all variables in \code{resp} have equal weights for mean
#'   performance and stability (i.e., \code{wresp = 50}). It must be a numeric
#'   vector of the same length of \code{resp} to assign different weights across
#'   variables, e.g., if for the first variable equal weights for mean
#'   performance and stability are assumed and for the second one, a higher
#'   weight for mean performance (e.g. 65) is assumed, then \code{wresp = c(50,
#'   65)} must be used. Numeric value of length 1 will be recycled with a
#'   warning message.
#' @param random The effects of the model assumed to be random. Defaults to
#'   \code{random = "gen"}. See \strong{Details} to see the random effects
#'   assumed depending on the experimental design of the trials.
#' @param prob The probability for estimating confidence interval for BLUP's
#'   prediction.
#' @param ind_anova Logical argument set to \code{TRUE}. If \code{FALSE} the
#'   within-environment ANOVA is not performed.
#' @param verbose Logical argument. If \code{verbose = FALSE} the code will run
#'   silently.
#' @param ... Arguments passed to the function
#'   \code{\link{impute_missing_val}()} for imputation of missing values in the
#'   matrix of BLUPs for genotype-environment interaction, thus allowing the
#'   computation of the WAASB index.
#' @references
#' Olivoto, T., A.D.C. L{\'{u}}cio, J.A.G. da silva, V.S. Marchioro, V.Q. de
#' Souza, and E. Jost. 2019. Mean performance and stability in multi-environment
#' trials I: Combining features of AMMI and BLUP techniques. Agron. J.
#' 111:2949-2960.
#' \href{https://dl.sciencesocieties.org/publications/aj/abstracts/0/0/agronj2019.03.0220?access=0&view=pdf}{doi:10.2134/agronj2019.03.0220}
#'
#' Mohring, J., E. Williams, and H.-P. Piepho. 2015. Inter-block information: to
#' recover or not to recover it? TAG. Theor. Appl. Genet. 128:1541-54.
#' \href{http://www.ncbi.nlm.nih.gov/pubmed/25972114}{doi:10.1007/s00122-015-2530-0}
#'
#' Patterson, H.D., and E.R. Williams. 1976. A new class of resolvable
#' incomplete block designs. Biometrika 63:83-92.
#' \href{https://doi.org/10.1093/biomet/63.1.83}{doi:10.1093/biomet/63.1.83}
#'
#'
#' @return An object of class \code{waasb} with the following items for each
#'   variable:
#'
#' * \strong{individual} A within-environments ANOVA considering a
#'   fixed-effect model.
#'
#' * \strong{fixed} Test for fixed effects.
#'
#' * \strong{random} Variance components for random effects.
#'
#' * \strong{LRT} The Likelihood Ratio Test for the random effects.
#'
#' * \strong{model} A tibble with the response variable, the scores of all
#' IPCAs, the estimates of Weighted Average of Absolute Scores, and WAASBY (the
#' index that considers the weights for stability and mean performance in the
#' genotype ranking), and their respective ranks.
#'
#' * \strong{BLUPgen} The random effects and estimated BLUPS for genotypes (If
#' \code{random = "gen"} or \code{random = "all"})
#'
#' * \strong{BLUPenv} The random effects and estimated BLUPS for environments,
#' (If \code{random = "env"} or \code{random = "all"}).
#'
#' * \strong{BLUPint} The random effects and estimated BLUPS of all genotypes in
#' all environments.
#'
#' * \strong{PCA} The results of Principal Component Analysis with the
#' eigenvalues and explained variance of the matrix of genotype-environment
#' effects estimated by the linear fixed-effect model.
#'
#' * \strong{MeansGxE} The phenotypic means of genotypes in the environments.
#'
#' * \strong{Details} A list summarizing the results. The following information
#' are shown: \code{Nenv}, the number of environments in the analysis;
#' \code{Ngen} the number of genotypes in the analysis; \code{mresp} The value
#' attributed to the highest value of the response variable after rescaling it;
#' \code{wresp} The weight of the response variable for estimating the WAASBY
#' index. \code{Mean} the grand mean; \code{SE} the standard error of the mean;
#' \code{SD} the standard deviation. \code{CV} the coefficient of variation of
#' the phenotypic means, estimating WAASB, \code{Min} the minimum value observed
#' (returning the genotype and environment), \code{Max} the maximum value
#' observed (returning the genotype and environment); \code{MinENV} the
#' environment with the lower mean, \code{MaxENV} the environment with the
#' larger mean observed, \code{MinGEN} the genotype with the lower mean,
#' \code{MaxGEN} the genotype with the larger.
#'
#' * \strong{ESTIMATES} A tibble with the genetic parameters (if \code{random =
#' "gen"} or \code{random = "all"}) with the following columns: \code{Phenotypic
#' variance} the phenotypic variance; \code{Heritability} the broad-sense
#' heritability; \code{GEr2} the coefficient of determination of the interaction
#' effects; \code{Heribatility of means} the heritability on the mean basis;
#' \code{Accuracy} the selective accuracy; \code{rge} the genotype-environment
#' correlation; \code{CVg} the genotypic coefficient of variation; \code{CVr}
#' the residual coefficient of variation; \code{CV ratio} the ratio between
#' genotypic and residual coefficient of variation.
#'
#'  * \strong{residuals} The residuals of the model.
#' @md
#' @author Tiago Olivoto \email{tiagoolivoto@@gmail.com}
#' @seealso \code{\link{mtsi}} \code{\link{waas}}
#'   \code{\link{get_model_data}} \code{\link{plot_scores}}
#' @export
#' @examples
#' \donttest{
#' library(metan)
#' #===============================================================#
#' # Example 1: Analyzing all numeric variables assuming genotypes #
#' # as random effects with equal weights for mean performance and #
#' # stability                                                     #
#' #===============================================================#
#'model <- waasb(data_ge,
#'               env = ENV,
#'               gen = GEN,
#'               rep = REP,
#'               resp = everything())
#' # Distribution of random effects (first variable)
#' plot(model, type = "re")
#'
#' # Genetic parameters
#' get_model_data(model, "genpar")
#'
#'
#'
#' #===============================================================#
#' # Example 2: Analyzing variables that starts with "N"           #
#' # assuming environment as random effects with higher weight for #
#' # response variable (65) for the three traits.                  #
#' #===============================================================#
#'
#'model2 <- waasb(data_ge2,
#'                env = ENV,
#'                gen = GEN,
#'                rep = REP,
#'                random = "env",
#'                resp = starts_with("N"),
#'                wresp = 65)
#'
#'
#' # Get the index WAASBY
#' get_model_data(model2, what = "WAASBY")
#'
#' # Plot the scores (response x WAASB)
#' plot_scores(model2, type = 3)
#'
#' #===============================================================#
#' # Example 3: Analyzing GY and HM assuming a random-effect model.#
#' # Smaller values for HM and higher values for GY are better.    #
#' # To estimate WAASBY, higher weight for the GY (60%) and lower  #
#' # weight for HM (40%) are considered for mean performance.      #
#' #===============================================================#
#'
#' model3 <- waasb(data_ge,
#'                 env = ENV,
#'                 gen = GEN,
#'                 rep = REP,
#'                 resp = c(GY, HM),
#'                 random = "all",
#'                 mresp = c(100, 0),
#'                 wresp = c(60, 40))
#'
#' # Get P-values for Likelihood-ratio test
#' get_model_data(model3, "pval_lrt")
#'
#' # Get the random effects
#' get_model_data(model3, what = "ranef")
#'
#' # Get the ranks for the WAASB index
#' get_model_data(model3, what = "OrWAASB")
#' }
#'
waasb <- function(.data,
                  env,
                  gen,
                  rep,
                  resp,
                  block = NULL,
                  mresp = NULL,
                  wresp = NULL,
                  random = "gen",
                  prob = 0.05,
                  ind_anova = TRUE,
                  verbose = TRUE,
                  ...) {
    if (!random %in% c("env", "gen", "all")) {
        stop("The argument 'random' must be one of the 'gen', 'env', or 'all'.")
    }
    block_test <- missing(block)
    if(!missing(block)){
        factors  <- .data %>%
            select({{env}},
                   {{gen}},
                   {{rep}},
                   {{block}}) %>%
            mutate_all(as.factor)
    } else{
        factors  <- .data %>%
            select({{env}},
                   {{gen}},
                   {{rep}}) %>%
            mutate_all(as.factor)
    }
    vars <- .data %>% select({{resp}}, -names(factors))
    vars %<>% select_numeric_cols()
    if(!missing(block)){
        factors %<>% set_names("ENV", "GEN", "REP", "BLOCK")
    } else{
        factors %<>% set_names("ENV", "GEN", "REP")
    }
    model_formula <-
        case_when(
            random == "gen" & block_test ~ paste("Y ~ ENV/REP + (1 | GEN) + (1 | GEN:ENV)"),
            random == "env" & block_test ~ paste("Y ~ GEN + (1 | ENV/REP) + (1 | GEN:ENV)"),
            random == "all" & block_test ~ paste("Y ~ (1 | GEN) + (1 | ENV/REP) + (1 | GEN:ENV)"),
            random == "gen" & !block_test ~ paste("Y ~  (1 | GEN) + ENV / REP + (1|BLOCK:(REP:ENV))  + (1 | GEN:ENV)"),
            random == "env" & !block_test ~ paste("Y ~ 0 + GEN + (1| ENV/REP/BLOCK)  + (1 | GEN:ENV)"),
            random == "all" & !block_test ~ paste("Y ~  (1 | GEN) + (1|ENV/REP/BLOCK) + (1 | GEN:ENV)")
        )
    lrt_groups <-
        strsplit(
            case_when(
                random == "gen" & block_test ~ c("COMPLETE GEN GEN:ENV"),
                random == "env" & block_test ~ c("COMPLETE REP(ENV) ENV GEN:ENV"),
                random == "all" & block_test ~ c("COMPLETE GEN REP(ENV) ENV GEN:ENV"),
                random == "gen" & !block_test ~ c("COMPLETE GEN BLOCK(ENV:REP) GEN:ENV"),
                random == "env" & !block_test ~ c("COMPLETE BLOCK(ENV:REP) REP(ENV) ENV GEN:ENV"),
                random == "all" & !block_test ~ c("COMPLETE GEN BLOCK(ENV:REP) REP(ENV) ENV GEN:ENV")
            ), " ")[[1]]
    mod1 <- random == "gen" & block_test
    mod2 <- random == "gen" & !block_test
    mod3 <- random == "env" & block_test
    mod4 <- random == "env" & !block_test
    mod5 <- random == "all" & block_test
    mod6 <- random == "all" & !block_test
    nvar <- ncol(vars)
    if (is.null(mresp)) {
        mresp <- replicate(nvar, 100)
        minresp <- 100 - mresp
    } else {
        mresp <- mresp
        minresp <- 100 - mresp
        if (length(mresp) != nvar) {
            warning("Invalid length in 'mresp'. Setting mresp = ", mresp[[1]],
                    " to all the ", nvar, " variables.", call. = FALSE)
            mresp <- replicate(nvar, mresp[[1]])
            minresp <- 100 - mresp
        }
        if (sum(mresp == 100) + sum(mresp == 0) != nvar) {
            stop("The values of the numeric vector 'mresp' must be 0 or 100.")
        }
    }
    if (is.null(wresp)) {
        PesoResp <- replicate(nvar, 50)
        PesoWAASB <- 100 - PesoResp
    } else {
        PesoResp <- wresp
        PesoWAASB <- 100 - PesoResp
        if (length(wresp) != nvar) {

            warning("Invalid length in 'wresp'. Setting wresp = ", wresp[[1]],
                    " to all the ", nvar, " variables.", call. = FALSE)
            PesoResp <- replicate(nvar, wresp[[1]])
            PesoWAASB <- 100 - PesoResp
        }
        if (min(wresp) < 0 | max(wresp) > 100) {
            stop("The range of the numeric vector 'wresp' must be equal between 0 and 100.")
        }
    }
    listres <- list()
    vin <- 0
    if (verbose == TRUE) {
        pb <- progress_bar$new(
            format = "Evaluating the variable :what [:bar]:percent",
            clear = FALSE, total = nvar, width = 90)
    }
    for (var in 1:nvar) {
        data <- factors %>%
            mutate(Y = vars[[var]])
        if(has_na(data)){
            data <- remove_rows_na(data)
            has_text_in_num(data)
        }
        Nenv <- nlevels(data$ENV)
        Ngen <- nlevels(data$GEN)
        Nrep <- nlevels(data$REP)
        minimo <- min(Nenv, Ngen) - 1
        vin <- vin + 1
        ovmean <- mean(data$Y)
        if (minimo < 2) {
            cat("\nWarning. The analysis is not possible.")
            cat("\nThe number of environments and number of genotypes must be greater than 2\n")
        }
        if(ind_anova == TRUE){
            if(missing(block)){
                individual <- data %>% anova_ind(ENV, GEN, REP, Y)
            } else{
                individual <- data %>% anova_ind(ENV, GEN, REP, Y, block = BLOCK)
            }
        } else{
            individual = NULL
        }
        Complete <- suppressWarnings(suppressMessages(lmerTest::lmer(model_formula, data = data)))
        LRT <- suppressWarnings(suppressMessages(lmerTest::ranova(Complete, reduce.terms = FALSE) %>%
                                                     mutate(model = lrt_groups) %>%
                                                     column_to_first(model)))
        fixed <- anova(Complete)
        var_eff <-
            lme4::VarCorr(Complete) %>%
            as.data.frame() %>%
            select_cols(1, 4) %>%
            arrange(grp) %>%
            rename(Group = grp, Variance = vcov) %>%
            add_cols(Percent = (Variance / sum(Variance)) * 100)
        if(random %in% c("gen", "all")){
            GV <- as.numeric(var_eff[which(var_eff[1] == "GEN"), 2])
            IV <- as.numeric(var_eff[which(var_eff[1] == "GEN:ENV"), 2])
            RV <- as.numeric(var_eff[which(var_eff[1] == "Residual"), 2])
            FV <- sum(var_eff$Variance)
            h2g <- GV/FV
            h2mg <- GV/(GV + IV/Nenv + RV/(Nenv * Nrep))
            GEr2 <- IV/(GV + IV + RV)
            AccuGen <- sqrt(h2mg)
            rge <- IV/(IV + RV)
            CVg <- (sqrt(GV)/ovmean) * 100
            CVr <- (sqrt(RV)/ovmean) * 100
            CVratio <- CVg/CVr
            PROB <- ((1 - (1 - prob))/2) + (1 - prob)
            t <- qt(PROB, 100)
            Limits <- t * sqrt(((1 - AccuGen) * GV))
            genpar <- tibble(Parameters = c("Phenotypic variance", "Heritability", "GEIr2", "Heribatility of means",
                                            "Accuracy", "rge", "CVg", "CVr", "CV ratio"),
                             Values = c(FV, h2g, GEr2, h2mg, AccuGen, rge, CVg, CVr, CVratio))
        } else{
            genpar <- NULL
        }
        bups <- lme4::ranef(Complete)
        bINT <-
            data.frame(Names = rownames(bups$`GEN:ENV`)) %>%
            separate(Names, into = c("GEN", "ENV")) %>%
            add_cols(BLUPge = bups[[1]][[1]]) %>%
            to_factor(1:2)
        intmatrix <- as.matrix(make_mat(bINT, GEN, ENV, BLUPge))
        if(has_na(intmatrix)){
            intmatrix <- impute_missing_val(intmatrix, verbose = verbose, ...)$.data
            warning("Data imputation used to fill the GxE matrix", call. = FALSE)
        }
        s <- svd(intmatrix)
        U <- s$u[, 1:minimo]
        LL <- diag(s$d[1:minimo])
        V <- s$v[, 1:minimo]
        Eigenvalue <- data.frame(Eigenvalue = s$d[1:minimo]^2) %>%
            add_cols(Proportion = s$d[1:minimo]^2/sum(s$d[1:minimo]^2) * 100,
                     Accumulated = cumsum(Proportion),
                     PC = paste("PC", 1:minimo, sep = "")) %>%
            column_to_first(PC)
        SCOREG <- U %*% LL^0.5
        SCOREE <- V %*% LL^0.5
        colnames(SCOREG) <- colnames(SCOREE) <- paste("PC", 1:minimo, sep = "")
        MEDIAS <- means_by(data, ENV, GEN)
        MGEN <- MEDIAS %>% means_by(GEN) %>% add_cols(type = "GEN")
        MGEN <- cbind(MGEN, SCOREG)
        MENV <- MEDIAS %>% means_by(ENV) %>% add_cols(type = "ENV")
        MENV <- cbind(MENV, SCOREE)
        MEDIAS <- suppressMessages(dplyr::mutate(MEDIAS,
                                                 envPC1 = left_join(MEDIAS, MENV %>% select(ENV, PC1))$PC1,
                                                 genPC1 = left_join(MEDIAS, MGEN %>% select(GEN, PC1))$PC1,
                                                 nominal = left_join(MEDIAS, MGEN %>% select(GEN, Y))$Y + genPC1 * envPC1))
        MGEN %<>% rename(Code = GEN)
        MENV %<>% rename(Code = ENV)
        Escores <- rbind(MGEN, MENV) %>%
            column_to_first(type)
        Pesos <- data.frame(Percent = Eigenvalue$Proportion)
        WAASB <- Escores %>%
            select(contains("PC")) %>%
            abs() %>%
            t() %>%
            as.data.frame() %>%
            mutate(Percent = Pesos$Percent)
        WAASAbs <- mutate(Escores, WAASB = sapply(WAASB[, -ncol(WAASB)], weighted.mean, w = WAASB$Percent)) %>%
            group_by(type) %>%
            mutate(PctResp = (mresp[vin] - minresp[vin])/(max(Y) - min(Y)) * (Y - max(Y)) + mresp[vin],
                   PctWAASB = (0 - 100)/(max(WAASB) - min(WAASB)) * (WAASB - max(WAASB)) + 0,
                   wRes = PesoResp[vin],
                   wWAASB = PesoWAASB[vin],
                   OrResp = rank(-Y),
                   OrWAASB = rank(WAASB),
                   OrPC1 = rank(abs(PC1)),
                   WAASBY = ((PctResp * wRes) + (PctWAASB * wWAASB))/(wRes + wWAASB),
                   OrWAASBY = rank(-WAASBY)) %>%
            ungroup()
        Details <- ge_details(data, ENV, GEN, Y) %>%
            add_rows(Parameters = "wresp", Y = PesoResp[vin], .before = 1) %>%
            add_rows(Parameters = "mresp", Y = mresp[vin], .before = 1) %>%
            add_rows(Parameters = "Ngen", Y = Ngen, .before = 1) %>%
            add_rows(Parameters = "Nenv", Y = Nenv, .before = 1) %>%
            rename(Values = Y)
        if(mod1){
            data_factors <- data %>% select_non_numeric_cols()
            BLUPgen <-
                data.frame(GEN = MGEN$Code,
                           BLUPg = bups$GEN$`(Intercept)`) %>%
                add_cols(Predicted = BLUPg + ovmean) %>%
                arrange(-Predicted) %>%
                add_cols(Rank = rank(-Predicted),
                         LL = Predicted - Limits,
                         UL = Predicted + Limits) %>%
                column_to_first(Rank)
            BLUPint <-
                left_join(data_factors, bINT, by = c("ENV", "GEN")) %>%
                left_join(BLUPgen, by = "GEN") %>%
                select(ENV, GEN, REP, BLUPg, BLUPge) %>%
                add_cols(`BLUPg+ge` = BLUPge + BLUPg,
                         Predicted = predict(Complete))
            BLUPenv <- NULL
        } else if(mod2){
            data_factors <- data %>% select_non_numeric_cols()
            BLUPgen <-
                data.frame(GEN = MGEN$Code,
                           BLUPg = bups$GEN$`(Intercept)`) %>%
                add_cols(Predicted = BLUPg + ovmean) %>%
                arrange(-Predicted) %>%
                add_cols(Rank = rank(-Predicted),
                         LL = Predicted - Limits,
                         UL = Predicted + Limits) %>%
                column_to_first(Rank)
            blupBRE <-
                data.frame(Names = rownames(bups$`BLOCK:(REP:ENV)`)) %>%
                separate(Names, into = c("BLOCK", "REP", "ENV")) %>%
                add_cols(BLUPbre = bups$`BLOCK:(REP:ENV)`[[1]]) %>%
                to_factor(1:3)
            BLUPint <-
                left_join(data_factors, bINT, by = c("ENV", "GEN")) %>%
                left_join(BLUPgen, by = "GEN") %>%
                left_join(blupBRE, by = c("ENV", "REP", "BLOCK")) %>%
                select(ENV, REP, BLOCK, GEN, BLUPg, BLUPge, BLUPbre) %>%
                add_cols(`BLUPg+ge+bre` = BLUPge + BLUPg + BLUPbre,
                         Predicted = `BLUPg+ge+bre` + left_join(data_factors, data %>% means_by(ENV, REP), by = c("ENV", "REP"))$Y)
            BLUPenv <- NULL
        } else if (mod3){
            data_factors <- data %>% select_non_numeric_cols()
            BLUPgen <- NULL
            BLUPenv <- data.frame(ENV = MENV$Code,
                                  BLUPe = bups$ENV$`(Intercept)`) %>%
                add_cols(Predicted = BLUPe + ovmean) %>%
                arrange(-Predicted) %>%
                add_cols(Rank = rank(-Predicted)) %>%
                column_to_first(Rank)
            blupRWE <-
                data.frame(Names = rownames(bups$`REP:ENV`)) %>%
                separate(Names, into = c("REP", "ENV")) %>%
                add_cols(BLUPre = bups$`REP:ENV`[[1]]) %>%
                to_factor(1:2)
            BLUPint <-
                left_join(data_factors, bINT, by = c("ENV", "GEN")) %>%
                left_join(BLUPenv, by = "ENV") %>%
                left_join(blupRWE, by = c("ENV", "REP")) %>%
                select(ENV, GEN, REP, BLUPe, BLUPge, BLUPre) %>%
                add_cols(`BLUPge+e+re` = BLUPge + BLUPe + BLUPre,
                         Predicted = `BLUPge+e+re` + left_join(data_factors, MGEN %>% select(Code, Y), by = c("GEN" = "Code"))$Y)
        } else if (mod4){
            data_factors <- data %>% select_non_numeric_cols()
            BLUPgen <- NULL
            BLUPenv <-
                data.frame(ENV = MENV$Code,
                           BLUPe = bups$ENV$`(Intercept)`) %>%
                add_cols(Predicted = BLUPe + ovmean) %>%
                arrange(-Predicted) %>%
                add_cols(Rank = rank(-Predicted)) %>%
                column_to_first(Rank)
            blupRWE <-
                data.frame(Names = rownames(bups$`REP:ENV`)) %>%
                separate(Names, into = c("REP", "ENV")) %>%
                add_cols(BLUPre = bups$`REP:ENV`[[1]]) %>%
                to_factor(1:2)
            blupBRE <-
                data.frame(Names = rownames(bups$`BLOCK:(REP:ENV)`)) %>%
                separate(Names, into = c("BLOCK", "REP", "ENV")) %>%
                add_cols(BLUPbre = bups$`BLOCK:(REP:ENV)`[[1]]) %>%
                to_factor(1:3)
            genCOEF <- summary(Complete)[["coefficients"]] %>%
                as_tibble(rownames = NA) %>%
                rownames_to_column("GEN") %>%
                replace_string(GEN, pattern = "GEN", new_var = GEN) %>%
                rename(Y = Estimate) %>%
                to_factor(1)
            BLUPint <-
                left_join(data_factors, bINT, by = c("ENV", "GEN")) %>%
                left_join(BLUPenv, by = "ENV") %>%
                left_join(blupRWE, by = c("ENV", "REP")) %>%
                left_join(blupBRE, by = c("ENV", "REP", "BLOCK")) %>%
                select(ENV, REP, BLOCK, GEN, BLUPe, BLUPge, BLUPre, BLUPbre) %>%
                add_cols(`BLUPe+ge+re+bre` = BLUPge + BLUPe + BLUPre + BLUPbre,
                         Predicted = `BLUPe+ge+re+bre` + left_join(data_factors, genCOEF, by = "GEN")$Y)
        } else if (mod5){
            data_factors <- data %>% select_non_numeric_cols()
            BLUPgen <-
                data.frame(GEN = MGEN$Code,
                           BLUPg = bups$GEN$`(Intercept)`) %>%
                add_cols(Predicted = BLUPg + ovmean) %>%
                arrange(-Predicted) %>%
                add_cols(Rank = rank(-Predicted),
                         LL = Predicted - Limits,
                         UL = Predicted + Limits) %>%
                column_to_first(Rank)
            BLUPenv <- data.frame(ENV = MENV$Code,
                                  BLUPe = bups$ENV$`(Intercept)`) %>%
                add_cols(Predicted = BLUPe + ovmean) %>%
                arrange(-Predicted) %>%
                add_cols(Rank = rank(-Predicted)) %>%
                column_to_first(Rank)
            blupRWE <- data.frame(Names = rownames(bups$`REP:ENV`)) %>%
                separate(Names, into = c("REP", "ENV")) %>%
                add_cols(BLUPre = bups$`REP:ENV`[[1]]) %>%
                arrange(ENV) %>%
                to_factor(1:2)
            BLUPint <-
                left_join(data_factors, bINT, by = c("ENV", "GEN")) %>%
                left_join(BLUPgen, by = "GEN") %>%
                left_join(BLUPenv, by = "ENV") %>%
                left_join(blupRWE, by = c("ENV", "REP")) %>%
                select(GEN, ENV, REP, BLUPe, BLUPg, BLUPge, BLUPre) %>%
                add_cols(`BLUPg+e+ge+re` = BLUPge + BLUPe + BLUPg + BLUPre,
                         Predicted = `BLUPg+e+ge+re` + ovmean)
        } else if (mod6){
            data_factors <- data %>% select_non_numeric_cols()
            BLUPgen <-
                data.frame(GEN = MGEN$Code,
                           BLUPg = bups$GEN$`(Intercept)`) %>%
                add_cols(Predicted = BLUPg + ovmean) %>%
                arrange(-Predicted) %>%
                add_cols(Rank = rank(-Predicted),
                         LL = Predicted - Limits,
                         UL = Predicted + Limits) %>%
                column_to_first(Rank)
            BLUPenv <- data.frame(ENV = MENV$Code,
                                  BLUPe = bups$ENV$`(Intercept)`) %>%
                add_cols(Predicted = BLUPe + ovmean) %>%
                arrange(-Predicted) %>%
                add_cols(Rank = rank(-Predicted)) %>%
                column_to_first(Rank)
            blupRWE <- data.frame(Names = rownames(bups$`REP:ENV`)) %>%
                separate(Names, into = c("REP", "ENV")) %>%
                add_cols(BLUPre = bups$`REP:ENV`[[1]]) %>%
                arrange(ENV) %>%
                to_factor(1:2)
            blupBRE <-
                data.frame(Names = rownames(bups$`BLOCK:(REP:ENV)`)) %>%
                separate(Names, into = c("BLOCK", "REP", "ENV")) %>%
                add_cols(BLUPbre = bups$`BLOCK:(REP:ENV)`[[1]]) %>%
                to_factor(1:3)
            BLUPint <-
                left_join(data_factors, bINT, by = c("ENV", "GEN")) %>%
                left_join(BLUPgen, by = "GEN") %>%
                left_join(BLUPenv, by = "ENV") %>%
                left_join(blupRWE, by = c("ENV", "REP")) %>%
                left_join(blupBRE, by = c("ENV", "REP", "BLOCK")) %>%
                select(GEN, ENV, REP, BLOCK, BLUPg, BLUPe, BLUPge, BLUPre, BLUPbre) %>%
                add_cols(`BLUPg+e+ge+re+bre` = BLUPg + BLUPge + BLUPe + BLUPre + BLUPbre,
                         Predicted = `BLUPg+e+ge+re+bre` + ovmean)
        }
        residuals <- data.frame(fortify.merMod(Complete))
        residuals$reff <- BLUPint$BLUPge
        temp <- structure(list(individual = individual[[1]],
                               fixed = fixed %>% rownames_to_column("SOURCE") %>% as_tibble(),
                               random = var_eff,
                               LRT = LRT,
                               model = as_tibble(WAASAbs),
                               BLUPgen = BLUPgen,
                               BLUPenv = BLUPenv,
                               BLUPint = BLUPint,
                               PCA = as_tibble(Eigenvalue),
                               modellme = Complete,
                               MeansGxE = as_tibble(MEDIAS),
                               Details = as_tibble(Details),
                               ESTIMATES = genpar,
                               residuals = as_tibble(residuals)), class = "waasb")
        if (verbose == TRUE) {
            pb$tick(tokens = list(what = names(vars[var])))
        }
        listres[[paste(names(vars[var]))]] <- temp
    }
    if (verbose == TRUE) {
        cat("Model: ", model_formula, "\n")
        cat("---------------------------------------------------------------------------\n")
        cat("P-values for Likelihood Ratio Test of the analyzed traits\n")
        cat("---------------------------------------------------------------------------\n")
        print.data.frame(sapply(listres, function(x){
            x$LRT[["Pr(>Chisq)"]]
        }) %>%
            as.data.frame() %>%
            add_cols(model = listres[[1]][["LRT"]][["model"]]) %>%
            column_to_first(model), row.names = FALSE, digits = 3)
        cat("---------------------------------------------------------------------------\n")
        if (length(which(unlist(lapply(listres, function(x) {
            x[["LRT"]] %>% dplyr::filter(model == "GEN:ENV") %>% pull(`Pr(>Chisq)`)
        })) > prob)) > 0) {
            cat("Variables with nonsignificant GxE interaction\n")
            cat(names(which(unlist(lapply(listres, function(x) {
                x[["LRT"]][which(x[["LRT"]][[1]] == "GEN:ENV"), 7]
            })) > prob)), "\n")
            cat("---------------------------------------------------------------------------\n")
        } else {
            cat("All variables with significant (p < 0.05) genotype-vs-environment interaction\n")
        }
    }
    invisible(structure(listres, class = "waasb"))
}









#' Several types of residual plots
#'
#' Residual plots for a output model of class \code{waas} and \code{waasb}. Six types
#' of plots are produced: (1) Residuals vs fitted, (2) normal Q-Q plot for the
#' residuals, (3) scale-location plot (standardized residuals vs Fitted
#' Values), (4) standardized residuals vs Factor-levels, (5) Histogram of raw
#' residuals and (6) standardized residuals vs observation order. For a \code{waasb}
#' object, normal Q-Q plot for random effects may also be obtained declaring
#' \code{type = 're'}
#'
#'
#' @param x An object of class \code{waasb}.
#' @param var The variable to plot. Defaults to \code{var = 1} the first
#'   variable of \code{x}.
#' @param type If \code{type = 're'}, normal Q-Q plots for the random effects
#' are obtained.
#' @param conf Level of confidence interval to use in the Q-Q plot (0.95 by
#' default).
#' @param out How the output is returned. Must be one of the 'print' (default)
#' or 'return'.
#' @param labels Logical argument. If \code{TRUE} labels the points outside
#' confidence interval limits.
#' @param plot_theme The graphical theme of the plot. Default is
#'   \code{plot_theme = theme_metan()}. For more details, see
#'   \code{\link[ggplot2]{theme}}.
#' @param alpha The transparency of confidence band in the Q-Q plot. Must be a
#' number between 0 (opaque) and 1 (full transparency).
#' @param fill.hist The color to fill the histogram. Default is 'gray'.
#' @param col.hist The color of the border of the the histogram. Default is
#' 'black'.
#' @param col.point The color of the points in the graphic. Default is 'black'.
#' @param col.line The color of the lines in the graphic. Default is 'red'.
#' @param col.lab.out The color of the labels for the 'outlying' points.
#' @param size.lab.out The size of the labels for the 'outlying' points.
#' @param size.tex.lab The size of the text in axis text and labels.
#' @param size.shape The size of the shape in the plots.
#' @param bins The number of bins to use in the histogram. Default is 30.
#' @param which Which graphics should be plotted. Default is \code{which =
#' c(1:4)} that means that the first four graphics will be plotted.
#' @param ncol,nrow The number of columns and rows of the plot pannel. Defaults
#'   to \code{NULL}
#' @param ... Additional arguments passed on to the function
#'   \code{\link[cowplot]{plot_grid}}
#' @author Tiago Olivoto \email{tiagoolivoto@@gmail.com}
#' @importFrom cowplot plot_grid
#' @importFrom dplyr distinct_all arrange_at
#' @method plot waasb
#' @export
#' @examples
#'\donttest{
#' library(metan)
#' model2 <- waasb(data_ge,
#'                 resp = GY,
#'                 gen = GEN,
#'                 env = ENV,
#'                 rep = REP)
#' plot(model2)
#'}
#'
plot.waasb <- function(x, var = 1, type = "res", conf = 0.95, out = "print",
                       labels = FALSE, plot_theme = theme_metan(), alpha = 0.2, fill.hist = "gray",
                       col.hist = "black", col.point = "black", col.line = "red",
                       col.lab.out = "red", size.lab.out = 2.5, size.tex.lab = 10,
                       size.shape = 1.5, bins = 30, which = c(1:4), ncol = NULL,
                       nrow = NULL, ...) {
    x <- x[[var]]
    if (type == "re" & max(which) >= 5) {
        stop("When type =\"re\", 'which' must be a value between 1 and 4")
    }
    if (type == "res") {
        df <- data.frame(x$residuals)
        df$id <- rownames(df)
        df <- data.frame(df[order(df$.scresid), ])
        P <- ppoints(nrow(df))
        df$z <- qnorm(P)
        n <- nrow(df)
        Q.x <- quantile(df$.scresid, c(0.25, 0.75))
        Q.z <- qnorm(c(0.25, 0.75))
        b <- diff(Q.x)/diff(Q.z)
        coef <- c(Q.x[1] - b * Q.z[1], b)
        zz <- qnorm(1 - (1 - conf)/2)
        SE <- (coef[2]/dnorm(df$z)) * sqrt(P * (1 - P)/n)
        fit.value <- coef[1] + coef[2] * df$z
        df$upper <- fit.value + zz * SE
        df$lower <- fit.value - zz * SE
        df$label <- ifelse(df$.scresid > df$.scresid | df$.scresid <
                               df$lower, rownames(df), "")
        df$factors <- paste(df$ENV, df$GEN)
        # Residuals vs .fitted
        p1 <- ggplot(df, aes(.fitted, .resid)) +
            geom_point(col = col.point, size = size.shape) +
            geom_smooth(se = F, method = "loess", col = col.line) +
            geom_hline(yintercept = 0, linetype = 2, col = "gray") +
            labs(x = "Fitted values", y = "Residual") +
            ggtitle("Residual vs fitted") + plot_theme %+replace%
            theme(axis.text = element_text(size = size.tex.lab, colour = "black"),
                  axis.title = element_text(size = size.tex.lab, colour = "black"),
                  plot.title = element_text(size = size.tex.lab, hjust = 0, vjust = 1))
        if (labels != FALSE) {
            p1 <- p1 +
                ggrepel::geom_text_repel(aes(.fitted, .resid, label = (label)),
                                         color = col.lab.out,
                                         size = size.lab.out)
        } else {
            p1 <- p1
        }
        # normal qq
        p2 <- ggplot(df, aes(z, .scresid)) +
            geom_point(col = col.point, size = size.shape) +
            geom_abline(intercept = coef[1],
                        slope = coef[2],
                        size = 1,
                        col = col.line) +
            geom_ribbon(aes_(ymin = ~lower, ymax = ~upper),
                        alpha = 0.2) +
            labs(x = "Theoretical quantiles", y = "Sample quantiles") +
            ggtitle("Normal Q-Q") +
            plot_theme %+replace%
            theme(axis.text = element_text(size = size.tex.lab, colour = "black"),
                  axis.title = element_text(size = size.tex.lab, colour = "black"),
                  plot.title = element_text(size = size.tex.lab, hjust = 0, vjust = 1))
        if (labels != FALSE) {
            p2 <- p2 + ggrepel::geom_text_repel(aes(z, .scresid, label = (label)),
                                                color = col.lab.out,
                                                size = size.lab.out)
        } else {
            p2 <- p2
        }
        # scale-location
        p3 <- ggplot(df, aes(.fitted, sqrt(abs(.resid)))) +
            geom_point(col = col.point, size = size.shape) +
            geom_smooth(se = F, method = "loess", col = col.line) +
            labs(x = "Fitted Values", y = expression(sqrt("|Standardized residuals|"))) +
            ggtitle("Scale-location") +
            plot_theme %+replace%
            theme(axis.text = element_text(size = size.tex.lab, colour = "black"),
                  axis.title = element_text(size = size.tex.lab, colour = "black"),
                  plot.title = element_text(size = size.tex.lab, hjust = 0, vjust = 1))
        if (labels != FALSE) {
            p3 <- p3 + ggrepel::geom_text_repel(aes(.fitted, sqrt(abs(.resid)),
                                                    label = (label)),
                                                color = col.lab.out,
                                                size = size.lab.out)
        } else {
            p3 <- p3
        }
        # Residuals vs Factor-levels
        p4 <- ggplot(df, aes(factors, .scresid)) +
            geom_point(col = col.point, size = size.shape) +
            geom_hline(yintercept = 0, linetype = 2, col = "gray") +
            labs(x = "Factor levels", y = "Standardized residuals") +
            ggtitle("Residuals vs factor-levels") +
            plot_theme %+replace%
            theme(axis.text = element_text(size = size.tex.lab, colour = "black"),
                  axis.title = element_text(size = size.tex.lab, colour = "black"),
                  panel.grid.major.x = element_blank(),
                  axis.text.x = element_text(color = "white"),
                  plot.title = element_text(size = size.tex.lab, hjust = 0, vjust = 1))
        if (labels != FALSE) {
            p4 <- p4 + ggrepel::geom_text_repel(aes(factors,
                                                    .scresid, label = (label)),
                                                color = col.lab.out,
                                                size = size.lab.out)
        } else {
            p4 <- p4
        }
        # Histogram of residuals
        p5 <- ggplot(df, aes(x = .resid)) +
            geom_histogram(bins = bins,
                           colour = col.hist,
                           fill = fill.hist,
                           aes(y = ..density..)) +
            stat_function(fun = dnorm,
                          color = col.line,
                          size = 1,
                          args = list(mean = mean(df$.resid),
                                      sd = sd(df$.resid))) +
            labs(x = "Raw residuals", y = "Density") +
            ggtitle("Histogram of residuals") +
            plot_theme %+replace%
            theme(axis.text = element_text(size = size.tex.lab, colour = "black"),
                  axis.title = element_text(size = size.tex.lab, colour = "black"),
                  plot.title = element_text(size = size.tex.lab, hjust = 0, vjust = 1))
        # Residuals vs order
        p6 <- ggplot(df, aes(as.numeric(id), .scresid, group = 1)) +
            geom_point(col = col.point, size = size.shape) +
            geom_line(col = col.line) +
            geom_hline(yintercept = 0,
                       linetype = 2,
                       col = col.line) +
            labs(x = "Observation order", y = "Standardized residuals") +
            ggtitle("Residuals vs observation order") +
            plot_theme %+replace%
            theme(axis.text = element_text(size = size.tex.lab, colour = "black"),
                  axis.title = element_text(size = size.tex.lab, colour = "black"),
                  plot.title = element_text(size = size.tex.lab, hjust = 0, vjust = 1))
        p7 <- ggplot(df, aes(.fitted, Y)) +
            geom_point(col = col.point, size = size.shape) +
            facet_wrap(~GEN) +
            geom_abline(intercept = 0, slope = 1, col = col.line) +
            labs(x = "Fitted values", y = "Observed values") +
            ggtitle("1:1 line plot") +
            plot_theme %+replace%
            theme(axis.text = element_text(size = size.tex.lab, colour = "black"),
                  axis.title = element_text(size = size.tex.lab, colour = "black"),
                  panel.grid.major.x = element_blank(),
                  panel.grid.major.y = element_blank(),
                  plot.title = element_text(size = size.tex.lab, hjust = 0, vjust = 1),
                  panel.spacing = unit(0, "cm"))
        plots <- list(p1, p2, p3, p4, p5, p6, p7)
    }
    if (type == "re") {
        blups <-
            x$BLUPint %>%
            select_cols(contains("BLUP"))
        fact <-x$BLUPint %>% select_non_numeric_cols()
        qlist <- list()
        for (i in 1:ncol(blups)) {
            df <-
                data.frame(blups[i]) %>%
                distinct_all() %>%
                rowid_to_column(var = "id") %>%
                arrange_at(2)
            P <- ppoints(nrow(df))
            df$z <- qnorm(P)
            n <- nrow(df)
            Q.x <- quantile(df[[2]], c(0.25, 0.75))
            Q.z <- qnorm(c(0.25, 0.75))
            b <- diff(Q.x)/diff(Q.z)
            coef <- c(Q.x[1] - b * Q.z[1], b)
            zz <- qnorm(1 - (1 - conf)/2)
            SE <- (coef[2]/dnorm(df$z)) * sqrt(P * (1 - P)/n)
            fit.value <- coef[1] + coef[2] * df$z
            df %<>% add_cols(upper = fit.value + zz * SE,
                             lower = fit.value - zz * SE,
                             label = ifelse(df[[2]] > upper | df[[2]] < lower, id, ""),
                             intercept = coef[1],
                             slope = coef[2],
                             var = paste(names(blups[i]))
            ) %>%
                set_names("id",    "blup", "z",     "upper", "lower", "label", "intercept", "slope", "var")
            qlist[[paste(names(blups[i]))]] <- df
        }

        df <- do.call(rbind, qlist)
        # normal qq GEI effects
        p1 <- ggplot(df, aes(z, blup)) +
            geom_point(col = col.point, size = size.shape) +
            geom_abline(aes(intercept = intercept,
                            slope = slope),
                        size = 1, col = col.line) +
            geom_ribbon(aes_(ymin = ~lower, ymax = ~upper),
                        alpha = 0.2) +
            labs(x = "Theoretical quantiles", y = "Sample quantiles")+
            facet_wrap( ~var,
                        scales = "free",
                        ncol = ncol,
                        nrow = nrow) +
            plot_theme %+replace%
            theme(axis.text = element_text(size = size.tex.lab, colour = "black"),
                  axis.title = element_text(size = size.tex.lab, colour = "black"),
                  plot.title = element_text(size = size.tex.lab, hjust = 0, vjust = 1))
        if (labels != FALSE) {
            p1 <- p1 + ggrepel::geom_text_repel(aes(z, blup, label = (label)),
                                                color = col.lab.out,
                                                size = size.lab.out)
        } else {
            p1 <- p1
        }
    }
    if(!type == "re"){
        plot_grid(plotlist = plots[c(which)],
                  ncol = ncol,
                  nrow = nrow,
                  ...)
    } else{
        print(p1)
    }
}








#' Print an object of class waasb
#'
#' Print a \code{waasb} object in two ways. By default, the results are shown in
#' the R console. The results can also be exported to the directory.
#'
#'
#' @param x An object of class \code{waasb}.
#' @param export A logical argument. If \code{TRUE|T}, a *.txt file is exported
#'   to the working directory
#' @param blup A logical argument. If \code{TRUE|T}, the blups are shown.
#' @param file.name The name of the file if \code{export = TRUE}
#' @param digits The significant digits to be shown.
#' @param ... Options used by the tibble package to format the output. See
#'   \code{\link[tibble:formatting]{tibble::print()}} for more details.
#' @author Tiago Olivoto \email{tiagoolivoto@@gmail.com}
#' @method print waasb
#' @export
#' @examples
#'\donttest{
#' library(metan)
#' model <- waasb(data_ge,
#'   resp = c(GY, HM),
#'   gen = GEN,
#'   env = ENV,
#'   rep = REP
#' )
#' print(model)
#' }
print.waasb <- function(x, export = FALSE, blup = FALSE, file.name = NULL, digits = 4, ...) {
    if (!class(x) == "waasb") {
        stop("The object must be of class 'waasb'")
    }
    if (export == TRUE) {
        file.name <- ifelse(is.null(file.name) == TRUE, "waasb print", file.name)
        sink(paste0(file.name, ".txt"))
    }
    opar <- options(pillar.sigfig = digits)
    on.exit(options(opar))
    for (i in 1:length(x)) {
        var <- x[[i]]
        cat("Variable", names(x)[i], "\n")
        cat("---------------------------------------------------------------------------\n")
        cat("Individual fixed-model analysis of variance\n")
        cat("---------------------------------------------------------------------------\n")
        print(var$individual$individual)
        cat("---------------------------------------------------------------------------\n")
        cat("Fixed effects\n")
        cat("---------------------------------------------------------------------------\n")
        print(var$fixed)
        cat("---------------------------------------------------------------------------\n")
        cat("Random effects\n")
        cat("---------------------------------------------------------------------------\n")
        print(var$random)
        cat("---------------------------------------------------------------------------\n")
        cat("Likelihood ratio test\n")
        cat("---------------------------------------------------------------------------\n")
        print(var$LRT)
        cat("---------------------------------------------------------------------------\n")
        cat("Variance components and genetic parameters\n")
        cat("---------------------------------------------------------------------------\n")
        print(var$ESTIMATES)
        cat("---------------------------------------------------------------------------\n")
        cat(" Principal component analysis of the G x E interaction matrix\n")
        cat("---------------------------------------------------------------------------\n")
        print(var$PCA)
        cat("---------------------------------------------------------------------------\n")
        if (blup == TRUE) {
            cat("BLUPs for genotypes\n")
            print(var$BLUPgen)
            cat("---------------------------------------------------------------------------\n")
            cat("BLUPs for genotypes-vs-environments\n")
            cat("---------------------------------------------------------------------------\n")
            print(var$BLUPgge)
            cat("---------------------------------------------------------------------------\n")
        }
        cat("Some information regarding the analysis\n")
        cat("---------------------------------------------------------------------------\n")
        print(var$Details)
        cat("\n\n\n")
    }
    if (export == TRUE) {
        sink()
    }
}










#' Predict method for waasb fits
#'
#' Obtains predictions from an object fitted with \code{\link{waasb}}.
#'
#'
#' @param object An object of class \code{waasb}
#' @param ... Currently not used
#' @return A tibble with the predicted values for each variable in the model
#' @author Tiago Olivoto \email{tiagoolivoto@@gmail.com}
#' @method predict waasb
#' @export
#' @examples
#'\donttest{
#' library(metan)
#' model <- waasb(data_ge,
#'                env = ENV,
#'                gen = GEN,
#'                rep = REP,
#'                resp = c(GY, HM))
#' predict(model)
#' }
#'
predict.waasb <- function(object, ...) {
    if (class(object) != "waasb") {
        stop("The objectin must be an objectin of the class 'waasb'")
    }
    factors <- object[[1]][["BLUPint"]] %>% select_non_numeric_cols()
    numeric <- sapply(object, function(x){
        x[["BLUPint"]][["Predicted"]]
    })
    return(cbind(factors, numeric) %>% as_tibble())
}

