\name{bkp}
\alias{bkp}
\title{
Draw block plots from block plot objects
}
\description{
\code{bkp} draws block plots based on block plot objects generated by \code{blockplot} and its methods. 
It is normally called from within \code{blockplot} but can be invoked directly.
}
\usage{
	bkp(x, labels = x$labels, xlim = NULL, ylim = NULL, 
            main = NULL, xlab = NULL, ylab = "Frequency", 
            square = FALSE, add = FALSE, offset = 0, 
            grp.spacing = 2, grp.at = NA, 
            fill = NA, border = NULL, density = NULL, angle = 45, 
            lty = 1, lwd = 2, label.col = 1, cex = NA, adj = c(0.5, 0.4), 
            uline = 2, uline.lwd = lwd, uline.lty = 1, 
            uline.col = if (!is.null(border)) border else par("fg"), 
            grp.labs = FALSE, grp.pos = 1, glab.control = list(), 
            axes = c(TRUE, FALSE), asp = NA, frame.plot = any(axes), 
            drop.unused = TRUE, unused.label="[Missing]", ...)
}
\arguments{
  \item{x}{ An R object. For the default method, a vector of values for which the blockplot is desired. 
            For the formula method, a valid formula, such as \code{y ~ grp} (see Details).
  }
  \item{labels}{Labels for data points, passed to \code{text}; in principle of any type acceptable to \code{text}. 
                Labels are placed inside boxes so should be short for readability.
  }
  \item{xlim}{ Limits \code{(x1, x2)} for the horizontal (\eqn{x}) range of the plot. The default is the range of \code{breaks}, after computation if necessary.
  }
  \item{ylim}{ limits for the vertical range of the plot. Will be overridden if \code{square=TRUE} (see below).
  }
  \item{main}{Main title for the plot, passed to \code{\link{plot}}.
  }
  \item{xlab, ylab}{ x- and y-axis labels for the plot. As usual, either can be expressions (see \code{\link{plotmath}}).
  }
  \item{square}{ Logical: If \code{square=TRUE}, the aspect ratio is set (via \code{asp}) to make the individual 
  	blocks appear square when rendered. Note that this generally overrides \code{ylim}. 
  }
  \item{add}{ Logical: If \code{TRUE}, the plot will be added to an existing plot.  
  }
  \item{offset}{Numeric scalar value. Vertical offset for the plot, in units of block height.  
    \code{offset} is added to the vertical position of the boxplot and any subplots. \code{offset} can be 
    considered as a number of vertical block heights to raise (or lower, if negative) the plot. 
    This, with \code{add}, is useful for adding further groups manually to an existing plot. 
  }
  \item{grp.spacing}{Numeric scalar, giving the minimum vertical spacing (in units of block height) between subplots when 
    there is more than one group.
  }
  \item{grp.at}{Optional vector specifying explicit vertical locations for subplot 
  	baselines (including the first group). The default (\code{grp.at=NA}) is 
  	to use \code{grp.spacing}. If specified, \code{grp.at} overrides \code{grp.spacing}. 
  }
  \item{fill}{Fill colour for the rectangles (\dQuote{blocks}) making up the plot. 
  	Recycled to length \code{length(x)}. Passed to \code{\link{rect}}.
  }
  \item{border}{Border colour for the rectangles (\dQuote{blocks}) making up the plot. 
  	Recycled to length \code{length(x)}. Passed to \code{\link{rect}}.
  }
  \item{density}{Shading line density for (\dQuote{blocks}) making up the plot. Recycled 
  	to length \code{length(x)}. Passed to \code{\link{rect}}.
  }
  \item{angle}{Shading line angle for (\dQuote{blocks}) making up the plot. Recycled 
  	to length \code{length(x)}. Passed to \code{\link{rect}}.
  }
  \item{lty}{Border line type for (\dQuote{blocks}) making up the plot. Recycled 
  	to length \code{length(x)}. Passed to \code{\link{rect}}.
  }
  \item{lwd}{Border line width for (\dQuote{blocks}) making up the plot. Recycled 
  	to length \code{length(x)}. Passed to \code{\link{rect}}.
  }
  \item{label.col}{Colour used for text labels in each (\dQuote{block}) making up the plot. 
  	Recycled to length \code{length(x)}. Passed to \code{\link{text}}.
  }
  \item{cex}{Size of text labels in each (\dQuote{block}) making up the plot. 
  	Recycled to length \code{length(x)} and passed to \code{\link{text}}. The 
  	default is to fit text inside each block automatically.
  }
  \item{adj}{Vector of two values giving text location adjustment for all block labels.
  	Passed to \code{\link{text}}
  }
  \item{uline}{Specification for the distance by which the baseline for each subplot 
  	extends beyond the data range for the group. See \sQuote{Graphical elements} 
  	for details. The default is two units either side. 
  }
  \item{uline.lwd}{Scalar: Line width for the subplot baseline(s). 
  }
  \item{uline.lty}{Scalar: Line type for the subplot baseline(s). 
  }
  \item{uline.col}{Scalar: Line colour for the subplot baseline(s). 
  }
  \item{grp.labs}{ Logical, determining whether group labels are plotted, or 
  	a vector of labels. See \sQuote{Details}.
  }
  \item{grp.pos}{Specification for the position of group labels. Follows \code{pos} in 
  	\code{\link{text}}. See \sQuote{Details}.  
  }
  \item{glab.control}{List of arguments to be passed to \code{\link{text}} 
  	when drawing group labels.
  }
  \item{axes}{Logical, indicating whether axes are drawn. Can be a vector 
        of two logicals, specifying horizontal and vertical axes 
  	respectively. See \sQuote{Graphical elements} for details.
  }
  \item{asp}{Aspect ratio, passed to \code{\link{plot}}
  }
  \item{frame.plot}{Logical, controlling whether a frame (box) is drawn round the plot.
  }
  \item{drop.unused}{Logical specification for the treatment of empty groups. If \code{TRUE},
                     groups (identified by levels in \code{x$groups}) that contain no non-missing
                     values will be omitted from the plot. If \code{FALSE}, a space is created
                     for the missing subplot but no subplot is generated. If \code{grp.labels}
                     requires group labelling, the group label is drawn in the space with 
                     \code{unused.label} appended.
  }
  \item{unused.label}{Character string appended to missing group labels.
  }
  \item{\dots}{Further parameters passed to \code{\link{plot}}
  }
}
\details{
   \code{bkp} provides considerable control of graphical elements. The main 
   elements, and the arguments controlling their location and appearance, are:
   
   \describe{
	\item{General appearance}{A block plot of a single group of data has the general 
		appearance of a histogram. However, instead of vertical bars (of possibly 
		variable width) indicating the number of data points within the bin interval,
		each bin is a stack of rectangles, each corresponding to a single data point
		and with an optional label identifying the datum.
		
		Block plots of this kind are useful for data sets of modest size; typically 10-100
		per group, as individual labels quickly become hard to identify in larger data sets.
		
		By default, \code{blockplot} produces one such plot for a set of data. If 
		a series of such plots is needed, this can be accommodated either by using
		\code{blockplot} with \code{add=TRUE} to build up a plot manually, setting
		\code{xlim}, \code{ylim} and \code{breaks} to accommodate all the required 
		groups. Alternatively, a grouping factor can be provided (via argument 
		\code{groups}) which will produce a series of subplots, laid out automatically.
		The use of \code{groups} and the corresponding layout options are detailed below
		(see \dQuote{Groups}).
		
		The vertical position of a single block plot within the figure 
		can be set using \code{offset}, which sets the baseline height, 
		in units of block height, from the figure origin. This is useful for 
		separating several groups that are added manually; just set \code{offset}
		appropriately for each separate plot. Note that setting \code{offset}
		has no effect on the automatic \code{ylim} setting, which means that 
		\code{ylim} must be set manually to accommodate the vertical offset.
	}
	\item{Blocks}{Each individual rectangle (\dQuote{block} in the plot corresponds to
		a single data point. In this implementation, blocks appear in rank order
		from left to right \emph{and} from bottom to top; that is, data are placed in 
		vertical bins as in a normal histogram but, in addition, the vertical ordering of 
		blocks corresponds to the data order within each bin, with blocks at the 
		bottom corresponding to lower values.  
		
		Blocks are always 1 unit high, so the total vertical height of each bin corresponds
		directly to frequency (not density) in a histogram. The block width is the interval 
		between \code{breaks}, which must be equispaced.
		
		By default, the apparent aspect ratio for blocks depends primarily on 
		\code{xlim} and \code{ylim} and the height and width of the plotting device. 
		However, setting \code{square=TRUE} will cause the plot aspect ration (\code{asp})
		to be set such that the bocks appear square in the current plot window.
		
		Fill colour, border colour and style, fill effects and text colour of individual 
		blocks can all be controlled using \code{fill}, \code{border}, 
		\code{density}, \code{angle}, \code{lty}, \code{lwd} and \code{label.col}, 
		as the relevant arguments can be vectors of length \code{length(x)}. This 
		allows conditional formatting, for example to identify
		a particular data point or some secondary grouping variable.

	}
	
	\item{Subplot baseline}{The baseline for each subplot is controlled by \code{uline},
		as follows:
		\describe{
		   \item{\code{TRUE}:}{ The line extends the full width of the plot;}

		   \item{\code{FALSE}:}{ No baseline is drawn;}

		   \item{numeric:}{ If numeric (as for the default), \code{uline} specifies 
			the distance that the baseline extends beyond each end of the data, 
			in units of block width. \code{uline} can be length 2 numeric vector,
			which specifies the baseline extension on the left and right sides 
			respectively.}
		}
		
		Colour, line type, and line width for the subplot baseline(s) can be controlled
		with \code{uline.col}, \code{uline.lty}, and \code{uline.lwd} respectively. 
	}

   	\item{Axes}{ Axes can be controlled with the \code{axes} argument, which 
   	     controls whether or not axes are drawn. If a vector of two logical 
   	     values (as for the default), \code{axes} specifies drawing for horizontal 
   	     and vertical axes respectively.
   	     
   	     The horizontal axis is normally continuous for the plot. If a 
   	     vertical (frequency) axis is requested (either by \code{axes=TRUE} or,
   	     for example, by \code{axes=c(TRUE, TRUE)}, a vertical axis is drawn
   	     for each subplot, starting at zero at the baseline and terminating 
   	     at the highest vertical value in the subplot. Vertical axes, 
   	     restarting at 0 at the next subplot baseline, are drawn if 
   	     there is more than one group.
	}
	
	\item{Groups}{\code{blockplot} provides a simple grouping mechanism to display
		separate subplots for different groups of data on the same figure. The 
		default method provides for a grouping variable specified via \code{groups}.
		The formula method provides a somewhat more flexible interface, allowing 
		specification of more than one grouping variable. Like \code{\link{boxplot}},
		if there is more than one goruping variable in the formula, subplots are drawn 
		for each (non-empty) level of the interaction term. 
		
		Subplots for different groups are arranged vertically. Vertical position can be
		specified explicitly via \code{grp.at} or, more simply, by setting 
		\code{grp.spacing}. The latter sets \code{grp.at} to equal vertical
		spacing such that the smallest vertical gap is \code{grp.spacing}.
		Both \code{grp.at} and \code{grp.spacing} are in units of block height;
		that is, \code{grpspacing=2} (the default) means that the smallest vertical 
		gap is equivalent to two blocks. 
	
	}
	
	\item{Group labels}{Labels can be added to each subplot. These are controlled by \code{grp.labs} ..
		\code{grp.labs} provides the specification for group labels, and can be a single logical or 
		a vector of labels. Effects of \code{grp.labs} are as follows:
		\itemize{
		\item{\code{FALSE}} (The default): No group labels are drawn. 
		\item{\code{TRUE}} Labels are taken as \code{levels(groups)}, 
			and set to \code{"1"} if there is only one group.
		\item{Vector} If a character vector (or expression) is provided, 
			these are used as labels for the groups plotted. 

			WARNING: If missing values in \code{x} cause group levels to be 
			dropped, those groups will not be plotted. \code{grp.labs} must
			have the same length as the number of groups plotted. An error
			is generated if the length of \code{labels} differs from the number
			of groups actually plotted.
		}
		
		\code{grp.pos} specifies the general positioning of group labels 
		relative to each subplot. \code{grp.pos} follows \code{pos} in \code{\link{text}}:
		Values of \code{1}, \code{2}, \code{3} and \code{4}, respectively 
		indicate positions below, to the left of, above and to the right of the plot.
          		
          	The detailed positioning of group labels is automatic; the four positions 
          	specified by \code{grp.pos} correspond approximately to the midpoints of the 
          	corresponding edge of each plot, where the \sQuote{edges} are the baseline, 
          	leftmost block, topmost block and rightmost block. Labels are placed a short 
          	distance outward from these edges. Labels are justified according to position;
          	\code{grp.pos} is re-used as the default \code{pos} argument to \code{\link{text}}.
          	
          	Further control of group label position is available via \code{grp.control}, 
          	which is a list (empty by default) of arguments passed to \code{\link{text}}.
          	Ths can include arguments such as \code{pos} and \code{adj}, as well as 
          	appearance elements such as \code{col}, \code{cex} etc.
	}
     }   

}
\value{
   \code{bxp} returns the original object \code{x} with additional elements:
	\item{grp.at}{The vertical coordinated for the subplot baselines.
	}
	\item{blockwidth}{The width of the blocks in the plot.
	}
}
\references{
	ISO 5725-2:1994 \emph{Accuracy (trueness and precision) of measurement methods 
	and results -- Part 2: Basic method for the determination of repeatability 
	and reproducibility of a standard measurement method.} ISO, Geneva, 1994.
}
\author{
	Stephen L R Ellison \email{s.ellison@lgcgroup.com}.

}

\seealso{
	For constructing breaks and grouping:
	\code{\link{blockplot}} 

	For graphical elements:
	\code{\link{text}}, \code{\link{rect}}

}
\examples{

#A simple blockplot
set.seed(55)
x<-rnorm(48, 15)
b <- blockplot(x)

#Aspect ratio control for square blocks
bkp(b, square=TRUE)

# Specifying groups 
grp <- gl(3, 16)
bg <- blockplot(x~grp)

# Add vertical axes ( axes=TRUE asks for horizontal and vertical axes)
bkp(bg, axes=TRUE ) 

#Axes both left and right
par(mar=c(5,4,4,4)+0.1)
bkp(bg, axes=c(TRUE, TRUE, FALSE, TRUE) ) 
		#Note that axes[3] is FALSE to suppress top axis
		
# Vectorised colour specification
bkp(bg, square=TRUE, fill=ifelse(1:48 \%in\% c(15, 23, 24), "gold", "white") )

# Group labelling
bkp(bg, square=TRUE, grp.labs=paste("Level", 1:3), grp.pos=2)

}
% Add one or more standard keywords, see file 'KEYWORDS' in the
% R documentation directory.
\keyword{ ~kwd1 }
\keyword{ hplot }
