% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/allStructure.R
\name{allStructure}
\alias{allStructure}
\title{Calculates all Structure and Functioning indicators}
\usage{
allStructure(X, X_length, LSI.group, LFI.group, max.length = 85,
  large.fish = 35, guild.groups, condition.groups, ratio.groups,
  species.table, speciesinfo.table, LenWt.table, years, raw = TRUE,
  std = TRUE, glob.env = TRUE, export.path = NULL,
  export.id = NULL)
}
\arguments{
\item{X}{A dataframe of fishery independent data derived from research vessel
survey data or model output, with columns \code{YEAR}, \code{ID},
\code{SPECIES}, and \code{BIOMASS}. \code{YEAR} indicates the year the
observation was recorded, \code{ID} is an area code indicating where the
observation was recorded, \code{SPECIES} is a numeric code indicating the
species sampled, and \code{BIOMASS} is the corresponding biomass (stratified
and corrected for catchability as required).}

\item{X_length}{A dataframe of fishery independent data derived from research
vessel survey data or model output, with columns \code{YEAR}, \code{ID},
\code{SPECIES}, \code{LENGTH}, \code{BIOMASS} and \code{ABUNDANCE}.
\code{YEAR} indicates the year the observation was recorded, \code{ID} is an
area code indicating where the observation was recorded, and \code{SPECIES}
is a numeric code indicating the species sampled. \code{LENGTH} is the
length class (cm) and \code{BIOMASS}/\code{ABUNDANCE} is the corresponding
biomass/abundance at length (stratified and corrected for catchability as
required). Species for which there are no length data should be assigned
\code{LENGTH = -99}. These observations are removed by the function.}

\item{LSI.group}{A character string indicating the species group for which to
calculate the Large Species Indicator. Must be set to \code{"ALL"} or match
a column name in \code{species.table}. If \code{LSI.group = NULL}, the Large
Species Indicator will not be calculated.}

\item{LFI.group}{A character string indicating the species group for which to
calculate the Large Fish Indicator. Must be set to \code{"ALL"} or match a
column name in \code{species.table}. If \code{LFI.group = NULL}, the Large
Fish Indicator will not be calculated.}

\item{max.length}{The threshold for large species (cm). Default is
\code{max.length = 85} (i.e., large species are those with \code{MAXLENGTH}
>= 85 cm).}

\item{large.fish}{Threshold for large fish (cm). Default is \code{large.fish =
35} (i.e., large fish are those with \code{X_length$LENGTH} >= 35 cm).}

\item{guild.groups}{A vector indicating the species groups for which to
calculate the resource potential. Each entry must be a character string
matching the name of a column in \code{species.table}. If \code{guild.groups
= NULL}, these indicators will not be calculated.}

\item{condition.groups}{A vector indicating the species groups for which to
calculate Fulton's community condition factor. Each entry must be a
character string matching the name of a column in \code{species.table}. If
\code{condition.groups = NULL}, the community condition indicators will not
be calculated.}

\item{ratio.groups}{A dataframe with two columns, which must be named
\code{group1} and \code{group2}. Each row holds the group names for one
biomass ratio, with the numerator in column \code{group1} and the
denominator in column \code{group2}. Each entry must be a character string
matching the name of a column in \code{species.table} or \code{"ALL"}. If
\code{ratio.groups = NULL}, biomass ratio indicators will not be calculated.}

\item{species.table}{A table where the column names match the entries in
\code{guild.groups}, \code{ratio.groups}, and/or \code{condition.groups}.
Column entries are the species codes indicating the species from \code{X}
(or \code{X_length}) included in each group. \code{species.table} may also
include columns for other species groups; these will be ignored.}

\item{speciesinfo.table}{A table with columns \code{SPECIES} and the
corresponding \code{TL} and \code{MAXLENGTH} (trophic level and maximum
recorded length). Entries in the \code{SPECIES} column should be the unique
values of species codes in \code{X}/\code{X_length} (or a subset thereof).
Other columns will be ignored.}

\item{LenWt.table}{A table of annual length at weight data with 5 columns.
\code{YEAR}, \code{ID}, and \code{SPECIES} are as described in
\code{X_length}. \code{LENGTH} is fish length at the corresponding
\code{WEIGHT} (fish weight).}

\item{years}{A vector of years for which to calculate indicators.}

\item{raw}{A logical value. If \code{raw = TRUE}, the raw indicator values are
returned by the function. If \code{raw = FALSE}, the raw indicator values are
not returned. Default is \code{raw = TRUE}. Either \code{raw} or \code{std}
must be \code{TRUE}.}

\item{std}{A logical value. If \code{std = TRUE}, the standardized indicator
values for each area \code{ID} are returned by the function. Indicators are
standardized using z-scores, i.e., by subtracting the mean and dividing by
the standard deviation (ignoring \code{NA} values). If \code{std = FALSE},
the standardized indicator values are not returned. Default is \code{std =
TRUE}. Either \code{raw} or \code{std} must be \code{TRUE}.}

\item{glob.env}{Logical value indicating whether to return output to global
environment. Default is \code{glob.env = TRUE}.}

\item{export.path}{File path indicating where to save a .csv file of
calculated indicators (named structure_export.id.csv; see below). If
\code{export.file = NULL}, the indicator dataframe will not be exported as a
.csv file. Default is \code{export.path = NULL}.}

\item{export.id}{Character string to modify the name of the .csv file (if
export.path is specified), for example an area name or date of analysis. The
exported .csv file is named structure_export.id.csv. Default is
\code{export.id = NULL}.}
}
\value{
Returns a dataframe with columns \code{ID}, \code{YEAR}, and
 indicators corresponding to the arguments supplied to the function.
 Standardized indicators are noted with \code{_s} in the name.
}
\description{
This function calculates all (or a subset) of the Structure and
 Functioning indicators for \eqn{j} areas and \eqn{i} years. The user can
 choose whether the function returns the indicator dataframe to the global
 environment, exports the dataframe to a .csv file, or both. The user can
 also choose whether the function returns the raw indicator values, the
 standardized (z-score) values, or both.
}
\details{
This function calculates the Structure and Functioning indicators:
 Biomass, Biomass Ratio(s), Large Species Indicator, Trophic Level of the
 Community, Large Fish Indicator, Mean Length (weighted by biomass and
 abundance), and Community Condition. If data are not available to calculate
 one or more of these indicators, a subset will be returned. See the help
 file for the individual functions for information on how each indicator is
 calculated.

 Notes on indicator calculations: In the individual functions, the user
 generally has control over which metric is used to calculate the indicator
 (e.g., \code{BIOMASS} or \code{ABUNDANCE}). Here, \code{BIOMASS} is used to
 calculate Biomass, Biomass Ratio(s), Large Species Indicator, Large Fish
 Indicator, Trophic Level of the Community, and Mean Length weighted by
 biomass. \code{ABUNDANCE} is used to calculate Mean Length weighted by
 abundance and Community Condition. See individual functions for more
 flexibility.
}
\examples{
# Compile data
data(X)
data(X_length)
data(species.table)
data(species.info)
data(Length_Weight)

# Species groups of interest
ratio.groups <- data.frame(rbind(c("PELAGIC", "GROUNDFISH"), c("PREDATORS", "ALL")))
names(ratio.groups) <- c("group1", "group2")
trophicguild.groups <- c("LBENTHIVORE", "MBENTHIVORE", "PISCIVORE", "PLANKTIVORE",
   "ZOOPISCIVORE")
condition.groups <- c("FINFISH", "LBENTHIVORE", "MBENTHIVORE", "PISCIVORE",
    "PLANKTIVORE", "ZOOPISCIVORE")

# Calculate raw indicators
allStructure(X = X, X_length = X_length,
   LSI.group = "ALL", LFI.group = "ALL",
    max.length = 85, large.fish = 35,
   guild.groups = trophicguild.groups, condition.groups = condition.groups,
   ratio.groups = ratio.groups,
   species.table = species.table, speciesinfo.table = species.info,
   LenWt.table = Length_Weight,
   years = c(2014:2019), raw = TRUE, std = FALSE)
}
\references{
Bundy A, Gomez C, Cook AM. 2017. Guidance framework for the
 selection and evaluation of ecological indicators. Can. Tech. Rep. Fish.
 Aquat. Sci. 3232: xii + 212 p.
}
\seealso{
Other ecosystem structure and function indicators: \code{\link{biomassRatio}},
  \code{\link{communityCondition}},
  \code{\link{largeFishIndicator}},
  \code{\link{largeSpeciesIndicator}},
  \code{\link{meanTLCommunity}}
}
\author{
Danielle Dempsey \email{Danielle.Dempsey@dfo-mpo.gc.ca}, Adam Cook,
 Catalina Gomez, Alida Bundy
}
\concept{ecosystem structure and function indicators}
