% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/loo_expectation.R
\name{E_loo}
\alias{E_loo}
\alias{E_loo.default}
\alias{E_loo.matrix}
\title{Compute weighted expectations}
\usage{
E_loo(x, lw, ...)

\method{E_loo}{default}(x, lw, ..., type = c("mean", "var", "quantile"),
  probs)

\method{E_loo}{matrix}(x, lw, ..., type = c("mean", "var", "quantile"), probs)
}
\arguments{
\item{x}{A numeric matrix or vector.}

\item{lw}{A numeric matrix (or vector) of smoothed log-weights with the same
dimensions (or length) as \code{x}. Typically this will be the
\code{"lw_smooth"} element of the list returned by \code{\link{psislw}}.}

\item{...}{For the generic function, arguments passed to the
individual methods.}

\item{type}{The type of expectation to compute. The options are
\code{"mean"}, \code{"var"} (variance), and \code{"quantile"}.}

\item{probs}{A vector of probabilities. Ignored unless \code{type} is
\code{"quantile"}.}
}
\value{
The matrix method returns a vector with \code{ncol(x)} elements, with
  one exception: when \code{type} is \code{"quantile"} and multiple values
  are specified in \code{probs} the returned object is a \code{length(probs)}
  by \code{ncol(x)} matrix.

  The default/vector method returns a scalar, with one exception: when
  \code{type} is \code{"quantile"} and multiple values are specified in
  \code{probs} the returned object is a vector with \code{length(probs)}
  elements.
}
\description{
The \code{E_loo} function computes weighted expectations (means, variances,
quantiles) using the smoothed importance weights obtained from the
\link[=psislw]{PSIS} procedure. The expectations estimated by the
\code{E_loo} function assume that the PSIS approximation is working well.
\strong{A small \link[=pareto-k-diagnostic]{Pareto k} estimate is necessary,
but not sufficient, for \code{E_loo} to give reliable estimates.} Additional
diagnostic checks for gauging the reliability of the estimates are in
development and will be added in a future release.
}
\examples{
\donttest{
# Use rstanarm package to quickly fit a model and get both a log-likelihood
# matrix and draws from the posterior predictive distribution
library("rstanarm")

# data from help("lm")
ctl <- c(4.17,5.58,5.18,6.11,4.50,4.61,5.17,4.53,5.33,5.14)
trt <- c(4.81,4.17,4.41,3.59,5.87,3.83,6.03,4.89,4.32,4.69)
d <- data.frame(
  weight = c(ctl, trt),
  group = gl(2, 10, 20, labels = c("Ctl","Trt"))
)
fit <- stan_glm(weight ~ group, data = d)
yrep <- posterior_predict(fit)
dim(yrep)

ll <- log_lik(fit)
lw <- psislw(-ll, cores = 2)$lw_smooth
dim(lw)

E_loo(yrep, lw, type = "mean")
E_loo(yrep, lw, type = "var")
E_loo(yrep, lw, type = "quantile", probs = 0.5) # median
E_loo(yrep, lw, type = "quantile", probs = c(0.1, 0.9))
}

}
