% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/loo.R
\name{loo}
\alias{loo}
\alias{loo.function}
\alias{loo.matrix}
\title{Leave-one-out cross-validation (LOO)}
\usage{
loo(x, ...)

\method{loo}{matrix}(x, ...)

\method{loo}{function}(x, ..., args)
}
\arguments{
\item{x}{A log-likelihood matrix or function. See the \strong{Methods (by
class)} section below for a detailed description.}

\item{...}{Optional arguments to pass to \code{\link{psislw}}. Possible
  arguments and their defaults are:
\describe{
 \item{\code{wcp = 0.2}}{The proportion of importance weights to use for the
   generalized Pareto fit. The \code{100*wcp}\% largest weights are used as the
   sample from which to estimate the parameters \eqn{k} and \eqn{\sigma} of
   the generalized Pareto distribution.}
 \item{\code{wtrunc = 3/4}}{For truncating very large weights to
   \eqn{S}^\code{wtrunc} (set to zero for no truncation).}
 \item{\code{cores = getOption("loo.cores", parallel::detectCores())}}{The
 number of cores to use for parallelization. This can be set for an entire R
 session by \code{options(loo.cores = NUMBER)}. The default is
 \code{\link[parallel]{detectCores}}().}
}

 We recommend using the default values for the \code{psislw} arguments unless
 there are problems (e.g. \code{NA} or \code{NaN} results).}

\item{args}{Only required if \code{x} is a function. A list containing
the data required to specify the arguments to the function. See the
\strong{Methods (by class)} section below for how \code{args} should be
specified.}
}
\value{
A named list with class \code{'loo'} and components:

\describe{
 \item{\code{elpd_loo, se_elpd_loo}}{Expected log pointwise predictive density
   and standard error.}
 \item{\code{p_loo, se_p_loo}}{Estimated effective number of parameters and
   standard error.}
 \item{\code{looic, se_looic}}{The LOO information criterion
   (\code{-2*elpd_loo}, i.e., converted to deviance scale) and standard
   error.}
 \item{\code{pointwise}}{A matrix containing the pointwise contributions of each
   of the above measures.}
 \item{\code{pareto_k}}{A vector containing the estimates of the shape
   parameter \eqn{k} for the generaelized Pareto fit to the importance ratios
   for each leave-one-out distribution. See PSIS-LOO section in
   \code{\link{loo-package}} for details about interpreting \eqn{k}.
   (By default, the \code{\link[=print.loo]{print}} method for \code{'loo'}
   objects will also provide warnings about problematic values of \eqn{k}.)}
}
}
\description{
Efficient approximate leave-one-out cross-validation for Bayesian models.
}
\details{
\subsection{PSIS-LOO}{
We approximate LOO using Pareto Smoothed Importance Sampling (PSIS). See
\code{\link{loo-package}} for details.
}
\subsection{Memory}{
For models fit to very large datasets we recommend the \code{loo.function}
method, which is much more memory efficient than the \code{loo.matrix}
method. However, the \code{loo.matrix} method is typically more convenient,
so it is usually worth trying \code{loo.matrix} and then switching to
\code{loo.function} if memory is an issue.
}
}
\section{Methods (by class)}{
\itemize{
\item \code{matrix}: An \eqn{S} by \eqn{N} matrix, where \eqn{S} is the size of the posterior
sample (the number of simulations) and \eqn{N} is the number of data points.
Typically (but not restricted to be) the object returned by
\code{\link{extract_log_lik}}.

\item \code{function}: A function \eqn{f} that takes arguments \code{i}, \code{data},
 and \code{draws} and returns a vector containing the log-likelihood for
 the \code{i}th observation evaluated at each posterior draw.

 The \code{args} argument must also be specified and should be a named list
 with the following components:
 \itemize{
   \item \code{draws}: An object containing the posterior draws for any
   parameters needed to compute the pointwise log-likelihood.
   \item \code{data}: An object containing data (e.g. observed outcome and
   predictors) needed to compute the pointwise log-likelihood. \code{data}
   should be in an appropriate form so that \eqn{f}\code{(i=i,
   data=data[i,,drop=FALSE], draws=draws)} returns the \code{S}-vector of
   log-likelihoods for the \code{i}th observation.
   \item \code{N}: The number of observations.
   \item \code{S}: The size of the posterior sample.
 }
}}
\examples{
\dontrun{
### Usage with stanfit objects
log_lik1 <- extract_log_lik(stanfit1) # see ?extract_log_lik
loo1 <- loo(log_lik1)
print(loo1, digits = 3)

log_lik2 <- extract_log_lik(stanfit2)
(loo2 <- loo(log_lik2))
compare(loo1, loo2)
}

### Using log-likelihood function instead of matrix
set.seed(024)

# Simulate data and draw from posterior
N <- 50; K <- 10; S <- 100; a0 <- 3; b0 <- 2
p <- rbeta(1, a0, b0)
y <- rbinom(N, size = K, prob = p)
a <- a0 + sum(y); b <- b0 + N * K - sum(y)
draws <- rbeta(S, a, b)
data <- data.frame(y,K)

llfun <- function(i, data, draws) {
  dbinom(data$y, size = data$K, prob = draws, log = TRUE)
}
loo_with_fn <- loo(llfun, args = nlist(data, draws, N, S), cores = 1)

# Check that we get same answer if using log-likelihood matrix
log_lik_mat <- sapply(1:N, function(i) llfun(i, data[i,, drop=FALSE], draws))
loo_with_mat <- loo(log_lik_mat, cores = 1)
all.equal(loo_with_mat, loo_with_fn)

}
\seealso{
\code{\link{loo-package}} for details on the Pareto Smoothed Importance
Sampling (PSIS) procedure used for approximating LOO.

\code{\link{print.loo}} for the \code{print} and \code{plot} methods for
\code{'loo'} objects.

\code{\link{compare}} for model comparison.
}

