#' Download SAILS data
#'
#' This function downloads data from SAILS (http://sails.clld.org/) and changes language names to the names from lingtypology database. You need the internet connection.
#'
#' @param features A character vector that define with a feature ids from SAILS (e. g. "and1", "argex4-1-3").
#' @param na.rm Logical. If TRUE function removes all languages not available in lingtypology database. By default is TRUE.
#' @param glottolog.source A character vector that define which glottolog database is used: 'original' or 'modified' (by default)
#' @author George Moroz <agricolamz@gmail.com>
#' @seealso \code{\link{afbo.feature}}, \code{\link{autotyp.feature}}, \code{\link{phoible.feature}}, \code{\link{wals.feature}}
#' @examples
#' # sails.feature(c("and1", "and11"))
#' @export
#'
#' @importFrom utils read.csv
#'

sails.feature <-
  function(features,
           na.rm = TRUE,
           glottolog.source = "modified") {
    features_set <-
      c(
        "AND1",
        "AND2",
        "AND3",
        "AND4",
        "AND5",
        "AND6",
        "AND7",
        "AND8",
        "AND9",
        "AND10",
        "AND11",
        "AND12",
        "AND13",
        "AND14",
        "AND15",
        "AND16",
        "AND17",
        "AND18",
        "AND19",
        "AND20",
        "AND21",
        "AND22",
        "AND23",
        "AND24",
        "AND25",
        "AND26",
        "ARGEX1-1",
        "ARGEX1-2",
        "ARGEX1-3",
        "ARGEX1-4",
        "ARGEX2-1",
        "ARGEX2-1-1",
        "ARGEX2-2",
        "ARGEX2-3",
        "ARGEX2-4",
        "ARGEX2-4-1",
        "ARGEX2-5",
        "ARGEX2-6",
        "ARGEX2-7",
        "ARGEX2-7-1",
        "ARGEX2-8",
        "ARGEX2-9",
        "ARGEX2-10",
        "ARGEX2-10-1",
        "ARGEX2-11",
        "ARGEX2-12",
        "ARGEX2-13",
        "ARGEX2-14",
        "ARGEX2-15",
        "ARGEX3-1",
        "ARGEX3-1-1",
        "ARGEX3-1-2",
        "ARGEX3-1-3",
        "ARGEX4-1",
        "ARGEX4-1-1",
        "ARGEX4-1-2",
        "ARGEX4-1-3",
        "ARGEX4-1-4",
        "ARGEX5-1",
        "ARGEX5-2",
        "ARGEX5-3",
        "ARGEX5-4",
        "ARGEX5-5",
        "ARGEX5-6",
        "ARGEX5-7",
        "ARGEX5-8",
        "ARGEX5-9",
        "ARGEX6-1",
        "ARGEX6-1-1",
        "ARGEX6-1-2",
        "ARGEX6-1-3",
        "ARGEX6-1-4",
        "ARGEX7-1",
        "ARGEX7-1-1",
        "ARGEX7-1-2",
        "ARGEX7-1-3",
        "ARGEX7-1-4",
        "ARGEX8-1",
        "ARGEX8-1-1",
        "ARGEX9-1",
        "ARGEX9-1-1",
        "ARGEX9-1-2",
        "ARGEX10-1",
        "ARGEX10-2",
        "ARGEX10-3",
        "ARGEX11-1",
        "ARGEX11-2",
        "ARGEX11-3",
        "ARGEX11-4",
        "ARGEX12-1",
        "ARGEX12-1-1",
        "ARW1",
        "ARW1-1-05",
        "ARW1-1-06",
        "ARW1-1-07",
        "ARW1-1-08",
        "ARW1-1-09",
        "ARW1-1-10",
        "ARW1-1-11",
        "ARW1-1-12",
        "ARW1-1-15",
        "ARW1-1-16",
        "ARW1-1-17",
        "ARW1-1-18",
        "ARW1-2-02",
        "ARW1-2-04",
        "ARW2",
        "ARW2-1-01",
        "ARW2-1-06",
        "ARW2-1-07",
        "ARW2-2-01",
        "ARW2-2-02",
        "ARW2-2-03",
        "ARW2-2-04",
        "ARW2-2-05",
        "ARW2-2-06",
        "ARW2-3-05",
        "ARW2-3-07",
        "ARW2-4-02",
        "ARW2-4-03",
        "ARW2-4-05",
        "ARW2-4-07",
        "ARW2-4-08",
        "ARW2-4-09",
        "ARW2-4-11",
        "ARW2-4-12",
        "ARW2-4-13",
        "ARW2-4-14",
        "ARW2-5-01",
        "ARW2-5-02",
        "ARW2-5-03",
        "ARW2-5-04",
        "ARW2-5-05",
        "ARW2-5-06",
        "ARW2-5-07",
        "ARW2-5-08",
        "ARW2-5-09",
        "ARW2-5-10",
        "ARW2-6-01",
        "ARW2-6-02",
        "ARW2-6-03",
        "ARW2-6-05",
        "ARW2-6-06",
        "ARW2-6-07",
        "ARW2-6-08",
        "ARW2-6-10",
        "ARW2-7-01",
        "ARW2-7-02",
        "ARW2-9-01",
        "ARW2-9-02",
        "ARW2-9-03",
        "ARW2-9-04",
        "ARW2-9-05",
        "ARW2-9-06",
        "ARW2-9-07",
        "ARW3",
        "ARW3-1-03",
        "ARW3-1-06",
        "ARW3-2-10",
        "ARW3-2-11",
        "ARW3-2-12",
        "ARW3-3-07",
        "ARW3-3-08",
        "ARW3-3-09",
        "ARW3-3-10",
        "ARW3-3-11",
        "ARW3-3-12",
        "ARW3-3-15",
        "ARW3-3-16",
        "ARW3-3-17",
        "ARW3-3-18",
        "ARW3-3-21",
        "ARW3-3-22",
        "ARW3-3-23",
        "ARW3-3-24",
        "ARW3-4-02",
        "ARW3-4-03",
        "ARW3-4-04",
        "ARW3-5-02",
        "ARW3-5-04",
        "ARW3-6-01",
        "ARW3-6-02",
        "ARW3-6-03",
        "ARW3-6-04",
        "ARW3-6-05",
        "ARW3-6-06",
        "ARW3-6-07",
        "ARW3-6-08",
        "ARW3-7-01",
        "ARW3-7-03",
        "ARW3-7-05",
        "ARW3-7-06",
        "ARW3-7-08",
        "ARW3-8-01",
        "ARW3-8-02",
        "ARW3-8-03",
        "ARW3-8-04",
        "ARW4",
        "ARW4-1-01",
        "ARW4-1-02",
        "ARW4-1-03",
        "ARW4-1-04",
        "ARW4-1-05",
        "ARW4-1-06",
        "ARW4-1-07",
        "ARW4-1-08",
        "ARW4-1-09",
        "ARW4-1-10",
        "ARW4-2-01",
        "ARW4-2-02",
        "ARW4-2-03",
        "ARW4-2-04",
        "ARW4-2-05",
        "ARW4-2-06",
        "ARW4-3-01",
        "ARW4-3-02",
        "ARW4-3-03",
        "ARW4-3-04",
        "ARW4-3-07",
        "ARW4-3-08",
        "ARW4-3-11",
        "ARW4-4-01",
        "ARW4-4-02",
        "ARW4-4-03",
        "ARW4-5-03",
        "ARW4-5-05",
        "ARW4-5-06",
        "ARW4-5-07",
        "ARW5",
        "ARW5-2-01",
        "ARW5-2-02",
        "ARW5-2-03",
        "ARW5-2-04",
        "ARW5-2-05",
        "ARW5-2-06",
        "ARW5-2-07",
        "ARW5-2-08",
        "ARW5-2-09",
        "ARW5-2-10",
        "ARW5-2-11",
        "ARW5-2-12",
        "ARW6",
        "ARW7",
        "ARW8",
        "ARW9",
        "ARW10",
        "ARW11",
        "ARW12",
        "ARW13",
        "ARW14",
        "ARW15",
        "ARW16",
        "ARW17",
        "ARW18",
        "ARW19",
        "ARW20",
        "ARW21",
        "ARW22",
        "ARW23",
        "ARW24",
        "ARW25",
        "ARW26",
        "ARW27",
        "ARW28",
        "ARW29",
        "ARW30",
        "ARW31",
        "ARW32",
        "ARW33",
        "ARW34",
        "ARW35",
        "ARW36",
        "ARW37",
        "ARW38",
        "ARW39",
        "ARW40",
        "ARW41",
        "ARW42",
        "ARW43",
        "ARW44",
        "ARW45",
        "ARW46",
        "ARW47",
        "ARW48",
        "ARW49",
        "ARW50",
        "ARW51",
        "ARW52",
        "ARW53",
        "ARW54",
        "ARW55",
        "ARW56",
        "ARW57",
        "ARW58",
        "ARW59",
        "ARW60",
        "ARW61",
        "ARW62",
        "ARW63",
        "ARW64",
        "ARW65",
        "ARW66",
        "ARW67",
        "ARW68",
        "ARW69",
        "ARW70",
        "ARW71",
        "ARW72",
        "ARW73",
        "ARW1-1-03a",
        "ARW1-1-09a",
        "ARW1-1-13a",
        "ARW1-2-03a",
        "ARW2-1-02a",
        "ARW2-1-03a",
        "ARW2-1-05a",
        "ARW2-2-01a",
        "ARW2-2-03a",
        "ARW2-3-02a",
        "ARW2-3-06a",
        "ARW2-4-11a",
        "ARW2-5-01a",
        "ARW2-5-04a",
        "ARW2-6-09a",
        "ARW2-7-03a",
        "ARW2-8-01a",
        "ARW2-9-01a",
        "ARW2-9-02a",
        "ARW3-1-04a",
        "ARW3-1-05a",
        "ARW3-2-01a",
        "ARW3-2-02a",
        "ARW3-4-02a",
        "ARW3-5-05a",
        "ARW3-6-01a",
        "ARW3-7-04a",
        "ARW3-7-07a",
        "ARW4-1-05a",
        "ARW4-3-05a",
        "ARW4-3-08a",
        "ARW4-3-09a",
        "ARW4-2a-01",
        "ARW4-2a-02",
        "ARW4-2a-03",
        "ARW4-2a-04",
        "ARW1-1-03b",
        "ARW1-1-09b",
        "ARW2-1-03b",
        "ARW2-1-05b",
        "ARW2-6-09b",
        "ARW2-7-03b",
        "ARW2-8-01b",
        "ARW3-2-02b",
        "ARW3-7-04b",
        "ARW4-1-05b",
        "ARW4-3-09b",
        "ARW2-1-05c",
        "ARW3-2-02c",
        "ARW3-7-04c",
        "ARW4-3-09c",
        "ARW2-1-05d",
        "ARW4-3-09d",
        "FFQ1",
        "FFQ2",
        "FFQ3",
        "FFQ4",
        "FFQ5",
        "FFQ6",
        "FFQ7",
        "FFQ8",
        "FFQ9",
        "FFQ10",
        "FFQ11",
        "FFQ12",
        "FFQ13",
        "FFQ14",
        "FFQ15",
        "FFQ16",
        "FFQ17",
        "FFQ18",
        "FFQ19",
        "FFQ20",
        "FFQ21",
        "FFQ22",
        "FFQ23",
        "ICS1",
        "ICS2",
        "ICS3",
        "ICS4",
        "ICS5",
        "ICS6",
        "ICS7",
        "ICS8",
        "ICS9",
        "ICS10",
        "ICS11",
        "ICS12",
        "ICS13",
        "ICS14",
        "ICS15",
        "ICS16",
        "ICS17",
        "ICS18",
        "ICS19",
        "ICS20",
        "ICS21",
        "ICS22",
        "ICS23",
        "ICS24",
        "ICS25",
        "ICS26",
        "ICS27",
        "ICS28",
        "ICS29",
        "ICS30",
        "ICS31",
        "ICS32",
        "ICS33",
        "ICS34",
        "ICS35",
        "ICS36",
        "ICS37",
        "ICS38",
        "ICS39",
        "ICS40",
        "ICS41",
        "ICS42",
        "ICS43",
        "ICS44",
        "ICU1",
        "ICU2",
        "ICU3",
        "ICU4",
        "ICU5",
        "ICU6",
        "ICU7",
        "ICU8",
        "ICU9",
        "ICU10",
        "ICU11",
        "ICU12",
        "ICU13",
        "ICU14",
        "ICU15",
        "ICU16",
        "ICU17",
        "ICU18",
        "ICU19",
        "ICU20",
        "ICU21",
        "ICU22",
        "ICU23",
        "ICU24",
        "ICU25",
        "ICU26",
        "ICU27",
        "ICU28",
        "ICU29",
        "ICU30",
        "ICU31",
        "ICU32",
        "ICU33",
        "ICU34",
        "ICU35",
        "ICU36",
        "ICU37",
        "ICU38",
        "ICU39",
        "ICU40",
        "ICU41",
        "ICU42",
        "ICU43",
        "ICU44",
        "ICU45",
        "ICU46",
        "NP1",
        "NP2",
        "NP4",
        "NP5",
        "NP6",
        "NP7",
        "NP8",
        "NP9",
        "NP10",
        "NP11",
        "NP12",
        "NP13",
        "NP110",
        "NP120",
        "NP130",
        "NP140",
        "NP150",
        "NP160",
        "NP210",
        "NP211",
        "NP212",
        "NP213",
        "NP220",
        "NP221",
        "NP222",
        "NP223",
        "NP230",
        "NP231",
        "NP232",
        "NP233",
        "NP301",
        "NP302",
        "NP303",
        "NP304",
        "NP305",
        "NP306",
        "NP307",
        "NP308",
        "NP309",
        "NP310",
        "NP311",
        "NP312",
        "NP313",
        "NP314",
        "NP315",
        "NP316",
        "NP317",
        "NP401",
        "NP402",
        "NP403",
        "NP404",
        "NP405",
        "NP406",
        "NP407",
        "NP408",
        "NP409",
        "NP501",
        "NP502",
        "NP503",
        "NP504",
        "NP505",
        "NP506",
        "NP507",
        "NP508",
        "NP610",
        "NP710",
        "NP711",
        "NP720",
        "NP730",
        "NP740",
        "NP810",
        "NP820",
        "NP830",
        "NP831",
        "NP832",
        "NP833",
        "NP834",
        "NP835",
        "NP836",
        "NP837",
        "NP838",
        "NP839",
        "NP901",
        "NP1010",
        "NP1011",
        "NP1012",
        "NP1013",
        "NP1014",
        "NP1015",
        "NP1020",
        "NP1021",
        "NP1022",
        "NP1023",
        "NP1024",
        "TAME1-1",
        "TAME1-2",
        "TAME1-2-1",
        "TAME1-3",
        "TAME1-3-1",
        "TAME2-1",
        "TAME2-2",
        "TAME2-3",
        "TAME2-4",
        "TAME2-5",
        "TAME2-6",
        "TAME2-7",
        "TAME2-8",
        "TAME2-9",
        "TAME2-10",
        "TAME2-11",
        "TAME2-12",
        "TAME2-13",
        "TAME2-14",
        "TAME2-15",
        "TAME3-1",
        "TAME3-2",
        "TAME3-3",
        "TAME3-4",
        "TAME3-5",
        "TAME3-6",
        "TAME3-7",
        "TAME3-8",
        "TAME4-1",
        "TAME4-2",
        "TAME4-3",
        "TAME4-4",
        "TAME4-5",
        "TAME4-6",
        "TAME4-7"
      )
    if (sum(!toupper(features) %in% features_set) < 1) {
      links <-
        paste0("http://sails.clld.org/parameters/",
               toupper(features),
               ".tab")
    } else {
      not_features <- features[which(!features %in% features_set)]
      stop(paste(
        "There is no features",
        paste0("'", not_features, "'", collapse = ", "),
        "in SAILS database."
      ))
    }
    datalist  <-  lapply(links, function(x) {
      utils::read.csv(
        x,
        sep = "\t",
        na.strings = "N/A",
        stringsAsFactors = FALSE
      )
    })
    oldw <- getOption("warn")
    options(warn = -1)
    final_df <- Reduce(function(x, y) {
      merge(
        x,
        y,
        all = TRUE,
        by = c("iso.639.3",
               "name",
               "latitude",
               "longitude",
               "family")
      )
    }, datalist)
    options(warn = oldw)
    colnames(final_df)[grep("description", colnames(final_df))] <-
      paste(features, "description", sep = "_")
    colnames(final_df)[grep("value", colnames(final_df))] <-
      paste(features, "value", sep = "_")
    final_df$language <- lingtypology::lang.iso(final_df$iso.639.3,
                                                glottolog.source = glottolog.source)
    final_df <-
      final_df[,-which(colnames(final_df) %in% c("iso.639.3", "family", "name"))]
    final_df <- final_df[, c(ncol(final_df), 1:(ncol(final_df) - 1))]
    if (na.rm == TRUE) {
      final_df <- final_df[is.glottolog(final_df$language), ]
    }
    return(final_df)
  }
