#ifndef AT_DATALET_H_
#define AT_DATALET_H_

/**
 * @brief LET tables and access routines
 */

/*
 *    AT_DataLET.h
 *    ==================
 *
 *    Copyright 2006, 2010 The libamtrack team
 *
 *    This file is part of the AmTrack program (libamtrack.sourceforge.net).
 *
 *    AmTrack is free software: you can redistribute it and/or modify
 *    it under the terms of the GNU General Public License as published by
 *    the Free Software Foundation, either version 3 of the License, or
 *    (at your option) any later version.
 *
 *    AmTrack is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *    GNU General Public License for more details.
 *
 *    You should have received a copy of the GNU General Public License
 *    along with AmTrack (file: copying.txt).
 *    If not, see <http://www.gnu.org/licenses/>
 */

#include <stdlib.h>

#include "AT_Error.h"
#include "AT_DataParticle.h"
#include "AT_PhysicsRoutines.h"
#include "AT_DataMaterial.h"

/**
 * Defines the total number of data entries in PSTAR structure below
 */
#define PSTAR_DATA_N_PER_MATERIAL       132
#define PSTAR_DATA_N                    PSTAR_DATA_N_PER_MATERIAL * MATERIAL_DATA_N


/**
 * Structure definition for PSTAR stopping power data
 */
typedef struct {
  const long     n;                                     /** number of items in local PSTAR data table */
  const double   kin_E_MeV[PSTAR_DATA_N];               /** vector of kinetic energies (of length 132) */
  const double   stp_pow_el_MeV_cm2_g[PSTAR_DATA_N];    /** Electronic (Collision) Stopping Power */
  const double   stp_pow_nuc_MeV_cm2_g[PSTAR_DATA_N];   /** Nuclear Stopping Power */
  const double   stp_pow_tot_MeV_cm2_g[PSTAR_DATA_N];   /** Total Stopping Power */
  const double   range_cdsa_g_cm2[PSTAR_DATA_N];        /** CSDA (continuous-slowing-down approximation) range */
  const double   range_proj_g_cm2[PSTAR_DATA_N];        /** Projected range */
  const double   detour_factor[PSTAR_DATA_N];           /** Detour factor (ratio of projected range to CSDA range) */
  const long     material_no[PSTAR_DATA_N];             /** Material number (see AT_DataMaterial.h) */
} AT_pstar_data_struct;

/**
 * Structure to hold PSTAR stopping power data
 * Tabulated data for protons with energies between 1keV and 10GeV (132 kinetic energies),
 * based on ICRU Report 49
 * See http://physics.nist.gov/PhysRefData/Star/Text/intro.html
 * TODO Should be moved to external file
*/
static const AT_pstar_data_struct AT_PSTAR_Data = {
    PSTAR_DATA_N,
    {
        1.0000e-3f,    1.5000e-3f,    2.0000e-3f,    2.5000e-3f,    3.0000e-3f,    4.0000e-3f,    5.0000e-3f,    6.0000e-3f,    7.0000e-3f,    8.0000e-3f,
        9.0000e-3f,    1.0000e-2f,    1.2500e-2f,    1.5000e-2f,    1.7500e-2f,    2.0000e-2f,    2.2500e-2f,    2.5000e-2f,    2.7500e-2f,    3.0000e-2f,
        3.5000e-2f,    4.0000e-2f,    4.5000e-2f,    5.0000e-2f,    5.5000e-2f,    6.0000e-2f,    6.5000e-2f,    7.0000e-2f,    7.5000e-2f,    8.0000e-2f,
        8.5000e-2f,    9.0000e-2f,    9.5000e-2f,    1.0000e-1f,    1.2500e-1f,    1.5000e-1f,    1.7500e-1f,    2.0000e-1f,    2.2500e-1f,    2.5000e-1f,
        2.7500e-1f,    3.0000e-1f,    3.5000e-1f,    4.0000e-1f,    4.5000e-1f,    5.0000e-1f,    5.5000e-1f,    6.0000e-1f,    6.5000e-1f,    7.0000e-1f,
        7.5000e-1f,    8.0000e-1f,    8.5000e-1f,    9.0000e-1f,    9.5000e-1f,    1.0000e0f,    1.2500e0f,    1.5000e0f,    1.7500e0f,    2.0000e0f,
        2.2500e0f,    2.5000e0f,    2.7500e0f,    3.0000e0f,    3.5000e0f,    4.0000e0f,    4.5000e0f,    5.0000e0f,    5.5000e0f,    6.0000e0f,
        6.5000e0f,    7.0000e0f,    7.5000e0f,    8.0000e0f,    8.5000e0f,    9.0000e0f,    9.5000e0f,    1.0000e1f,    1.2500e1f,    1.5000e1f,
        1.7500e1f,    2.0000e1f,    2.5000e1f,    2.7500e1f,    3.0000e1f,    3.5000e1f,    4.0000e1f,    4.5000e1f,    5.0000e1f,    5.5000e1f,
        6.0000e1f,    6.5000e1f,    7.0000e1f,    7.5000e1f,    8.0000e1f,    8.5000e1f,    9.0000e1f,    9.5000e1f,    1.0000e2f,    1.2500e2f,
        1.5000e2f,    1.7500e2f,    2.0000e2f,    2.2500e2f,    2.5000e2f,    2.7500e2f,    3.0000e2f,    3.5000e2f,    4.0000e2f,    4.5000e2f,
        5.0000e2f,    5.5000e2f,    6.0000e2f,    6.5000e2f,    7.0000e2f,    7.5000e2f,    8.0000e2f,    8.5000e2f,    9.0000e2f,    9.5000e2f,
        1.0000e3f,    1.5000e3f,    2.0000e3f,    2.5000e3f,    3.0000e3f,    4.0000e3f,    5.0000e3f,    6.0000e3f,    7.0000e3f,    8.0000e3f,
        9.0000e3f,    1.0000e4f,    1.0000e-3f,    1.5000e-3f,    2.0000e-3f,    2.5000e-3f,    3.0000e-3f,    4.0000e-3f,    5.0000e-3f,    6.0000e-3f,
        7.0000e-3f,    8.0000e-3f,    9.0000e-3f,    1.0000e-2f,    1.2500e-2f,    1.5000e-2f,    1.7500e-2f,    2.0000e-2f,    2.2500e-2f,    2.5000e-2f,
        2.7500e-2f,    3.0000e-2f,    3.5000e-2f,    4.0000e-2f,    4.5000e-2f,    5.0000e-2f,    5.5000e-2f,    6.0000e-2f,    6.5000e-2f,    7.0000e-2f,
        7.5000e-2f,    8.0000e-2f,    8.5000e-2f,    9.0000e-2f,    9.5000e-2f,    1.0000e-1f,    1.2500e-1f,    1.5000e-1f,    1.7500e-1f,    2.0000e-1f,
        2.2500e-1f,    2.5000e-1f,    2.7500e-1f,    3.0000e-1f,    3.5000e-1f,    4.0000e-1f,    4.5000e-1f,    5.0000e-1f,    5.5000e-1f,    6.0000e-1f,
        6.5000e-1f,    7.0000e-1f,    7.5000e-1f,    8.0000e-1f,    8.5000e-1f,    9.0000e-1f,    9.5000e-1f,    1.0000e0f,    1.2500e0f,    1.5000e0f,
        1.7500e0f,    2.0000e0f,    2.2500e0f,    2.5000e0f,    2.7500e0f,    3.0000e0f,    3.5000e0f,    4.0000e0f,    4.5000e0f,    5.0000e0f,
        5.5000e0f,    6.0000e0f,    6.5000e0f,    7.0000e0f,    7.5000e0f,    8.0000e0f,    8.5000e0f,    9.0000e0f,    9.5000e0f,    1.0000e1f,
        1.2500e1f,    1.5000e1f,    1.7500e1f,    2.0000e1f,    2.5000e1f,    2.7500e1f,    3.0000e1f,    3.5000e1f,    4.0000e1f,    4.5000e1f,
        5.0000e1f,    5.5000e1f,    6.0000e1f,    6.5000e1f,    7.0000e1f,    7.5000e1f,    8.0000e1f,    8.5000e1f,    9.0000e1f,    9.5000e1f,
        1.0000e2f,    1.2500e2f,    1.5000e2f,    1.7500e2f,    2.0000e2f,    2.2500e2f,    2.5000e2f,    2.7500e2f,    3.0000e2f,    3.5000e2f,
        4.0000e2f,    4.5000e2f,    5.0000e2f,    5.5000e2f,    6.0000e2f,    6.5000e2f,    7.0000e2f,    7.5000e2f,    8.0000e2f,    8.5000e2f,
        9.0000e2f,    9.5000e2f,    1.0000e3f,    1.5000e3f,    2.0000e3f,    2.5000e3f,    3.0000e3f,    4.0000e3f,    5.0000e3f,    6.0000e3f,
        7.0000e3f,    8.0000e3f,    9.0000e3f,    1.0000e4f,    1.0000e-3f,    1.5000e-3f,    2.0000e-3f,    2.5000e-3f,    3.0000e-3f,    4.0000e-3f,
        5.0000e-3f,    6.0000e-3f,    7.0000e-3f,    8.0000e-3f,    9.0000e-3f,    1.0000e-2f,    1.2500e-2f,    1.5000e-2f,    1.7500e-2f,    2.0000e-2f,
        2.2500e-2f,    2.5000e-2f,    2.7500e-2f,    3.0000e-2f,    3.5000e-2f,    4.0000e-2f,    4.5000e-2f,    5.0000e-2f,    5.5000e-2f,    6.0000e-2f,
        6.5000e-2f,    7.0000e-2f,    7.5000e-2f,    8.0000e-2f,    8.5000e-2f,    9.0000e-2f,    9.5000e-2f,    1.0000e-1f,    1.2500e-1f,    1.5000e-1f,
        1.7500e-1f,    2.0000e-1f,    2.2500e-1f,    2.5000e-1f,    2.7500e-1f,    3.0000e-1f,    3.5000e-1f,    4.0000e-1f,    4.5000e-1f,    5.0000e-1f,
        5.5000e-1f,    6.0000e-1f,    6.5000e-1f,    7.0000e-1f,    7.5000e-1f,    8.0000e-1f,    8.5000e-1f,    9.0000e-1f,    9.5000e-1f,    1.0000e0f,
        1.2500e0f,    1.5000e0f,    1.7500e0f,    2.0000e0f,    2.2500e0f,    2.5000e0f,    2.7500e0f,    3.0000e0f,    3.5000e0f,    4.0000e0f,
        4.5000e0f,    5.0000e0f,    5.5000e0f,    6.0000e0f,    6.5000e0f,    7.0000e0f,    7.5000e0f,    8.0000e0f,    8.5000e0f,    9.0000e0f,
        9.5000e0f,    1.0000e1f,    1.2500e1f,    1.5000e1f,    1.7500e1f,    2.0000e1f,    2.5000e1f,    2.7500e1f,    3.0000e1f,    3.5000e1f,
        4.0000e1f,    4.5000e1f,    5.0000e1f,    5.5000e1f,    6.0000e1f,    6.5000e1f,    7.0000e1f,    7.5000e1f,    8.0000e1f,    8.5000e1f,
        9.0000e1f,    9.5000e1f,    1.0000e2f,    1.2500e2f,    1.5000e2f,    1.7500e2f,    2.0000e2f,    2.2500e2f,    2.5000e2f,    2.7500e2f,
        3.0000e2f,    3.5000e2f,    4.0000e2f,    4.5000e2f,    5.0000e2f,    5.5000e2f,    6.0000e2f,    6.5000e2f,    7.0000e2f,    7.5000e2f,
        8.0000e2f,    8.5000e2f,    9.0000e2f,    9.5000e2f,    1.0000e3f,    1.5000e3f,    2.0000e3f,    2.5000e3f,    3.0000e3f,    4.0000e3f,
        5.0000e3f,    6.0000e3f,    7.0000e3f,    8.0000e3f,    9.0000e3f,    1.0000e4f,    1.0000e-3f,    1.5000e-3f,    2.0000e-3f,    2.5000e-3f,
        3.0000e-3f,    4.0000e-3f,    5.0000e-3f,    6.0000e-3f,    7.0000e-3f,    8.0000e-3f,    9.0000e-3f,    1.0000e-2f,    1.2500e-2f,    1.5000e-2f,
        1.7500e-2f,    2.0000e-2f,    2.2500e-2f,    2.5000e-2f,    2.7500e-2f,    3.0000e-2f,    3.5000e-2f,    4.0000e-2f,    4.5000e-2f,    5.0000e-2f,
        5.5000e-2f,    6.0000e-2f,    6.5000e-2f,    7.0000e-2f,    7.5000e-2f,    8.0000e-2f,    8.5000e-2f,    9.0000e-2f,    9.5000e-2f,    1.0000e-1f,
        1.2500e-1f,    1.5000e-1f,    1.7500e-1f,    2.0000e-1f,    2.2500e-1f,    2.5000e-1f,    2.7500e-1f,    3.0000e-1f,    3.5000e-1f,    4.0000e-1f,
        4.5000e-1f,    5.0000e-1f,    5.5000e-1f,    6.0000e-1f,    6.5000e-1f,    7.0000e-1f,    7.5000e-1f,    8.0000e-1f,    8.5000e-1f,    9.0000e-1f,
        9.5000e-1f,    1.0000e0f,    1.2500e0f,    1.5000e0f,    1.7500e0f,    2.0000e0f,    2.2500e0f,    2.5000e0f,    2.7500e0f,    3.0000e0f,
        3.5000e0f,    4.0000e0f,    4.5000e0f,    5.0000e0f,    5.5000e0f,    6.0000e0f,    6.5000e0f,    7.0000e0f,    7.5000e0f,    8.0000e0f,
        8.5000e0f,    9.0000e0f,    9.5000e0f,    1.0000e1f,    1.2500e1f,    1.5000e1f,    1.7500e1f,    2.0000e1f,    2.5000e1f,    2.7500e1f,
        3.0000e1f,    3.5000e1f,    4.0000e1f,    4.5000e1f,    5.0000e1f,    5.5000e1f,    6.0000e1f,    6.5000e1f,    7.0000e1f,    7.5000e1f,
        8.0000e1f,    8.5000e1f,    9.0000e1f,    9.5000e1f,    1.0000e2f,    1.2500e2f,    1.5000e2f,    1.7500e2f,    2.0000e2f,    2.2500e2f,
        2.5000e2f,    2.7500e2f,    3.0000e2f,    3.5000e2f,    4.0000e2f,    4.5000e2f,    5.0000e2f,    5.5000e2f,    6.0000e2f,    6.5000e2f,
        7.0000e2f,    7.5000e2f,    8.0000e2f,    8.5000e2f,    9.0000e2f,    9.5000e2f,    1.0000e3f,    1.5000e3f,    2.0000e3f,    2.5000e3f,
        3.0000e3f,    4.0000e3f,    5.0000e3f,    6.0000e3f,    7.0000e3f,    8.0000e3f,    9.0000e3f,    1.0000e4f,
        1.000e-03f,             1.500e-03f,             2.000e-03f,             2.500e-03f,             3.000e-03f,             4.000e-03f,             5.000e-03f,             6.000e-03f,             7.000e-03f,             8.000e-03f,
        9.000e-03f,             1.000e-02f,             1.250e-02f,             1.500e-02f,             1.750e-02f,             2.000e-02f,             2.250e-02f,             2.500e-02f,             2.750e-02f,             3.000e-02f,
        3.500e-02f,             4.000e-02f,             4.500e-02f,             5.000e-02f,             5.500e-02f,             6.000e-02f,             6.500e-02f,             7.000e-02f,             7.500e-02f,             8.000e-02f,
        8.500e-02f,             9.000e-02f,             9.500e-02f,             1.000e-01f,             1.250e-01f,             1.500e-01f,             1.750e-01f,             2.000e-01f,             2.250e-01f,             2.500e-01f,
        2.750e-01f,             3.000e-01f,             3.500e-01f,             4.000e-01f,             4.500e-01f,             5.000e-01f,             5.500e-01f,             6.000e-01f,             6.500e-01f,             7.000e-01f,
        7.500e-01f,             8.000e-01f,             8.500e-01f,             9.000e-01f,             9.500e-01f,             1.000e+00f,             1.250e+00f,             1.500e+00f,             1.750e+00f,             2.000e+00f,
        2.250e+00f,             2.500e+00f,             2.750e+00f,             3.000e+00f,             3.500e+00f,             4.000e+00f,             4.500e+00f,             5.000e+00f,             5.500e+00f,             6.000e+00f,
        6.500e+00f,             7.000e+00f,             7.500e+00f,             8.000e+00f,             8.500e+00f,             9.000e+00f,             9.500e+00f,             1.000e+01f,             1.250e+01f,             1.500e+01f,
        1.750e+01f,             2.000e+01f,             2.500e+01f,             2.750e+01f,             3.000e+01f,             3.500e+01f,             4.000e+01f,             4.500e+01f,             5.000e+01f,             5.500e+01f,
        6.000e+01f,             6.500e+01f,             7.000e+01f,             7.500e+01f,             8.000e+01f,             8.500e+01f,             9.000e+01f,             9.500e+01f,             1.000e+02f,             1.250e+02f,
        1.500e+02f,             1.750e+02f,             2.000e+02f,             2.250e+02f,             2.500e+02f,             2.750e+02f,             3.000e+02f,             3.500e+02f,             4.000e+02f,             4.500e+02f,
        5.000e+02f,             5.500e+02f,             6.000e+02f,             6.500e+02f,             7.000e+02f,             7.500e+02f,             8.000e+02f,             8.500e+02f,             9.000e+02f,             9.500e+02f,
        1.000e+03f,             1.500e+03f,             2.000e+03f,             2.500e+03f,             3.000e+03f,             4.000e+03f,             5.000e+03f,             6.000e+03f,             7.000e+03f,             8.000e+03f,
        9.000e+03f,             1.000e+04f,
        1.000e-03f,             1.500e-03f,             2.000e-03f,             2.500e-03f,             3.000e-03f,             4.000e-03f,             5.000e-03f,             6.000e-03f,             7.000e-03f,             8.000e-03f,
        9.000e-03f,             1.000e-02f,             1.250e-02f,             1.500e-02f,             1.750e-02f,             2.000e-02f,             2.250e-02f,             2.500e-02f,             2.750e-02f,             3.000e-02f,
        3.500e-02f,             4.000e-02f,             4.500e-02f,             5.000e-02f,             5.500e-02f,             6.000e-02f,             6.500e-02f,             7.000e-02f,             7.500e-02f,             8.000e-02f,
        8.500e-02f,             9.000e-02f,             9.500e-02f,             1.000e-01f,             1.250e-01f,             1.500e-01f,             1.750e-01f,             2.000e-01f,             2.250e-01f,             2.500e-01f,
        2.750e-01f,             3.000e-01f,             3.500e-01f,             4.000e-01f,             4.500e-01f,             5.000e-01f,             5.500e-01f,             6.000e-01f,             6.500e-01f,             7.000e-01f,
        7.500e-01f,             8.000e-01f,             8.500e-01f,             9.000e-01f,             9.500e-01f,             1.000e+00f,             1.250e+00f,             1.500e+00f,             1.750e+00f,             2.000e+00f,
        2.250e+00f,             2.500e+00f,             2.750e+00f,             3.000e+00f,             3.500e+00f,             4.000e+00f,             4.500e+00f,             5.000e+00f,             5.500e+00f,             6.000e+00f,
        6.500e+00f,             7.000e+00f,             7.500e+00f,             8.000e+00f,             8.500e+00f,             9.000e+00f,             9.500e+00f,             1.000e+01f,             1.250e+01f,             1.500e+01f,
        1.750e+01f,             2.000e+01f,             2.500e+01f,             2.750e+01f,             3.000e+01f,             3.500e+01f,             4.000e+01f,             4.500e+01f,             5.000e+01f,             5.500e+01f,
        6.000e+01f,             6.500e+01f,             7.000e+01f,             7.500e+01f,             8.000e+01f,             8.500e+01f,             9.000e+01f,             9.500e+01f,             1.000e+02f,             1.250e+02f,
        1.500e+02f,             1.750e+02f,             2.000e+02f,             2.250e+02f,             2.500e+02f,             2.750e+02f,             3.000e+02f,             3.500e+02f,             4.000e+02f,             4.500e+02f,
        5.000e+02f,             5.500e+02f,             6.000e+02f,             6.500e+02f,             7.000e+02f,             7.500e+02f,             8.000e+02f,             8.500e+02f,             9.000e+02f,             9.500e+02f,
        1.000e+03f,             1.500e+03f,             2.000e+03f,             2.500e+03f,             3.000e+03f,             4.000e+03f,             5.000e+03f,             6.000e+03f,             7.000e+03f,             8.000e+03f,
        9.000e+03f,             1.000e+04f,
		1.000e-03f,		1.500e-03f,		2.000e-03f,		2.500e-03f,		3.000e-03f,		4.000e-03f,		5.000e-03f,		6.000e-03f,		7.000e-03f,		8.000e-03f, 
		9.000e-03f,		1.000e-02f,		1.250e-02f,		1.500e-02f,		1.750e-02f,		2.000e-02f,		2.250e-02f,		2.500e-02f,		2.750e-02f,		3.000e-02f, 
		3.500e-02f,		4.000e-02f,		4.500e-02f,		5.000e-02f,		5.500e-02f,		6.000e-02f,		6.500e-02f,		7.000e-02f,		7.500e-02f,		8.000e-02f, 
		8.500e-02f,		9.000e-02f,		9.500e-02f,		1.000e-01f,		1.250e-01f,		1.500e-01f,		1.750e-01f,		2.000e-01f,		2.250e-01f,		2.500e-01f, 
		2.750e-01f,		3.000e-01f,		3.500e-01f,		4.000e-01f,		4.500e-01f,		5.000e-01f,		5.500e-01f,		6.000e-01f,		6.500e-01f,		7.000e-01f, 
		7.500e-01f,		8.000e-01f,		8.500e-01f,		9.000e-01f,		9.500e-01f,		1.000e+00f,		1.250e+00f,		1.500e+00f,		1.750e+00f,		2.000e+00f, 
		2.250e+00f,		2.500e+00f,		2.750e+00f,		3.000e+00f,		3.500e+00f,		4.000e+00f,		4.500e+00f,		5.000e+00f,		5.500e+00f,		6.000e+00f, 
		6.500e+00f,		7.000e+00f,		7.500e+00f,		8.000e+00f,		8.500e+00f,		9.000e+00f,		9.500e+00f,		1.000e+01f,		1.250e+01f,		1.500e+01f, 
		1.750e+01f,		2.000e+01f,		2.500e+01f,		2.750e+01f,		3.000e+01f,		3.500e+01f,		4.000e+01f,		4.500e+01f,		5.000e+01f,		5.500e+01f, 
		6.000e+01f,		6.500e+01f,		7.000e+01f,		7.500e+01f,		8.000e+01f,		8.500e+01f,		9.000e+01f,		9.500e+01f,		1.000e+02f,		1.250e+02f, 
		1.500e+02f,		1.750e+02f,		2.000e+02f,		2.250e+02f,		2.500e+02f,		2.750e+02f,		3.000e+02f,		3.500e+02f,		4.000e+02f,		4.500e+02f, 
		5.000e+02f,		5.500e+02f,		6.000e+02f,		6.500e+02f,		7.000e+02f,		7.500e+02f,		8.000e+02f,		8.500e+02f,		9.000e+02f,		9.500e+02f, 
		1.000e+03f,		1.500e+03f,		2.000e+03f,		2.500e+03f,		3.000e+03f,		4.000e+03f,		5.000e+03f,		6.000e+03f,		7.000e+03f,		8.000e+03f, 
		9.000e+03f,		1.000e+04f 

    },
    {
        1.3370e2f,    1.6380e2f,    1.8910e2f,    2.1140e2f,    2.3160e2f,    2.6750e2f,    2.9900e2f,    3.2760e2f,    3.5380e2f,    3.7820e2f,
        4.0120e2f,    4.2290e2f,    4.6600e2f,    5.0360e2f,    5.3720e2f,    5.6730e2f,    5.9460e2f,    6.1950e2f,    6.4210e2f,    6.6280e2f,
        6.9890e2f,    7.2900e2f,    7.5380e2f,    7.7400e2f,    7.9010e2f,    8.0260e2f,    8.1190e2f,    8.1830e2f,    8.2230e2f,    8.2410e2f,
        8.2390e2f,    8.2220e2f,    8.1900e2f,    8.1450e2f,    7.8010e2f,    7.3600e2f,    6.9590e2f,    6.6040e2f,    6.2860e2f,    5.9990e2f,
        5.7370e2f,    5.4970e2f,    5.0750e2f,    4.7140e2f,    4.4010e2f,    4.1280e2f,    3.8880e2f,    3.6760e2f,    3.4890e2f,    3.3220e2f,
        3.1720e2f,    3.0370e2f,    2.9140e2f,    2.8030e2f,    2.7000e2f,    2.6060e2f,    2.2280e2f,    1.9550e2f,    1.7480e2f,    1.5850e2f,
        1.4530e2f,    1.3430e2f,    1.2500e2f,    1.1710e2f,    1.0410e2f,    9.3980e1f,    8.5800e1f,    7.9060e1f,    7.3390e1f,    6.8540e1f,
        6.4340e1f,    6.0680e1f,    5.7440e1f,    5.4560e1f,    5.1990e1f,    4.9660e1f,    4.7560e1f,    4.5640e1f,    3.8130e1f,    3.2900e1f,
        2.9040e1f,    2.6050e1f,    2.1740e1f,    2.0120e1f,    1.8750e1f,    1.6560e1f,    1.4870e1f,    1.3530e1f,    1.2440e1f,    1.1540e1f,
        1.0780e1f,    1.0120e1f,    9.5550e0f,    9.0590e0f,    8.6220e0f,    8.2330e0f,    7.8840e0f,    7.5700e0f,    7.2860e0f,    6.1900e0f,
        5.4430e0f,    4.9010e0f,    4.4910e0f,    4.1690e0f,    3.9100e0f,    3.6970e0f,    3.5190e0f,    3.2400e0f,    3.0310e0f,    2.8700e0f,
        2.7430e0f,    2.6400e0f,    2.5550e0f,    2.4850e0f,    2.4260e0f,    2.3750e0f,    2.3330e0f,    2.2960e0f,    2.2640e0f,    2.2360e0f,
        2.2110e0f,    2.0700e0f,    2.0210e0f,    2.0040e0f,    2.0010e0f,    2.0120e0f,    2.0310e0f,    2.0520e0f,    2.0720e0f,    2.0910e0f,
        2.1090e0f,    2.1260e0f,    7.3510e1f,    9.0030e1f,    1.0400e2f,    1.1620e2f,    1.2730e2f,    1.4700e2f,    1.6440e2f,    1.8010e2f,
        1.9450e2f,    2.0790e2f,    2.2050e2f,    2.3250e2f,    2.5610e2f,    2.7680e2f,    2.9530e2f,    3.1190e2f,    3.2700e2f,    3.4080e2f,
        3.5340e2f,    3.6490e2f,    3.8530e2f,    4.0250e2f,    4.1700e2f,    4.2920e2f,    4.3930e2f,    4.4760e2f,    4.5430e2f,    4.5960e2f,
        4.6370e2f,    4.6660e2f,    4.6860e2f,    4.6980e2f,    4.7020e2f,    4.7000e2f,    4.6190e2f,    4.4720e2f,    4.3010e2f,    4.1270e2f,
        3.9650e2f,    3.8140e2f,    3.6750e2f,    3.5450e2f,    3.3110e2f,    3.1060e2f,    2.9260e2f,    2.7650e2f,    2.6220e2f,    2.4940e2f,
        2.3810e2f,    2.2790e2f,    2.1870e2f,    2.1020e2f,    2.0260e2f,    1.9570e2f,    1.8920e2f,    1.8320e2f,    1.5890e2f,    1.4090e2f,
        1.2700e2f,    1.1600e2f,    1.0690e2f,    9.9260e1f,    9.2790e1f,    8.7200e1f,    7.8030e1f,    7.0780e1f,    6.4900e1f,    6.0020e1f,
        5.5900e1f,    5.2360e1f,    4.9280e1f,    4.6580e1f,    4.4190e1f,    4.2060e1f,    4.0150e1f,    3.8420e1f,    3.6840e1f,    3.5410e1f,
        2.9760e1f,    2.5790e1f,    2.2840e1f,    2.0550e1f,    1.7220e1f,    1.5970e1f,    1.4910e1f,    1.3200e1f,    1.1880e1f,    1.0830e1f,
        9.9730e0f,    9.2610e0f,    8.6580e0f,    8.1410e0f,    7.6930e0f,    7.3000e0f,    6.9530e0f,    6.6440e0f,    6.3670e0f,    6.1180e0f,
        5.8910e0f,    5.0160e0f,    4.4190e0f,    3.9850e0f,    3.6560e0f,    3.3970e0f,    3.1890e0f,    3.0180e0f,    2.8750e0f,    2.6500e0f,
        2.4820e0f,    2.3520e0f,    2.2480e0f,    2.1640e0f,    2.0940e0f,    2.0360e0f,    1.9870e0f,    1.9460e0f,    1.9100e0f,    1.8790e0f,
        1.8520e0f,    1.8290e0f,    1.8090e0f,    1.6960e0f,    1.6600e0f,    1.6510e0f,    1.6530e0f,    1.6690e0f,    1.6900e0f,    1.7110e0f,
        1.7310e0f,    1.7490e0f,    1.7660e0f,    1.7820e0f,    9.2380e1f,    1.1310e2f,    1.3060e2f,    1.4610e2f,    1.6000e2f,    1.8480e2f,
        2.0660e2f,    2.2630e2f,    2.4440e2f,    2.6130e2f,    2.7710e2f,    2.9210e2f,    3.2060e2f,    3.4480e2f,    3.6570e2f,    3.8380e2f,
        3.9960e2f,    4.1320e2f,    4.2500e2f,    4.3510e2f,    4.5100e2f,    4.6200e2f,    4.6920e2f,    4.7340e2f,    4.7520e2f,    4.7510e2f,
        4.7370e2f,    4.7120e2f,    4.6800e2f,    4.6420e2f,    4.6010e2f,    4.5580e2f,    4.5130e2f,    4.4680e2f,    4.2450e2f,    4.0450e2f,
        3.8670e2f,    3.7100e2f,    3.5680e2f,    3.4400e2f,    3.3230e2f,    3.2150e2f,    3.0170e2f,    2.8420e2f,    2.6860e2f,    2.5480e2f,
        2.4250e2f,    2.3140e2f,    2.2150e2f,    2.1240e2f,    2.0420e2f,    1.9660e2f,    1.8970e2f,    1.8330e2f,    1.7740e2f,    1.7190e2f,
        1.4940e2f,    1.3270e2f,    1.1980e2f,    1.0940e2f,    1.0090e2f,    9.3770e1f,    8.7690e1f,    8.2450e1f,    7.3830e1f,    6.7030e1f,
        6.1510e1f,    5.6910e1f,    5.3030e1f,    4.9700e1f,    4.6810e1f,    4.4280e1f,    4.2030e1f,    4.0020e1f,    3.8220e1f,    3.6580e1f,
        3.5100e1f,    3.3750e1f,    2.8410e1f,    2.4650e1f,    2.1850e1f,    1.9680e1f,    1.6510e1f,    1.5320e1f,    1.4300e1f,    1.2670e1f,
        1.1410e1f,    1.0410e1f,    9.5900e0f,    8.9080e0f,    8.3300e0f,    7.8350e0f,    7.4050e0f,    7.0290e0f,    6.6960e0f,    6.3990e0f,
        6.1330e0f,    5.8930e0f,    5.6760e0f,    4.8350e0f,    4.2610e0f,    3.8430e0f,    3.5250e0f,    3.2760e0f,    3.0750e0f,    2.9100e0f,
        2.7720e0f,    2.5550e0f,    2.3920e0f,    2.2660e0f,    2.1660e0f,    2.0860e0f,    2.0190e0f,    1.9640e0f,    1.9180e0f,    1.8780e0f,
        1.8450e0f,    1.8160e0f,    1.7900e0f,    1.7680e0f,    1.7490e0f,    1.6470e0f,    1.6170e0f,    1.6130e0f,    1.6190e0f,    1.6420e0f,
        1.6680e0f,    1.6920e0f,    1.7140e0f,    1.7340e0f,    1.7520e0f,    1.7680e0f,    1.7490e2f,    2.1420e2f,    2.4740e2f,    2.7660e2f,
        3.0300e2f,    3.4590e2f,    3.8410e2f,    4.1870e2f,    4.5040e2f,    4.7900e2f,    5.0520e2f,    5.2980e2f,    5.7910e2f,    6.2080e2f,
        6.5710e2f,    6.8930e2f,    7.1790e2f,    7.4320e2f,    7.6570e2f,    7.8580e2f,    8.2080e2f,    8.5050e2f,    8.7500e2f,    8.9450e2f,
        9.0930e2f,    9.2010e2f,    9.2740e2f,    9.3190e2f,    9.3390e2f,    9.3370e2f,    9.3160e2f,    9.2800e2f,    9.2290e2f,    9.1680e2f,
        8.7440e2f,    8.2320e2f,    7.7130e2f,    7.2240e2f,    6.7620e2f,    6.3400e2f,    5.9620e2f,    5.6280e2f,    5.0720e2f,    4.6340e2f,
        4.2850e2f,    4.0020e2f,    3.7670e2f,    3.5610e2f,    3.3800e2f,    3.2190e2f,    3.0750e2f,    2.9450e2f,    2.8280e2f,    2.7200e2f,
        2.6210e2f,    2.5300e2f,    2.1660e2f,    1.9040e2f,    1.7030e2f,    1.5450e2f,    1.4170e2f,    1.3100e2f,    1.2200e2f,    1.1430e2f,
        1.0160e2f,    9.1730e1f,    8.3740e1f,    7.7140e1f,    7.1600e1f,    6.6860e1f,    6.2760e1f,    5.9180e1f,    5.6020e1f,    5.3210e1f,
        5.0700e1f,    4.8430e1f,    4.6370e1f,    4.4500e1f,    3.7170e1f,    3.2060e1f,    2.8290e1f,    2.5380e1f,    2.1170e1f,    1.9600e1f,
        1.8260e1f,    1.6120e1f,    1.4480e1f,    1.3180e1f,    1.2120e1f,    1.1230e1f,    1.0490e1f,    9.8540e0f,    9.3020e0f,    8.8190e0f,
        8.3930e0f,    8.0140e0f,    7.6750e0f,    7.3690e0f,    7.0930e0f,    6.0250e0f,    5.2980e0f,    4.7700e0f,    4.3710e0f,    4.0570e0f,
        3.8050e0f,    3.5980e0f,    3.4250e0f,    3.1530e0f,    2.9500e0f,    2.7930e0f,    2.6690e0f,    2.5690e0f,    2.4860e0f,    2.4180e0f,
        2.3600e0f,    2.3110e0f,    2.2690e0f,    2.2310e0f,    2.1990e0f,    2.1700e0f,    2.1450e0f,    2.0040e0f,    1.9540e0f,    1.9370e0f,
        1.9340e0f,    1.9450e0f,    1.9640e0f,    1.9850e0f,    2.0050e0f,    2.0240e0f,    2.0420e0f,    2.0590e0f,
        1.707e+02f,             2.091e+02f,             2.414e+02f,             2.699e+02f,             2.957e+02f,             3.387e+02f,             3.770e+02f,             4.116e+02f,             4.432e+02f,             4.722e+02f,
        4.990e+02f,             5.241e+02f,             5.739e+02f,             6.163e+02f,             6.534e+02f,             6.863e+02f,             7.156e+02f,             7.417e+02f,             7.648e+02f,             7.857e+02f,
        8.217e+02f,             8.517e+02f,             8.761e+02f,             8.955e+02f,             9.101e+02f,             9.208e+02f,             9.281e+02f,             9.325e+02f,             9.343e+02f,             9.340e+02f,
        9.319e+02f,             9.281e+02f,             9.230e+02f,             9.167e+02f,             8.740e+02f,             8.226e+02f,             7.705e+02f,             7.214e+02f,             6.766e+02f,             6.362e+02f,
        6.004e+02f,             5.689e+02f,             5.166e+02f,             4.750e+02f,             4.407e+02f,             4.119e+02f,             3.872e+02f,             3.655e+02f,             3.466e+02f,             3.298e+02f,
        3.147e+02f,             3.012e+02f,             2.890e+02f,             2.778e+02f,             2.676e+02f,             2.581e+02f,             2.206e+02f,             1.936e+02f,             1.730e+02f,             1.568e+02f,
        1.437e+02f,             1.328e+02f,             1.237e+02f,             1.158e+02f,             1.029e+02f,             9.284e+01f,             8.473e+01f,             7.803e+01f,             7.241e+01f,             6.760e+01f,
        6.345e+01f,             5.981e+01f,             5.661e+01f,             5.376e+01f,             5.122e+01f,             4.892e+01f,             4.684e+01f,             4.495e+01f,             3.753e+01f,             3.236e+01f,
        2.855e+01f,             2.561e+01f,             2.135e+01f,             1.976e+01f,             1.841e+01f,             1.625e+01f,             1.459e+01f,             1.328e+01f,             1.221e+01f,             1.132e+01f,
        1.057e+01f,             9.925e+00f,             9.369e+00f,             8.881e+00f,             8.452e+00f,             8.070e+00f,             7.727e+00f,             7.419e+00f,             7.140e+00f,             6.063e+00f,
        5.330e+00f,             4.798e+00f,             4.395e+00f,             4.078e+00f,             3.824e+00f,             3.615e+00f,             3.440e+00f,             3.166e+00f,             2.960e+00f,             2.802e+00f,
        2.675e+00f,             2.574e+00f,             2.491e+00f,             2.420e+00f,             2.362e+00f,             2.312e+00f,             2.269e+00f,             2.232e+00f,             2.200e+00f,             2.173e+00f,
        2.148e+00f,             2.015e+00f,             1.976e+00f,             1.967e+00f,             1.973e+00f,             2.000e+00f,             2.032e+00f,             2.064e+00f,             2.094e+00f,             2.122e+00f,
        2.148e+00f,             2.173e+00f,
        8.087e+01f,             9.904e+01f,             1.144e+02f,             1.279e+02f,             1.401e+02f,             1.617e+02f,             1.808e+02f,             1.981e+02f,             2.140e+02f,             2.287e+02f,
        2.426e+02f,             2.557e+02f,             2.819e+02f,             3.049e+02f,             3.254e+02f,             3.440e+02f,             3.609e+02f,             3.765e+02f,             3.908e+02f,             4.040e+02f,
        4.276e+02f,             4.478e+02f,             4.653e+02f,             4.803e+02f,             4.931e+02f,             5.041e+02f,             5.133e+02f,             5.211e+02f,             5.274e+02f,             5.325e+02f,
        5.365e+02f,             5.396e+02f,             5.417e+02f,             5.430e+02f,             5.403e+02f,             5.274e+02f,             5.092e+02f,             4.888e+02f,             4.674e+02f,             4.465e+02f,
        4.269e+02f,             4.087e+02f,             3.769e+02f,             3.504e+02f,             3.280e+02f,             3.091e+02f,             2.927e+02f,             2.781e+02f,             2.651e+02f,             2.533e+02f,
        2.425e+02f,             2.328e+02f,             2.239e+02f,             2.156e+02f,             2.082e+02f,             2.012e+02f,             1.731e+02f,             1.525e+02f,             1.368e+02f,             1.244e+02f,
        1.142e+02f,             1.057e+02f,             9.853e+01f,             9.238e+01f,             8.233e+01f,             7.443e+01f,             6.804e+01f,             6.276e+01f,             5.831e+01f,             5.451e+01f,
        5.122e+01f,             4.834e+01f,             4.579e+01f,             4.352e+01f,             4.149e+01f,             3.966e+01f,             3.799e+01f,             3.648e+01f,             3.053e+01f,             2.638e+01f,
        2.330e+01f,             2.093e+01f,             1.748e+01f,             1.619e+01f,             1.510e+01f,             1.334e+01f,             1.199e+01f,             1.092e+01f,             1.004e+01f,             9.317e+00f,
        8.704e+00f,             8.178e+00f,             7.723e+00f,             7.324e+00f,             6.972e+00f,             6.659e+00f,             6.379e+00f,             6.126e+00f,             5.897e+00f,             5.013e+00f,
        4.411e+00f,             3.974e+00f,             3.642e+00f,             3.382e+00f,             3.173e+00f,             3.001e+00f,             2.858e+00f,             2.632e+00f,             2.463e+00f,             2.332e+00f,
        2.228e+00f,             2.144e+00f,             2.075e+00f,             2.017e+00f,             1.968e+00f,             1.927e+00f,             1.891e+00f,             1.860e+00f,             1.833e+00f,             1.810e+00f,
        1.789e+00f,             1.674e+00f,             1.635e+00f,             1.622e+00f,             1.621e+00f,             1.631e+00f,             1.648e+00f,             1.665e+00f,             1.682e+00f,             1.697e+00f,
        1.712e+00f,             1.726e+00f,
		1.197e+02f,		1.467e+02f,		1.693e+02f,		1.893e+02f,		2.074e+02f,		2.395e+02f,		2.678e+02f,		2.933e+02f,		3.168e+02f,		3.387e+02f, 
		3.592e+02f,		3.787e+02f,		4.170e+02f,		4.504e+02f,		4.801e+02f,		5.067e+02f,		5.307e+02f,		5.526e+02f,		5.724e+02f,		5.905e+02f, 
		6.221e+02f,		6.483e+02f,		6.700e+02f,		6.877e+02f,		7.020e+02f,		7.132e+02f,		7.217e+02f,		7.278e+02f,		7.319e+02f,		7.341e+02f, 
		7.348e+02f,		7.340e+02f,		7.320e+02f,		7.290e+02f,		7.029e+02f,		6.672e+02f,		6.291e+02f,		5.922e+02f,		5.583e+02f,		5.278e+02f, 
		5.006e+02f,		4.763e+02f,		4.349e+02f,		4.012e+02f,		3.733e+02f,		3.498e+02f,		3.297e+02f,		3.121e+02f,		2.964e+02f,		2.824e+02f, 
		2.699e+02f,		2.587e+02f,		2.485e+02f,		2.391e+02f,		2.306e+02f,		2.227e+02f,		1.911e+02f,		1.682e+02f,		1.508e+02f,		1.370e+02f, 
		1.258e+02f,		1.164e+02f,		1.085e+02f,		1.017e+02f,		9.063e+01f,		8.192e+01f,		7.488e+01f,		6.905e+01f,		6.414e+01f,		5.994e+01f, 
		5.630e+01f,		5.312e+01f,		5.031e+01f,		4.781e+01f,		4.557e+01f,		4.355e+01f,		4.171e+01f,		4.004e+01f,		3.349e+01f,		2.892e+01f, 
		2.554e+01f,		2.293e+01f,		1.914e+01f,		1.773e+01f,		1.652e+01f,		1.460e+01f,		1.312e+01f,		1.194e+01f,		1.098e+01f,		1.019e+01f, 
		9.514e+00f,		8.938e+00f,		8.440e+00f,		8.003e+00f,		7.618e+00f,		7.275e+00f,		6.968e+00f,		6.691e+00f,		6.441e+00f,		5.474e+00f, 
		4.815e+00f,		4.337e+00f,		3.975e+00f,		3.690e+00f,		3.462e+00f,		3.274e+00f,		3.117e+00f,		2.870e+00f,		2.686e+00f,		2.544e+00f, 
		2.431e+00f,		2.340e+00f,		2.265e+00f,		2.203e+00f,		2.151e+00f,		2.107e+00f,		2.069e+00f,		2.036e+00f,		2.008e+00f,		1.984e+00f, 
		1.962e+00f,		1.850e+00f,		1.820e+00f,		1.818e+00f,		1.828e+00f,		1.861e+00f,		1.898e+00f,		1.933e+00f,		1.967e+00f,		1.998e+00f, 
		2.026e+00f,		2.052e+00f 
    },
    {
        4.3150e1f,    3.4600e1f,    2.9270e1f,    2.5570e1f,    2.2810e1f,    1.8940e1f,    1.6310e1f,    1.4390e1f,    1.2920e1f,    1.1750e1f,
        1.0800e1f,    1.0000e1f,    8.4850e0f,    7.4000e0f,    6.5810e0f,    5.9390e0f,    5.4210e0f,    4.9930e0f,    4.6330e0f,    4.3250e0f,
        3.8260e0f,    3.4370e0f,    3.1260e0f,    2.8700e0f,    2.6550e0f,    2.4730e0f,    2.3160e0f,    2.1780e0f,    2.0580e0f,    1.9510e0f,
        1.8550e0f,    1.7690e0f,    1.6910e0f,    1.6200e0f,    1.3430e0f,    1.1520e0f,    1.0100e0f,    9.0160e-1f,    8.1520e-1f,    7.4470e-1f,
        6.8550e-1f,    6.3510e-1f,    5.5450e-1f,    4.9280e-1f,    4.4390e-1f,    4.0430e-1f,    3.7150e-1f,    3.4380e-1f,    3.2010e-1f,    2.9960e-1f,
        2.8170e-1f,    2.6580e-1f,    2.5160e-1f,    2.3900e-1f,    2.2760e-1f,    2.1730e-1f,    1.7750e-1f,    1.5040e-1f,    1.3070e-1f,    1.1570e-1f,
        1.0380e-1f,    9.4300e-2f,    8.6400e-2f,    7.9700e-2f,    6.9200e-2f,    6.1100e-2f,    5.4800e-2f,    4.9700e-2f,    4.5500e-2f,    4.2000e-2f,
        3.8900e-2f,    3.6300e-2f,    3.4100e-2f,    3.2100e-2f,    3.0300e-2f,    2.8700e-2f,    2.7300e-2f,    2.6000e-2f,    2.1100e-2f,    1.7800e-2f,
        1.5400e-2f,    1.3600e-2f,    1.1000e-2f,    1.0000e-2f,    9.2000e-3f,    8.0000e-3f,    7.0000e-3f,    6.3000e-3f,    5.7000e-3f,    5.2000e-3f,
        4.8000e-3f,    4.4000e-3f,    4.1000e-3f,    3.9000e-3f,    3.6000e-3f,    3.4000e-3f,    3.3000e-3f,    3.1000e-3f,    2.9000e-3f,    2.4000e-3f,
        2.0000e-3f,    1.7000e-3f,    1.5000e-3f,    1.4000e-3f,    1.2000e-3f,    1.1000e-3f,    1.0000e-3f,    9.0000e-4f,    8.0000e-4f,    7.0000e-4f,
        6.0000e-4f,    6.0000e-4f,    5.0000e-4f,    5.0000e-4f,    5.0000e-4f,    4.0000e-4f,    4.0000e-4f,    4.0000e-4f,    4.0000e-4f,    3.0000e-4f,
        3.0000e-4f,    2.0000e-4f,    2.0000e-4f,    1.0000e-4f,    1.0000e-4f,    1.0000e-4f,    1.0000e-4f,    1.0000e-4f,    1.0000e-4f,    0.0000e0f,
        0.0000e0f,    0.0000e0f,    1.5780e1f,    1.3780e1f,    1.2310e1f,    1.1180e1f,    1.0270e1f,    8.9000e0f,    7.9000e0f,    7.1310e0f,
        6.5180e0f,    6.0160e0f,    5.5950e0f,    5.2370e0f,    4.5340e0f,    4.0150e0f,    3.6140e0f,    3.2940e0f,    3.0310e0f,    2.8110e0f,
        2.6240e0f,    2.4620e0f,    2.1980e0f,    1.9890e0f,    1.8190e0f,    1.6790e0f,    1.5600e0f,    1.4590e0f,    1.3710e0f,    1.2930e0f,
        1.2250e0f,    1.1640e0f,    1.1100e0f,    1.0610e0f,    1.0160e0f,    9.7510e-1f,    8.1480e-1f,    7.0260e-1f,    6.1930e-1f,    5.5480e-1f,
        5.0320e-1f,    4.6100e-1f,    4.2580e-1f,    3.9590e-1f,    3.4780e-1f,    3.1070e-1f,    2.8120e-1f,    2.5710e-1f,    2.3710e-1f,    2.2010e-1f,
        2.0550e-1f,    1.9280e-1f,    1.8170e-1f,    1.7190e-1f,    1.6310e-1f,    1.5520e-1f,    1.4800e-1f,    1.4160e-1f,    1.1650e-1f,    9.9200e-2f,
        8.6600e-2f,    7.6900e-2f,    6.9300e-2f,    6.3000e-2f,    5.7900e-2f,    5.3600e-2f,    4.6600e-2f,    4.1300e-2f,    3.7200e-2f,    3.3800e-2f,
        3.1000e-2f,    2.8600e-2f,    2.6600e-2f,    2.4800e-2f,    2.3300e-2f,    2.2000e-2f,    2.0800e-2f,    1.9700e-2f,    1.8800e-2f,    1.7900e-2f,
        1.4600e-2f,    1.2300e-2f,    1.0700e-2f,    9.4000e-3f,    7.6000e-3f,    7.0000e-3f,    6.4000e-3f,    5.6000e-3f,    4.9000e-3f,    4.4000e-3f,
        4.0000e-3f,    3.6000e-3f,    3.4000e-3f,    3.1000e-3f,    2.9000e-3f,    2.7000e-3f,    2.6000e-3f,    2.4000e-3f,    2.3000e-3f,    2.2000e-3f,
        2.1000e-3f,    1.7000e-3f,    1.4000e-3f,    1.2000e-3f,    1.1000e-3f,    1.0000e-3f,    9.0000e-4f,    8.0000e-4f,    7.0000e-4f,    6.0000e-4f,
        6.0000e-4f,    5.0000e-4f,    4.0000e-4f,    4.0000e-4f,    4.0000e-4f,    4.0000e-4f,    3.0000e-4f,    3.0000e-4f,    3.0000e-4f,    3.0000e-4f,
        3.0000e-4f,    2.0000e-4f,    2.0000e-4f,    2.0000e-4f,    1.0000e-4f,    1.0000e-4f,    1.0000e-4f,    1.0000e-4f,    1.0000e-4f,    0.0000e0f,
        0.0000e0f,    0.0000e0f,    0.0000e0f,    0.0000e0f,    1.1970e1f,    1.0720e1f,    9.7490e0f,    8.9670e0f,    8.3240e0f,    7.3220e0f,
        6.5710e0f,    5.9820e0f,    5.5050e0f,    5.1100e0f,    4.7750e0f,    4.4880e0f,    3.9170e0f,    3.4910e0f,    3.1570e0f,    2.8890e0f,
        2.6670e0f,    2.4800e0f,    2.3210e0f,    2.1830e0f,    1.9550e0f,    1.7740e0f,    1.6270e0f,    1.5040e0f,    1.4010e0f,    1.3110e0f,
        1.2340e0f,    1.1660e0f,    1.1060e0f,    1.0520e0f,    1.0030e0f,    9.5970e-1f,    9.2000e-1f,    8.8370e-1f,    7.4060e-1f,    6.4000e-1f,
        5.6510e-1f,    5.0700e-1f,    4.6040e-1f,    4.2220e-1f,    3.9030e-1f,    3.6320e-1f,    3.1950e-1f,    2.8580e-1f,    2.5890e-1f,    2.3690e-1f,
        2.1850e-1f,    2.0300e-1f,    1.8970e-1f,    1.7810e-1f,    1.6790e-1f,    1.5890e-1f,    1.5090e-1f,    1.4360e-1f,    1.3710e-1f,    1.3120e-1f,
        1.0820e-1f,    9.2300e-2f,    8.0700e-2f,    7.1700e-2f,    6.4700e-2f,    5.8900e-2f,    5.4100e-2f,    5.0100e-2f,    4.3700e-2f,    3.8800e-2f,
        3.4900e-2f,    3.1700e-2f,    2.9100e-2f,    2.6900e-2f,    2.5000e-2f,    2.3400e-2f,    2.2000e-2f,    2.0800e-2f,    1.9600e-2f,    1.8600e-2f,
        1.7700e-2f,    1.6900e-2f,    1.3800e-2f,    1.1700e-2f,    1.0100e-2f,    8.9000e-3f,    7.3000e-3f,    6.6000e-3f,    6.1000e-3f,    5.3000e-3f,
        4.7000e-3f,    4.2000e-3f,    3.8000e-3f,    3.5000e-3f,    3.2000e-3f,    3.0000e-3f,    2.8000e-3f,    2.6000e-3f,    2.4000e-3f,    2.3000e-3f,
        2.2000e-3f,    2.1000e-3f,    2.0000e-3f,    1.6000e-3f,    1.3000e-3f,    1.2000e-3f,    1.0000e-3f,    9.0000e-4f,    8.0000e-4f,    8.0000e-4f,
        7.0000e-4f,    6.0000e-4f,    5.0000e-4f,    5.0000e-4f,    4.0000e-4f,    4.0000e-4f,    4.0000e-4f,    3.0000e-4f,    3.0000e-4f,    3.0000e-4f,
        3.0000e-4f,    3.0000e-4f,    2.0000e-4f,    2.0000e-4f,    2.0000e-4f,    2.0000e-4f,    1.0000e-4f,    1.0000e-4f,    1.0000e-4f,    1.0000e-4f,
        0.0000e0f,    0.0000e0f,    0.0000e0f,    0.0000e0f,    0.0000e0f,    0.0000e0f,    3.9730e1f,    3.2030e1f,    2.7180e1f,    2.3790e1f,
        2.1250e1f,    1.7660e1f,    1.5220e1f,    1.3440e1f,    1.2070e1f,    1.0980e1f,    1.0090e1f,    9.3420e0f,    7.9240e0f,    6.9100e0f,
        6.1450e0f,    5.5450e0f,    5.0610e0f,    4.6610e0f,    4.3240e0f,    4.0360e0f,    3.5700e0f,    3.2070e0f,    2.9160e0f,    2.6770e0f,
        2.4770e0f,    2.3070e0f,    2.1600e0f,    2.0320e0f,    1.9190e0f,    1.8190e0f,    1.7300e0f,    1.6500e0f,    1.5770e0f,    1.5110e0f,
        1.2530e0f,    1.0740e0f,    9.4190e-1f,    8.4050e-1f,    7.5990e-1f,    6.9420e-1f,    6.3920e-1f,    5.9230e-1f,    5.1740e-1f,    4.6000e-1f,
        4.1460e-1f,    3.7770e-1f,    3.4720e-1f,    3.2130e-1f,    2.9910e-1f,    2.7990e-1f,    2.6310e-1f,    2.4830e-1f,    2.3510e-1f,    2.2330e-1f,
        2.1270e-1f,    2.0300e-1f,    1.6590e-1f,    1.4050e-1f,    1.2210e-1f,    1.0810e-1f,    9.7000e-2f,    8.8000e-2f,    8.0600e-2f,    7.4400e-2f,
        6.4600e-2f,    5.7000e-2f,    5.1100e-2f,    4.6400e-2f,    4.2400e-2f,    3.9100e-2f,    3.6300e-2f,    3.3900e-2f,    3.1800e-2f,    2.9900e-2f,
        2.8200e-2f,    2.6800e-2f,    2.5400e-2f,    2.4200e-2f,    1.9700e-2f,    1.6600e-2f,    1.4300e-2f,    1.2600e-2f,    1.0200e-2f,    9.3000e-3f,
        8.6000e-3f,    7.4000e-3f,    6.5000e-3f,    5.8000e-3f,    5.3000e-3f,    4.8000e-3f,    4.5000e-3f,    4.1000e-3f,    3.8000e-3f,    3.6000e-3f,
        3.4000e-3f,    3.2000e-3f,    3.0000e-3f,    2.9000e-3f,    2.7000e-3f,    2.2000e-3f,    1.9000e-3f,    1.6000e-3f,    1.4000e-3f,    1.3000e-3f,
        1.1000e-3f,    1.0000e-3f,    1.0000e-3f,    8.0000e-4f,    7.0000e-4f,    7.0000e-4f,    6.0000e-4f,    5.0000e-4f,    5.0000e-4f,    5.0000e-4f,
        4.0000e-4f,    4.0000e-4f,    4.0000e-4f,    4.0000e-4f,    3.0000e-4f,    3.0000e-4f,    3.0000e-4f,    2.0000e-4f,    2.0000e-4f,    1.0000e-4f,
        1.0000e-4f,    1.0000e-4f,    1.0000e-4f,    1.0000e-4f,    0.0000e0f,    0.0000e0f,    0.0000e0f,    0.0000e0f,
        3.881e+01f,             3.135e+01f,             2.664e+01f,             2.332e+01f,             2.085e+01f,             1.735e+01f,             1.497e+01f,             1.322e+01f,             1.188e+01f,             1.081e+01f,
        9.935e+00f,             9.205e+00f,             7.813e+00f,             6.817e+00f,             6.064e+00f,             5.474e+00f,             4.997e+00f,             4.604e+00f,             4.271e+00f,             3.988e+00f,
        3.528e+00f,             3.171e+00f,             2.884e+00f,             2.648e+00f,             2.450e+00f,             2.282e+00f,             2.137e+00f,             2.011e+00f,             1.899e+00f,             1.800e+00f,
        1.712e+00f,             1.633e+00f,             1.561e+00f,             1.496e+00f,             1.240e+00f,             1.063e+00f,             9.331e-01f,             8.327e-01f,             7.529e-01f,             6.879e-01f,
        6.335e-01f,             5.870e-01f,             5.128e-01f,             4.560e-01f,             4.110e-01f,             3.746e-01f,             3.443e-01f,             3.186e-01f,             2.967e-01f,             2.776e-01f,
        2.611e-01f,             2.463e-01f,             2.333e-01f,             2.216e-01f,             2.110e-01f,             2.015e-01f,             1.647e-01f,             1.396e-01f,             1.213e-01f,             1.073e-01f,
        9.636e-02f,             8.748e-02f,             8.014e-02f,             7.397e-02f,             6.416e-02f,             5.670e-02f,             5.084e-02f,             4.610e-02f,             4.219e-02f,             3.891e-02f,
        3.611e-02f,             3.369e-02f,             3.159e-02f,             2.974e-02f,             2.810e-02f,             2.663e-02f,             2.532e-02f,             2.413e-02f,             1.956e-02f,             1.648e-02f,
        1.424e-02f,             1.256e-02f,             1.017e-02f,             9.292e-03f,             8.557e-03f,             7.393e-03f,             6.514e-03f,             5.825e-03f,             5.270e-03f,             4.814e-03f,
        4.431e-03f,             4.107e-03f,             3.827e-03f,             3.584e-03f,             3.371e-03f,             3.182e-03f,             3.013e-03f,             2.862e-03f,             2.726e-03f,             2.204e-03f,
        1.853e-03f,             1.600e-03f,             1.409e-03f,             1.259e-03f,             1.139e-03f,             1.041e-03f,             9.579e-04f,             8.273e-04f,             7.286e-04f,             6.514e-04f,
        5.894e-04f,             5.384e-04f,             4.957e-04f,             4.595e-04f,             4.283e-04f,             4.012e-04f,             3.773e-04f,             3.563e-04f,             3.375e-04f,             3.207e-04f,
        3.055e-04f,             2.083e-04f,             1.588e-04f,             1.287e-04f,             1.084e-04f,             8.277e-05f,             6.714e-05f,             5.660e-05f,             4.899e-05f,             4.323e-05f,
        3.871e-05f,             3.508e-05f,
        1.991e+01f,             1.669e+01f,             1.452e+01f,             1.293e+01f,             1.171e+01f,             9.926e+00f,             8.675e+00f,             7.739e+00f,             7.008e+00f,             6.418e+00f,
        5.931e+00f,             5.521e+00f,             4.728e+00f,             4.153e+00f,             3.714e+00f,             3.367e+00f,             3.085e+00f,             2.850e+00f,             2.652e+00f,             2.482e+00f,
        2.204e+00f,             1.987e+00f,             1.812e+00f,             1.667e+00f,             1.546e+00f,             1.442e+00f,             1.353e+00f,             1.275e+00f,             1.206e+00f,             1.144e+00f,
        1.089e+00f,             1.040e+00f,             9.948e-01f,             9.539e-01f,             7.939e-01f,             6.824e-01f,             6.001e-01f,             5.365e-01f,             4.858e-01f,             4.444e-01f,
        4.097e-01f,             3.804e-01f,             3.333e-01f,             2.971e-01f,             2.684e-01f,             2.449e-01f,             2.255e-01f,             2.090e-01f,             1.949e-01f,             1.827e-01f,
        1.720e-01f,             1.625e-01f,             1.540e-01f,             1.464e-01f,             1.396e-01f,             1.334e-01f,             1.094e-01f,             9.290e-02f,             8.089e-02f,             7.172e-02f,
        6.448e-02f,             5.861e-02f,             5.375e-02f,             4.966e-02f,             4.315e-02f,             3.819e-02f,             3.428e-02f,             3.112e-02f,             2.850e-02f,             2.630e-02f,
        2.443e-02f,             2.281e-02f,             2.140e-02f,             2.016e-02f,             1.905e-02f,             1.807e-02f,             1.718e-02f,             1.638e-02f,             1.330e-02f,             1.121e-02f,
        9.706e-03f,             8.561e-03f,             6.939e-03f,             6.343e-03f,             5.843e-03f,             5.051e-03f,             4.452e-03f,             3.983e-03f,             3.604e-03f,             3.293e-03f,
        3.032e-03f,             2.811e-03f,             2.620e-03f,             2.454e-03f,             2.308e-03f,             2.179e-03f,             2.064e-03f,             1.960e-03f,             1.867e-03f,             1.510e-03f,
        1.270e-03f,             1.097e-03f,             9.661e-04f,             8.637e-04f,             7.814e-04f,             7.137e-04f,             6.571e-04f,             5.675e-04f,             4.999e-04f,             4.470e-04f,
        4.045e-04f,             3.695e-04f,             3.403e-04f,             3.154e-04f,             2.940e-04f,             2.754e-04f,             2.591e-04f,             2.446e-04f,             2.318e-04f,             2.202e-04f,
        2.098e-04f,             1.431e-04f,             1.091e-04f,             8.845e-05f,             7.453e-05f,             5.690e-05f,             4.616e-05f,             3.892e-05f,             3.369e-05f,             2.973e-05f,
        2.663e-05f,             2.413e-05f,
		2.163e+01f,		1.840e+01f,		1.614e+01f,		1.446e+01f,		1.314e+01f,		1.120e+01f,		9.825e+00f,		8.786e+00f,		7.970e+00f,		7.310e+00f, 
		6.762e+00f,		6.300e+00f,		5.404e+00f,		4.751e+00f,		4.253e+00f,		3.858e+00f,		3.536e+00f,		3.269e+00f,		3.042e+00f,		2.848e+00f, 
		2.531e+00f,		2.282e+00f,		2.082e+00f,		1.917e+00f,		1.777e+00f,		1.659e+00f,		1.556e+00f,		1.466e+00f,		1.387e+00f,		1.316e+00f, 
		1.253e+00f,		1.196e+00f,		1.145e+00f,		1.098e+00f,		9.142e-01f,		7.861e-01f,		6.914e-01f,		6.183e-01f,		5.600e-01f,		5.124e-01f, 
		4.727e-01f,		4.390e-01f,		3.850e-01f,		3.435e-01f,		3.105e-01f,		2.836e-01f,		2.612e-01f,		2.423e-01f,		2.261e-01f,		2.119e-01f, 
		1.995e-01f,		1.885e-01f,		1.787e-01f,		1.699e-01f,		1.620e-01f,		1.548e-01f,		1.270e-01f,		1.080e-01f,		9.404e-02f,		8.340e-02f, 
		7.500e-02f,		6.818e-02f,		6.254e-02f,		5.778e-02f,		5.021e-02f,		4.444e-02f,		3.989e-02f,		3.621e-02f,		3.317e-02f,		3.061e-02f, 
		2.843e-02f,		2.654e-02f,		2.490e-02f,		2.345e-02f,		2.217e-02f,		2.102e-02f,		1.999e-02f,		1.905e-02f,		1.547e-02f,		1.304e-02f, 
		1.128e-02f,		9.953e-03f,		8.066e-03f,		7.372e-03f,		6.790e-03f,		5.870e-03f,		5.173e-03f,		4.627e-03f,		4.187e-03f,		3.826e-03f, 
		3.523e-03f,		3.265e-03f,		3.043e-03f,		2.850e-03f,		2.681e-03f,		2.531e-03f,		2.397e-03f,		2.277e-03f,		2.169e-03f,		1.754e-03f, 
		1.475e-03f,		1.274e-03f,		1.122e-03f,		1.003e-03f,		9.075e-04f,		8.289e-04f,		7.631e-04f,		6.591e-04f,		5.806e-04f,		5.191e-04f, 
		4.697e-04f,		4.291e-04f,		3.951e-04f,		3.663e-04f,		3.414e-04f,		3.198e-04f,		3.009e-04f,		2.841e-04f,		2.691e-04f,		2.557e-04f, 
		2.436e-04f,		1.661e-04f,		1.267e-04f,		1.027e-04f,		8.655e-05f,		6.608e-05f,		5.362e-05f,		4.520e-05f,		3.913e-05f,		3.453e-05f, 
		3.093e-05f,		2.803e-05f 
    },
    {
        1.7690e2f,    1.9840e2f,    2.1840e2f,    2.3700e2f,    2.5440e2f,    2.8640e2f,    3.1530e2f,    3.4200e2f,    3.6670e2f,    3.9000e2f,
        4.1200e2f,    4.3290e2f,    4.7450e2f,    5.1100e2f,    5.4370e2f,    5.7330e2f,    6.0010e2f,    6.2450e2f,    6.4670e2f,    6.6710e2f,
        7.0280e2f,    7.3240e2f,    7.5690e2f,    7.7680e2f,    7.9270e2f,    8.0500e2f,    8.1420e2f,    8.2050e2f,    8.2430e2f,    8.2600e2f,
        8.2580e2f,    8.2390e2f,    8.2060e2f,    8.1610e2f,    7.8140e2f,    7.3710e2f,    6.9690e2f,    6.6130e2f,    6.2940e2f,    6.0060e2f,
        5.7440e2f,    5.5040e2f,    5.0800e2f,    4.7190e2f,    4.4060e2f,    4.1320e2f,    3.8910e2f,    3.6800e2f,    3.4920e2f,    3.3250e2f,
        3.1750e2f,    3.0390e2f,    2.9170e2f,    2.8050e2f,    2.7020e2f,    2.6080e2f,    2.2290e2f,    1.9570e2f,    1.7490e2f,    1.5860e2f,
        1.4540e2f,    1.3440e2f,    1.2510e2f,    1.1720e2f,    1.0420e2f,    9.4040e1f,    8.5860e1f,    7.9110e1f,    7.3430e1f,    6.8580e1f,
        6.4380e1f,    6.0710e1f,    5.7470e1f,    5.4600e1f,    5.2020e1f,    4.9690e1f,    4.7590e1f,    4.5670e1f,    3.8150e1f,    3.2920e1f,
        2.9050e1f,    2.6070e1f,    2.1750e1f,    2.0130e1f,    1.8760e1f,    1.6560e1f,    1.4880e1f,    1.3540e1f,    1.2450e1f,    1.1540e1f,
        1.0780e1f,    1.0130e1f,    9.5590e0f,    9.0630e0f,    8.6250e0f,    8.2360e0f,    7.8880e0f,    7.5730e0f,    7.2890e0f,    6.1920e0f,
        5.4450e0f,    4.9030e0f,    4.4920e0f,    4.1700e0f,    3.9110e0f,    3.6980e0f,    3.5200e0f,    3.2410e0f,    3.0320e0f,    2.8710e0f,
        2.7430e0f,    2.6400e0f,    2.5560e0f,    2.4850e0f,    2.4260e0f,    2.3760e0f,    2.3330e0f,    2.2960e0f,    2.2640e0f,    2.2360e0f,
        2.2110e0f,    2.0700e0f,    2.0210e0f,    2.0040e0f,    2.0010e0f,    2.0120e0f,    2.0310e0f,    2.0520e0f,    2.0720e0f,    2.0910e0f,
        2.1090e0f,    2.1260e0f,    8.9290e1f,    1.0380e2f,    1.1630e2f,    1.2740e2f,    1.3760e2f,    1.5590e2f,    1.7230e2f,    1.8720e2f,
        2.0100e2f,    2.1390e2f,    2.2610e2f,    2.3770e2f,    2.6070e2f,    2.8090e2f,    2.9890e2f,    3.1520e2f,    3.3000e2f,    3.4360e2f,
        3.5600e2f,    3.6740e2f,    3.8750e2f,    4.0450e2f,    4.1880e2f,    4.3080e2f,    4.4080e2f,    4.4900e2f,    4.5570e2f,    4.6090e2f,
        4.6490e2f,    4.6780e2f,    4.6970e2f,    4.7080e2f,    4.7120e2f,    4.7090e2f,    4.6270e2f,    4.4790e2f,    4.3070e2f,    4.1330e2f,
        3.9700e2f,    3.8190e2f,    3.6790e2f,    3.5490e2f,    3.3140e2f,    3.1090e2f,    2.9280e2f,    2.7680e2f,    2.6250e2f,    2.4970e2f,
        2.3830e2f,    2.2810e2f,    2.1880e2f,    2.1040e2f,    2.0280e2f,    1.9580e2f,    1.8940e2f,    1.8340e2f,    1.5900e2f,    1.4100e2f,
        1.2710e2f,    1.1600e2f,    1.0690e2f,    9.9330e1f,    9.2850e1f,    8.7260e1f,    7.8070e1f,    7.0820e1f,    6.4940e1f,    6.0050e1f,
        5.5930e1f,    5.2380e1f,    4.9310e1f,    4.6610e1f,    4.4220e1f,    4.2080e1f,    4.0170e1f,    3.8440e1f,    3.6860e1f,    3.5430e1f,
        2.9770e1f,    2.5800e1f,    2.2850e1f,    2.0560e1f,    1.7230e1f,    1.5980e1f,    1.4910e1f,    1.3200e1f,    1.1880e1f,    1.0830e1f,
        9.9770e0f,    9.2650e0f,    8.6620e0f,    8.1450e0f,    7.6960e0f,    7.3030e0f,    6.9560e0f,    6.6470e0f,    6.3700e0f,    6.1200e0f,
        5.8930e0f,    5.0180e0f,    4.4210e0f,    3.9870e0f,    3.6570e0f,    3.3980e0f,    3.1900e0f,    3.0190e0f,    2.8760e0f,    2.6510e0f,
        2.4830e0f,    2.3530e0f,    2.2480e0f,    2.1640e0f,    2.0940e0f,    2.0360e0f,    1.9880e0f,    1.9460e0f,    1.9100e0f,    1.8790e0f,
        1.8530e0f,    1.8290e0f,    1.8090e0f,    1.6970e0f,    1.6600e0f,    1.6510e0f,    1.6530e0f,    1.6690e0f,    1.6900e0f,    1.7110e0f,
        1.7310e0f,    1.7490e0f,    1.7660e0f,    1.7820e0f,    1.0430e2f,    1.2390e2f,    1.4040e2f,    1.5500e2f,    1.6830e2f,    1.9210e2f,
        2.1310e2f,    2.3230e2f,    2.4990e2f,    2.6640e2f,    2.8190e2f,    2.9660e2f,    3.2450e2f,    3.4830e2f,    3.6890e2f,    3.8670e2f,
        4.0220e2f,    4.1570e2f,    4.2730e2f,    4.3730e2f,    4.5290e2f,    4.6380e2f,    4.7090e2f,    4.7490e2f,    4.7660e2f,    4.7640e2f,
        4.7490e2f,    4.7240e2f,    4.6910e2f,    4.6530e2f,    4.6110e2f,    4.5670e2f,    4.5220e2f,    4.4770e2f,    4.2530e2f,    4.0510e2f,
        3.8730e2f,    3.7150e2f,    3.5730e2f,    3.4440e2f,    3.3270e2f,    3.2180e2f,    3.0200e2f,    2.8440e2f,    2.6890e2f,    2.5500e2f,
        2.4270e2f,    2.3160e2f,    2.2160e2f,    2.1260e2f,    2.0430e2f,    1.9680e2f,    1.8990e2f,    1.8350e2f,    1.7750e2f,    1.7200e2f,
        1.4950e2f,    1.3280e2f,    1.1990e2f,    1.0950e2f,    1.0100e2f,    9.3830e1f,    8.7750e1f,    8.2500e1f,    7.3880e1f,    6.7070e1f,
        6.1540e1f,    5.6950e1f,    5.3060e1f,    4.9730e1f,    4.6840e1f,    4.4300e1f,    4.2050e1f,    4.0040e1f,    3.8240e1f,    3.6600e1f,
        3.5120e1f,    3.3760e1f,    2.8420e1f,    2.4660e1f,    2.1860e1f,    1.9690e1f,    1.6520e1f,    1.5320e1f,    1.4310e1f,    1.2680e1f,
        1.1420e1f,    1.0410e1f,    9.5940e0f,    8.9110e0f,    8.3340e0f,    7.8380e0f,    7.4080e0f,    7.0310e0f,    6.6980e0f,    6.4010e0f,
        6.1350e0f,    5.8950e0f,    5.6780e0f,    4.8370e0f,    4.2620e0f,    3.8440e0f,    3.5260e0f,    3.2770e0f,    3.0760e0f,    2.9110e0f,
        2.7730e0f,    2.5550e0f,    2.3930e0f,    2.2670e0f,    2.1670e0f,    2.0860e0f,    2.0200e0f,    1.9650e0f,    1.9180e0f,    1.8790e0f,
        1.8450e0f,    1.8160e0f,    1.7910e0f,    1.7690e0f,    1.7500e0f,    1.6470e0f,    1.6180e0f,    1.6130e0f,    1.6190e0f,    1.6420e0f,
        1.6680e0f,    1.6920e0f,    1.7140e0f,    1.7340e0f,    1.7520e0f,    1.7680e0f,    2.1470e2f,    2.4630e2f,    2.7460e2f,    3.0040e2f,
        3.2420e2f,    3.6350e2f,    3.9940e2f,    4.3220e2f,    4.6240e2f,    4.8990e2f,    5.1530e2f,    5.3910e2f,    5.8700e2f,    6.2770e2f,
        6.6320e2f,    6.9480e2f,    7.2290e2f,    7.4790e2f,    7.7000e2f,    7.8980e2f,    8.2440e2f,    8.5370e2f,    8.7790e2f,    8.9720e2f,
        9.1180e2f,    9.2240e2f,    9.2960e2f,    9.3400e2f,    9.3580e2f,    9.3550e2f,    9.3340e2f,    9.2960e2f,    9.2450e2f,    9.1830e2f,
        8.7570e2f,    8.2430e2f,    7.7230e2f,    7.2320e2f,    6.7690e2f,    6.3460e2f,    5.9690e2f,    5.6340e2f,    5.0770e2f,    4.6390e2f,
        4.2890e2f,    4.0060e2f,    3.7700e2f,    3.5640e2f,    3.3830e2f,    3.2220e2f,    3.0780e2f,    2.9480e2f,    2.8300e2f,    2.7220e2f,
        2.6230e2f,    2.5320e2f,    2.1680e2f,    1.9050e2f,    1.7050e2f,    1.5460e2f,    1.4180e2f,    1.3110e2f,    1.2210e2f,    1.1430e2f,
        1.0170e2f,    9.1790e1f,    8.3790e1f,    7.7190e1f,    7.1640e1f,    6.6900e1f,    6.2800e1f,    5.9210e1f,    5.6050e1f,    5.3240e1f,
        5.0730e1f,    4.8450e1f,    4.6400e1f,    4.4520e1f,    3.7190e1f,    3.2080e1f,    2.8310e1f,    2.5390e1f,    2.1180e1f,    1.9610e1f,
        1.8270e1f,    1.6130e1f,    1.4490e1f,    1.3180e1f,    1.2120e1f,    1.1240e1f,    1.0500e1f,    9.8580e0f,    9.3060e0f,    8.8230e0f,
        8.3970e0f,    8.0180e0f,    7.6780e0f,    7.3720e0f,    7.0950e0f,    6.0270e0f,    5.3000e0f,    4.7720e0f,    4.3720e0f,    4.0580e0f,
        3.8060e0f,    3.5990e0f,    3.4260e0f,    3.1540e0f,    2.9510e0f,    2.7940e0f,    2.6700e0f,    2.5690e0f,    2.4870e0f,    2.4180e0f,
        2.3610e0f,    2.3120e0f,    2.2690e0f,    2.2320e0f,    2.1990e0f,    2.1700e0f,    2.1450e0f,    2.0040e0f,    1.9540e0f,    1.9370e0f,
        1.9340e0f,    1.9450e0f,    1.9640e0f,    1.9850e0f,    2.0050e0f,    2.0240e0f,    2.0420e0f,    2.0590e0f,
        2.095e+02f,             2.404e+02f,             2.680e+02f,             2.932e+02f,             3.165e+02f,             3.561e+02f,             3.920e+02f,             4.247e+02f,             4.551e+02f,             4.830e+02f,
        5.088e+02f,             5.333e+02f,             5.817e+02f,             6.231e+02f,             6.595e+02f,             6.917e+02f,             7.206e+02f,             7.462e+02f,             7.691e+02f,             7.896e+02f,
        8.252e+02f,             8.548e+02f,             8.790e+02f,             8.981e+02f,             9.126e+02f,             9.231e+02f,             9.303e+02f,             9.345e+02f,             9.363e+02f,             9.359e+02f,
        9.336e+02f,             9.297e+02f,             9.246e+02f,             9.183e+02f,             8.753e+02f,             8.237e+02f,             7.715e+02f,             7.224e+02f,             6.773e+02f,             6.369e+02f,
        6.011e+02f,             5.694e+02f,             5.172e+02f,             4.754e+02f,             4.412e+02f,             4.123e+02f,             3.875e+02f,             3.659e+02f,             3.469e+02f,             3.301e+02f,
        3.150e+02f,             3.015e+02f,             2.892e+02f,             2.780e+02f,             2.677e+02f,             2.584e+02f,             2.207e+02f,             1.937e+02f,             1.731e+02f,             1.570e+02f,
        1.438e+02f,             1.330e+02f,             1.237e+02f,             1.158e+02f,             1.030e+02f,             9.289e+01f,             8.478e+01f,             7.808e+01f,             7.245e+01f,             6.764e+01f,
        6.348e+01f,             5.985e+01f,             5.664e+01f,             5.379e+01f,             5.125e+01f,             4.895e+01f,             4.687e+01f,             4.496e+01f,             3.755e+01f,             3.238e+01f,
        2.856e+01f,             2.562e+01f,             2.136e+01f,             1.977e+01f,             1.842e+01f,             1.626e+01f,             1.460e+01f,             1.328e+01f,             1.221e+01f,             1.132e+01f,
        1.057e+01f,             9.929e+00f,             9.372e+00f,             8.886e+00f,             8.455e+00f,             8.073e+00f,             7.730e+00f,             7.422e+00f,             7.142e+00f,             6.065e+00f,
        5.331e+00f,             4.800e+00f,             4.396e+00f,             4.080e+00f,             3.825e+00f,             3.616e+00f,             3.441e+00f,             3.167e+00f,             2.961e+00f,             2.802e+00f,
        2.677e+00f,             2.575e+00f,             2.491e+00f,             2.421e+00f,             2.362e+00f,             2.313e+00f,             2.270e+00f,             2.233e+00f,             2.200e+00f,             2.173e+00f,
        2.148e+00f,             2.016e+00f,             1.976e+00f,             1.968e+00f,             1.973e+00f,             2.000e+00f,             2.032e+00f,             2.064e+00f,             2.094e+00f,             2.122e+00f,
        2.149e+00f,             2.173e+00f,
        1.008e+02f,             1.157e+02f,             1.289e+02f,             1.408e+02f,             1.518e+02f,             1.717e+02f,             1.895e+02f,             2.058e+02f,             2.210e+02f,             2.352e+02f,
         2.485e+02f,             2.613e+02f,             2.866e+02f,             3.090e+02f,             3.291e+02f,             3.474e+02f,             3.640e+02f,             3.793e+02f,             3.934e+02f,             4.065e+02f,
         4.298e+02f,             4.498e+02f,             4.671e+02f,             4.820e+02f,             4.947e+02f,             5.055e+02f,             5.147e+02f,             5.223e+02f,             5.286e+02f,             5.337e+02f,
         5.376e+02f,             5.406e+02f,             5.427e+02f,             5.440e+02f,             5.411e+02f,             5.281e+02f,             5.098e+02f,             4.893e+02f,             4.679e+02f,             4.469e+02f,
         4.273e+02f,             4.091e+02f,             3.773e+02f,             3.507e+02f,             3.283e+02f,             3.093e+02f,             2.930e+02f,             2.783e+02f,             2.652e+02f,             2.534e+02f,
         2.427e+02f,             2.329e+02f,             2.240e+02f,             2.158e+02f,             2.083e+02f,             2.014e+02f,             1.732e+02f,             1.526e+02f,             1.369e+02f,             1.244e+02f,
         1.142e+02f,             1.058e+02f,             9.859e+01f,             9.243e+01f,             8.237e+01f,             7.447e+01f,             6.807e+01f,             6.279e+01f,             5.834e+01f,             5.453e+01f,
         5.124e+01f,             4.836e+01f,             4.581e+01f,             4.354e+01f,             4.151e+01f,             3.967e+01f,             3.801e+01f,             3.649e+01f,             3.054e+01f,             2.639e+01f,
         2.331e+01f,             2.093e+01f,             1.749e+01f,             1.620e+01f,             1.510e+01f,             1.334e+01f,             1.199e+01f,             1.092e+01f,             1.005e+01f,             9.320e+00f,
         8.707e+00f,             8.181e+00f,             7.725e+00f,             7.327e+00f,             6.975e+00f,             6.661e+00f,             6.381e+00f,             6.128e+00f,             5.899e+00f,             5.015e+00f,
         4.412e+00f,             3.975e+00f,             3.643e+00f,             3.383e+00f,             3.174e+00f,             3.002e+00f,             2.858e+00f,             2.633e+00f,             2.464e+00f,             2.333e+00f,
         2.229e+00f,             2.144e+00f,             2.075e+00f,             2.017e+00f,             1.968e+00f,             1.927e+00f,             1.891e+00f,             1.860e+00f,             1.833e+00f,             1.810e+00f,
         1.789e+00f,             1.675e+00f,             1.635e+00f,             1.622e+00f,             1.621e+00f,             1.631e+00f,             1.648e+00f,             1.665e+00f,             1.682e+00f,             1.697e+00f,
         1.712e+00f,             1.726e+00f,
		1.414e+02f,		1.651e+02f,		1.855e+02f,		2.038e+02f,		2.206e+02f,		2.507e+02f,		2.776e+02f,		3.021e+02f,		3.248e+02f,		3.460e+02f, 
		3.660e+02f,		3.850e+02f,		4.224e+02f,		4.552e+02f,		4.843e+02f,		5.106e+02f,		5.343e+02f,		5.558e+02f,		5.755e+02f,		5.934e+02f, 
		6.246e+02f,		6.506e+02f,		6.721e+02f,		6.897e+02f,		7.038e+02f,		7.149e+02f,		7.233e+02f,		7.293e+02f,		7.333e+02f,		7.355e+02f, 
		7.360e+02f,		7.352e+02f,		7.332e+02f,		7.301e+02f,		7.038e+02f,		6.680e+02f,		6.298e+02f,		5.928e+02f,		5.589e+02f,		5.284e+02f, 
		5.011e+02f,		4.767e+02f,		4.353e+02f,		4.015e+02f,		3.736e+02f,		3.501e+02f,		3.300e+02f,		3.123e+02f,		2.967e+02f,		2.826e+02f, 
		2.701e+02f,		2.589e+02f,		2.486e+02f,		2.393e+02f,		2.308e+02f,		2.229e+02f,		1.912e+02f,		1.683e+02f,		1.509e+02f,		1.371e+02f, 
		1.258e+02f,		1.165e+02f,		1.086e+02f,		1.018e+02f,		9.068e+01f,		8.197e+01f,		7.492e+01f,		6.909e+01f,		6.417e+01f,		5.997e+01f, 
		5.633e+01f,		5.315e+01f,		5.033e+01f,		4.783e+01f,		4.559e+01f,		4.357e+01f,		4.173e+01f,		4.006e+01f,		3.351e+01f,		2.894e+01f, 
		2.555e+01f,		2.294e+01f,		1.915e+01f,		1.773e+01f,		1.653e+01f,		1.460e+01f,		1.312e+01f,		1.194e+01f,		1.099e+01f,		1.019e+01f, 
		9.517e+00f,		8.942e+00f,		8.443e+00f,		8.006e+00f,		7.620e+00f,		7.277e+00f,		6.970e+00f,		6.693e+00f,		6.443e+00f,		5.475e+00f, 
		4.816e+00f,		4.338e+00f,		3.976e+00f,		3.691e+00f,		3.462e+00f,		3.275e+00f,		3.118e+00f,		2.871e+00f,		2.687e+00f,		2.544e+00f, 
		2.431e+00f,		2.340e+00f,		2.266e+00f,		2.203e+00f,		2.151e+00f,		2.107e+00f,		2.069e+00f,		2.037e+00f,		2.008e+00f,		1.984e+00f, 
		1.963e+00f,		1.850e+00f,		1.820e+00f,		1.818e+00f,		1.828e+00f,		1.861e+00f,		1.898e+00f,		1.934e+00f,		1.967e+00f,		1.998e+00f, 
		2.026e+00f,		2.052e+00f 
   },
    {
        0.0000e0f,    0.0000e0f,    0.0000e0f,    0.0000e0f,    0.0000e0f,    0.0000e0f,    0.0000e0f,    0.0000e0f,    0.0000e0f,    0.0000e0f,
        0.0000e0f,    0.0000e0f,    0.0000e0f,    0.0000e0f,    1.0000e-4f,    1.0000e-4f,    1.0000e-4f,    1.0000e-4f,    1.0000e-4f,    1.0000e-4f,
        1.0000e-4f,    1.0000e-4f,    1.0000e-4f,    1.0000e-4f,    1.0000e-4f,    1.0000e-4f,    1.0000e-4f,    1.0000e-4f,    1.0000e-4f,    1.0000e-4f,
        1.0000e-4f,    1.0000e-4f,    2.0000e-4f,    2.0000e-4f,    2.0000e-4f,    2.0000e-4f,    3.0000e-4f,    3.0000e-4f,    3.0000e-4f,    4.0000e-4f,
        4.0000e-4f,    5.0000e-4f,    6.0000e-4f,    7.0000e-4f,    8.0000e-4f,    9.0000e-4f,    1.0000e-3f,    1.1000e-3f,    1.3000e-3f,    1.4000e-3f,
        1.6000e-3f,    1.7000e-3f,    1.9000e-3f,    2.1000e-3f,    2.3000e-3f,    2.5000e-3f,    3.5000e-3f,    4.7000e-3f,    6.1000e-3f,    7.6000e-3f,
        9.2000e-3f,    1.1000e-2f,    1.2900e-2f,    1.5000e-2f,    1.9500e-2f,    2.4600e-2f,    3.0200e-2f,    3.6200e-2f,    4.2800e-2f,    4.9800e-2f,
        5.7400e-2f,    6.5400e-2f,    7.3800e-2f,    8.2800e-2f,    9.2200e-2f,    1.0200e-1f,    1.1230e-1f,    1.2300e-1f,    1.8320e-1f,    2.5390e-1f,
        3.3500e-1f,    4.2600e-1f,    6.3700e-1f,    7.5660e-1f,    8.8530e-1f,    1.1700e0f,    1.4890e0f,    1.8410e0f,    2.2270e0f,    2.6440e0f,
        3.0930e0f,    3.5720e0f,    4.0800e0f,    4.6180e0f,    5.1840e0f,    5.7770e0f,    6.3980e0f,    7.0450e0f,    7.7180e0f,    1.1460e1f,
        1.5770e1f,    2.0620e1f,    2.5960e1f,    3.1740e1f,    3.7940e1f,    4.4520e1f,    5.1450e1f,    6.6280e1f,    8.2250e1f,    9.9210e1f,
        1.1700e2f,    1.3560e2f,    1.5490e2f,    1.7470e2f,    1.9510e2f,    2.1590e2f,    2.3720e2f,    2.5880e2f,    2.8070e2f,    3.0290e2f,
        3.2540e2f,    5.6050e2f,    8.0540e2f,    1.0540e3f,    1.3040e3f,    1.8020e3f,    2.2970e3f,    2.7870e3f,    3.2720e3f,    3.7520e3f,
        4.2280e3f,    4.7000e3f,    0.0000e0f,    0.0000e0f,    0.0000e0f,    0.0000e0f,    0.0000e0f,    0.0000e0f,    0.0000e0f,    1.0000e-4f,
        1.0000e-4f,    1.0000e-4f,    1.0000e-4f,    1.0000e-4f,    1.0000e-4f,    1.0000e-4f,    1.0000e-4f,    1.0000e-4f,    1.0000e-4f,    1.0000e-4f,
        1.0000e-4f,    1.0000e-4f,    1.0000e-4f,    2.0000e-4f,    2.0000e-4f,    2.0000e-4f,    2.0000e-4f,    2.0000e-4f,    2.0000e-4f,    2.0000e-4f,
        2.0000e-4f,    3.0000e-4f,    3.0000e-4f,    3.0000e-4f,    3.0000e-4f,    3.0000e-4f,    3.0000e-4f,    4.0000e-4f,    5.0000e-4f,    5.0000e-4f,
        6.0000e-4f,    6.0000e-4f,    7.0000e-4f,    8.0000e-4f,    9.0000e-4f,    1.1000e-3f,    1.2000e-3f,    1.4000e-3f,    1.6000e-3f,    1.8000e-3f,
        2.0000e-3f,    2.2000e-3f,    2.4000e-3f,    2.7000e-3f,    2.9000e-3f,    3.2000e-3f,    3.4000e-3f,    3.7000e-3f,    5.2000e-3f,    6.8000e-3f,
        8.7000e-3f,    1.0800e-2f,    1.3000e-2f,    1.5400e-2f,    1.8000e-2f,    2.0800e-2f,    2.6900e-2f,    3.3600e-2f,    4.1000e-2f,    4.9000e-2f,
        5.7700e-2f,    6.6900e-2f,    7.6800e-2f,    8.7200e-2f,    9.8200e-2f,    1.0980e-1f,    1.2200e-1f,    1.3470e-1f,    1.4800e-1f,    1.6180e-1f,
        2.3910e-1f,    3.2960e-1f,    4.3280e-1f,    5.4830e-1f,    8.1510e-1f,    9.6590e-1f,    1.1280e0f,    1.4850e0f,    1.8850e0f,    2.3260e0f,
        2.8080e0f,    3.3280e0f,    3.8870e0f,    4.4820e0f,    5.1140e0f,    5.7810e0f,    6.4830e0f,    7.2190e0f,    7.9870e0f,    8.7890e0f,
        9.6210e0f,    1.4240e1f,    1.9560e1f,    2.5530e1f,    3.2090e1f,    3.9190e1f,    4.6790e1f,    5.4850e1f,    6.3340e1f,    8.1480e1f,
        1.0100e2f,    1.2170e2f,    1.4350e2f,    1.6610e2f,    1.8960e2f,    2.1390e2f,    2.3870e2f,    2.6420e2f,    2.9010e2f,    3.1650e2f,
        3.4330e2f,    3.7040e2f,    3.9790e2f,    6.8500e2f,    9.8350e2f,    1.2860e3f,    1.5890e3f,    2.1910e3f,    2.7860e3f,    3.3740e3f,
        3.9550e3f,    4.5300e3f,    5.0990e3f,    5.6630e3f,    0.0000e0f,    0.0000e0f,    0.0000e0f,    0.0000e0f,    0.0000e0f,    0.0000e0f,
        0.0000e0f,    0.0000e0f,    0.0000e0f,    1.0000e-4f,    1.0000e-4f,    1.0000e-4f,    1.0000e-4f,    1.0000e-4f,    1.0000e-4f,    1.0000e-4f,
        1.0000e-4f,    1.0000e-4f,    1.0000e-4f,    1.0000e-4f,    1.0000e-4f,    1.0000e-4f,    1.0000e-4f,    2.0000e-4f,    2.0000e-4f,    2.0000e-4f,
        2.0000e-4f,    2.0000e-4f,    2.0000e-4f,    2.0000e-4f,    2.0000e-4f,    2.0000e-4f,    3.0000e-4f,    3.0000e-4f,    3.0000e-4f,    4.0000e-4f,
        4.0000e-4f,    5.0000e-4f,    6.0000e-4f,    6.0000e-4f,    7.0000e-4f,    8.0000e-4f,    1.0000e-3f,    1.1000e-3f,    1.3000e-3f,    1.5000e-3f,
        1.7000e-3f,    1.9000e-3f,    2.1000e-3f,    2.4000e-3f,    2.6000e-3f,    2.9000e-3f,    3.1000e-3f,    3.4000e-3f,    3.7000e-3f,    3.9000e-3f,
        5.5000e-3f,    7.3000e-3f,    9.3000e-3f,    1.1500e-2f,    1.3800e-2f,    1.6400e-2f,    1.9200e-2f,    2.2100e-2f,    2.8500e-2f,    3.5600e-2f,
        4.3400e-2f,    5.1900e-2f,    6.1000e-2f,    7.0700e-2f,    8.1100e-2f,    9.2100e-2f,    1.0370e-1f,    1.1580e-1f,    1.2860e-1f,    1.4200e-1f,
        1.5590e-1f,    1.7050e-1f,    2.5150e-1f,    3.4620e-1f,    4.5410e-1f,    5.7480e-1f,    8.5330e-1f,    1.0110e0f,    1.1800e0f,    1.5520e0f,
        1.9680e0f,    2.4270e0f,    2.9280e0f,    3.4690e0f,    4.0500e0f,    4.6690e0f,    5.3250e0f,    6.0180e0f,    6.7470e0f,    7.5110e0f,
        8.3090e0f,    9.1410e0f,    1.0010e1f,    1.4800e1f,    2.0320e1f,    2.6510e1f,    3.3310e1f,    4.0670e1f,    4.8550e1f,    5.6910e1f,
        6.5720e1f,    8.4540e1f,    1.0480e2f,    1.2630e2f,    1.4890e2f,    1.7240e2f,    1.9680e2f,    2.2190e2f,    2.4760e2f,    2.7400e2f,
        3.0080e2f,    3.2820e2f,    3.5590e2f,    3.8400e2f,    4.1240e2f,    7.0870e2f,    1.0160e3f,    1.3250e3f,    1.6350e3f,    2.2480e3f,
        2.8530e3f,    3.4480e3f,    4.0350e3f,    4.6150e3f,    5.1890e3f,    5.7570e3f,    0.0000e0f,    0.0000e0f,    0.0000e0f,    0.0000e0f,
        0.0000e0f,    0.0000e0f,    0.0000e0f,    0.0000e0f,    0.0000e0f,    0.0000e0f,    0.0000e0f,    0.0000e0f,    0.0000e0f,    0.0000e0f,
        0.0000e0f,    0.0000e0f,    0.0000e0f,    1.0000e-4f,    1.0000e-4f,    1.0000e-4f,    1.0000e-4f,    1.0000e-4f,    1.0000e-4f,    1.0000e-4f,
        1.0000e-4f,    1.0000e-4f,    1.0000e-4f,    1.0000e-4f,    1.0000e-4f,    1.0000e-4f,    1.0000e-4f,    1.0000e-4f,    1.0000e-4f,    1.0000e-4f,
        2.0000e-4f,    2.0000e-4f,    2.0000e-4f,    3.0000e-4f,    3.0000e-4f,    3.0000e-4f,    4.0000e-4f,    4.0000e-4f,    5.0000e-4f,    6.0000e-4f,
        7.0000e-4f,    8.0000e-4f,    1.0000e-3f,    1.1000e-3f,    1.3000e-3f,    1.4000e-3f,    1.6000e-3f,    1.7000e-3f,    1.9000e-3f,    2.1000e-3f,
        2.3000e-3f,    2.5000e-3f,    3.5000e-3f,    4.8000e-3f,    6.2000e-3f,    7.7000e-3f,    9.4000e-3f,    1.1200e-2f,    1.3200e-2f,    1.5300e-2f,
        2.0000e-2f,    2.5200e-2f,    3.0900e-2f,    3.7100e-2f,    4.3800e-2f,    5.1000e-2f,    5.8800e-2f,    6.7000e-2f,    7.5600e-2f,    8.4800e-2f,
        9.4400e-2f,    1.0450e-1f,    1.1510e-1f,    1.2610e-1f,    1.8780e-1f,    2.6040e-1f,    3.4350e-1f,    4.3690e-1f,    6.5350e-1f,    7.7630e-1f,
        9.0850e-1f,    1.2000e0f,    1.5280e0f,    1.8900e0f,    2.2860e0f,    2.7150e0f,    3.1760e0f,    3.6680e0f,    4.1900e0f,    4.7420e0f,
        5.3230e0f,    5.9330e0f,    6.5700e0f,    7.2350e0f,    7.9270e0f,    1.1770e1f,    1.6200e1f,    2.1190e1f,    2.6670e1f,    3.2610e1f,
        3.8980e1f,    4.5740e1f,    5.2860e1f,    6.8100e1f,    8.4510e1f,    1.0190e2f,    1.2030e2f,    1.3940e2f,    1.5920e2f,    1.7960e2f,
        2.0050e2f,    2.2190e2f,    2.4370e2f,    2.6600e2f,    2.8850e2f,    3.1140e2f,    3.3460e2f,    5.7720e2f,    8.3030e2f,    1.0880e3f,
        1.3460e3f,    1.8620e3f,    2.3740e3f,    2.8800e3f,    3.3810e3f,    3.8770e3f,    4.3690e3f,    4.8570e3f,
        9.096e-06f,             1.213e-05f,             1.481e-05f,             1.722e-05f,             1.944e-05f,             2.345e-05f,             2.706e-05f,             3.037e-05f,             3.343e-05f,             3.631e-05f,
        3.903e-05f,             4.163e-05f,             4.769e-05f,             5.330e-05f,             5.856e-05f,             6.354e-05f,             6.831e-05f,             7.288e-05f,             7.730e-05f,             8.161e-05f,
        8.983e-05f,             9.772e-05f,             1.053e-04f,             1.127e-04f,             1.199e-04f,             1.269e-04f,             1.339e-04f,             1.407e-04f,             1.475e-04f,             1.543e-04f,
        1.610e-04f,             1.677e-04f,             1.744e-04f,             1.812e-04f,             2.153e-04f,             2.509e-04f,             2.884e-04f,             3.280e-04f,             3.699e-04f,             4.142e-04f,
        4.611e-04f,             5.103e-04f,             6.157e-04f,             7.303e-04f,             8.534e-04f,             9.853e-04f,             1.125e-03f,             1.273e-03f,             1.429e-03f,             1.594e-03f,
        1.766e-03f,             1.946e-03f,             2.132e-03f,             2.327e-03f,             2.529e-03f,             2.739e-03f,             3.892e-03f,             5.218e-03f,             6.710e-03f,             8.362e-03f,
        1.017e-02f,             1.213e-02f,             1.425e-02f,             1.651e-02f,             2.146e-02f,             2.698e-02f,             3.305e-02f,             3.966e-02f,             4.681e-02f,             5.448e-02f,
        6.267e-02f,             7.137e-02f,             8.057e-02f,             9.027e-02f,             1.005e-01f,             1.112e-01f,             1.223e-01f,             1.340e-01f,             1.992e-01f,             2.759e-01f,
        3.637e-01f,             4.623e-01f,             6.907e-01f,             8.201e-01f,             9.595e-01f,             1.267e+00f,             1.612e+00f,             1.993e+00f,             2.410e+00f,             2.861e+00f,
        3.346e+00f,             3.863e+00f,             4.413e+00f,             4.993e+00f,             5.605e+00f,             6.245e+00f,             6.916e+00f,             7.614e+00f,             8.341e+00f,             1.238e+01f,
        1.704e+01f,             2.228e+01f,             2.803e+01f,             3.428e+01f,             4.097e+01f,             4.807e+01f,             5.556e+01f,             7.158e+01f,             8.884e+01f,             1.072e+02f,
        1.265e+02f,             1.466e+02f,             1.674e+02f,             1.889e+02f,             2.109e+02f,             2.335e+02f,             2.565e+02f,             2.799e+02f,             3.037e+02f,             3.278e+02f,
        3.522e+02f,             6.070e+02f,             8.716e+02f,             1.139e+03f,             1.407e+03f,             1.938e+03f,             2.461e+03f,             2.976e+03f,             3.484e+03f,             3.984e+03f,
        4.479e+03f,             4.967e+03f,
        1.271e-05f,             1.730e-05f,             2.139e-05f,             2.510e-05f,             2.852e-05f,             3.470e-05f,             4.024e-05f,             4.530e-05f,             4.999e-05f,             5.437e-05f,
        5.851e-05f,             6.243e-05f,             7.154e-05f,             7.994e-05f,             8.777e-05f,             9.516e-05f,             1.022e-04f,             1.089e-04f,             1.154e-04f,             1.216e-04f,
        1.336e-04f,             1.450e-04f,             1.559e-04f,             1.664e-04f,             1.766e-04f,             1.866e-04f,             1.964e-04f,             2.061e-04f,             2.156e-04f,             2.250e-04f,
        2.343e-04f,             2.436e-04f,             2.528e-04f,             2.620e-04f,             3.080e-04f,             3.547e-04f,             4.029e-04f,             4.529e-04f,             5.052e-04f,             5.598e-04f,
        6.171e-04f,             6.769e-04f,             8.043e-04f,             9.418e-04f,             1.089e-03f,             1.246e-03f,             1.412e-03f,             1.588e-03f,             1.772e-03f,             1.965e-03f,
        2.166e-03f,             2.377e-03f,             2.596e-03f,             2.823e-03f,             3.059e-03f,             3.303e-03f,             4.646e-03f,             6.187e-03f,             7.920e-03f,             9.838e-03f,
        1.194e-02f,             1.421e-02f,             1.666e-02f,             1.928e-02f,             2.503e-02f,             3.142e-02f,             3.845e-02f,             4.611e-02f,             5.437e-02f,             6.324e-02f,
        7.271e-02f,             8.276e-02f,             9.338e-02f,             1.046e-01f,             1.163e-01f,             1.287e-01f,             1.416e-01f,             1.550e-01f,             2.302e-01f,             3.185e-01f,
        4.196e-01f,             5.329e-01f,             7.955e-01f,             9.441e-01f,             1.104e+00f,             1.457e+00f,             1.853e+00f,             2.290e+00f,             2.768e+00f,             3.285e+00f,
        3.841e+00f,             4.434e+00f,             5.063e+00f,             5.728e+00f,             6.428e+00f,             7.161e+00f,             7.929e+00f,             8.728e+00f,             9.560e+00f,             1.418e+01f,
        1.951e+01f,             2.549e+01f,             3.207e+01f,             3.920e+01f,             4.684e+01f,             5.494e+01f,             6.348e+01f,             8.174e+01f,             1.014e+02f,             1.223e+02f,
        1.442e+02f,             1.671e+02f,             1.908e+02f,             2.153e+02f,             2.404e+02f,             2.660e+02f,             2.922e+02f,             3.189e+02f,             3.460e+02f,             3.734e+02f,
        4.012e+02f,             6.918e+02f,             9.945e+02f,             1.302e+03f,             1.610e+03f,             2.226e+03f,             2.836e+03f,             3.439e+03f,             4.037e+03f,             4.629e+03f,
        5.215e+03f,             5.797e+03f,
		9.857e-06f,		1.310e-05f,		1.595e-05f,		1.852e-05f,		2.088e-05f,		2.512e-05f,		2.891e-05f,		3.236e-05f,		3.555e-05f,		3.853e-05f, 
		4.134e-05f,		4.400e-05f,		5.019e-05f,		5.588e-05f,		6.120e-05f,		6.623e-05f,		7.101e-05f,		7.560e-05f,		8.002e-05f,		8.430e-05f, 
		9.250e-05f,		1.003e-04f,		1.079e-04f,		1.152e-04f,		1.224e-04f,		1.295e-04f,		1.364e-04f,		1.433e-04f,		1.501e-04f,		1.569e-04f, 
		1.637e-04f,		1.705e-04f,		1.773e-04f,		1.842e-04f,		2.190e-04f,		2.554e-04f,		2.940e-04f,		3.349e-04f,		3.783e-04f,		4.244e-04f, 
		4.730e-04f,		5.241e-04f,		6.340e-04f,		7.538e-04f,		8.830e-04f,		1.021e-03f,		1.169e-03f,		1.324e-03f,		1.489e-03f,		1.661e-03f, 
		1.842e-03f,		2.032e-03f,		2.229e-03f,		2.434e-03f,		2.646e-03f,		2.867e-03f,		4.082e-03f,		5.479e-03f,		7.051e-03f,		8.792e-03f, 
		1.070e-02f,		1.276e-02f,		1.499e-02f,		1.737e-02f,		2.258e-02f,		2.839e-02f,		3.478e-02f,		4.173e-02f,		4.925e-02f,		5.731e-02f, 
		6.592e-02f,		7.506e-02f,		8.474e-02f,		9.493e-02f,		1.056e-01f,		1.169e-01f,		1.286e-01f,		1.408e-01f,		2.094e-01f,		2.899e-01f, 
		3.820e-01f,		4.855e-01f,		7.252e-01f,		8.609e-01f,		1.007e+00f,		1.330e+00f,		1.691e+00f,		2.091e+00f,		2.528e+00f,		3.001e+00f, 
		3.509e+00f,		4.052e+00f,		4.628e+00f,		5.236e+00f,		5.876e+00f,		6.548e+00f,		7.250e+00f,		7.983e+00f,		8.744e+00f,		1.297e+01f, 
		1.786e+01f,		2.334e+01f,		2.937e+01f,		3.590e+01f,		4.290e+01f,		5.033e+01f,		5.816e+01f,		7.490e+01f,		9.293e+01f,		1.121e+02f, 
		1.322e+02f,		1.532e+02f,		1.749e+02f,		1.973e+02f,		2.203e+02f,		2.437e+02f,		2.677e+02f,		2.921e+02f,		3.168e+02f,		3.418e+02f, 
		3.672e+02f,		6.311e+02f,		9.041e+02f,		1.179e+03f,		1.454e+03f,		1.996e+03f,		2.528e+03f,		3.050e+03f,		3.563e+03f,		4.067e+03f, 
		4.564e+03f,		5.054e+03f 
    },
    {
        0.0000e0f,    0.0000e0f,    0.0000e0f,    0.0000e0f,    0.0000e0f,    0.0000e0f,    0.0000e0f,    0.0000e0f,    0.0000e0f,    0.0000e0f,
        0.0000e0f,    0.0000e0f,    0.0000e0f,    0.0000e0f,    0.0000e0f,    0.0000e0f,    0.0000e0f,    1.0000e-4f,    1.0000e-4f,    1.0000e-4f,
        1.0000e-4f,    1.0000e-4f,    1.0000e-4f,    1.0000e-4f,    1.0000e-4f,    1.0000e-4f,    1.0000e-4f,    1.0000e-4f,    1.0000e-4f,    1.0000e-4f,
        1.0000e-4f,    1.0000e-4f,    1.0000e-4f,    1.0000e-4f,    2.0000e-4f,    2.0000e-4f,    2.0000e-4f,    3.0000e-4f,    3.0000e-4f,    4.0000e-4f,
        4.0000e-4f,    4.0000e-4f,    5.0000e-4f,    6.0000e-4f,    8.0000e-4f,    9.0000e-4f,    1.0000e-3f,    1.1000e-3f,    1.3000e-3f,    1.4000e-3f,
        1.6000e-3f,    1.7000e-3f,    1.9000e-3f,    2.1000e-3f,    2.2000e-3f,    2.4000e-3f,    3.5000e-3f,    4.7000e-3f,    6.0000e-3f,    7.5000e-3f,
        9.2000e-3f,    1.1000e-2f,    1.2900e-2f,    1.4900e-2f,    1.9500e-2f,    2.4500e-2f,    3.0100e-2f,    3.6100e-2f,    4.2700e-2f,    4.9700e-2f,
        5.7200e-2f,    6.5200e-2f,    7.3700e-2f,    8.2600e-2f,    9.2000e-2f,    1.0180e-1f,    1.1200e-1f,    1.2280e-1f,    1.8280e-1f,    2.5350e-1f,
        3.3440e-1f,    4.2520e-1f,    6.3590e-1f,    7.5530e-1f,    8.8390e-1f,    1.1680e0f,    1.4860e0f,    1.8390e0f,    2.2240e0f,    2.6410e0f,
        3.0890e0f,    3.5670e0f,    4.0750e0f,    4.6110e0f,    5.1760e0f,    5.7690e0f,    6.3890e0f,    7.0350e0f,    7.7070e0f,    1.1440e1f,
        1.5760e1f,    2.0600e1f,    2.5930e1f,    3.1710e1f,    3.7900e1f,    4.4470e1f,    5.1390e1f,    6.6210e1f,    8.2170e1f,    9.9120e1f,
        1.1690e2f,    1.3550e2f,    1.5470e2f,    1.7460e2f,    1.9490e2f,    2.1580e2f,    2.3700e2f,    2.5860e2f,    2.8050e2f,    3.0270e2f,
        3.2520e2f,    5.6010e2f,    8.0490e2f,    1.0530e3f,    1.3030e3f,    1.8020e3f,    2.2960e3f,    2.7860e3f,    3.2710e3f,    3.7510e3f,
        4.2270e3f,    4.6990e3f,    0.0000e0f,    0.0000e0f,    0.0000e0f,    0.0000e0f,    0.0000e0f,    0.0000e0f,    0.0000e0f,    0.0000e0f,
        0.0000e0f,    0.0000e0f,    0.0000e0f,    0.0000e0f,    0.0000e0f,    1.0000e-4f,    1.0000e-4f,    1.0000e-4f,    1.0000e-4f,    1.0000e-4f,
        1.0000e-4f,    1.0000e-4f,    1.0000e-4f,    1.0000e-4f,    1.0000e-4f,    1.0000e-4f,    2.0000e-4f,    2.0000e-4f,    2.0000e-4f,    2.0000e-4f,
        2.0000e-4f,    2.0000e-4f,    2.0000e-4f,    2.0000e-4f,    2.0000e-4f,    2.0000e-4f,    3.0000e-4f,    4.0000e-4f,    4.0000e-4f,    5.0000e-4f,
        5.0000e-4f,    6.0000e-4f,    7.0000e-4f,    7.0000e-4f,    9.0000e-4f,    1.0000e-3f,    1.2000e-3f,    1.4000e-3f,    1.5000e-3f,    1.7000e-3f,
        1.9000e-3f,    2.2000e-3f,    2.4000e-3f,    2.6000e-3f,    2.9000e-3f,    3.1000e-3f,    3.4000e-3f,    3.6000e-3f,    5.1000e-3f,    6.8000e-3f,
        8.6000e-3f,    1.0700e-2f,    1.2900e-2f,    1.5300e-2f,    1.7900e-2f,    2.0700e-2f,    2.6700e-2f,    3.3400e-2f,    4.0800e-2f,    4.8800e-2f,
        5.7400e-2f,    6.6600e-2f,    7.6400e-2f,    8.6800e-2f,    9.7800e-2f,    1.0930e-1f,    1.2150e-1f,    1.3420e-1f,    1.4740e-1f,    1.6120e-1f,
        2.3830e-1f,    3.2850e-1f,    4.3130e-1f,    5.4650e-1f,    8.1270e-1f,    9.6310e-1f,    1.1250e0f,    1.4810e0f,    1.8800e0f,    2.3200e0f,
        2.8000e0f,    3.3190e0f,    3.8770e0f,    4.4710e0f,    5.1010e0f,    5.7670e0f,    6.4670e0f,    7.2010e0f,    7.9680e0f,    8.7680e0f,
        9.5990e0f,    1.4210e1f,    1.9520e1f,    2.5480e1f,    3.2020e1f,    3.9110e1f,    4.6700e1f,    5.4740e1f,    6.3220e1f,    8.1330e1f,
        1.0080e2f,    1.2150e2f,    1.4320e2f,    1.6590e2f,    1.8930e2f,    2.1350e2f,    2.3840e2f,    2.6380e2f,    2.8970e2f,    3.1600e2f,
        3.4280e2f,    3.6990e2f,    3.9740e2f,    6.8430e2f,    9.8250e2f,    1.2850e3f,    1.5870e3f,    2.1890e3f,    2.7850e3f,    3.3720e3f,
        3.9530e3f,    4.5280e3f,    5.0970e3f,    5.6600e3f,    0.0000e0f,    0.0000e0f,    0.0000e0f,    0.0000e0f,    0.0000e0f,    0.0000e0f,
        0.0000e0f,    0.0000e0f,    0.0000e0f,    0.0000e0f,    0.0000e0f,    0.0000e0f,    0.0000e0f,    0.0000e0f,    1.0000e-4f,    1.0000e-4f,
        1.0000e-4f,    1.0000e-4f,    1.0000e-4f,    1.0000e-4f,    1.0000e-4f,    1.0000e-4f,    1.0000e-4f,    1.0000e-4f,    1.0000e-4f,    1.0000e-4f,
        1.0000e-4f,    2.0000e-4f,    2.0000e-4f,    2.0000e-4f,    2.0000e-4f,    2.0000e-4f,    2.0000e-4f,    2.0000e-4f,    3.0000e-4f,    3.0000e-4f,
        4.0000e-4f,    5.0000e-4f,    5.0000e-4f,    6.0000e-4f,    7.0000e-4f,    8.0000e-4f,    9.0000e-4f,    1.1000e-3f,    1.3000e-3f,    1.4000e-3f,
        1.6000e-3f,    1.9000e-3f,    2.1000e-3f,    2.3000e-3f,    2.5000e-3f,    2.8000e-3f,    3.0000e-3f,    3.3000e-3f,    3.6000e-3f,    3.9000e-3f,
        5.4000e-3f,    7.2000e-3f,    9.2000e-3f,    1.1300e-2f,    1.3700e-2f,    1.6300e-2f,    1.9000e-2f,    2.1900e-2f,    2.8300e-2f,    3.5400e-2f,
        4.3200e-2f,    5.1600e-2f,    6.0600e-2f,    7.0300e-2f,    8.0600e-2f,    9.1600e-2f,    1.0310e-1f,    1.1530e-1f,    1.2800e-1f,    1.4130e-1f,
        1.5520e-1f,    1.6970e-1f,    2.5040e-1f,    3.4480e-1f,    4.5230e-1f,    5.7260e-1f,    8.5010e-1f,    1.0070e0f,    1.1750e0f,    1.5460e0f,
        1.9610e0f,    2.4190e0f,    2.9180e0f,    3.4580e0f,    4.0370e0f,    4.6540e0f,    5.3090e0f,    6.0000e0f,    6.7270e0f,    7.4880e0f,
        8.2840e0f,    9.1140e0f,    9.9760e0f,    1.4760e1f,    2.0260e1f,    2.6440e1f,    3.3220e1f,    4.0570e1f,    4.8430e1f,    5.6780e1f,
        6.5560e1f,    8.4340e1f,    1.0460e2f,    1.2600e2f,    1.4850e2f,    1.7200e2f,    1.9640e2f,    2.2140e2f,    2.4720e2f,    2.7350e2f,
        3.0030e2f,    3.2760e2f,    3.5530e2f,    3.8330e2f,    4.1170e2f,    7.0770e2f,    1.0140e3f,    1.3240e3f,    1.6330e3f,    2.2460e3f,
        2.8500e3f,    3.4450e3f,    4.0320e3f,    4.6120e3f,    5.1860e3f,    5.7540e3f,    0.0000e0f,    0.0000e0f,    0.0000e0f,    0.0000e0f,
        0.0000e0f,    0.0000e0f,    0.0000e0f,    0.0000e0f,    0.0000e0f,    0.0000e0f,    0.0000e0f,    0.0000e0f,    0.0000e0f,    0.0000e0f,
        0.0000e0f,    0.0000e0f,    0.0000e0f,    0.0000e0f,    0.0000e0f,    0.0000e0f,    1.0000e-4f,    1.0000e-4f,    1.0000e-4f,    1.0000e-4f,
        1.0000e-4f,    1.0000e-4f,    1.0000e-4f,    1.0000e-4f,    1.0000e-4f,    1.0000e-4f,    1.0000e-4f,    1.0000e-4f,    1.0000e-4f,    1.0000e-4f,
        2.0000e-4f,    2.0000e-4f,    2.0000e-4f,    2.0000e-4f,    3.0000e-4f,    3.0000e-4f,    4.0000e-4f,    4.0000e-4f,    5.0000e-4f,    6.0000e-4f,
        7.0000e-4f,    8.0000e-4f,    1.0000e-3f,    1.1000e-3f,    1.2000e-3f,    1.4000e-3f,    1.6000e-3f,    1.7000e-3f,    1.9000e-3f,    2.1000e-3f,
        2.3000e-3f,    2.4000e-3f,    3.5000e-3f,    4.7000e-3f,    6.1000e-3f,    7.7000e-3f,    9.4000e-3f,    1.1200e-2f,    1.3200e-2f,    1.5300e-2f,
        1.9900e-2f,    2.5100e-2f,    3.0800e-2f,    3.7000e-2f,    4.3700e-2f,    5.0900e-2f,    5.8600e-2f,    6.6800e-2f,    7.5500e-2f,    8.4600e-2f,
        9.4200e-2f,    1.0430e-1f,    1.1490e-1f,    1.2580e-1f,    1.8750e-1f,    2.6000e-1f,    3.4300e-1f,    4.3630e-1f,    6.5260e-1f,    7.7520e-1f,
        9.0720e-1f,    1.1990e0f,    1.5260e0f,    1.8880e0f,    2.2830e0f,    2.7120e0f,    3.1720e0f,    3.6630e0f,    4.1850e0f,    4.7360e0f,
        5.3170e0f,    5.9260e0f,    6.5620e0f,    7.2260e0f,    7.9170e0f,    1.1750e1f,    1.6190e1f,    2.1160e1f,    2.6640e1f,    3.2580e1f,
        3.8940e1f,    4.5690e1f,    5.2810e1f,    6.8040e1f,    8.4430e1f,    1.0190e2f,    1.2020e2f,    1.3930e2f,    1.5900e2f,    1.7940e2f,
        2.0030e2f,    2.2170e2f,    2.4350e2f,    2.6580e2f,    2.8830e2f,    3.1120e2f,    3.3440e2f,    5.7680e2f,    8.2990e2f,    1.0870e3f,
        1.3450e3f,    1.8610e3f,    2.3730e3f,    2.8790e3f,    3.3800e3f,    3.8760e3f,    4.3680e3f,    4.8560e3f,
        -1.000e+04f,            -1.000e+04f,            -1.000e+04f,            -1.000e+04f,            -1.000e+04f,            -1.000e+04f,            -1.000e+04f,            -1.000e+04f,            -1.000e+04f,            -1.000e+04f,
        -1.000e+04f,            -1.000e+04f,            -1.000e+04f,            -1.000e+04f,            -1.000e+04f,            -1.000e+04f,            -1.000e+04f,            -1.000e+04f,            -1.000e+04f,            -1.000e+04f,
        -1.000e+04f,            -1.000e+04f,            -1.000e+04f,            -1.000e+04f,            -1.000e+04f,            -1.000e+04f,            -1.000e+04f,            -1.000e+04f,            -1.000e+04f,            -1.000e+04f,
        -1.000e+04f,            -1.000e+04f,            -1.000e+04f,            -1.000e+04f,            -1.000e+04f,            -1.000e+04f,            -1.000e+04f,            -1.000e+04f,            -1.000e+04f,            -1.000e+04f,
        -1.000e+04f,            -1.000e+04f,            -1.000e+04f,            -1.000e+04f,            -1.000e+04f,            -1.000e+04f,            -1.000e+04f,            -1.000e+04f,            -1.000e+04f,            -1.000e+04f,
        -1.000e+04f,            -1.000e+04f,            -1.000e+04f,            -1.000e+04f,            -1.000e+04f,            -1.000e+04f,            -1.000e+04f,            -1.000e+04f,            -1.000e+04f,            -1.000e+04f,
        -1.000e+04f,            -1.000e+04f,            -1.000e+04f,            -1.000e+04f,            -1.000e+04f,            -1.000e+04f,            -1.000e+04f,            -1.000e+04f,            -1.000e+04f,            -1.000e+04f,
        -1.000e+04f,            -1.000e+04f,            -1.000e+04f,            -1.000e+04f,            -1.000e+04f,            -1.000e+04f,            -1.000e+04f,            -1.000e+04f,            -1.000e+04f,            -1.000e+04f,
        -1.000e+04f,            -1.000e+04f,            -1.000e+04f,            -1.000e+04f,            -1.000e+04f,            -1.000e+04f,            -1.000e+04f,            -1.000e+04f,            -1.000e+04f,            -1.000e+04f,
        -1.000e+04f,            -1.000e+04f,            -1.000e+04f,            -1.000e+04f,            -1.000e+04f,            -1.000e+04f,            -1.000e+04f,            -1.000e+04f,            -1.000e+04f,            -1.000e+04f,
        -1.000e+04f,            -1.000e+04f,            -1.000e+04f,            -1.000e+04f,            -1.000e+04f,            -1.000e+04f,            -1.000e+04f,            -1.000e+04f,            -1.000e+04f,            -1.000e+04f,
        -1.000e+04f,            -1.000e+04f,            -1.000e+04f,            -1.000e+04f,            -1.000e+04f,            -1.000e+04f,            -1.000e+04f,            -1.000e+04f,            -1.000e+04f,            -1.000e+04f,
        -1.000e+04f,            -1.000e+04f,            -1.000e+04f,            -1.000e+04f,            -1.000e+04f,            -1.000e+04f,            -1.000e+04f,            -1.000e+04f,            -1.000e+04f,            -1.000e+04f,
        -1.000e+04f,            -1.000e+04f,
        3.947e-06f,             6.091e-06f,             8.255e-06f,             1.039e-05f,             1.250e-05f,             1.660e-05f,             2.052e-05f,             2.430e-05f,             2.792e-05f,             3.142e-05f,
        3.479e-05f,             3.805e-05f,             4.582e-05f,             5.318e-05f,             6.018e-05f,             6.689e-05f,             7.333e-05f,             7.956e-05f,             8.559e-05f,             9.146e-05f,
        1.028e-04f,             1.136e-04f,             1.240e-04f,             1.342e-04f,             1.441e-04f,             1.538e-04f,             1.633e-04f,             1.727e-04f,             1.820e-04f,             1.912e-04f,
        2.003e-04f,             2.094e-04f,             2.185e-04f,             2.275e-04f,             2.728e-04f,             3.189e-04f,             3.666e-04f,             4.162e-04f,             4.680e-04f,             5.222e-04f,
        5.790e-04f,             6.384e-04f,             7.651e-04f,             9.019e-04f,             1.049e-03f,             1.205e-03f,             1.370e-03f,             1.545e-03f,             1.728e-03f,             1.920e-03f,
        2.121e-03f,             2.330e-03f,             2.549e-03f,             2.775e-03f,             3.010e-03f,             3.254e-03f,             4.592e-03f,             6.129e-03f,             7.856e-03f,             9.769e-03f,
        1.186e-02f,             1.413e-02f,             1.658e-02f,             1.919e-02f,             2.492e-02f,             3.130e-02f,             3.831e-02f,             4.595e-02f,             5.420e-02f,             6.305e-02f,
        7.249e-02f,             8.252e-02f,             9.312e-02f,             1.043e-01f,             1.160e-01f,             1.283e-01f,             1.412e-01f,             1.546e-01f,             2.297e-01f,             3.178e-01f,
        4.187e-01f,             5.318e-01f,             7.939e-01f,             9.423e-01f,             1.102e+00f,             1.454e+00f,             1.850e+00f,             2.286e+00f,             2.763e+00f,             3.280e+00f,
        3.834e+00f,             4.426e+00f,             5.055e+00f,             5.718e+00f,             6.417e+00f,             7.150e+00f,             7.916e+00f,             8.715e+00f,             9.545e+00f,             1.416e+01f,
        1.948e+01f,             2.545e+01f,             3.203e+01f,             3.915e+01f,             4.677e+01f,             5.487e+01f,             6.340e+01f,             8.164e+01f,             1.013e+02f,             1.221e+02f,
        1.441e+02f,             1.669e+02f,             1.906e+02f,             2.150e+02f,             2.401e+02f,             2.658e+02f,             2.920e+02f,             3.186e+02f,             3.457e+02f,             3.731e+02f,
        4.009e+02f,             6.912e+02f,             9.939e+02f,             1.301e+03f,             1.609e+03f,             2.224e+03f,             2.834e+03f,             3.438e+03f,             4.035e+03f,             4.627e+03f,
        5.214e+03f,             5.795e+03f,
		3.257e-06f,		4.925e-06f,		6.577e-06f,		8.189e-06f,		9.759e-06f,		1.277e-05f,		1.563e-05f,		1.834e-05f,		2.094e-05f,		2.342e-05f, 
		2.580e-05f,		2.810e-05f,		3.356e-05f,		3.869e-05f,		4.356e-05f,		4.822e-05f,		5.269e-05f,		5.701e-05f,		6.120e-05f,		6.527e-05f, 
		7.313e-05f,		8.069e-05f,		8.800e-05f,		9.513e-05f,		1.021e-04f,		1.090e-04f,		1.158e-04f,		1.226e-04f,		1.293e-04f,		1.360e-04f, 
		1.426e-04f,		1.493e-04f,		1.561e-04f,		1.628e-04f,		1.972e-04f,		2.333e-04f,		2.715e-04f,		3.121e-04f,		3.553e-04f,		4.010e-04f, 
		4.493e-04f,		5.002e-04f,		6.095e-04f,		7.287e-04f,		8.573e-04f,		9.951e-04f,		1.142e-03f,		1.297e-03f,		1.461e-03f,		1.633e-03f, 
		1.813e-03f,		2.002e-03f,		2.198e-03f,		2.402e-03f,		2.614e-03f,		2.834e-03f,		4.046e-03f,		5.438e-03f,		7.006e-03f,		8.742e-03f, 
		1.064e-02f,		1.270e-02f,		1.492e-02f,		1.730e-02f,		2.250e-02f,		2.829e-02f,		3.466e-02f,		4.161e-02f,		4.910e-02f,		5.715e-02f, 
		6.574e-02f,		7.486e-02f,		8.452e-02f,		9.469e-02f,		1.054e-01f,		1.166e-01f,		1.283e-01f,		1.405e-01f,		2.089e-01f,		2.893e-01f, 
		3.813e-01f,		4.845e-01f,		7.238e-01f,		8.593e-01f,		1.005e+00f,		1.327e+00f,		1.688e+00f,		2.088e+00f,		2.524e+00f,		2.996e+00f, 
		3.504e+00f,		4.045e+00f,		4.620e+00f,		5.228e+00f,		5.867e+00f,		6.538e+00f,		7.239e+00f,		7.970e+00f,		8.731e+00f,		1.295e+01f, 
		1.783e+01f,		2.330e+01f,		2.933e+01f,		3.585e+01f,		4.284e+01f,		5.026e+01f,		5.809e+01f,		7.481e+01f,		9.282e+01f,		1.119e+02f, 
		1.320e+02f,		1.530e+02f,		1.747e+02f,		1.971e+02f,		2.200e+02f,		2.435e+02f,		2.674e+02f,		2.918e+02f,		3.165e+02f,		3.415e+02f, 
		3.668e+02f,		6.306e+02f,		9.035e+02f,		1.178e+03f,		1.453e+03f,		1.995e+03f,		2.527e+03f,		3.049e+03f,		3.561e+03f,		4.066e+03f, 
		4.563e+03f,		5.053e+03f 
    },
    {
        4.5550e-1f,    4.9060e-1f,    5.1970e-1f,    5.4400e-1f,    5.6470e-1f,    5.9860e-1f,    6.2540e-1f,    6.4730e-1f,    6.6560e-1f,    6.8130e-1f,
        6.9500e-1f,    7.0700e-1f,    7.3180e-1f,    7.5140e-1f,    7.6740e-1f,    7.8080e-1f,    7.9230e-1f,    8.0220e-1f,    8.1090e-1f,    8.1870e-1f,
        8.3190e-1f,    8.4290e-1f,    8.5220e-1f,    8.6020e-1f,    8.6730e-1f,    8.7350e-1f,    8.7910e-1f,    8.8420e-1f,    8.8890e-1f,    8.9310e-1f,
        8.9710e-1f,    9.0070e-1f,    9.0410e-1f,    9.0730e-1f,    9.2070e-1f,    9.3100e-1f,    9.3930e-1f,    9.4600e-1f,    9.5150e-1f,    9.5620e-1f,
        9.6010e-1f,    9.6350e-1f,    9.6890e-1f,    9.7310e-1f,    9.7640e-1f,    9.7900e-1f,    9.8110e-1f,    9.8290e-1f,    9.8440e-1f,    9.8570e-1f,
        9.8680e-1f,    9.8770e-1f,    9.8860e-1f,    9.8930e-1f,    9.8990e-1f,    9.9050e-1f,    9.9250e-1f,    9.9380e-1f,    9.9460e-1f,    9.9520e-1f,
        9.9570e-1f,    9.9600e-1f,    9.9630e-1f,    9.9650e-1f,    9.9680e-1f,    9.9710e-1f,    9.9730e-1f,    9.9740e-1f,    9.9750e-1f,    9.9760e-1f,
        9.9770e-1f,    9.9770e-1f,    9.9780e-1f,    9.9780e-1f,    9.9790e-1f,    9.9790e-1f,    9.9790e-1f,    9.9800e-1f,    9.9810e-1f,    9.9820e-1f,
        9.9820e-1f,    9.9830e-1f,    9.9830e-1f,    9.9840e-1f,    9.9840e-1f,    9.9840e-1f,    9.9850e-1f,    9.9850e-1f,    9.9850e-1f,    9.9850e-1f,
        9.9860e-1f,    9.9860e-1f,    9.9860e-1f,    9.9860e-1f,    9.9860e-1f,    9.9860e-1f,    9.9860e-1f,    9.9860e-1f,    9.9870e-1f,    9.9870e-1f,
        9.9870e-1f,    9.9880e-1f,    9.9880e-1f,    9.9880e-1f,    9.9890e-1f,    9.9890e-1f,    9.9890e-1f,    9.9890e-1f,    9.9900e-1f,    9.9900e-1f,
        9.9900e-1f,    9.9910e-1f,    9.9910e-1f,    9.9910e-1f,    9.9910e-1f,    9.9910e-1f,    9.9920e-1f,    9.9920e-1f,    9.9920e-1f,    9.9920e-1f,
        9.9920e-1f,    9.9930e-1f,    9.9940e-1f,    9.9950e-1f,    9.9950e-1f,    9.9960e-1f,    9.9960e-1f,    9.9970e-1f,    9.9970e-1f,    9.9970e-1f,
        9.9970e-1f,    9.9980e-1f,    2.4840e-1f,    2.8570e-1f,    3.1680e-1f,    3.4320e-1f,    3.6630e-1f,    4.0500e-1f,    4.3660e-1f,    4.6310e-1f,
        4.8600e-1f,    5.0590e-1f,    5.2350e-1f,    5.3930e-1f,    5.7260e-1f,    5.9970e-1f,    6.2220e-1f,    6.4150e-1f,    6.5810e-1f,    6.7270e-1f,
        6.8570e-1f,    6.9730e-1f,    7.1740e-1f,    7.3420e-1f,    7.4870e-1f,    7.6130e-1f,    7.7240e-1f,    7.8230e-1f,    7.9120e-1f,    7.9930e-1f,
        8.0670e-1f,    8.1360e-1f,    8.1990e-1f,    8.2580e-1f,    8.3130e-1f,    8.3640e-1f,    8.5800e-1f,    8.7480e-1f,    8.8830e-1f,    8.9940e-1f,
        9.0870e-1f,    9.1660e-1f,    9.2330e-1f,    9.2920e-1f,    9.3880e-1f,    9.4640e-1f,    9.5240e-1f,    9.5730e-1f,    9.6140e-1f,    9.6480e-1f,
        9.6770e-1f,    9.7020e-1f,    9.7240e-1f,    9.7430e-1f,    9.7590e-1f,    9.7730e-1f,    9.7860e-1f,    9.7980e-1f,    9.8400e-1f,    9.8670e-1f,
        9.8860e-1f,    9.8990e-1f,    9.9090e-1f,    9.9170e-1f,    9.9230e-1f,    9.9280e-1f,    9.9350e-1f,    9.9410e-1f,    9.9450e-1f,    9.9480e-1f,
        9.9510e-1f,    9.9530e-1f,    9.9540e-1f,    9.9560e-1f,    9.9570e-1f,    9.9580e-1f,    9.9590e-1f,    9.9600e-1f,    9.9610e-1f,    9.9620e-1f,
        9.9640e-1f,    9.9660e-1f,    9.9670e-1f,    9.9680e-1f,    9.9700e-1f,    9.9700e-1f,    9.9710e-1f,    9.9720e-1f,    9.9720e-1f,    9.9730e-1f,
        9.9730e-1f,    9.9740e-1f,    9.9740e-1f,    9.9750e-1f,    9.9750e-1f,    9.9750e-1f,    9.9760e-1f,    9.9760e-1f,    9.9760e-1f,    9.9760e-1f,
        9.9760e-1f,    9.9770e-1f,    9.9780e-1f,    9.9790e-1f,    9.9790e-1f,    9.9800e-1f,    9.9800e-1f,    9.9810e-1f,    9.9810e-1f,    9.9820e-1f,
        9.9820e-1f,    9.9830e-1f,    9.9830e-1f,    9.9840e-1f,    9.9840e-1f,    9.9840e-1f,    9.9850e-1f,    9.9850e-1f,    9.9850e-1f,    9.9860e-1f,
        9.9860e-1f,    9.9860e-1f,    9.9870e-1f,    9.9880e-1f,    9.9900e-1f,    9.9910e-1f,    9.9920e-1f,    9.9930e-1f,    9.9940e-1f,    9.9940e-1f,
        9.9950e-1f,    9.9950e-1f,    9.9950e-1f,    9.9960e-1f,    2.5550e-1f,    2.9330e-1f,    3.2450e-1f,    3.5090e-1f,    3.7380e-1f,    4.1220e-1f,
        4.4340e-1f,    4.6960e-1f,    4.9210e-1f,    5.1170e-1f,    5.2910e-1f,    5.4450e-1f,    5.7730e-1f,    6.0400e-1f,    6.2630e-1f,    6.4540e-1f,
        6.6200e-1f,    6.7670e-1f,    6.8980e-1f,    7.0150e-1f,    7.2200e-1f,    7.3940e-1f,    7.5440e-1f,    7.6760e-1f,    7.7930e-1f,    7.8980e-1f,
        7.9940e-1f,    8.0810e-1f,    8.1610e-1f,    8.2340e-1f,    8.3020e-1f,    8.3660e-1f,    8.4250e-1f,    8.4800e-1f,    8.7080e-1f,    8.8800e-1f,
        9.0130e-1f,    9.1190e-1f,    9.2050e-1f,    9.2770e-1f,    9.3370e-1f,    9.3890e-1f,    9.4710e-1f,    9.5350e-1f,    9.5850e-1f,    9.6260e-1f,
        9.6590e-1f,    9.6870e-1f,    9.7110e-1f,    9.7310e-1f,    9.7490e-1f,    9.7640e-1f,    9.7780e-1f,    9.7900e-1f,    9.8000e-1f,    9.8100e-1f,
        9.8450e-1f,    9.8680e-1f,    9.8840e-1f,    9.8950e-1f,    9.9040e-1f,    9.9110e-1f,    9.9160e-1f,    9.9210e-1f,    9.9280e-1f,    9.9330e-1f,
        9.9370e-1f,    9.9400e-1f,    9.9420e-1f,    9.9440e-1f,    9.9460e-1f,    9.9480e-1f,    9.9490e-1f,    9.9500e-1f,    9.9510e-1f,    9.9520e-1f,
        9.9530e-1f,    9.9530e-1f,    9.9560e-1f,    9.9580e-1f,    9.9600e-1f,    9.9610e-1f,    9.9630e-1f,    9.9630e-1f,    9.9640e-1f,    9.9650e-1f,
        9.9660e-1f,    9.9660e-1f,    9.9670e-1f,    9.9680e-1f,    9.9680e-1f,    9.9680e-1f,    9.9690e-1f,    9.9690e-1f,    9.9700e-1f,    9.9700e-1f,
        9.9700e-1f,    9.9700e-1f,    9.9710e-1f,    9.9720e-1f,    9.9730e-1f,    9.9730e-1f,    9.9740e-1f,    9.9750e-1f,    9.9750e-1f,    9.9760e-1f,
        9.9760e-1f,    9.9770e-1f,    9.9780e-1f,    9.9790e-1f,    9.9790e-1f,    9.9800e-1f,    9.9800e-1f,    9.9810e-1f,    9.9810e-1f,    9.9820e-1f,
        9.9820e-1f,    9.9820e-1f,    9.9830e-1f,    9.9830e-1f,    9.9830e-1f,    9.9860e-1f,    9.9870e-1f,    9.9890e-1f,    9.9900e-1f,    9.9910e-1f,
        9.9920e-1f,    9.9930e-1f,    9.9930e-1f,    9.9940e-1f,    9.9940e-1f,    9.9950e-1f,    4.7130e-1f,    5.1250e-1f,    5.4510e-1f,    5.7140e-1f,
        5.9340e-1f,    6.2870e-1f,    6.5600e-1f,    6.7790e-1f,    6.9610e-1f,    7.1160e-1f,    7.2490e-1f,    7.3650e-1f,    7.6040e-1f,    7.7910e-1f,
        7.9430e-1f,    8.0690e-1f,    8.1770e-1f,    8.2700e-1f,    8.3510e-1f,    8.4240e-1f,    8.5460e-1f,    8.6480e-1f,    8.7330e-1f,    8.8070e-1f,
        8.8710e-1f,    8.9280e-1f,    8.9780e-1f,    9.0240e-1f,    9.0660e-1f,    9.1040e-1f,    9.1390e-1f,    9.1710e-1f,    9.2020e-1f,    9.2300e-1f,
        9.3470e-1f,    9.4360e-1f,    9.5060e-1f,    9.5630e-1f,    9.6110e-1f,    9.6500e-1f,    9.6840e-1f,    9.7130e-1f,    9.7590e-1f,    9.7930e-1f,
        9.8200e-1f,    9.8420e-1f,    9.8590e-1f,    9.8720e-1f,    9.8840e-1f,    9.8930e-1f,    9.9010e-1f,    9.9080e-1f,    9.9140e-1f,    9.9200e-1f,
        9.9240e-1f,    9.9280e-1f,    9.9430e-1f,    9.9520e-1f,    9.9580e-1f,    9.9620e-1f,    9.9650e-1f,    9.9680e-1f,    9.9700e-1f,    9.9710e-1f,
        9.9740e-1f,    9.9750e-1f,    9.9770e-1f,    9.9780e-1f,    9.9780e-1f,    9.9790e-1f,    9.9800e-1f,    9.9800e-1f,    9.9810e-1f,    9.9810e-1f,
        9.9810e-1f,    9.9820e-1f,    9.9820e-1f,    9.9820e-1f,    9.9830e-1f,    9.9840e-1f,    9.9840e-1f,    9.9850e-1f,    9.9850e-1f,    9.9850e-1f,
        9.9860e-1f,    9.9860e-1f,    9.9860e-1f,    9.9870e-1f,    9.9870e-1f,    9.9870e-1f,    9.9870e-1f,    9.9870e-1f,    9.9870e-1f,    9.9880e-1f,
        9.9880e-1f,    9.9880e-1f,    9.9880e-1f,    9.9880e-1f,    9.9880e-1f,    9.9880e-1f,    9.9890e-1f,    9.9890e-1f,    9.9890e-1f,    9.9900e-1f,
        9.9900e-1f,    9.9900e-1f,    9.9900e-1f,    9.9910e-1f,    9.9910e-1f,    9.9910e-1f,    9.9910e-1f,    9.9920e-1f,    9.9920e-1f,    9.9920e-1f,
        9.9920e-1f,    9.9920e-1f,    9.9920e-1f,    9.9930e-1f,    9.9930e-1f,    9.9930e-1f,    9.9930e-1f,    9.9940e-1f,    9.9950e-1f,    9.9950e-1f,
        9.9960e-1f,    9.9960e-1f,    9.9970e-1f,    9.9970e-1f,    9.9970e-1f,    9.9970e-1f,    9.9980e-1f,    9.9980e-1f,
        -1.000e+04f,            -1.000e+04f,            -1.000e+04f,            -1.000e+04f,            -1.000e+04f,            -1.000e+04f,            -1.000e+04f,            -1.000e+04f,            -1.000e+04f,            -1.000e+04f,
        -1.000e+04f,            -1.000e+04f,            -1.000e+04f,            -1.000e+04f,            -1.000e+04f,            -1.000e+04f,            -1.000e+04f,            -1.000e+04f,            -1.000e+04f,            -1.000e+04f,
        -1.000e+04f,            -1.000e+04f,            -1.000e+04f,            -1.000e+04f,            -1.000e+04f,            -1.000e+04f,            -1.000e+04f,            -1.000e+04f,            -1.000e+04f,            -1.000e+04f,
        -1.000e+04f,            -1.000e+04f,            -1.000e+04f,            -1.000e+04f,            -1.000e+04f,            -1.000e+04f,            -1.000e+04f,            -1.000e+04f,            -1.000e+04f,            -1.000e+04f,
        -1.000e+04f,            -1.000e+04f,            -1.000e+04f,            -1.000e+04f,            -1.000e+04f,            -1.000e+04f,            -1.000e+04f,            -1.000e+04f,            -1.000e+04f,            -1.000e+04f,
        -1.000e+04f,            -1.000e+04f,            -1.000e+04f,            -1.000e+04f,            -1.000e+04f,            -1.000e+04f,            -1.000e+04f,            -1.000e+04f,            -1.000e+04f,            -1.000e+04f,
        -1.000e+04f,            -1.000e+04f,            -1.000e+04f,            -1.000e+04f,            -1.000e+04f,            -1.000e+04f,            -1.000e+04f,            -1.000e+04f,            -1.000e+04f,            -1.000e+04f,
        -1.000e+04f,            -1.000e+04f,            -1.000e+04f,            -1.000e+04f,            -1.000e+04f,            -1.000e+04f,            -1.000e+04f,            -1.000e+04f,            -1.000e+04f,            -1.000e+04f,
        -1.000e+04f,            -1.000e+04f,            -1.000e+04f,            -1.000e+04f,            -1.000e+04f,            -1.000e+04f,            -1.000e+04f,            -1.000e+04f,            -1.000e+04f,            -1.000e+04f,
        -1.000e+04f,            -1.000e+04f,            -1.000e+04f,            -1.000e+04f,            -1.000e+04f,            -1.000e+04f,            -1.000e+04f,            -1.000e+04f,            -1.000e+04f,            -1.000e+04f,
        -1.000e+04f,            -1.000e+04f,            -1.000e+04f,            -1.000e+04f,            -1.000e+04f,            -1.000e+04f,            -1.000e+04f,            -1.000e+04f,            -1.000e+04f,            -1.000e+04f,
        -1.000e+04f,            -1.000e+04f,            -1.000e+04f,            -1.000e+04f,            -1.000e+04f,            -1.000e+04f,            -1.000e+04f,            -1.000e+04f,            -1.000e+04f,            -1.000e+04f,
        -1.000e+04f,            -1.000e+04f,            -1.000e+04f,            -1.000e+04f,            -1.000e+04f,            -1.000e+04f,            -1.000e+04f,            -1.000e+04f,            -1.000e+04f,            -1.000e+04f,
        -1.000e+04f,            -1.000e+04f,
        3.105e-01f,             3.520e-01f,             3.859e-01f,             4.141e-01f,             4.384e-01f,             4.782e-01f,             5.101e-01f,             5.364e-01f,             5.586e-01f,             5.778e-01f,
        5.946e-01f,             6.095e-01f,             6.405e-01f,             6.653e-01f,             6.857e-01f,             7.029e-01f,             7.176e-01f,             7.305e-01f,             7.418e-01f,             7.519e-01f,
        7.692e-01f,             7.836e-01f,             7.959e-01f,             8.064e-01f,             8.157e-01f,             8.240e-01f,             8.314e-01f,             8.381e-01f,             8.442e-01f,             8.498e-01f,
        8.549e-01f,             8.597e-01f,             8.641e-01f,             8.683e-01f,             8.857e-01f,             8.991e-01f,             9.099e-01f,             9.188e-01f,             9.264e-01f,             9.328e-01f,
        9.384e-01f,             9.432e-01f,             9.513e-01f,             9.576e-01f,             9.626e-01f,             9.667e-01f,             9.701e-01f,             9.729e-01f,             9.753e-01f,             9.773e-01f,
        9.791e-01f,             9.806e-01f,             9.819e-01f,             9.831e-01f,             9.841e-01f,             9.850e-01f,             9.884e-01f,             9.905e-01f,             9.919e-01f,             9.930e-01f,
        9.937e-01f,             9.943e-01f,             9.948e-01f,             9.951e-01f,             9.957e-01f,             9.961e-01f,             9.963e-01f,             9.966e-01f,             9.967e-01f,             9.969e-01f,
        9.970e-01f,             9.971e-01f,             9.972e-01f,             9.973e-01f,             9.973e-01f,             9.974e-01f,             9.974e-01f,             9.975e-01f,             9.977e-01f,             9.978e-01f,
        9.979e-01f,             9.979e-01f,             9.980e-01f,             9.980e-01f,             9.981e-01f,             9.981e-01f,             9.982e-01f,             9.982e-01f,             9.982e-01f,             9.983e-01f,
        9.983e-01f,             9.983e-01f,             9.983e-01f,             9.984e-01f,             9.984e-01f,             9.984e-01f,             9.984e-01f,             9.984e-01f,             9.984e-01f,             9.985e-01f,
        9.985e-01f,             9.986e-01f,             9.986e-01f,             9.986e-01f,             9.987e-01f,             9.987e-01f,             9.987e-01f,             9.988e-01f,             9.988e-01f,             9.988e-01f,
        9.989e-01f,             9.989e-01f,             9.989e-01f,             9.989e-01f,             9.990e-01f,             9.990e-01f,             9.990e-01f,             9.990e-01f,             9.991e-01f,             9.991e-01f,
        9.991e-01f,             9.992e-01f,             9.993e-01f,             9.994e-01f,             9.994e-01f,             9.995e-01f,             9.996e-01f,             9.996e-01f,             9.996e-01f,             9.997e-01f,
        9.997e-01f,             9.997e-01f,
		3.304e-01f,		3.760e-01f,		4.123e-01f,		4.421e-01f,		4.674e-01f,		5.084e-01f,		5.406e-01f,		5.669e-01f,		5.889e-01f,		6.078e-01f, 
		6.242e-01f,		6.387e-01f,		6.686e-01f,		6.923e-01f,		7.118e-01f,		7.281e-01f,		7.420e-01f,		7.542e-01f,		7.648e-01f,		7.743e-01f, 
		7.906e-01f,		8.041e-01f,		8.156e-01f,		8.256e-01f,		8.343e-01f,		8.420e-01f,		8.490e-01f,		8.553e-01f,		8.611e-01f,		8.664e-01f, 
		8.712e-01f,		8.758e-01f,		8.800e-01f,		8.839e-01f,		9.005e-01f,		9.133e-01f,		9.236e-01f,		9.320e-01f,		9.390e-01f,		9.450e-01f, 
		9.500e-01f,		9.544e-01f,		9.614e-01f,		9.668e-01f,		9.710e-01f,		9.743e-01f,		9.770e-01f,		9.793e-01f,		9.812e-01f,		9.827e-01f, 
		9.841e-01f,		9.852e-01f,		9.862e-01f,		9.871e-01f,		9.879e-01f,		9.886e-01f,		9.910e-01f,		9.926e-01f,		9.936e-01f,		9.943e-01f, 
		9.949e-01f,		9.953e-01f,		9.956e-01f,		9.959e-01f,		9.963e-01f,		9.966e-01f,		9.968e-01f,		9.969e-01f,		9.971e-01f,		9.972e-01f, 
		9.973e-01f,		9.973e-01f,		9.974e-01f,		9.975e-01f,		9.975e-01f,		9.976e-01f,		9.976e-01f,		9.976e-01f,		9.978e-01f,		9.979e-01f, 
		9.979e-01f,		9.980e-01f,		9.981e-01f,		9.981e-01f,		9.981e-01f,		9.982e-01f,		9.982e-01f,		9.983e-01f,		9.983e-01f,		9.983e-01f, 
		9.983e-01f,		9.984e-01f,		9.984e-01f,		9.984e-01f,		9.984e-01f,		9.984e-01f,		9.984e-01f,		9.985e-01f,		9.985e-01f,		9.985e-01f, 
		9.986e-01f,		9.986e-01f,		9.986e-01f,		9.987e-01f,		9.987e-01f,		9.987e-01f,		9.987e-01f,		9.988e-01f,		9.988e-01f,		9.989e-01f, 
		9.989e-01f,		9.989e-01f,		9.990e-01f,		9.990e-01f,		9.990e-01f,		9.990e-01f,		9.990e-01f,		9.991e-01f,		9.991e-01f,		9.991e-01f, 
		9.991e-01f,		9.992e-01f,		9.993e-01f,		9.994e-01f,		9.994e-01f,		9.995e-01f,		9.996e-01f,		9.996e-01f,		9.997e-01f,		9.997e-01f, 
		9.997e-01f,		9.997e-01f 
    },
    {
        Water_Liquid,    Water_Liquid,    Water_Liquid,    Water_Liquid,    Water_Liquid,    Water_Liquid,    Water_Liquid,    Water_Liquid,    Water_Liquid,    Water_Liquid,
        Water_Liquid,    Water_Liquid,    Water_Liquid,    Water_Liquid,    Water_Liquid,    Water_Liquid,    Water_Liquid,    Water_Liquid,    Water_Liquid,    Water_Liquid,
        Water_Liquid,    Water_Liquid,    Water_Liquid,    Water_Liquid,    Water_Liquid,    Water_Liquid,    Water_Liquid,    Water_Liquid,    Water_Liquid,    Water_Liquid,
        Water_Liquid,    Water_Liquid,    Water_Liquid,    Water_Liquid,    Water_Liquid,    Water_Liquid,    Water_Liquid,    Water_Liquid,    Water_Liquid,    Water_Liquid,
        Water_Liquid,    Water_Liquid,    Water_Liquid,    Water_Liquid,    Water_Liquid,    Water_Liquid,    Water_Liquid,    Water_Liquid,    Water_Liquid,    Water_Liquid,
        Water_Liquid,    Water_Liquid,    Water_Liquid,    Water_Liquid,    Water_Liquid,    Water_Liquid,    Water_Liquid,    Water_Liquid,    Water_Liquid,    Water_Liquid,
        Water_Liquid,    Water_Liquid,    Water_Liquid,    Water_Liquid,    Water_Liquid,    Water_Liquid,    Water_Liquid,    Water_Liquid,    Water_Liquid,    Water_Liquid,
        Water_Liquid,    Water_Liquid,    Water_Liquid,    Water_Liquid,    Water_Liquid,    Water_Liquid,    Water_Liquid,    Water_Liquid,    Water_Liquid,    Water_Liquid,
        Water_Liquid,    Water_Liquid,    Water_Liquid,    Water_Liquid,    Water_Liquid,    Water_Liquid,    Water_Liquid,    Water_Liquid,    Water_Liquid,    Water_Liquid,
        Water_Liquid,    Water_Liquid,    Water_Liquid,    Water_Liquid,    Water_Liquid,    Water_Liquid,    Water_Liquid,    Water_Liquid,    Water_Liquid,    Water_Liquid,
        Water_Liquid,    Water_Liquid,    Water_Liquid,    Water_Liquid,    Water_Liquid,    Water_Liquid,    Water_Liquid,    Water_Liquid,    Water_Liquid,    Water_Liquid,
        Water_Liquid,    Water_Liquid,    Water_Liquid,    Water_Liquid,    Water_Liquid,    Water_Liquid,    Water_Liquid,    Water_Liquid,    Water_Liquid,    Water_Liquid,
        Water_Liquid,    Water_Liquid,    Water_Liquid,    Water_Liquid,    Water_Liquid,    Water_Liquid,    Water_Liquid,    Water_Liquid,    Water_Liquid,    Water_Liquid,
        Water_Liquid,    Water_Liquid,    Aluminum_Oxide,    Aluminum_Oxide,    Aluminum_Oxide,    Aluminum_Oxide,    Aluminum_Oxide,    Aluminum_Oxide,    Aluminum_Oxide,    Aluminum_Oxide,
        Aluminum_Oxide,    Aluminum_Oxide,    Aluminum_Oxide,    Aluminum_Oxide,    Aluminum_Oxide,    Aluminum_Oxide,    Aluminum_Oxide,    Aluminum_Oxide,    Aluminum_Oxide,    Aluminum_Oxide,
        Aluminum_Oxide,    Aluminum_Oxide,    Aluminum_Oxide,    Aluminum_Oxide,    Aluminum_Oxide,    Aluminum_Oxide,    Aluminum_Oxide,    Aluminum_Oxide,    Aluminum_Oxide,    Aluminum_Oxide,
        Aluminum_Oxide,    Aluminum_Oxide,    Aluminum_Oxide,    Aluminum_Oxide,    Aluminum_Oxide,    Aluminum_Oxide,    Aluminum_Oxide,    Aluminum_Oxide,    Aluminum_Oxide,    Aluminum_Oxide,
        Aluminum_Oxide,    Aluminum_Oxide,    Aluminum_Oxide,    Aluminum_Oxide,    Aluminum_Oxide,    Aluminum_Oxide,    Aluminum_Oxide,    Aluminum_Oxide,    Aluminum_Oxide,    Aluminum_Oxide,
        Aluminum_Oxide,    Aluminum_Oxide,    Aluminum_Oxide,    Aluminum_Oxide,    Aluminum_Oxide,    Aluminum_Oxide,    Aluminum_Oxide,    Aluminum_Oxide,    Aluminum_Oxide,    Aluminum_Oxide,
        Aluminum_Oxide,    Aluminum_Oxide,    Aluminum_Oxide,    Aluminum_Oxide,    Aluminum_Oxide,    Aluminum_Oxide,    Aluminum_Oxide,    Aluminum_Oxide,    Aluminum_Oxide,    Aluminum_Oxide,
        Aluminum_Oxide,    Aluminum_Oxide,    Aluminum_Oxide,    Aluminum_Oxide,    Aluminum_Oxide,    Aluminum_Oxide,    Aluminum_Oxide,    Aluminum_Oxide,    Aluminum_Oxide,    Aluminum_Oxide,
        Aluminum_Oxide,    Aluminum_Oxide,    Aluminum_Oxide,    Aluminum_Oxide,    Aluminum_Oxide,    Aluminum_Oxide,    Aluminum_Oxide,    Aluminum_Oxide,    Aluminum_Oxide,    Aluminum_Oxide,
        Aluminum_Oxide,    Aluminum_Oxide,    Aluminum_Oxide,    Aluminum_Oxide,    Aluminum_Oxide,    Aluminum_Oxide,    Aluminum_Oxide,    Aluminum_Oxide,    Aluminum_Oxide,    Aluminum_Oxide,
        Aluminum_Oxide,    Aluminum_Oxide,    Aluminum_Oxide,    Aluminum_Oxide,    Aluminum_Oxide,    Aluminum_Oxide,    Aluminum_Oxide,    Aluminum_Oxide,    Aluminum_Oxide,    Aluminum_Oxide,
        Aluminum_Oxide,    Aluminum_Oxide,    Aluminum_Oxide,    Aluminum_Oxide,    Aluminum_Oxide,    Aluminum_Oxide,    Aluminum_Oxide,    Aluminum_Oxide,    Aluminum_Oxide,    Aluminum_Oxide,
        Aluminum_Oxide,    Aluminum_Oxide,    Aluminum_Oxide,    Aluminum_Oxide,    Aluminum_Oxide,    Aluminum_Oxide,    Aluminum_Oxide,    Aluminum_Oxide,    Aluminum_Oxide,    Aluminum_Oxide,
        Aluminum_Oxide,    Aluminum_Oxide,    Aluminum_Oxide,    Aluminum_Oxide,    Aluminum,    Aluminum,    Aluminum,    Aluminum,    Aluminum,    Aluminum,
        Aluminum,    Aluminum,    Aluminum,    Aluminum,    Aluminum,    Aluminum,    Aluminum,    Aluminum,    Aluminum,    Aluminum,
        Aluminum,    Aluminum,    Aluminum,    Aluminum,    Aluminum,    Aluminum,    Aluminum,    Aluminum,    Aluminum,    Aluminum,
        Aluminum,    Aluminum,    Aluminum,    Aluminum,    Aluminum,    Aluminum,    Aluminum,    Aluminum,    Aluminum,    Aluminum,
        Aluminum,    Aluminum,    Aluminum,    Aluminum,    Aluminum,    Aluminum,    Aluminum,    Aluminum,    Aluminum,    Aluminum,
        Aluminum,    Aluminum,    Aluminum,    Aluminum,    Aluminum,    Aluminum,    Aluminum,    Aluminum,    Aluminum,    Aluminum,
        Aluminum,    Aluminum,    Aluminum,    Aluminum,    Aluminum,    Aluminum,    Aluminum,    Aluminum,    Aluminum,    Aluminum,
        Aluminum,    Aluminum,    Aluminum,    Aluminum,    Aluminum,    Aluminum,    Aluminum,    Aluminum,    Aluminum,    Aluminum,
        Aluminum,    Aluminum,    Aluminum,    Aluminum,    Aluminum,    Aluminum,    Aluminum,    Aluminum,    Aluminum,    Aluminum,
        Aluminum,    Aluminum,    Aluminum,    Aluminum,    Aluminum,    Aluminum,    Aluminum,    Aluminum,    Aluminum,    Aluminum,
        Aluminum,    Aluminum,    Aluminum,    Aluminum,    Aluminum,    Aluminum,    Aluminum,    Aluminum,    Aluminum,    Aluminum,
        Aluminum,    Aluminum,    Aluminum,    Aluminum,    Aluminum,    Aluminum,    Aluminum,    Aluminum,    Aluminum,    Aluminum,
        Aluminum,    Aluminum,    Aluminum,    Aluminum,    Aluminum,    Aluminum,    Aluminum,    Aluminum,    Aluminum,    Aluminum,
        Aluminum,    Aluminum,    Aluminum,    Aluminum,    Aluminum,    Aluminum,    PMMA,    PMMA,    PMMA,    PMMA,
        PMMA,    PMMA,    PMMA,    PMMA,    PMMA,    PMMA,    PMMA,    PMMA,    PMMA,    PMMA,
        PMMA,    PMMA,    PMMA,    PMMA,    PMMA,    PMMA,    PMMA,    PMMA,    PMMA,    PMMA,
        PMMA,    PMMA,    PMMA,    PMMA,    PMMA,    PMMA,    PMMA,    PMMA,    PMMA,    PMMA,
        PMMA,    PMMA,    PMMA,    PMMA,    PMMA,    PMMA,    PMMA,    PMMA,    PMMA,    PMMA,
        PMMA,    PMMA,    PMMA,    PMMA,    PMMA,    PMMA,    PMMA,    PMMA,    PMMA,    PMMA,
        PMMA,    PMMA,    PMMA,    PMMA,    PMMA,    PMMA,    PMMA,    PMMA,    PMMA,    PMMA,
        PMMA,    PMMA,    PMMA,    PMMA,    PMMA,    PMMA,    PMMA,    PMMA,    PMMA,    PMMA,
        PMMA,    PMMA,    PMMA,    PMMA,    PMMA,    PMMA,    PMMA,    PMMA,    PMMA,    PMMA,
        PMMA,    PMMA,    PMMA,    PMMA,    PMMA,    PMMA,    PMMA,    PMMA,    PMMA,    PMMA,
        PMMA,    PMMA,    PMMA,    PMMA,    PMMA,    PMMA,    PMMA,    PMMA,    PMMA,    PMMA,
        PMMA,    PMMA,    PMMA,    PMMA,    PMMA,    PMMA,    PMMA,    PMMA,    PMMA,    PMMA,
        PMMA,    PMMA,    PMMA,    PMMA,    PMMA,    PMMA,    PMMA,    PMMA,    PMMA,    PMMA,
        PMMA,    PMMA,    PMMA,    PMMA,    PMMA,    PMMA,    PMMA,    PMMA,
        Alanine,    Alanine,    Alanine,    Alanine,    Alanine,    Alanine,    Alanine,    Alanine,    Alanine,    Alanine,
        Alanine,    Alanine,    Alanine,    Alanine,    Alanine,    Alanine,    Alanine,    Alanine,    Alanine,    Alanine,
        Alanine,    Alanine,    Alanine,    Alanine,    Alanine,    Alanine,    Alanine,    Alanine,    Alanine,    Alanine,
        Alanine,    Alanine,    Alanine,    Alanine,    Alanine,    Alanine,    Alanine,    Alanine,    Alanine,    Alanine,
        Alanine,    Alanine,    Alanine,    Alanine,    Alanine,    Alanine,    Alanine,    Alanine,    Alanine,    Alanine,
        Alanine,    Alanine,    Alanine,    Alanine,    Alanine,    Alanine,    Alanine,    Alanine,    Alanine,    Alanine,
        Alanine,    Alanine,    Alanine,    Alanine,    Alanine,    Alanine,    Alanine,    Alanine,    Alanine,    Alanine,
        Alanine,    Alanine,    Alanine,    Alanine,    Alanine,    Alanine,    Alanine,    Alanine,    Alanine,    Alanine,
        Alanine,    Alanine,    Alanine,    Alanine,    Alanine,    Alanine,    Alanine,    Alanine,    Alanine,    Alanine,
        Alanine,    Alanine,    Alanine,    Alanine,    Alanine,    Alanine,    Alanine,    Alanine,    Alanine,    Alanine,
        Alanine,    Alanine,    Alanine,    Alanine,    Alanine,    Alanine,    Alanine,    Alanine,    Alanine,    Alanine,
        Alanine,    Alanine,    Alanine,    Alanine,    Alanine,    Alanine,    Alanine,    Alanine,    Alanine,    Alanine,
        Alanine,    Alanine,    Alanine,    Alanine,    Alanine,    Alanine,    Alanine,    Alanine,    Alanine,    Alanine,
        Alanine,    Alanine,
        LiF,    LiF,    LiF,    LiF,    LiF,    LiF,    LiF,    LiF,    LiF,    LiF,
        LiF,    LiF,    LiF,    LiF,    LiF,    LiF,    LiF,    LiF,    LiF,    LiF,
        LiF,    LiF,    LiF,    LiF,    LiF,    LiF,    LiF,    LiF,    LiF,    LiF,
        LiF,    LiF,    LiF,    LiF,    LiF,    LiF,    LiF,    LiF,    LiF,    LiF,
        LiF,    LiF,    LiF,    LiF,    LiF,    LiF,    LiF,    LiF,    LiF,    LiF,
        LiF,    LiF,    LiF,    LiF,    LiF,    LiF,    LiF,    LiF,    LiF,    LiF,
        LiF,    LiF,    LiF,    LiF,    LiF,    LiF,    LiF,    LiF,    LiF,    LiF,
        LiF,    LiF,    LiF,    LiF,    LiF,    LiF,    LiF,    LiF,    LiF,    LiF,
        LiF,    LiF,    LiF,    LiF,    LiF,    LiF,    LiF,    LiF,    LiF,    LiF,
        LiF,    LiF,    LiF,    LiF,    LiF,    LiF,    LiF,    LiF,    LiF,    LiF,
        LiF,    LiF,    LiF,    LiF,    LiF,    LiF,    LiF,    LiF,    LiF,    LiF,
        LiF,    LiF,    LiF,    LiF,    LiF,    LiF,    LiF,    LiF,    LiF,    LiF,
        LiF,    LiF,    LiF,    LiF,    LiF,    LiF,    LiF,    LiF,    LiF,    LiF,
        LiF,    LiF,
        Air,    Air,    Air,    Air,    Air,    Air,    Air,    Air,    Air,    Air,
        Air,    Air,    Air,    Air,    Air,    Air,    Air,    Air,    Air,    Air,
        Air,    Air,    Air,    Air,    Air,    Air,    Air,    Air,    Air,    Air,
        Air,    Air,    Air,    Air,    Air,    Air,    Air,    Air,    Air,    Air,
        Air,    Air,    Air,    Air,    Air,    Air,    Air,    Air,    Air,    Air,
        Air,    Air,    Air,    Air,    Air,    Air,    Air,    Air,    Air,    Air,
        Air,    Air,    Air,    Air,    Air,    Air,    Air,    Air,    Air,    Air,
        Air,    Air,    Air,    Air,    Air,    Air,    Air,    Air,    Air,    Air,
        Air,    Air,    Air,    Air,    Air,    Air,    Air,    Air,    Air,    Air,
        Air,    Air,    Air,    Air,    Air,    Air,    Air,    Air,    Air,    Air,
        Air,    Air,    Air,    Air,    Air,    Air,    Air,    Air,    Air,    Air,
        Air,    Air,    Air,    Air,    Air,    Air,    Air,    Air,    Air,    Air,
        Air,    Air,    Air,    Air,    Air,    Air,    Air,    Air,    Air,    Air,
        Air,    Air
    }
};

/**
 * TODO find better name for this function
 * it just finds a vector of interpolated values
 * and is not related only to PSTAR tables
 *
 * @param[in]  n             number of points to interpolate
 * @param[in]  x             array of x values for which interpolation is done
 * @param[in]  subset_no     TODO
 * @param[in]  x_table       x part of data table
 * @param[in]  y_table       y part of data table
 * @param[out] y             array of interpolated y values
 */
void get_table_value( const long    n,
    const double  x[],
    const long    subset_no,
    const double  x_table[],
    const double  y_table[],
    double        y[]);


/**
 * Returns CSDA range (in g/cm2) from pstar tables for given energy.
 * In case of ions a simple scaling procedure (A/Z^2) will be used (even effective charge will be neglected)
 * @param[in]   E_MeV_u                  energy of particle
 * @param[in]   particle_no              type of the particle
 * @see          AT_DataParticle.h for definition
 * @param[in]   material_no              material index
 * @see          AT_DataMaterial.h for definition
 * @return      CSDA_range_g_cm2         CSDA range
  */
double AT_CSDA_range_g_cm2_single(   const double  E_MeV_u,
    const long    particle_no,
    const long    material_no);


/**
 * Returns CSDA range (in g/cm2) from pstar tables for given energy.
 * In case of ions a simple scaling procedure (A/Z^2) will be used (even effective charge will be neglected)
 * @param[in]   number_of_particles      number of particle types in the mixed particle field
 * @param[in]   E_MeV_u                  energy of particles in the mixed particle field (array of size number_of_particles)
 * @param[in]   particle_no              type of the particles in the mixed particle field (array of size number_of_particles)
 * @see          AT_DataParticle.h for definition
 * @param[in]   material_no              material index
 * @see          AT_DataMaterial.h for definition
 * @param[out]  CSDA_range_g_cm2         (array of size number_of_particles) to be allocated by the user which will be used to return the results
 */
void AT_CSDA_range_g_cm2(  const long  number_of_particles,
    const double   E_MeV_u[],
    const long     particle_no[],
    const long     material_no,
    double         CSDA_range_g_cm2[]);


/**
 * Returns CSDA range (in m) from pstar tables for given energy.
 * In case of ions a simple scaling procedure (A/Z^2) will be used (even effective charge will be neglected)
 * @param[in]   E_MeV_u                  energy of particle
 * @param[in]   particle_no              type of the particle
 * @see          AT_DataParticle.h for definition
 * @param[in]   material_no              material index
 * @see          AT_DataMaterial.h for definition
 * @return      CSDA_range_m             CSDA range
 */
double AT_CSDA_range_m_single(  const double  E_MeV_u,
    const long    particle_no,
    const long    material_no);

/**
 * Returns CSDA range (in m) from pstar tables for given energy.
 * In case of ions a simple scaling procedure (A/Z^2) will be used (even effective charge will be neglected)
 * @param[in]   number_of_particles      number of particle types in the mixed particle field
 * @param[in]   E_MeV_u                  energy of particles in the mixed particle field (array of size number_of_particles)
 * @param[in]   particle_no              type of the particles in the mixed particle field (array of size number_of_particles)
 * @see          AT_DataParticle.h for definition
 * @param[in]   material_no              material index
 * @see          AT_DataMaterial.h for definition
 * @param[out]  CSDA_range_m            (array of size number_of_particles) to be allocated by the user which will be used to return the results
 */
void AT_CSDA_range_m(  const long  number_of_particles,
	    const double  E_MeV_u[],
	    const long    particle_no[],
	    const long    material_no,
	    double        CSDA_range_m[]);


#endif /* AT_DATALET_H_ */
