% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/simves.R
\name{sim.ves}
\alias{sim.ves}
\title{Simulate Vector Exponential Smoothing}
\usage{
sim.ves(model = "ANN", obs = 10, nsim = 1, nvariate = 2,
  frequency = 1, persistence = NULL, phi = 1, transition = NULL,
  initial = NULL, initialSeason = NULL,
  seasonal = c("individual, common"), weights = rep(1/nvariate, nvariate),
  bounds = c("usual", "admissible", "restricted"), randomizer = c("rnorm",
  "rt", "rlaplace", "rs"), ...)
}
\arguments{
\item{model}{Type of ETS model. This can consist of 3 or 4 chars:
\code{ANN}, \code{AAN}, \code{AAdN}, \code{AAA}, \code{AAdA} etc.
Only pure additive models are supported. If you want to have multiplicative
one, then just take exponent of the generated data.}

\item{obs}{Number of observations in each generated time series.}

\item{nsim}{Number of series to generate (number of simulations to do).}

\item{nvariate}{Number of series in each generated group of series.}

\item{frequency}{Frequency of generated data. In cases of seasonal models
must be greater than 1.}

\item{persistence}{Matrix of smoothing parameters for all the components
of all the generated time series.}

\item{phi}{Value of damping parameter. If trend is not chosen in the model,
the parameter is ignored. If vector is provided, then several parameters
are used for different series.}

\item{transition}{Transition matrix. This should have the size appropriate
to the selected model and \code{nvariate}. e.g. if ETS(A,A,N) is selected
and \code{nvariate=3}, then the transition matrix should be 6 x 6. In case
of damped trend, the phi parameter should be placed in the matrix manually.
if \code{NULL}, then the default transition matrix for the selected type
of model is used. If both \code{phi} and \code{transition} are provided,
then the value of \code{phi} is ignored.}

\item{initial}{Vector of initial states of level and trend. The minimum
length is one (in case of ETS(A,N,N), the initial is used for all the
series), the maximum length is 2 x nvariate. If \code{NULL}, values are
generated for each series.}

\item{initialSeason}{Vector or matrix of initial states for seasonal
coefficients. Should have number of rows equal to \code{frequency}
parameter. If \code{NULL}, values are generated for each series.}

\item{seasonal}{The type of seasonal component across the series. Can be
\code{"individual"}, so that each series has its own component or \code{"common"},
so that the component is shared across the series.}

\item{weights}{The weights for the errors between the series with the common
seasonal component. Ignored if \code{seasonal="individual"}.}

\item{bounds}{Type of bounds to use for persistence vector if values are
generated. \code{"usual"} - bounds from p.156 by Hyndman et. al., 2008.
\code{"restricted"} - similar to \code{"usual"} but with upper bound equal
to 0.3. \code{"admissible"} - bounds from tables 10.1 and 10.2 of Hyndman
et. al., 2008. Using first letter of the type of bounds also works.}

\item{randomizer}{Type of random number generator function used for error
term. Defaults are: \code{rnorm}, \code{rt}, \code{rlaplace}, \code{rs}. But
any function from \link[stats]{Distributions} will do the trick if the
appropriate parameters are passed. \code{mvrnorm} from MASS package can also
be used.}

\item{...}{Additional parameters passed to the chosen randomizer. All the
parameters should be passed in the order they are used in chosen randomizer.
For example, passing just \code{sd=0.5} to \code{rnorm} function will lead
to the call \code{rnorm(obs, mean=0.5, sd=1)}. ATTENTION! When generating
the multiplicative errors some tuning might be needed to obtain meaningful
data. \code{sd=0.1} is usually already a high value for such models.}
}
\value{
List of the following values is returned:
\itemize{
\item \code{model} - Name of ETS model.
\item \code{data} - The matrix (or an array if \code{nsim>1}) of the
generated series.
\item \code{states} - The matrix (or array if \code{nsim>1}) of states.
States are in columns, time is in rows.
\item \code{persistence} - The matrix (or array if \code{nsim>1}) of
smoothing parameters used in the simulation.
\item \code{transition} - The transition matrix (or array if \code{nsim>1}).
\item \code{initial} - Vector (or matrix) of initial values.
\item \code{initialSeason} - Vector (or matrix) of initial seasonal
coefficients.
\item \code{residuals} - Error terms used in the simulation. Either matrix
or array, depending on \code{nsim}.
}
}
\description{
Function generates data using VES model as a data generating process.
}
\examples{

# Create 40 observations of quarterly data using AAA model with errors
# from normal distribution
\donttest{VESAAA <- sim.ves(model="AAA",frequency=4,obs=40,nvariate=3,
                   randomizer="rnorm",mean=0,sd=100)}

# You can also use mvrnorm function from MASS package as randomizer,
# but you need to provide mu and Sigma explicitly
\dontrun{VESANN <- sim.ves(model="ANN",frequency=4,obs=40,nvariate=2,
                   randomizer="mvrnorm",mu=c(100,50),sigma=matrix(c(40,20,20,30),2,2))}

# When generating the data with multiplicative model a more diligent definitiion
# of parameters is needed. Here's an example with MMM model:

VESMMM <- sim.ves("AAA", obs=120, nvariate=2, frequency=12, initial=c(10,0),
          initialSeason=runif(12,-1,1), persistence=c(0.06,0.05,0.2), mean=0, sd=0.03)
VESMMM$data <- exp(VESMMM$data)

# Note that smoothing parameters should be low and the standard diviation should
# definitely be less than 0.1. Otherwise you might face the explosions.

}
\references{
\itemize{
\item de Silva A., Hyndman R.J. and Snyder, R.D. (2010). The vector
innovations structural time series framework: a simple approach to
multivariate forecasting. Statistical Modelling, 10 (4), pp.353-374
\item Hyndman, R.J., Koehler, A.B., Ord, J.K., and Snyder, R.D. (2008)
Forecasting with exponential smoothing: the state space approach,
Springer-Verlag.
\item Lütkepohl, H. (2005). New Introduction to Multiple Time Series
Analysis. New introduction to Multiple Time Series Analysis. Berlin,
Heidelberg: Springer Berlin Heidelberg.
\doi{10.1007/978-3-540-27752-1}
\item Chen H., Svetunkov I., Boylan J. (2021). A New Taxonomy for
Vector Exponential Smoothing and Its Application to Seasonal Time
Series.
}
}
\seealso{
\code{\link[legion]{ves}, \link[stats]{Distributions}}
}
\author{
Ivan Svetunkov, \email{ivan@svetunkov.ru}
}
\keyword{models}
\keyword{multivariate}
\keyword{nonlinear}
\keyword{regression}
\keyword{ts}
