### iid2.R --- 
#----------------------------------------------------------------------
## author: Brice Ozenne
## created: okt 12 2017 (13:16) 
## Version: 
## last-updated: jan 12 2018 (11:46) 
##           By: Brice Ozenne
##     Update #: 366
#----------------------------------------------------------------------
## 
### Commentary: 
## 
### Change Log:
#----------------------------------------------------------------------
## 
### Code:

## * Documentation - iid2
#' @title  Extract i.i.d. decomposition from linear and latent variable models
#' @description  Extract i.i.d. decomposition from linear and latent variable models using leverage adjusted fitted residuals.
#' @name iid2
#'
#' @param x a linear model or a latent variable model
#' @param p [optional] the fitted parameters.
#' @param data [optional] the data set.
#' @param adjust.residuals Should the leverage-adjusted residuals be used to compute the score? Otherwise the raw residuals will be used. 
#' @param ... arguments to be passed to \code{score2}.
#'
#' @details
#' Leverage-adjusted residuals have been shown to improve the coverage of robust standard errors in small samples.
#' They are computed according to the formula:
#' \eqn{e_adj = \frac{e}{(1-h_{ii})^(1/2)}}
#' 
#' @references
#' Bell, R. M., & McCaffrey, D. F. Bias reduction in standard errors for linear regression with multi-stage samples. Survey Methodology, 28(2), 169-181 (2002). \cr
#' Kauermann G. and Carroll R. J. A note on the efficiency of sandwich covariance matrix estimation. Journal of the American Statistical Association. Vol. 96, No. 456 (2001).
#'
#' @return A matrix.
#' 
#' @examples
#' n <- 5e1
#' p <- 3
#' X.name <- paste0("X",1:p)
#' link.lvm <- paste0("Y~",X.name)
#' formula.lvm <- as.formula(paste0("Y~",paste0(X.name,collapse="+")))
#'
#' m <- lvm(formula.lvm)
#' distribution(m,~Id) <- sequence.lvm(0)
#' set.seed(10)
#' d <- sim(m,n)
#'
#' ## linear model
#' e.lm <- lm(formula.lvm,data=d)
#' iid.tempo <- iid2(e.lm, adjust.residuals = FALSE)
#' range(iid.tempo[,1:4]-iid(e.lm))
#'
#' ## latent variable model
#' e.lvm <- estimate(lvm(formula.lvm),data=d)
#' iid.tempo <- iid2(e.lvm, adjust.residuals = FALSE)
#' range(iid.tempo-iid(e.lvm))
#' ## difference due to the use of the observed info matrix vs. the expected one.
#'
#' @export
`iid2` <-
  function(x, ...) UseMethod("iid2")

## * iid2.lm
#' @rdname iid2
#' @export
iid2.lm <- function(x, p = NULL, data = NULL, 
                    adjust.residuals = TRUE, 
                    ...){
    
    ### ** compute the score
    e.score <- score2(x, p = p, data = data,
                      adjust.residuals = adjust.residuals,
                      indiv = TRUE, return.vcov.param = TRUE, ...)
    vcov.param <- attr(e.score,"vcov.param")

    ### ** compute the iid
    iid0 <- e.score %*% vcov.param

    ### ** export
    colnames(iid0) <- colnames(e.score)
    
    return(iid0)
}

## * iid2.gls
#' @rdname iid2
#' @export
iid2.gls <- iid2.lm

## * iid2.lme
#' @rdname iid2
#' @export
iid2.lme <- iid2.lm

## * iid2.lvmfit
#' @rdname iid2
#' @export
iid2.lvmfit <- iid2.lm

##----------------------------------------------------------------------
### iid2.R ends here
