% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/semivariogram.R
\name{semivariogram}
\alias{semivariogram}
\alias{semivariogram.krige}
\alias{plot.krige}
\alias{semivariogram.lm}
\alias{semivariogram.default}
\alias{semivariogram.semivariance}
\alias{plot.semivariance}
\title{Semivariogram for Geospatial Data}
\usage{
semivariogram(x, ...)

\method{semivariogram}{krige}(x, ..., bins = 13, terms = "all", type, pch, lty, legend, col)

\method{plot}{krige}(...)

\method{semivariogram}{lm}(
  x,
  ...,
  coords,
  bins = 13,
  terms = c("raw", "residual"),
  east,
  north,
  type,
  legend,
  col,
  pch,
  lty
)

\method{semivariogram}{default}(
  x,
  ...,
  coords,
  data,
  bins = 13,
  east,
  north,
  type,
  pch,
  lty,
  col
)

\method{semivariogram}{semivariance}(x, ..., type, pch, lty, legend, col)

\method{plot}{semivariance}(x, ..., type, pch, lty, legend, col)
}
\arguments{
\item{x}{An object for which a semivariogram is desired. The object can 
be a \code{krige} object, a \code{semivariance} object, a \code{lm} object, 
or a vector of variables (or variable names in the \code{data}).}

\item{\dots}{Additional arguments to be passed to \code{semivariogram} methods. 
Further arguments that can passed to \code{plot()} function can be specified
here.}

\item{bins}{Number of bins into which distances should be divided. The observed 
distances will be split at equal intervals, and the semivariance will be computed 
within each interval. Defaults to 13 intervals.}

\item{terms}{A vector of strings specifies for which the semivariogram is created. 
Options are "raw" (the semivariogram for raw data), "residual" (the semivariogram
for residuals from linear regression). "parametric" (the parametric powered 
exponential semivariogram based on the estimation from \code{metropolis.krige}). 
Defaults will create all the applicable plots.}

\item{type}{A vector specify the type of plots for each term. Options are "l" 
(line plot) and "p" (scatter plot). Defaults to \code{c(raw = "p", residual = "p", 
parametric = "l")}}

\item{pch}{A vector specify the points symbols for scatter plot. Suppressed for 
line plot.}

\item{lty}{A vector specify the line type for line plot. Suppressed for 
scatter plot.}

\item{legend}{A logical argument for whether legend should be presented. Defaults to \code{TRUE}.}

\item{col}{A vector specify the color for each term. Defaults to \code{c(raw = "black", 
residual = "blue", parametric = "red")}}

\item{coords}{A matrix of coordinates for all observations or a vector of variable 
names indicating the coordinates variables in the data. Alternatively, the 
coordinates can also be specified separately using \code{east} and \code{north}.}

\item{east}{Alternative specification for the vector of eastings for all observations.}

\item{north}{Alternative specification for the vector of northing for all observations.}

\item{data}{If object is a variable name, a data frame must be provided.}
}
\value{
An semivariogram plot. For \code{krige} object, it will return empirical 
  semivariograms for raw data and residuals of linear regression as well as a 
  parametric powered exponential semivariogram given the values of the estimates 
  from \code{metropolis.krige} as default.
}
\description{
This function creates semivariogram plots. It creates empirical semivariogram 
for raw data and \code{lm} object or parametric exponential semivariogram based 
on the estimation from \code{metropolis.krige}. Based on the user's chosen level 
of coarsening, the semivariogram is presented for various distances.
}
\details{
The function creates semivariograms for diagnosing the spatial relationship
  that best describes the data and for diagnosing the degree to which the model 
  fits the spatial relationship. With a view of the empirical semivariogram, 
  a user can consult images of parametric semivariograms to determine whether 
  an exponential, Gaussian, or other powered expoential function fit the data 
  well, or if another style of semivariogram works better. Examining this also 
  allows the user to develop priors such as the approximate split in variance 
  between the nugget and partial sill as well as the approximate distance of 
  the effective range. Semivariograms are explicitly tied to a corresponding 
  spatial correlation function, so determining the former automatically implies 
  the latter. See Banerjee, Carlin, and Gelfand for a fuller explanation, as 
  well as a guidebook to semivariogram diagnosis (2015, 26-30).
  
  The function can be applied to a variable, a fitted linear model (\code{lm} 
  object) before fitting a spatial model or to a \code{krige} object or \code{semivariance}
  object to assess the model fit. When applying to a variable, it will describes 
  the raw data; for a \code{lm} object, the default will present empirical 
  semivariogram for both the raw data and linear residuals; when applying to a 
  \code{krige} object, the default will present empirical semivariogram for the 
  raw data and the residuals from linear fit, and the parametric semivariogram 
  given the estimates from the geospatial model fitted in \code{metropolis.krige}; 
  for a \code{semivariance} object, it will present a plot(s) for whichever the
  semivariance is calculated. Users can also specify which semivariogram is needed 
  in the \code{terms} argument if there are multiple kinds of semivariogram can 
  be plotted. The plots are compatible to the arguments used in base \code{R} 
  base graphics.
}
\examples{
\dontrun{
# Summarize Data
summary(ContrivedData)

# Empirical semivariagram for variable y
semivariogram(x=ContrivedData$y, coords = cbind(ContrivedData$s.1, ContrivedData$s.2))

# Initial OLS Model
contrived.ols<-lm(y~x.1+x.2,data=ContrivedData)

# Empirical semivariagram for ols fit
semivariogram(contrived.ols, coords = c("s.1","s.2"), bins=13)

# Set seed
set.seed(1241060320)

M <- 100
#M<-10000

contrived.run <- metropolis.krige(y ~ x.1 + x.2, coords = c("s.1","s.2"), 
   data = ContrivedData, n.iter = M, range.tol = 0.05)
   
# Parametric semivariagram
semivariogram(contrived.run, bins=13, terms = c("raw", "residual", "parametric"),
  type= c(raw = "p", residual = "p", parametric = "l"), legend = TRUE, col = c("black", 
  "blue", "red"), pch = c(1,3,NA), lty = c(NA,NA,1))
  
# Alternatively, the generic function plot can also be applied to krige object
plot(contrived.run)

# Plot semivariance object
plot(semivariance(contrived.run, bins=13))
}

}
\references{
Sudipto Banerjee, Bradley P. Carlin, and Alan E. Gelfand. 2015. \emph{Hierarchical 
  Modeling and Analysis for Spatial Data}. 2nd ed. Boca Raton, FL: CRC Press.
}
\seealso{
\code{\link{semivariance}}, \code{\link{exponential.semivariance}}
}
